/*
 * Java Template Project
 * 
 * Copyright (C) 2006 Satoshi Nagashiba, All Rights Reserved.
 */
package org.jtp.web.download;

import java.io.IOException;
import java.io.InputStream;

import javax.faces.FacesException;
import javax.faces.context.ExternalContext;
import javax.faces.context.FacesContext;
import javax.faces.event.AbortProcessingException;
import javax.faces.event.ActionEvent;
import javax.servlet.ServletContext;
import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import javax.servlet.http.HttpSessionBindingEvent;
import javax.servlet.http.HttpSessionBindingListener;

import org.jtp.web.csv.Factory;


/**
 * ダウンロードイベントを処理するバッキングビーンクラスです。 
 * 
 * @author <a href="mailto:sato-vista@jade.plala.or.jp">Satoshi Nagashiba</a>
 */
public class DownloadActionBean implements HttpSessionBindingListener {

    /** レスポンスヘッダ名称 */
    private static final String HEADER_DISPOSITION = "Content-Disposition";
    
    /** ダウンロード用のバッキングビーン */
    private DownloadBean bean;
    
    /**
     * コンストラクタ
     * @param bean ダウンロード用のバッキングビーン
     */
    public DownloadActionBean(DownloadBean bean) {
    	this.bean = bean;
    }
    
    /**
     * セッションに結びつけられようとしていることをオブジェクトに通知し、 セッションを識別します
     * @param event セッションを識別するためのイベントオブジェクト
     */
    public void valueBound(HttpSessionBindingEvent event) {}

    /**
     * セッションから取り外されようとしていることをオブジェクトに通知し、 セッションを識別します
     * @param event セッションを識別するためのイベントオブジェクト
     */
    public void valueUnbound(HttpSessionBindingEvent event) {}
    
    /**
     * ダウンロードする
     * @param event イベント
     * @throws FacesException JSF例外
     */
    public final void download(ActionEvent event) throws FacesException {
        try {
        	String filename = this.bean.getFilename();

            //レスポンス設定
            HttpServletResponse response = getResponse();
            response.setContentType(getContentType());
            response.setHeader(HEADER_DISPOSITION, getContentDisposition(filename));

            //ストリーム作成
            Factory csvfactory = this.bean.getFactory();
            InputStream is = csvfactory.getStream(getServletContext().getRealPath(filename));
            ServletOutputStream out = response.getOutputStream();

            //new ByteTransferor().transfer(is, out);
            
        	byte _buffer[] = new byte[1024];

        	int partial;
            for(long result = 0L; (partial = is.read(_buffer, 0, _buffer.length)) > 0; result += partial) {
                out.write(_buffer, 0, partial);
            }

            FacesContext.getCurrentInstance().responseComplete();

        }
        catch (IOException e) {
//            FacesUtils.addExceptionMessages(e);
            throw new AbortProcessingException(e);
        }
    }
    
    /**
     * HTTPリクエストを取得する
     * @return HTTPリクエスト
     */
    protected HttpServletRequest getRequest() {
        return (HttpServletRequest) getExternalContext().getRequest();
    }

    /**
     * HTTPセッションを取得する
     * @return  HTTPセッション
     */
    protected HttpSession getSession() {
        return getRequest().getSession();
    }

    /**
     * HTTPレスポンスを取得する
     * @return HTTPレスポンス
     */
    protected HttpServletResponse getResponse() {
        return (HttpServletResponse) getExternalContext().getResponse();
    }

    /**
     * サーブレットコンテキストを取得する
     * @return サーブレットコンテキスト
     */
    protected ServletContext getServletContext() {
        return (ServletContext) getExternalContext().getContext();
    }
    
    /**
     * クライアントに送り返されるレスポンスのコンテントタイプを返します
     * @return クライアントに送り返されるレスポンスのコンテントタイプ
     */
    protected String getContentType() {
    	return "application/csv";
    }
    
    /**
     * レスポンスヘッダへ追加する値を返します
     * @param filename ファイル名
     * @return レスポンスヘッダへ追加する値
     */
    protected String getContentDisposition(String filename) {
    	return "attachment; filename=" + filename;
    }
    
    /**
     * コンテキストを取得する
     * @return コンテキスト
     */
    private ExternalContext getExternalContext() {
    	return FacesContext.getCurrentInstance().getExternalContext();
    }
}
