unit tn_vd;

interface

uses
	tn_utils, tn_classes;

type
	TNVirtualDir	=	class
    	private
        	FValidPath : Boolean;
        	FilePath : String;
            FDataExt : String;
            Ids : TNIntList;
            Places : TNIntList;
            Names : TNStringList;
			procedure Load;
			procedure SaveNameTable;
			procedure SavePosTable;
    	public
        	constructor Create(Path : String); virtual;
            destructor Destroy; override;
			function LoadFromFile(Id : Integer) : String; virtual;
			function SaveToFile(const Folder : String ; Id : Integer ; const Data : String) : Boolean; virtual;
			function CreateNewId : Integer; virtual;
			function CreateFolder(const Folder : String) : Boolean;
			function GetFolderList(const Folder : String) : TNStringList;
			function IsFolderExists(const Folder : String) : Boolean;
			procedure SaveSettings(const Section, Data : String);
			function LoadSettings(const Section : String) : String;
        published
        	property DataExt : String read FDataExt write FDataExt;
        	property ValidPath : Boolean read FValidPath;
    end;

implementation

constructor TNVirtualDir.Create(Path : String);
begin
	Randomize;
	FilePath := SetPathEndsBackSlash(Path);
    FValidPath := False;
    Ids := TNIntList.Create;
    Places := TNIntList.Create;
    Names := TNStringList.Create;
    FDataExt := '.dat';

	if not DirectoryExists(FilePath) then
    begin
    	//쐬
        MkDir(FilePath);
		SavePosTable;
        SaveNameTable;
    end;

    //݃`FbN
    if (not FileExists(FilePath + '_entry.vd'))
    	or (not FileExists(FilePath + '_position.vd'))
        or (not FileExists(FilePath + '_alias.vd')) then Exit;

    //ǂݍ
    Load;

    FValidPath := True;
end;

destructor TNVirtualDir.Destroy;
begin
	if FValidPath then
    begin
    	//쐬
		SavePosTable;
        SaveNameTable;
    end;

	Ids.Free;
	Places.Free;
	Names.Free;

	inherited;
end;

procedure TNVirtualDir.Load;
begin
	Names.LoadFromFile(FilePath + '_alias.vd');
    Ids.LoadFromFile(FilePath + '_entry.vd');
    Places.LoadFromFile(FilePath + '_position.vd');
end;

procedure TNVirtualDir.SaveNameTable;
begin
	Names.SaveToFile(FilePath + '_alias.vd');
end;

procedure TNVirtualDir.SavePosTable;
begin
    Ids.SaveToFile(FilePath + '_entry.vd');
    Places.SaveToFile(FilePath + '_position.vd');
end;

function TNVirtualDir.LoadFromFile(Id : Integer) : String;
begin
	Result := tn_utils.LoadFromFile(FilePath + IntToHex(Id) + FDataExt);
end;

function TNVirtualDir.SaveToFile(const Folder : String ; Id : Integer
									; const Data : String) : Boolean;
begin
	Result := False;
	if Ids.IndexOf(Id) <> -1 then Exit;
    if Names.IndexOf(Folder) = -1 then Exit;

	Ids.Add(Id);
    Places.Add(Names.IndexOf(Folder));
	tn_utils.SaveToFile(FilePath + IntToHex(Id) + FDataExt, Data);
end;

function TNVirtualDir.CreateNewId : Integer;
begin
    repeat
		Result := Random($80000000);
    until Ids.IndexOf(Result) = -1;
end;

function TNVirtualDir.CreateFolder(const Folder : String) : Boolean;
var
	path : String;
    temp : String;
    list : TNStringList;
    n : Integer;
begin
	Result := False;
    path := SetPathEndsBackSlash(Folder);
    if path = '' then Exit;
    if path[1] <> '\' then path := '\' + path;

    temp := '\';
    list := SplitString(Folder, '\');

    for n := 0 to list.Count - 2 do
    begin
    	if list.Strings[n] <> '' then
        	temp := temp + list.Strings[n] + '\';

        if not IsFolderExists(temp) then
        begin
        	list.Free;
        	Exit;
        end;
    end;

	list.Free;

    if Names.IndexOf(path) <> -1 then Exit;
	Names.Add(path);
    SaveNameTable;
end;

function TNVirtualDir.GetFolderList(const Folder : String) : TNStringList;
var
	path : String;
    temp : String;
    n : Integer;
    level : Integer;

    function CountBackSlash(Path : WideString) : Integer;
	var
		n : Integer;
    begin
    	Result := 0;

		for n := 1 to Length(Path) do
        begin
        	if Path[n] = '\' then Inc(Result);
        end;
    end;

begin
	Result := TNStringList.Create;

	path := SetPathEndsBackSlash(Folder);
    if path = '' then Exit;
    if path[1] <> '\' then path := '\' + path;
	level := CountBackSlash(path) + 1;

    for n := 0 to Names.Count - 1 do
    begin
    	temp := Names.Strings[n];
    	if (Pos(path, temp) = 1) and (CountBackSlash(temp) = level) then
        	Result.Add(temp);
    end;
end;

function TNVirtualDir.IsFolderExists(const Folder : String) : Boolean;
begin
	Result := Names.IndexOf(Folder) <> -1;
end;

procedure TNVirtualDir.SaveSettings(const Section, Data : String);
var
	filename : String;
begin
	filename := FilePath + '_settings.vd';
    tn_utils.SaveSettings(filename, Section, Data);
end;

function TNVirtualDir.LoadSettings(const Section : String) : String;
var
	filename : String;
begin
	filename := FilePath + '_settings.vd';
    Result := tn_utils.GetSettings(filename, Section);
end;

end.
