unit tn_internet;

interface

uses
    windows, tn_classes, tn_utils, tn_crc;

type
    THeader     =   class(TNVirtualGrid)
        public
            constructor Create;
            destructor Destroy; override;
            procedure SetHeader(Header : String);
            function GetHeader(Name : String) : String;
    end;

    TFormData	=	class(TNVirtualGrid)
		public
            constructor Create;
            destructor Destroy; override;
			procedure SetData(const ContentType, Data : String);
			function GetData(const Name : String) : String;
			function GetDatas(const Name : String) : TNStringList;
			function GetFileName(const Name : String) : String;
    end;

    TAuthData	=	class(TNVirtualGrid)
		public
            constructor Create;
            destructor Destroy; override;
			procedure SetAuth(const Path, Data : String);
			function GetAuth(const Path : String) : String;
    end;

//procedure
    procedure DecodeMIME(const Source : String ; var Header : THeader);
    function Base64Encode(const input : String) : String;
    function Base64Decode(const input : String) : String;
    function BinHexEncode( const input, Rcode: String ): string;
    function BinHexDecode( const input : string ): String;
    function QPEncode(const Source : String) : String;
    function QPDecode(const Source : String): String;
    function UrlDecode(const Source : String): String;
    function UrlEncode(const Source : String): String;
    function GenerateBoundary : String;
    function GenerateTempFile(const Path : String) : String;
    function IsIncludeNonAscii(const Source : String) : Boolean;
    function AutoReturn(const Source : String ; Width : Integer) : String;
    function HTMLtoText(const Source : String) : String;
    function TexttoHTML(const Source : String) : String;
    function DeleteTag(const Source : String) : String;
	function AnalyzeFormData(const ContentType, Data : String) : TFormData;
	function JisToSjis(const Data : String) : String;
	function SJisToJis(const Data : String) : String;

const
    Code64 : PChar = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/';
    CodeBinHex : PChar = '!"#$%&''()*+,-012345689@ABCDEFGHIJKLMNPQRSTUVXYZ[`abcdefhijklmpqr';
    DecBinHex : array[0..81] of BYTE = (
        $00,$01,$02,$03,$04,$05,$06,$07,$08,$09,$0A,$0B,$0C,$FF,$FF,$0D,
        $0E,$0F,$10,$11,$12,$13,$FF,$14,$15,$FF,$FF,$FF,$FF,$FF,$FF,$16,
        $17,$18,$19,$1A,$1B,$1C,$1D,$1E,$1F,$20,$21,$22,$23,$24,$FF,$25,
        $26,$27,$28,$29,$2A,$2B,$FF,$2C,$2D,$2E,$2F,$FF,$FF,$FF,$FF,$30,
        $31,$32,$33,$34,$35,$36,$FF,$37,$38,$39,$3A,$3B,$3C,$FF,$FF,$3D,
        $3E,$3F );
    CodeQuoted: string[16]=('0123456789ABCDEF');

    ISO2022JP	=	'ISO-2022-JP';
    US_ASCII	=	'US-ASCII';

implementation

//******************************************************************************
//THeader

constructor THeader.Create;
begin
    inherited Create(2);
end;

destructor THeader.Destroy;
begin
    inherited Destroy;
end;

procedure THeader.SetHeader(Header : String);
begin
    Clear;
    DecodeMIME(Header, Self);
end;

function THeader.GetHeader(Name : String) : String;
var
    n : Integer;
begin
    Result := '';

    for n := 0 to Datas[0].Count - 1 do
    begin
        if UpperCase(Datas[0].Strings[n]) = UpperCase(Trim(Name)) then
        begin
            if Result <> '' then
                Result := Result + #13#10;

            Result := Result + Datas[1].Strings[n];
        end;
    end;
end;

//******************************************************************************
// TFormData

constructor TFormData.Create;
begin
    inherited Create(3);
end;

destructor TFormData.Destroy;
begin
	inherited;
end;

procedure TFormData.SetData(const ContentType, Data : String);
var
	multipart : Boolean;
    boundary : String;
    list : TNStringList;
    temp, stemp, ttemp : String;
    m, n : Integer;
    ps : Integer;
    p : PChar;
    ch : Char;
    name : String;
    dat : String;
    filename : String;
    ln : Integer;
begin
	//f[^NA
    Clear;
    
	multipart := Pos('multipart/', ContentType) <> 0;
    ps := Pos('boundary=', ContentType);

    if ps <> 0 then
    begin
    	boundary := Trim(Replace(Copy(ContentType, ps + 9
        							, Length(ContentType) - ps - 8), '"', ''));
    end;

    if not multipart then
    begin
		list := SplitString(Data, '&');

        for n := 0 to list.Count - 1 do
        begin
			temp := list.Strings[n];
            ps := Pos('=', temp);

            if ps <> 0 then
            begin
            	Add([UrlDecode(Copy(temp, 1, ps - 1))
                		, UrlDecode(Copy(temp, ps + 1, Length(temp) - ps))
                        , '']);
            end;
        end;

    end
    else
    begin
    	list := SplitString(Data, '--' + boundary);

        for n := 0 to list.Count - 2 do
        begin
			temp := list.Strings[n];
            p := PChar(temp) + 2;
            m := 2;
            name := '';
            dat := '';
            filename := '';
            ln := Length(temp);

            //wb_݂̂擾
            while m < ln do
            begin
            	stemp := '';

            	while m < ln do
                begin
					ch := p^;
                	stemp := stemp + ch;
	            	Inc(m);
    	            Inc(p);
                    if ch = #10 then Break;
                end;

                stemp := Trim(stemp);
                if stemp = '' then Break;

                if UpperCase(Copy(stemp, 1, 20)) = 'CONTENT-DISPOSITION:' then
                begin
                	//O擾
                	ps := Pos('NAME=', UpperCase(stemp));
                    if ps = 0 then Continue;
					ttemp := Copy(stemp, ps + 5, Length(stemp) - ps - 4);
                    ps := Pos(';', ttemp);
                    if ps <> 0 then ttemp := Trim(Copy(ttemp, 1, ps - 1));
                  	name := Replace(ttemp, '"', '');

                    //t@C擾
                	ps := Pos('FILENAME=', UpperCase(stemp));
                    if ps <> 0 then
                    begin
						ttemp := Copy(stemp, ps + 9, Length(stemp) - ps - 8);
    	                ps := Pos(';', ttemp);
        	            if ps <> 0 then ttemp := Trim(Copy(ttemp, 1, ps - 1));
            	      	filename := Replace(ttemp, '"', '');
                    end;
                end;
            end;

            //f[^擾
            dat := Copy(temp, m + 1, ln - m - 2);

            //ǉ
            Add([name, dat, filename]);
        end;
    end;

    list.Free;
end;

function TFormData.GetData(const Name : String) : String;
var
    n : Integer;
begin
    Result := '';

    for n := 0 to Datas[0].Count - 1 do
    begin
        if UpperCase(Datas[0].Strings[n]) = UpperCase(Trim(Name)) then
        begin
            Result := Datas[1].Strings[n];
        	Exit;
        end;
    end;
end;

function TFormData.GetDatas(const Name : String) : TNStringList;
var
    n : Integer;
begin
    Result := TNStringList.Create;

    for n := 0 to Datas[0].Count - 1 do
    begin
        if UpperCase(Datas[0].Strings[n]) = UpperCase(Trim(Name)) then
        begin
            Result.Add(Datas[1].Strings[n]);
        end;
    end;
end;

function TFormData.GetFileName(const Name : String) : String;
var
    n : Integer;
begin
    Result := '';

    for n := 0 to Datas[0].Count - 1 do
    begin
        if UpperCase(Datas[0].Strings[n]) = UpperCase(Trim(Name)) then
        begin
            Result := Datas[2].Strings[n];
        	Exit;
        end;
    end;
end;

//******************************************************************************
// TAuthData

constructor TAuthData.Create;
begin
	inherited Create(2);
end;

destructor TAuthData.Destroy;
begin
	inherited;
end;

procedure TAuthData.SetAuth(const Path, Data : String);
var
    n : Integer;
begin
    for n := 0 to Datas[0].Count - 1 do
    begin
		if Datas[0].Strings[n] = Path then
        begin
        	Datas[1].Strings[n] := Data;
        	Exit;
        end;
    end;

	Add([Path, Data]);
end;

function TAuthData.GetAuth(const Path : String) : String;
var
    n : Integer;
begin
    Result := '';

    for n := 0 to Datas[0].Count - 1 do
    begin
        if Pos(UpperCase(Trim(Path)), UpperCase(Datas[0].Strings[n])) <> 0 then
        begin
            Result := Datas[1].Strings[n];
        	Exit;
        end;
    end;
end;

//******************************************************************************
//Procedure

procedure DecodeMIME(const Source : String ; var Header : THeader);
var
    n : Integer;
    ps : Integer;
    temp : String;
    Name : String;
    Data : String;
    List : TNStringList;
begin
    List := TNStringList.Create;
    List.Text := Source;

    for n := 0 to List.Count - 1 do
    begin
        temp := List.Strings[n];
        ps := Pos(':', temp);

        if ps <> 0 then
        begin
            Name := Trim(Copy(temp, 1, ps - 1));
            Data := Trim(Copy(temp, ps + 1, Length(temp) - ps));
            Header.Add([Name, Data]);
        end
        else
        begin
            if Header.Datas[1].Count = 0 then Continue;
            Header.Datas[1].Strings[Header.Datas[1].Count - 1]
                := Header.Datas[1].Strings[Header.Datas[1].Count - 1] + temp;
        end;
    end;

    List.Free;
end;

function Base64Encode(const input: String) : String;
var
    i,j,iLen: Integer;
    a,b,c: BYTE;
begin
    Result := '';

    //GR[h̑傫vZ
    iLen := Length(input);
    i := iLen mod 3;

    if i <> 0 then
        i := 4;

    SetLength( Result, ( iLen div 3 ) * 4 + i);

    i:=1; j:=1;

    while i <= iLen -2 do
    begin
        a := BYTE(input[i]);
        b:= BYTE(input[i+1]);
        c := BYTE(input[i+2]);

        Result[j] := Code64[ ((a and $FC) shr 2) ];
        Inc(j);
        Result[j] := Code64[ ( ((a and $03) shl 4) or ((b and $F0) shr 4) ) ];
        Inc(j);
        Result[j] := Code64[ ( ((b and $0F) shl 2) or ((c and $C0) shr 6) ) ];
        Inc(j);
        Result[j] := Code64[ (c and $3F) ];
        Inc(j);
        i := i + 3;
    end;

    if (iLen mod 3) = 1 then
    begin
        a := BYTE(input[iLen]);
        b:=0;
        Result[j] := Code64[ ((a and $FC) shr 2) ];
        Inc(j);
        Result[j] := Code64[ ( ((a and $03) shl 4) or ((b and $F0) shr 4) ) ];
        Inc(j);
        Result[j] := '=';
        Inc(j);
        Result[j] := '=';
    end
    else if (iLen mod 3) = 2 then
    begin
        a := BYTE(input[iLen -1]);
        b := BYTE(input[iLen]);
        c := 0;
        Result[j] := Code64[ ((a and $FC) shr 2) ];
        Inc(j);
        Result[j] := Code64[ ( ((a and $03) shl 4) or ((b and $F0) shr 4) ) ];
        Inc(j);
        Result[j] := Code64[ ( ((b and $0F) shl 2) or ((c and $C0) shr 6) ) ];
        Inc(j);
        Result[j] := '=';
    end;
end;

function Base64Decode(const input : String) : String;
var
    i,j,k,iLen: Integer;
    dbuf: array[0..3] of BYTE;
begin

    iLen := Length( input );
    Result := '';

    //fR[h̑傫vZ
    j := 0;

    for i:=1 to iLen do
    begin
        if (input[i] = #$0D) or (input[i] = #$0A) or (input[i] = '=') then
            Inc(j);
    end;

    iLen := iLen -j;
    i :=  iLen mod 4;

    if i <> 0 then
        Dec(i);

    iLen := (iLen div 4) * 3 +i;
    SetLength( Result, iLen); //̂

    iLen := Length( input );
    i := 1;
    k := 1;

    while i <= iLen do
    begin
        if (input[i] = #$0D) or (input[i] = #$0A) then
        begin
            Inc(i);
            Continue;
        end;

        for j:=0 to 3 do
        begin
            case (input[i]) of
                'A'..'Z':
                    dbuf[j] := BYTE(input[i]) - $41;

                'a'..'z':
                    dbuf[j] := BYTE(input[i]) - $47;

                '0'..'9':
                    dbuf[j] := BYTE(input[i]) + 4;

                '+'     :
                    dbuf[j] := 62;

                '/'     :
                    dbuf[j] := 63;

                '='     :
                    dbuf[j] := $FF;
            end;

            Inc(i);
        end;

        if dbuf[2] = $FF then
        begin
            Result[k] := Char( (dbuf[0] shl 2) or (dbuf[1] shr 4) );
        end
        else if dbuf[3] = $FF then
        begin
            Result[k] := Char( (dbuf[0] shl 2) or (dbuf[1] shr 4) );
            Inc(k);
            Result[k] := Char( (dbuf[1] shl 4) or (dbuf[2] shr 2) );
        end
        else
        begin
            Result[k] := Char( (dbuf[0] shl 2) or (dbuf[1] shr 4) );
            Inc(k);
            Result[k] := Char( (dbuf[1] shl 4) or (dbuf[2] shr 2) );
            Inc(k);
            Result[k] := Char( (dbuf[2] shl 6) or dbuf[3] );
        end;

        Inc(k);
    end;
end;

function BinHexEncode( const input, Rcode: string ): string;
var
    i,j,k,l,iLen: Integer;
    a,b,c: BYTE;
begin
    Result := '';

    //GR[h̑傫vZ
    iLen := (Length(input) div 3) * 4;
    i := iLen mod 3;

    if i <> 0 then
        Inc(i);

    iLen := iLen + i +2;  // +2 nI[L
    iLen := iLen + (iLen div 64) * Length(Rcode); // nI[LƉsR[h
    SetLength( Result, iLen );

    iLen := Length(input);
    i:=1;
    j:=2;
    k:=1;
    Result[1] := ':';

    while i <= iLen -2 do
    begin
        a := BYTE(input[i]);
        b:= BYTE(input[i+1]);
        c := BYTE(input[i+2]);

        Result[j] := CodeBinHex[ ((a and $FC) shr 2) ];
        Inc(j);
        Result[j] := CodeBinHex[ ( ((a and $03) shl 4) or ((b and $F0) shr 4) ) ];
        Inc(j);
        Result[j] := CodeBinHex[ ( ((b and $0F) shl 2) or ((c and $C0) shr 6) ) ];
        Inc(j);
        k := k + 3;

        if k = 64 then
        begin
            for l:=1 to Length(Rcode) do
            begin
                Result[j] := Rcode[l]; Inc(j);
            end;

            k := 0;
        end;

        Result[j] := CodeBinHex[ (c and $3F) ];
        Inc(j);
        Inc(k);
        i := i + 3;
    end;

    if (iLen mod 3) <> 0 then
    begin
        if (iLen mod 3) = 1 then
        begin
            a := BYTE(input[iLen]);
            b:=0;
            Result[j] := CodeBinHex[ ((a and $FC) shr 2) ];
            Inc(j);
            Result[j] := CodeBinHex[ ( ((a and $03) shl 4)
                                        or ((b and $F0) shr 4) ) ];
            Inc(j);
        end
        else if (iLen mod 3) = 2 then
        begin
            a := BYTE(input[iLen -1]);
            b := BYTE(input[iLen]);
            c := 0;
            Result[j] := CodeBinHex[ ((a and $FC) shr 2) ];
            Inc(j);
            Result[j] := CodeBinHex[ ( ((a and $03) shl 4)
                                        or ((b and $F0) shr 4) ) ];
            Inc(j);
            Result[j] := CodeBinHex[ ( ((b and $0F) shl 2)
                                        or ((c and $C0) shr 6) ) ];
            Inc(j);

            if k = 64 then
            begin
                for l:=1 to Length(Rcode) do
                begin
                    Result[j] := Rcode[l];
                    Inc(j);
                end;
            end;
        end;
    end;

    Result[j] := ':';
end;

function BinHexDecode( const input: string ): string;
var
    i,j,k,iLen: Integer;
    dbuf: array[0..3] of BYTE;
begin
    iLen := Length( input );
    Result := '';

    //fR[h̑傫vZ
    j := 0;

    for i:=1 to iLen do
    begin
        if (input[i] = #$0D) or (input[i] = #$0A) then
        begin
            Inc(j);
        end;
    end;

    iLen := iLen -j -2;
    i := iLen mod 4;

    if i <> 0 then
        Dec(i);

    iLen := (iLen div 4) * 3 + i;
    SetLength( Result, iLen); //̂

    iLen := Length( input );
    i := 2;
    k := 1;

    while i <= iLen do
    begin
        dbuf[0] := DecBinHex[ Ord(input[i]) -$21];
        Inc(i);

        if dbuf[0] = $FF then
            Break;

        dbuf[1] := DecBinHex[ Ord(input[i]) -$21];
        Inc(i);
        dbuf[2] := DecBinHex[ Ord(input[i]) -$21];
        Inc(i);

        if (input[i] = #$0D) or (input[i] = #$0A) then
        begin
            // s̉\
            Inc(i);

            if input[i] = #$0A then
                Inc(i);
        end;

        dbuf[3] := DecBinHex[ Ord(input[i]) -$21]; Inc(i);

        if dbuf[2] = $FF then
        begin
            Result[k] := Char( (dbuf[0] shl 2) or (dbuf[1] shr 4) );
            Break;
        end
        else if dbuf[3] = $FF then
        begin
            Result[k] := Char( (dbuf[0] shl 2) or (dbuf[1] shr 4) );
            Inc(k);
            Result[k] := Char( (dbuf[1] shl 4) or (dbuf[2] shr 2) );
            Break;
        end
        else
        begin
            Result[k] := Char( (dbuf[0] shl 2) or (dbuf[1] shr 4) );
            Inc(k);
            Result[k] := Char( (dbuf[1] shl 4) or (dbuf[2] shr 2) );
            Inc(k);
            Result[k] := Char( (dbuf[2] shl 6) or dbuf[3] );
        end;

        Inc(k);
    end;
end;

function QPCatString(Source : String) : String;
const
    lf = '=' + #13#10;
var
    n : Integer;
begin
    Result := Source;
    n := Pos(lf,Result);
    while n <> 0 do
    begin
        Result := Copy(Result,1,n-1) + Copy(Result,n+3,Length(Result)-n-2);
        n := Pos(lf,Result);
    end;
end;

function QPEncode(const Source : String) : String;
var
    n : Integer;
    curqlen: integer;
begin
    Result := '';
    curqlen := 0;
    for n:=1 to Length(Source) do
    begin
        case Source[n] of
            #0..#32, #61, #127..#255:
                begin
                    Result := Result + '=' + CodeQuoted[(Byte(Source[n]) div 16)+1]
                        + CodeQuoted[(Byte(Source[n]) mod 16)+1];
                    Inc(curqlen,3);
                end
            else
                begin
                    Result := Result + Source[n];
                    Inc(curqlen);
                end;
        end;

        if curqlen>70 then
        begin
            curqlen:=0;
            Result := Result + '=' + #13#10 ;
        end;
    end;
end;

function QPDecode(const Source : String): String;

    function LineQuotedDecode(Line: string): String;
    var
        i , j ,hex : integer;
    begin
        j := 0;
        hex := 0;
        Result := '';

        for i := 1 to Length(Line) do
        begin
            case j of
                0:
                    begin
                        if line[i]<>'=' then
                            Result := Result+Line[i]
                        else
                            j:=1;
                    end;
                1:
                    begin
                        if line[i]<='9' then
                            hex := (Ord(Line[i]) - Ord('0')) * 16
                        else
                            hex := (Ord(Line[i]) - 55) * 16;

                        j:=2;
                    end;
                2:
                    begin
                        if line[i]<='9' then
                            hex := hex + (Ord(Line[i]) - Ord('0'))
                        else
                            hex := hex + Ord(line[i]) - 55;

                        Result := Result + Chr(hex);
                        j:=0;
                    end;
            end;
        end;
    end;

var
    k :      integer;
    temp , line : String;
begin
    Result := '';
    temp := QPCatString(Trim(Source));
    k := Pos(#13#10 , temp);

    while k <> 0 do
    begin
        line := Copy( temp , 1 , k - 1 );
        Result := Result + LineQuotedDecode(Line) + #13#10;
        temp := Copy( temp , k + 2 , Length(temp) - k - 1 );
        k := Pos(#13#10 , temp);
    end;

    if temp <>'' then
        Result := Result + LineQuotedDecode(temp);
end;

function UrlDecode(const Source: String): String;
var
    Sp, Cp: PChar;
    HexStr : String;
    Value : Integer;
    Code : Integer;
    mem : TNMemory;
begin
	mem := TNMemory.Create(256);
    Sp := PChar(Source);

    while Sp^ <> #0 do
    begin
        if not (Sp^ in ['+','%']) then
        	mem.PushChar(Sp^)
        else if Sp^ = '+' then
        	mem.PushChar(' ')
        else
        begin
        	Inc(Sp);

            if Sp^ = '%' then
            	mem.PushChar('%')
            else
            begin
                Cp := Sp;
                Inc(Sp);

                HexStr := '$' + Cp^ + Sp^;
                Val(HexStr, Value, Code);
				mem.PushChar(Chr(Value));
            end;
        end;

        Inc(Sp);
    end;

    //NULLPUSH
    mem.PushChar(#0);

    Result := String(PChar(mem.Data));
    mem.Free;
end;

function UrlEncode(const Source: String): String;
const
    NoConversion = ['A'..'Z', 'a'..'z', '*', '@', '.', '-',
                  '0'..'9', '$', '!', '''', '(', ')', ',', '_'];
var
    Sp : PChar;
    temp : String;
    mem : TNMemory;
begin
    Sp := PChar(Source);
    mem := TNMemory.Create(256);

    while Sp^ <> #0 do
    begin
        if Sp^ in NoConversion then
            mem.PushChar(Sp^)
        else
            if Sp^ = ' ' then
                mem.PushChar('+')
            else
            begin
                temp := BytetoHex(Byte(Sp^));
                mem.PushChar('%');
                mem.PushChar(temp[1]);
                mem.PushChar(temp[2]);
            end;

            Inc(Sp);
    end;

    //NULLPUSH
	mem.PushChar(#0);
    Result := String(PChar(mem.Data));
end;

function GenerateBoundary : String;
var
    SysTime : SYSTEMTIME;
begin
    GetSystemTime(SysTime);
    Result := Format('-TS-JUPITER5-%04d%02d%02d%02d%02d%02d%02d----'
                        ,[SysTime.wYear, SysTime.wMonth, SysTime.wDay
                            , SysTime.wHour, SysTime.wMinute, SysTime.wSecond
                                , SysTime.wMilliseconds]);
end;

function GenerateTempFile(const Path : String) : String;
var
    temp : String;
begin
    temp := GenerateBoundary;

    while True do
    begin
        Result := Path + GetCrc(temp + IntToStr(Random(65536))) + '.tmp';
        if not FileExists(Result) then Break;
    end;
end;

function IsIncludeNonAscii(const Source : String) : Boolean;
var
    n : Integer;
begin
    Result := True;

    for n := 1 to Length(Source) do
    begin
        if Source[n] > Char(127) then
            Exit;
    end;

    Result := False;
end;

function AutoReturn(const Source : String ; Width : Integer) : String;
var
    SourceSize : Integer;
    ExpectSize : Integer;
    NewSize : Integer;
    Src : PChar;
    Res : PChar;
    n : Integer;

begin
    SourceSize := Length(Source);
    ExpectSize := SourceSize + (SourceSize div Width) * 2 + 2;
    NewSize := 0;
    SetLength(Result, ExpectSize);
    Src := PChar(Source);
    Res := PChar(Result);

    for n := 0 to SourceSize - 1 do
    begin
        if n mod Width = 0 then
        begin
            (Res + NewSize)^ := #13;
            NewSize := NewSize + 1;
            (Res + NewSize)^ := #10;
            NewSize := NewSize + 1;
        end;

        (Res + NewSize)^ := (Src + n)^;
        NewSize := NewSize + 1;
    end;

    SetLength(Result, NewSize);
end;

function HTMLtoText(const Source : String) : String;
var
	l , n : Integer;
	temp, stemp : String;
    p : PChar;
    mem : TNMemory;
    ch : Char;

    function ConvHtmlCode(Value : Integer) : String;
    var
        temp : WideString;
    begin
        temp := WideChar(WORD(Value));
        Result := temp;
    end;

begin
    temp := Source + '         ';
    l := Length(Source);
    p := PChar(temp);
    mem := TNMemory.Create(256);
    n := 0;

    while n < l do
    begin
        if p^ = '&' then
        begin
            stemp := '';
            Inc(p);
            Inc(n);
            ch := #0;

            //&`;Ԃ̕擾
            while n < l do
            begin
                ch := p^;
                if (ch = ';') or (ch = ' ') then Break;
                stemp := stemp + ch;
                Inc(n);
                Inc(p);
            end;

            if n >= l then
            begin
            	mem.PushChar('&');
                mem.Push(PChar(stemp), Length(stemp));
            	Continue;
            end;

            Inc(n);
            Inc(p);

            if stemp = '' then
            begin
                mem.PushChar('&');
                mem.PushChar(ch);
                Continue;
            end;

            if stemp[1] = '#' then
            begin
                stemp := Trim(Copy(stemp, 2, Length(stemp) - 1));
                stemp := ConvHtmlCode(StrToInt(stemp));
                mem.Push(PChar(stemp), Length(stemp));
            end
            else
            begin
                if (stemp = 'quot') or (stemp = 'QUOT') then
                    mem.PushChar('"')
                else if (stemp = 'amp') or (stemp = 'AMP') then
                    mem.PushChar('&')
                else if (stemp = 'lt') or (stemp = 'LT') then
                    mem.PushChar('<')
                else if (stemp = 'gt') or (stemp = 'GT') then
                    mem.PushChar('>')
                else if (stemp = 'ndash') or (stemp = 'NDASH') then
                    mem.PushChar('-')
                else if (stemp = 'nbsp') or (stemp = 'NBSP') then
                    mem.PushChar(' ')
                else if (stemp = 'circ') or (stemp = 'CIRC') then
                    mem.PushChar('^')

                //ȍ~͈ʓIłȂ
                else if stemp <> '' then
                begin
                    case stemp[1] of
                        'A':
                            begin
                                if stemp = 'ACUTE' then
                                    stemp := ConvHtmlCode(180)
                                else if stemp = 'Agrave' then
                                    stemp := ConvHtmlCode(192)
                                else if stemp = 'Aacute' then
                                    stemp := ConvHtmlCode(193)
                                else if stemp = 'Acric' then
                                    stemp := ConvHtmlCode(194)
                                else if stemp = 'Atilde' then
                                    stemp := ConvHtmlCode(195)
                                else if stemp = 'Auml' then
                                    stemp := ConvHtmlCode(196)
                                else if stemp = 'Aring' then
                                    stemp := ConvHtmlCode(197)
                                else if stemp = 'AElig' then
                                    stemp := ConvHtmlCode(198)
                                else if stemp = 'Alpha' then
                                    stemp := ConvHtmlCode(913)
                                else
                            		stemp := '&' + stemp + ';';
                            end;

                        'a':
                            begin
                                //A
                                if stemp = 'acute' then
                                    stemp := ConvHtmlCode(180)
                                else if stemp = 'agrave' then
                                    stemp := ConvHtmlCode(224)
                                else if stemp = 'aacute' then
                                    stemp := ConvHtmlCode(225)
                                else if stemp = 'acirc' then
                                    stemp := ConvHtmlCode(226)
                                else if stemp = 'arilde' then
                                    stemp := ConvHtmlCode(227)
                                else if stemp = 'auml' then
                                    stemp := ConvHtmlCode(228)
                                else if stemp = 'aring' then
                                    stemp := ConvHtmlCode(229)
                                else if stemp = 'aeling' then
                                    stemp := ConvHtmlCode(230)
                                else if stemp = 'alpha' then
                                    stemp := ConvHtmlCode(945)
                                else if stemp = 'alefsym' then
                                    stemp := ConvHtmlCode(8501)
                                else if stemp = 'ang' then
                                    stemp := ConvHtmlCode(8736)
                                else if stemp = 'and' then
                                    stemp := ConvHtmlCode(8743)
                                else if stemp = 'asymp' then
                                    stemp := ConvHtmlCode(8776)
                                else
                            		stemp := '&' + stemp + ';';
                            end;

                        'B', 'b':
                            begin
                                //B
                                if (stemp = 'brvbar') or (stemp = 'BRVBAR') then
                                    stemp := ConvHtmlCode(166)
                                else if stemp = 'Beta' then
                                    stemp := ConvHtmlCode(914)
                                else if stemp = 'beta' then
                                    stemp := ConvHtmlCode(946)
                                else if stemp = 'bdquo' then
                                    stemp := ConvHtmlCode(8222)
                                else if stemp = 'bull' then
                                    stemp := ConvHtmlCode(8226)
                                else
                            		stemp := '&' + stemp + ';';
                            end;

                        'C', 'c':
                            begin
                                //C
                                if (stemp = 'cent') or (stemp = 'CENT') then
                                    stemp := ConvHtmlCode(162)
                                else if (stemp = 'curren') or (stemp = 'CURREN') then
                                    stemp := ConvHtmlCode(164)
                                else if (stemp = 'copy') or (stemp = 'COPY') then
                                    stemp := '(C)'
                                else if (stemp = 'cedil') or (stemp = 'CEDIL') then
                                    stemp := ConvHtmlCode(184)
                                else if stemp = 'Ccedil' then
                                    stemp := ConvHtmlCode(199)
                                else if stemp = 'ccedil' then
                                    stemp := ConvHtmlCode(231)
                                else if stemp = 'Chi' then
                                    stemp := ConvHtmlCode(935)
                                else if stemp = 'chi' then
                                    stemp := ConvHtmlCode(967)
                                else if stemp = 'crarr' then
                                    stemp := ConvHtmlCode(8629)
                                else if stemp = 'cap' then
                                    stemp := ConvHtmlCode(8745)
                                else if stemp = 'cup' then
                                    stemp := ConvHtmlCode(8746)
                                else if stemp = 'cong' then
                                    stemp := ConvHtmlCode(8773)
                                else if stemp = 'clubs' then
                                    stemp := ConvHtmlCode(9827)
                                else
                            		stemp := '&' + stemp + ';';
                            end;

                        'D', 'd':
                            begin
                                //D
                                if (stemp = 'deg') or (stemp = 'DEG') then
                                    stemp := ConvHtmlCode(176)
                                else if stemp = 'divide' then
                                    stemp := ConvHtmlCode(247)
                                else if stemp = 'Delta' then
                                    stemp := ConvHtmlCode(916)
                                else if stemp = 'delta' then
                                    stemp := ConvHtmlCode(948)
                                else if stemp = 'darr' then
                                    stemp := ConvHtmlCode(8595)
                                else if stemp = 'dArr' then
                                    stemp := ConvHtmlCode(8659)
                                else if stemp = 'diams' then
                                    stemp := ConvHtmlCode(9830)
                                else if stemp = 'dagger' then
                                    stemp := ConvHtmlCode(8224)
                                else if stemp = 'Dagger' then
                                    stemp := ConvHtmlCode(8225)
                                else
                            		stemp := '&' + stemp + ';';
                            end;

                        'E':
                            begin
                                //E
                                if stemp = 'Egrave' then
                                    stemp := ConvHtmlCode(200)
                                else if stemp = 'Eacute' then
                                    stemp := ConvHtmlCode(201)
                                else if stemp = 'Ecric' then
                                    stemp := ConvHtmlCode(202)
                                else if stemp = 'Euml' then
                                    stemp := ConvHtmlCode(203)
                                else if stemp = 'ETH' then
                                    stemp := ConvHtmlCode(208)
                                else if stemp = 'Epsilon' then
                                    stemp := ConvHtmlCode(917)
                                else if stemp = 'Eta' then
                                    stemp := ConvHtmlCode(919)
                                else
                            		stemp := '&' + stemp + ';';
                            end;

                        'e':
                            begin
                                if stemp = 'egrave' then
                                    stemp := ConvHtmlCode(232)
                                else if stemp = 'eacute' then
                                    stemp := ConvHtmlCode(233)
                                else if stemp = 'ecirc' then
                                    stemp := ConvHtmlCode(234)
                                else if stemp = 'euml' then
                                    stemp := ConvHtmlCode(235)
                                else if stemp = 'eth' then
                                    stemp := ConvHtmlCode(240)
                                else if stemp = 'epsilon' then
                                    stemp := ConvHtmlCode(949)
                                else if stemp = 'eta' then
                                    stemp := ConvHtmlCode(951)
                                else if stemp = 'exit' then
                                    stemp := ConvHtmlCode(8707)
                                else if stemp = 'empty' then
                                    stemp := ConvHtmlCode(8709)
                                else if stemp = 'equiv' then
                                    stemp := ConvHtmlCode(8801)
                                else if stemp = 'ensp' then
                                    stemp := ConvHtmlCode(8194)
                                else if stemp = 'emsp' then
                                    stemp := ConvHtmlCode(8195)
                                else if stemp = 'euro' then
                                    stemp := ConvHtmlCode(8364)
                                else
                            		stemp := '&' + stemp + ';';
                            end;

                        'F', 'f':
                            begin
                                //F
                                if (stemp = 'frac14') or (stemp = 'FRAC14') then
                                    stemp := ConvHtmlCode(188)
                                else if (stemp = 'frac12') or (stemp = 'FRAC12') then
                                    stemp := ConvHtmlCode(189)
                                else if (stemp = 'frac34') or (stemp = 'FRAC34') then
                                    stemp := ConvHtmlCode(190)
                                else if stemp = 'frasl' then
                                    stemp := ConvHtmlCode(8260)
                                else if stemp = 'forall' then
                                    stemp := ConvHtmlCode(8704)
                                else
                            		stemp := '&' + stemp + ';';
                            end;

                        'G', 'g':
                            begin
                                //G
                                if stemp = 'Gamma' then
                                    stemp := ConvHtmlCode(915)
                                else if stemp = 'gamma' then
                                    stemp := ConvHtmlCode(947)
                                else if stemp = 'ge' then
                                    stemp := ConvHtmlCode(8805)
                                else
                            		stemp := '&' + stemp + ';';
                            end;

                        'H', 'h':
                            begin
                                //H
                                if stemp = 'hellip' then
                                    stemp := ConvHtmlCode(8230)
                                else if stemp = 'harr' then
                                    stemp := ConvHtmlCode(8596)
                                else if stemp = 'hArr' then
                                    stemp := ConvHtmlCode(8660)
                                else if stemp = 'hearts' then
                                    stemp := ConvHtmlCode(9829)
                                else
                            		stemp := '&' + stemp + ';';
                            end;

                        'I':
                            begin
                                if stemp = 'IXCEL' then
                                    stemp := ConvHtmlCode(161)
                                else if stemp = 'IQUEST' then
                                    stemp := ConvHtmlCode(191)
                                else if stemp = 'Igrave' then
                                    stemp := ConvHtmlCode(204)
                                else if stemp = 'Iacute' then
                                    stemp := ConvHtmlCode(205)
                                else if stemp = 'Icirc' then
                                    stemp := ConvHtmlCode(206)
                                else if stemp = 'Iuml' then
                                    stemp := ConvHtmlCode(207)
                                else if stemp = 'Iota' then
                                    stemp := ConvHtmlCode(921)
                                else
                            		stemp := '&' + stemp + ';';
                            end;

                        'i':
                            begin
                                //I
                                if stemp = 'ixcel' then
                                    stemp := ConvHtmlCode(161)
                                else if stemp = 'iquest' then
                                    stemp := ConvHtmlCode(191)
                                else if stemp = 'igrave' then
                                    stemp := ConvHtmlCode(236)
                                else if stemp = 'iacute' then
                                    stemp := ConvHtmlCode(237)
                                else if stemp = 'icirc' then
                                    stemp := ConvHtmlCode(238)
                                else if stemp = 'iuml' then
                                    stemp := ConvHtmlCode(239)
                                else if stemp = 'iota' then
                                    stemp := ConvHtmlCode(953)
                                else if stemp = 'image' then
                                    stemp := ConvHtmlCode(8465)
                                else if stemp = 'isin' then
                                    stemp := ConvHtmlCode(8712)
                                else if stemp = 'infin' then
                                    stemp := ConvHtmlCode(8734)
                                else if stemp = 'int' then
                                    stemp := ConvHtmlCode(8747)
                                else
                            		stemp := '&' + stemp + ';';
                            end;

                        'K', 'k':
                            begin
                                //K
                                if (stemp = 'knof') or (stemp = 'KNOF') then
                                    stemp := ConvHtmlCode(402)
                                else if stemp = 'Kappa' then
                                    stemp := ConvHtmlCode(922)
                                else if stemp = 'kappa' then
                                    stemp := ConvHtmlCode(954)
                                else
                            		stemp := '&' + stemp + ';';
                            end;

                        'L', 'l':
                            begin
                                //L
                                if (stemp = 'laquo') or (stemp = 'LAQUO') then
                                    stemp := ConvHtmlCode(171)
                                else if stemp = 'Lambda' then
                                    stemp := ConvHtmlCode(923)
                                else if stemp = 'lambda' then
                                    stemp := ConvHtmlCode(955)
                                else if stemp = 'larr' then
                                    stemp := ConvHtmlCode(8592)
                                else if stemp = 'lArr' then
                                    stemp := ConvHtmlCode(8656)
                                else if stemp = 'lowast' then
                                    stemp := ConvHtmlCode(8727)
                                else if stemp = 'le' then
                                    stemp := ConvHtmlCode(8804)
                                else if stemp = 'lceil' then
                                    stemp := ConvHtmlCode(8968)
                                else if stemp = 'lfloor' then
                                    stemp := ConvHtmlCode(8970)
                                else if stemp = 'lang' then
                                    stemp := ConvHtmlCode(9001)
                                else if stemp = 'loz' then
                                    stemp := ConvHtmlCode(9674)
                                else if stemp = 'lrm' then
                                    stemp := ConvHtmlCode(8206)
                                else if stemp = 'lsquo' then
                                    stemp := ConvHtmlCode(8216)
                                else if stemp = 'ldquo' then
                                    stemp := ConvHtmlCode(8220)
                                else if stemp = 'lsaquo' then
                                    stemp := ConvHtmlCode(8249)
                                else
                            		stemp := '&' + stemp + ';';
                            end;

                        'M', 'm':
                            begin
                                //M
                                if (stemp = 'macr') or (stemp = 'MACR') then
                                    stemp := ConvHtmlCode(175)
                                else if (stemp = 'micro') or (stemp = 'MICRO') then
                                    stemp := ConvHtmlCode(181)
                                else if (stemp = 'middot') or (stemp = 'MIDDOT') then
                                    stemp := ConvHtmlCode(183)
                                else if stemp = 'Mu' then
                                    stemp := ConvHtmlCode(924)
                                else if stemp = 'mu' then
                                    stemp := ConvHtmlCode(956)
                                else if stemp = 'mdash' then
                                    stemp := ConvHtmlCode(8212)
                                else if stemp = 'minus' then
                                    stemp := ConvHtmlCode(8722)
                                else
                            		stemp := '&' + stemp + ';';
                            end;

                        'N', 'n':
                            begin
                                //N
                                if (stemp = 'not') or (stemp = 'NOT') then
                                    stemp := ConvHtmlCode(172)
                                else if stemp = 'Ntilde' then
                                    stemp := ConvHtmlCode(209)
                                else if stemp = 'ntilde' then
                                    stemp := ConvHtmlCode(241)
                                else if stemp = 'Nu' then
                                    stemp := ConvHtmlCode(925)
                                else if stemp = 'nu' then
                                    stemp := ConvHtmlCode(957)
                                else if stemp = 'nabla' then
                                    stemp := ConvHtmlCode(8711)
                                else if stemp = 'notin' then
                                    stemp := ConvHtmlCode(8713)
                                else if stemp = 'ni' then
                                    stemp := ConvHtmlCode(8715)
                                else if stemp = 'ne' then
                                    stemp := ConvHtmlCode(8800)
                                else if stemp = 'nsub' then
                                    stemp := ConvHtmlCode(8836)
                                else
                            		stemp := '&' + stemp + ';';
                            end;

                        'O':
                            begin
                                if stemp = 'ORDF' then
                                    stemp := ConvHtmlCode(170)
                                else if stemp = 'ORDM' then
                                    stemp := ConvHtmlCode(186)
                                else if stemp = 'Ograve' then
                                    stemp := ConvHtmlCode(210)
                                else if stemp = 'Oacute' then
                                    stemp := ConvHtmlCode(211)
                                else if stemp = 'Ocirc' then
                                    stemp := ConvHtmlCode(212)
                                else if stemp = 'Otilde' then
                                    stemp := ConvHtmlCode(213)
                                else if stemp = 'Ouml' then
                                    stemp := ConvHtmlCode(214)
                                else if stemp = 'Oslash' then
                                    stemp := ConvHtmlCode(216)
                                else if stemp = 'Omicron' then
                                    stemp := ConvHtmlCode(927)
                                else if stemp = 'Omega' then
                                    stemp := ConvHtmlCode(937)
                                else if stemp = 'OElig' then
                                    stemp := ConvHtmlCode(338)
                                else
                            		stemp := '&' + stemp + ';';
                            end;

                        'o':
                            begin
                                //O
                                if stemp = 'ordf' then
                                    stemp := ConvHtmlCode(170)
                                else if stemp = 'ordm' then
                                    stemp := ConvHtmlCode(186)
                                else if stemp = 'ograve' then
                                    stemp := ConvHtmlCode(242)
                                else if stemp = 'oacute' then
                                    stemp := ConvHtmlCode(243)
                                else if stemp = 'ocirc' then
                                    stemp := ConvHtmlCode(244)
                                else if stemp = 'otilde' then
                                    stemp := ConvHtmlCode(245)
                                else if stemp = 'ouml' then
                                    stemp := ConvHtmlCode(246)
                                else if stemp = 'oslash' then
                                    stemp := ConvHtmlCode(248)
                                else if stemp = 'omicron' then
                                    stemp := ConvHtmlCode(959)
                                else if stemp = 'omega' then
                                    stemp := ConvHtmlCode(969)
                                else if stemp = 'oline' then
                                    stemp := ConvHtmlCode(8254)
                                else if stemp = 'or' then
                                    stemp := ConvHtmlCode(8744)
                                else if stemp = 'sub' then
                                    stemp := ConvHtmlCode(8834)
                                else if stemp = 'oplus' then
                                    stemp := ConvHtmlCode(8853)
                                else if stemp = 'otimes' then
                                    stemp := ConvHtmlCode(8855)
                                else if stemp = 'oelig' then
                                    stemp := ConvHtmlCode(339)
                                else
                            		stemp := '&' + stemp + ';';
                            end;

                        'P', 'p':
                            begin
                                //P
                                if (stemp = 'pound') or (stemp = 'POUND') then
                                    stemp := ConvHtmlCode(163)
                                else if (stemp = 'plusmn') or (stemp = 'PLUSMN') then
                                    stemp := ConvHtmlCode(177)
                                else if (stemp = 'para') or (stemp = 'PARA') then
                                    stemp := ConvHtmlCode(182)
                                else if stemp = 'Pi' then
                                    stemp := ConvHtmlCode(928)
                                else if stemp = 'Phi' then
                                    stemp := ConvHtmlCode(934)
                                else if stemp = 'Psi' then
                                    stemp := ConvHtmlCode(936)
                                else if stemp = 'pi' then
                                    stemp := ConvHtmlCode(960)
                                else if stemp = 'phi' then
                                    stemp := ConvHtmlCode(966)
                                else if stemp = 'psi' then
                                    stemp := ConvHtmlCode(968)
                                else if stemp = 'piv' then
                                    stemp := ConvHtmlCode(982)
                                else if stemp = 'prime' then
                                    stemp := ConvHtmlCode(8242)
                                else if stemp = 'Prime' then
                                    stemp := ConvHtmlCode(8243)
                                else if stemp = 'part' then
                                    stemp := ConvHtmlCode(8706)
                                else if stemp = 'prod' then
                                    stemp := ConvHtmlCode(8719)
                                else if stemp = 'prop' then
                                    stemp := ConvHtmlCode(8733)
                                else if stemp = 'perp' then
                                    stemp := ConvHtmlCode(8869)
                                else if stemp = 'permil' then
                                    stemp := ConvHtmlCode(8240)
                                else
                            		stemp := '&' + stemp + ';';
                            end;

                        'R', 'r':
                            begin
                                //R
                                if (stemp = 'reg') or (stemp = 'REG') then
                                    stemp := '(R)'
                                else if (stemp = 'raquo') or (stemp = 'RAQUO') then
                                    stemp := ConvHtmlCode(187)
                                else if stemp = 'Rho' then
                                    stemp := ConvHtmlCode(929)
                                else if stemp = 'rho' then
                                    stemp := ConvHtmlCode(961)
                                else if stemp = 'real' then
                                    stemp := ConvHtmlCode(8476)
                                else if stemp = 'rarr' then
                                    stemp := ConvHtmlCode(8594)
                                else if stemp = 'rArr' then
                                    stemp := ConvHtmlCode(8658)
                                else if stemp = 'radic' then
                                    stemp := ConvHtmlCode(8730)
                                else if stemp = 'rceil' then
                                    stemp := ConvHtmlCode(8969)
                                else if stemp = 'rfloor' then
                                    stemp := ConvHtmlCode(8971)
                                else if stemp = 'rang' then
                                    stemp := ConvHtmlCode(9002)
                                else if stemp = 'rlm' then
                                    stemp := ConvHtmlCode(8207)
                                else if stemp = 'rsquo' then
                                    stemp := ConvHtmlCode(8217)
                                else if stemp = 'rdquo' then
                                    stemp := ConvHtmlCode(8221)
                                else if stemp = 'rsaquo' then
                                    stemp := ConvHtmlCode(8250)
                                else
                            		stemp := '&' + stemp + ';';
                            end;

                        'S', 's':
                            begin
                                //S
                                if (stemp = 'sent') or (stemp = 'SENT') then
                                    stemp := ConvHtmlCode(167)
                                else if (stemp = 'shy') or (stemp = 'SHY') then
                                    stemp := ConvHtmlCode(173)
                                else if (stemp = 'sup2') or (stemp = 'SUP2') then
                                    stemp := ConvHtmlCode(178)
                                else if (stemp = 'sup3') or (stemp = 'SUP3') then
                                    stemp := ConvHtmlCode(179)
                                else if (stemp = 'sup1') or (stemp = 'SUP1') then
                                    stemp := ConvHtmlCode(185)
                                else if stemp = 'szlig' then
                                    stemp := ConvHtmlCode(223)
                                else if stemp = 'Scaron' then
                                    stemp := ConvHtmlCode(352)
                                else if stemp = 'scaron' then
                                    stemp := ConvHtmlCode(353)
                                else if stemp = 'Sigma' then
                                    stemp := ConvHtmlCode(931)
                                else if stemp = 'sigmaf' then
                                    stemp := ConvHtmlCode(962)
                                else if stemp = 'sigma' then
                                    stemp := ConvHtmlCode(963)
                                else if stemp = 'sum' then
                                    stemp := ConvHtmlCode(8721)
                                else if stemp = 'sim' then
                                    stemp := ConvHtmlCode(8764)
                                else if stemp = 'sub' then
                                	stemp := ConvHtmlCode(8834)
                                else if stemp = 'sup' then
                                    stemp := ConvHtmlCode(8835)
                                else if stemp = 'sube' then
                                    stemp := ConvHtmlCode(8838)
                                else if stemp = 'supe' then
                                    stemp := ConvHtmlCode(8839)
                                else if stemp = 'sdot' then
                                    stemp := ConvHtmlCode(8901)
                                else if stemp = 'spades' then
                                    stemp := ConvHtmlCode(9824)
                                else if stemp = 'sbquo' then
                                    stemp := ConvHtmlCode(8218)
                                else
                            		stemp := '&' + stemp + ';';
                            end;

                        'T', 't':
                            begin
                                //T
                                if (stemp = 'tilde') or (stemp = 'TILDE') then
                                    stemp := '~'
                                else if (stemp = 'times') or (stemp = 'TIMES') then
                                    stemp := ConvHtmlCode(215)
                                else if stemp = 'THORN' then
                                    stemp := ConvHtmlCode(222)
                                else if stemp = 'thorn' then
                                    stemp := ConvHtmlCode(254)
                                else if stemp = 'Theta' then
                                    stemp := ConvHtmlCode(920)
                                else if stemp = 'Tau' then
                                    stemp := ConvHtmlCode(932)
                                else if stemp = 'theta' then
                                    stemp := ConvHtmlCode(952)
                                else if stemp = 'tau' then
                                    stemp := ConvHtmlCode(964)
                                else if stemp = 'thetasym' then
                                    stemp := ConvHtmlCode(977)
                                else if stemp = 'trade' then
                                    stemp := '(TM)'
                                else if stemp = 'there4' then
                                    stemp := ConvHtmlCode(8756)
                                else if stemp = 'thinsp' then
                                    stemp := ConvHtmlCode(8201)
                                else
                            		stemp := '&' + stemp + ';';
                            end;

                        'U', 'u':
                            begin
                                //U
                                if (stemp = 'uml') or (stemp = 'UML') then
                                    stemp := ConvHtmlCode(168)
                                else if stemp = 'Ugrave' then
                                    stemp := ConvHtmlCode(217)
                                else if stemp = 'Uacute' then
                                    stemp := ConvHtmlCode(218)
                                else if stemp = 'Ucirc' then
                                    stemp := ConvHtmlCode(219)
                                else if stemp = 'Uuml' then
                                    stemp := ConvHtmlCode(220)
                                else if stemp = 'ugrave' then
                                    stemp := ConvHtmlCode(249)
                                else if stemp = 'uacute' then
                                    stemp := ConvHtmlCode(250)
                                else if stemp = 'ucirc' then
                                    stemp := ConvHtmlCode(251)
                                else if stemp = 'uuml' then
                                    stemp := ConvHtmlCode(252)
                                else if stemp = 'Upsilon' then
                                    stemp := ConvHtmlCode(933)
                                else if stemp = 'upsilon' then
                                    stemp := ConvHtmlCode(965)
                                else if stemp = 'upsih' then
                                    stemp := ConvHtmlCode(978)
                                else if stemp = 'uarr' then
                                    stemp := ConvHtmlCode(8593)
                                else if stemp = 'uArr' then
                                    stemp := ConvHtmlCode(8657)
                                else
                            		stemp := '&' + stemp + ';';
                            end;

                        'W', 'w':
                            begin
                                //W
                                if stemp = 'weierp' then
                                    stemp := ConvHtmlCode(8472)
                                else
                            		stemp := '&' + stemp + ';';
                            end;

                        'X', 'x':
                            begin
                                //X
                                if stemp = 'Xi' then
                                    stemp := ConvHtmlCode(926)
                                else if stemp = 'xi' then
                                    stemp := ConvHtmlCode(958)
                                else
                            		stemp := '&' + stemp + ';';
                            end;

                        'Y', 'y':
                            begin
                                //Y
                                if (stemp = 'yen') or (stemp = 'YEN') then
                                    stemp := ConvHtmlCode(165)
                                else if stemp = 'Yacute' then
                                    stemp := ConvHtmlCode(221)
                                else if stemp = 'yacute' then
                                    stemp := ConvHtmlCode(253)
                                else if stemp = 'yuml' then
                                    stemp := ConvHtmlCode(255)
                                else if stemp = 'Yuml' then
                                    stemp := ConvHtmlCode(376)
                                else
                            		stemp := '&' + stemp + ';';
                            end;

                        'Z', 'z':
                            begin
                                //Z
                                if stemp = 'Zeta' then
                                    stemp := ConvHtmlCode(918)
                                else if stemp  = 'zeta' then
                                    stemp := ConvHtmlCode(950)
                                else if stemp = 'zwnj' then
                                    stemp := ConvHtmlCode(8204)
                                else if stemp = 'zwj' then
                                    stemp := ConvHtmlCode(8205)
                                else
                            		stemp := '&' + stemp + ';';
                            end;

                        else
                        	begin
                            	stemp := '&' + stemp + ';';
                            end;
                    end;

                    mem.Push(PChar(stemp), Length(stemp));
                end;
            end;
        end
        else
        begin
            mem.PushChar(p^);
            Inc(n);
            Inc(p);
        end;
    end;

    //NULLPUSH
    mem.PushChar(#0);

    Result := String(PChar(mem.Data));
    mem.Free;
end;

function TexttoHTML(const Source : String) : String;
var
    r , s : PChar;
    n , size : Integer;
begin
    SetLength(Result,Length(Source)*2);
    r := PChar(Result);
    s := PChar(Source);
    size := 0;

    for n := 0 to Length(Source) - 1 do
    begin
        if s^ = '&' then
        begin
            r^ := '&';
            (r+1)^ := 'a';
            (r+2)^ := 'm';
            (r+3)^ := 'p';
            (r+4)^ := ';';
            s := s + 1;
            r := r + 5;
            size := size + 5;
        end
        else
        if s^ = '<' then
        begin
            r^ := '&';
            (r+1)^ := 'l';
            (r+2)^ := 't';
            (r+3)^ := ';';
            s := s + 1;
            r := r + 4;
            size := size + 4;
        end
        else
        if s^ = '>' then
        begin
            r^ := '&';
            (r+1)^ := 'g';
            (r+2)^ := 't';
            (r+3)^ := ';';
            s := s + 1;
            r := r + 4;
            size := size + 4;
        end
        else
        if s^ = '"' then
        begin
            r^ := '&';
            (r+1)^ := 'q';
            (r+2)^ := 'u';
            (r+3)^ := 'o';
            (r+4)^ := 't';
            (r+5)^ := ';';
            s := s + 1;
            r := r + 6;
            size := size + 6;
        end
        else
        begin
            r^ := s^;
            s := s + 1;
            r := r + 1;
            size := size + 1;
        end;
    end;

    Result := Copy(Result,1,size);
end;

function DeleteTag(const Source : String) : String;
var
    r , s : PChar;
    ln , n , size : Integer;
    TagOn : Boolean;
begin
    size := 0;
    TagOn := False;
    ln := Length(Source);
    Setlength(Result,ln);
    s := PChar(Source);
    r := PChar(Result);
    for n := 1 to ln do
    begin
        if s^ = '<' then
            TagOn := True
        else if s^ = '>' then
            TagOn := False
        else if TagOn=False then
        begin
            r^ := s^;
            r := r + 1;
            size := size + 1;
        end;
        s := s + 1;
    end;
    Result := Copy(Result , 1 , size);
end;

function AnalyzeFormData(const ContentType, Data : String) : TFormData;
begin
	Result := TFormData.Create;
    Result.SetData(ContentType, Data);
end;

function JisToSjis(const Data : String) : String;
var
	source : String;
    p : PChar;
    n : Integer;
    ln : Integer;
    mem : TNMemory;
    ch : Char;
    db : Boolean;
    lb, tb, shift : Byte;
begin
    n := 0;
    db := False;
	source := Data + '          ';
    p := PChar(Source);
    mem := TNMemory.Create(256);
    ln := Length(Data);

	while n < ln do
    begin
		ch := p^;
        Inc(p);
        Inc(n);

        if ch = #27 then
        begin
			ch := p^;

    		if ch = '$' then
            begin
            	db := True;
                Inc(p, 2);
                Inc(n, 2);
            end
            else if ch = '(' then
            begin
            	db := False;
                Inc(p, 2);
                Inc(n, 2);
            end;
        end
        else if db then
        begin
        	lb := Byte(ch);
            tb := Byte(p^);

            if (lb < 33) or (lb > 126) then
            begin
            	mem.PushChar(ch);
            	Continue;
            end
            else
            begin
            	shift := 126;
                if lb mod 2 = 1 then
                if tb < 96 then shift := 31 else shift := 32;
                tb := tb + shift;
                if lb < 95 then shift := 112 else shift := 176;
                lb := ((lb + 1) shr 1) + shift;

                mem.PushChar(Char(lb));
                mem.PushChar(Char(tb));

                Inc(p);
                Inc(n);
            end;
        end
        else
        	mem.PushChar(ch);
	end;

    mem.PushChar(#0);
    Result := String(PChar(mem.Data));
    mem.Free;
end;

function SJisToJis(const Data : String) : String;
const
	HAN_1	=	'';
	CON_1	=	'BuvAE@BDFHb[ACEGIijklm}~JK';

	HAN_2	=	'';
	CON_2	=	'JLNPRTVXZ\^`ceg';
	CON_2D	=	'KMOQSUWY[]_adfh';

    HAN_3	=	'';
	CON_3	=	'nqtwz';
	CON_3D	=	'orux{';
	CON_3HD	=	'psvy|';
var
    src : String;
    p : PChar;
    size : Integer;
    ps : Integer;
	n : Longint;
    mem : TNMemory;
    tb : Boolean;
    fb , sb , shift , adj :Byte;

	procedure Convertion(const MBChar : String);
	var
		fb, sb, shift , adj : Byte;
    begin
		if not tb then //QoCg̊Jn
      	begin
        	tb := True;
            mem.PushChar(#27);
            mem.PushChar('$');
            mem.PushChar('B');
        end;

    	fb := Byte(MBChar[1]);
        sb := Byte(MBChar[2]);

		if fb <= 159 then
			shift := 112
		else
			shift := 176;

		if sb < 159  then
			adj := 1
		else
			adj := 0;

		fb := ((fb - shift) shl 1) - adj;
		shift := 126;

		if sb < 127 then
			shift := 31
		else if sb < 159 then
			shift := 32;

		sb := sb - shift;

        mem.PushChar(Char(fb));
        mem.PushChar(Char(sb));
    end;

begin
  	//Init
    n := 0;
  	tb := False;

    src := Data + ' ';
    p := PChar(src);
    size := Length(Data);

    mem := TNMemory.Create(64);

    while n < size do
    begin
		if IsDBCSLeadByteEx(0, Byte(p^)) then
        begin
      		if not tb then //QoCg̊Jn
      		begin
        		tb := True;
                mem.PushChar(#27);
                mem.PushChar('$');
                mem.PushChar('B');
      		end;

			//LeadoCg擾
  			fb := Byte(p^);

			//TrailoCg擾
            Inc(p);
            Inc(n);
        	sb := Byte(p^);

  			if fb <= 159 then
                shift := 112
            else
                shift := 176;

  			if sb < 159  then
                adj := 1
            else
                adj := 0;

  			fb := ((fb - shift) shl 1) - adj;
  			shift := 126;

  			if sb < 127 then
                shift := 31
            else if sb < 159 then
                shift := 32;

  			sb := sb - shift;
            mem.PushChar(Char(fb));
            mem.PushChar(Char(sb));
        end
        else
        begin
			//̕ϊ
			ps := Pos(p^, HAN_1);

            if ps <> 0 then
			begin
				Inc(p);
				Inc(n);
            	Dec(ps);
            	Convertion(Copy(CON_1, ps * 2 + 1, 2));
                Continue;
            end;

			//E̕ϊ
            ps := Pos(p^, HAN_2);

            if ps <> 0 then
            begin
				Inc(p);
				Inc(n);
            	Dec(ps);

                if p^ = '' then
                begin
					Inc(p);
					Inc(n);
                	Convertion(Copy(CON_2D, ps * 2 + 1, 2));
                end
                else
                	Convertion(Copy(CON_2, ps * 2 + 1, 2));

                Continue;
            end;

			//͍s̕ϊ
			//E̕ϊ
			ps := Pos(p^, HAN_3);

			if ps <> 0 then
			begin
				Inc(p);
				Inc(n);
		    	Dec(ps);

				if p^ = '' then
                begin
					Inc(p);
					Inc(n);
                	Convertion(Copy(CON_3D, ps * 2 + 1, 2));
                end
                else if p^ = '' then
                begin
					Inc(p);
					Inc(n);
                	Convertion(Copy(CON_3HD, ps * 2 + 1, 2));
                end
                else
                	Convertion(Copy(CON_3, ps * 2 + 1, 2));

                Continue;
            end;

      		if tb then
      		begin
        		tb := False;
                mem.PushChar(#27);
                mem.PushChar('(');
                mem.PushChar('B');
            end;

			mem.PushChar(p^);
        end;

		Inc(p);
		Inc(n);
    end;

	if tb then
	begin
		mem.PushChar(#27);
		mem.PushChar('(');
		mem.PushChar('B');
	end;

	//NULL
    mem.PushChar(#0);

	Result := String(PChar(mem.Data));
    mem.Free;
end;

end.
