unit tn_fw;

//Window Frame Works
//Developed by T.Nak

//1st Edition   2001/10/13

interface

uses
	windows, messages, tn_utils, tn_classes, shellapi, commctrl;

const
    WM_TRAY     =   WM_USER + 8100;

type
	WObject     =   TObject;
    WHandle     =   THandle;
    WResult     =   LRESULT;
    WDefProc    =   function (hWnd : HWND; Msg: UINT; wParam: WPARAM; lParam: LPARAM): WResult; stdcall;

    WApp    =   class(WObject)
        public
            ExeName : String;
            Path : String;
            Icon : hIcon;
            constructor Create;
            destructor Destroy; override;
            procedure Run; virtual;
            procedure MessageLoop;
            procedure Quit;
    end;

    WWindow =   class(WObject)
        private
            function GetCaption : String;
            procedure SetCaption(Value : String);
            function GetHeight : Integer;
            procedure SetHeight(Value : Integer);
            function GetWidth : Integer;
            procedure SetWidth(Value : Integer);
            function GetLeft : Integer;
            procedure SetLeft(Value : Integer);
            function GetTop : Integer;
            procedure SetTop(Value : Integer);

        public
            App : WApp;
            Handle : WHandle;
            ParentHandle : WHandle;
            AutoClose : Boolean;

            procedure OnInit(Sender : WObject ; var Return : WResult); virtual;
            procedure OnKeyDown(Sender : WObject ; Key : Integer ; var Return : WResult); virtual;
            procedure OnKeyUp(Sender : WObject ; Key : Integer ; var Return : WResult); virtual;
            procedure OnMouseDown(Sender : WObject ; Button : Integer ; x : Integer ; y : Integer ; var Return : WResult); virtual;
            procedure OnMouseMove(Sender : WObject ; Button : Integer ; x : Integer ; y : Integer ; var Return : WResult); virtual;
            procedure OnMouseUp(Sender : WObject ; Button : Integer ; x : Integer ; y : Integer ; var Return : WResult); virtual;
            procedure OnShow(Sender : WObject ; var Return : WResult); virtual;
            procedure OnDestroy(Sender : WObject ; var Return : WResult); virtual;
            procedure OnCommand(Sender : WObject ; wParam : Integer ; lParam : Integer ; var Return : WResult); virtual;
            procedure OnClose(Sender : WObject ; var DoClose : Boolean); virtual;
            procedure OnPaint(Sender : WObject ; var Return : WResult); virtual;
            procedure OnClick(Sender : WObject ; var Return : WResult); virtual;

            procedure Hide;
            procedure Close; virtual;
            procedure SetCenter;

            property Caption : String read GetCaption write SetCaption;
            property Height : Integer read GetHeight write SetHeight;
            property Left : Integer read GetLeft write SetLeft;
            property Width : Integer read GetWidth write SetWidth;
            property Top : Integer read GetTop write SetTop;
            procedure DefaultHandler(var Message); override;
    end;

    WModalDialog    =   class(WWindow)
        private
            ResourceID : String;
        public
            constructor Create(AppObj : WApp ; Parent : WWindow ; ResID : String); overload; virtual;
            constructor Create(Parent : WHandle ; ResID : String); overload; virtual;
            destructor Destroy; override;
            function GetHandle(ID : Integer) : WHandle;
            procedure Show; virtual;
            procedure ShowModal; virtual;
            procedure Close; override;
            procedure DefaultHandler(var Message); override;
    end;

    WFrame  =   class(WWindow)
        private
            NowShown : Boolean;
        public
            constructor Create(AppObj : WApp ; Parent : WWindow); overload; virtual;
            constructor Create(Parent : WHandle); overload; virtual;
            destructor Destroy; override;
            procedure Show; virtual;
            procedure ShowModal; virtual;
            procedure Close; override;
    end;

    WTrayAction =   (NIM_ADD, NIM_MODIFY, NIM_DELETE);

    WTrayIcon   =   class(WFrame)
        private
            FIcon : HICON;
            FTitle : String;
            FVisible : Boolean;
            procedure SetIcon(Value : HICON);
            procedure SetTitle(Value : String);
            procedure SetVisible(Value : Boolean);
            procedure SetupIcon(Action : WTrayAction);
        public
            constructor Create(AppObj : WApp ; Parent : WWindow); override;
            destructor Destroy; override;
            procedure PopupMenu(Menu : HMENU ; x : Integer ; y : Integer);
            //Cxg
            procedure OnTray(var Message: TMessage); message WM_TRAY;
            //vpeB
            property Icon : HICON read FIcon write SetIcon;
            property Title : String read FTitle write SetTitle;
            property Visible : Boolean read FVisible write SetVisible;
    end;

    WAlign      =   (alNone, alTop, alBottom, alLeft, alRight, alClient);
    WControl    =   class(WWindow)
        private
            Assigned : Boolean;
            DefProc : WDefProc;
        protected
            FAlign : WAlign;
            procedure SetAlign(Value : WAlign); virtual;
        public
            constructor Create(Parent : WWindow); overload; virtual;
            constructor Create(AssignHandle : WHandle); overload; virtual;
            destructor Destroy; override;
            procedure DefaultHandler(var Message); override;
            property Align : WAlign read FAlign write SetAlign;

            procedure OnPaint(Sender : WObject ; var Return : WResult); override;
    end;

    WImageList  =   class
        public
            Handle : WHandle;
            Height : Integer;
            Width : Integer;
            MaskColor : COLORREF;
            constructor Create(x : Integer ; y : Integer ; ColorType : Cardinal);
            destructor Destroy; override;
            procedure LoadResBitmap(ResID : String);
    end;

    WListView   =   class(WControl)
    end;

//procedures
    function ExtractWindow(Handle : WHandle) : WWindow;

    function Ctl_SetText(Handle : THandle ; Value : String) : Integer;
    function Ctl_GetText(Handle : THandle) : String;
    procedure Ctl_SetEnabled(Handle : THandle ; Value : Boolean);

    procedure LV_AddColumn(Handle : THandle ; Text : String ; Index, Size : Integer);
    procedure LV_SetImageList(Handle : THandle ; ImgList : HIMAGELIST);
    procedure LV_AddItem(Handle : THandle ; var List : TNStringList ; ImgIndex : Integer);
    procedure LV_SetStyle(Handle : THandle ; AdditionStyle : Cardinal);
    function LV_GetSelectedCount(Handle : THandle) : Integer;
    function LV_GetIndex(Handle : THandle) : Integer;
    function LV_GetItem(Handle : THandle ; Index : Integer ; SubIndex : Integer) : String;
    procedure LV_SetItem(Handle : THandle ; var List : TNStringList ; ImgIndex : Integer ; Index : Integer);

var
    WindowQue : TNIntList;      //֘At҂

implementation

//******************************************************************************
// WApp

constructor WApp.Create;
begin
    ExeName := ExtractFileName(ParamStr(0));
    Path := ExtractFilePath(ParamStr(0));
    //Icon := LoadIcon(hInstance, IDI_APPLICATION);
    Icon := LoadIcon(hInstance, 'MAINICON');
    Run;
end;

destructor WApp.Destroy;
begin
    DestroyIcon(Icon);
end;

procedure WApp.Run;
begin
    //This procedure is used for initializing application.
    //Please override here.
end;

procedure WApp.MessageLoop;
var
    msg : TMsg;
begin
    while GetMessage(msg, 0, 0, 0) do
    begin
        TranslateMessage(msg);
        DispatchMessage(msg);
    end;
end;

procedure WApp.Quit;
begin
    PostQuitMessage(0);
end;

//******************************************************************************
// WWindow

function WWindow.GetCaption : String;
var
    Buf : array [0..1024] of Char;
begin
    FillChar(Buf, SizeOf(Buf), #0);
    GetWindowText(Handle, @Buf, SizeOf(Buf));
    Result := String(@Buf);
end;

procedure WWindow.SetCaption(Value : String);
begin
    SetWindowText(Handle, PChar(Value));
end;

function WWindow.GetHeight : Integer;
var
    rect : TRect;
begin
	GetWindowRect(Handle, rect);
	//Result := rect.Bottom - rect.Top + 1;
	Result := rect.Bottom - rect.Top;
end;

procedure WWindow.SetHeight(Value : Integer);
begin
	MoveWindow(Handle, GetLeft, GetTop, GetWidth, Value, True);
end;

function WWindow.GetWidth : Integer;
var
    rect : TRect;
begin
	GetWindowRect(Handle, rect);
	//Result := rect.Right - rect.Left + 1;
	Result := rect.Right - rect.Left;
end;

procedure WWindow.SetWidth(Value : Integer);
begin
	MoveWindow(Handle, GetLeft, GetTop, Value, GetHeight, True);
end;

function WWindow.GetLeft : Integer;
var
    rect : TRect;
begin
	GetWindowRect(Handle, rect);
	Result := rect.Left;
end;

procedure WWindow.SetLeft(Value : Integer);
begin
	MoveWindow(Handle, Value, GetTop, GetWidth, GetHeight, True);
end;

function WWindow.GetTop : Integer;
var
    rect : TRect;
begin
	GetWindowRect(Handle, rect);
	Result := rect.Top;
end;

procedure WWindow.SetTop(Value : Integer);
begin
	MoveWindow(Handle, GetLeft, Value, GetWidth, GetHeight, True);
end;

procedure WWindow.OnInit(Sender : WObject ; var Return : WResult);
begin
    //This procedure is used for initializing application.
    //Please override here.
end;

procedure WWindow.OnKeyDown(Sender : WObject ; Key : Integer
                                            ; var Return : WResult);
begin
    //This procedure is used for initializing application.
    //Please override here.
end;

procedure WWindow.OnKeyUp(Sender : WObject ; Key : Integer
                                            ; var Return : WResult);
begin
    //This procedure is used for initializing application.
    //Please override here.
end;

procedure WWindow.OnMouseDown(Sender : WObject ; Button : Integer ; x : Integer
                                        ; y : Integer ; var Return : WResult);
begin
    //This procedure is used for initializing application.
    //Please override here.
end;

procedure WWindow.OnMouseMove(Sender : WObject ; Button : Integer ; x : Integer
                                ; y : Integer ; var Return : WResult);
begin
    //This procedure is used for initializing application.
    //Please override here.
end;

procedure WWindow.OnMouseUp(Sender : WObject ; Button : Integer ; x : Integer
                                        ; y : Integer ; var Return : WResult);
begin
    //This procedure is used for initializing application.
    //Please override here.
end;

procedure WWindow.OnShow(Sender : WObject ; var Return : WResult);
begin
    //This procedure is used for initializing application.
    //Please override here.
end;

procedure WWindow.OnDestroy(Sender : WObject ; var Return : WResult);
begin
    //This procedure is used for initializing application.
    //Please override here.
end;

procedure WWindow.OnCommand(Sender : WObject ; wParam : Integer
                                    ; lParam : Integer ; var Return : WResult);
begin
    //This procedure is used for initializing application.
    //Please override here.

    if ((wParam = IDOK) or (wParam = IDCANCEL)) and AutoClose then
    begin
        Close;
    end;
end;

procedure WWindow.OnClose(Sender : WObject ; var DoClose : Boolean);
begin
    //This procedure is used for initializing application.
    //Please override here.
end;

procedure WWindow.OnPaint(Sender : WObject ; var Return : WResult);
begin
    Return := DefWindowProc(Handle, WM_PAINT, 0, 0);
end;

procedure WWindow.OnClick(Sender : WObject ; var Return : WResult);
begin
    //This procedure is used for initializing application.
    //Please override here.
end;

procedure WWindow.DefaultHandler(var Message);
begin
    TMessage(Message).Result := DefWindowProc(Handle, TMessage(Message).Msg
                        , TMessage(Message).WParam, TMessage(Message).LParam);
end;

procedure WWindow.Hide;
begin
    if Handle = 0 then Exit;
    ShowWindow(Handle, SW_HIDE);
end;

procedure WWindow.Close;
begin
    //This procedure is used for initializing application.
    //Please override here.
end;

procedure WWindow.SetCenter;
var
    Rect : TRect;
begin
    GetWindowRect(GetDesktopWindow, Rect);
    Left := (Rect.Right - Rect.Left - Width + 1) div 2;
    Top := (Rect.Bottom - Rect.Top - Height + 1) div 2;
end;

//******************************************************************************
// WModalDialog

function ModalDialogProc(hDlg : WHandle ; uMsg : Cardinal ; wParam : Integer
                                        ; lParam : Integer) : Integer; stdcall;
var
    Window : WModalDialog;
    Return : WResult;
    Msg : TMessage;
    DoClose : Boolean;
begin
    Result := 0;
    Window := WModalDialog(ExtractWindow(hDlg));

    case uMsg of
        WM_INITDIALOG:
            begin
                if Window = nil then
                begin
                    if WindowQue.Count = 0 then Exit;
                    Window := Pointer(WindowQue.GetValue(0));
                    Window.Handle := hDlg;
                    WindowQue.Delete(0);
                    SetProp(hDlg, 'Handle', Cardinal(Window));
                    Window.OnInit(Window, Return);
                end;

                Result := Return;
            end;

        WM_KEYDOWN:
            begin
                Return := HRESULT(False);
                if Window <> nil then
                    Window.OnKeyDown(Window, wParam, Return);
                Result := Return;
            end;

        WM_KEYUP:
            begin
                Return := HRESULT(False);
                if Window <> nil then
                    Window.OnKeyUp(Window, wParam, Return);
                Result := Return;
            end;

        WM_LBUTTONDOWN, WM_RBUTTONDOWN:
            begin
                Return := HRESULT(False);
                if Window <> nil then
                    Window.OnMouseDown(Window, wParam, LOWORD(lParam)
                                        , HIWORD(lParam), Return);
                Result := Return;
            end;

        WM_MOUSEMOVE:
            begin
                Return := HRESULT(False);
                if Window <> nil then
                    Window.OnMouseMove(Window, wParam, LOWORD(lParam)
                                        , HIWORD(lParam), Return);
                Result := Return;
            end;

        WM_LBUTTONUP, WM_RBUTTONUP:
            begin
                Return := HRESULT(False);
                if Window <> nil then
                    Window.OnMouseUp(Window, wParam, LOWORD(lParam)
                                    , HIWORD(lParam), Return);
                Result := Return;
            end;

        WM_SHOWWINDOW:
            begin
                Return := HRESULT(False);
                if Window <> nil then
                    Window.OnShow(Window, Return);
                Result := Return;
                SetActiveWindow(hDlg);
            end;

        WM_DESTROY:
            begin
                RemoveProp(hDlg, 'Handle');
                Return := HRESULT(False);
                if Window <> nil then
                    Window.OnDestroy(Window, Return);
                Result := Return;
            end;

        WM_COMMAND:
            begin
                Return := HRESULT(False);
                if Window <> nil then
                    Window.OnCommand(Window, wParam, lParam, Return);
                Result := Return;
            end;

        WM_CLOSE:
            begin
                DoClose := True;
                if Window <> nil then
                    Window.OnClose(Window, DoClose);
                //if DoClose then EndDialog(hDlg, 0);
            end;

        else
            begin
                Msg.Msg := uMsg;
                Msg.WParam := wParam;
                Msg.LParam := lParam;
                Msg.Result := HRESULT(False);

                if Window <> nil then
                    Window.Dispatch(Msg);

                Result := Msg.Result;
            end;
    end;
end;

constructor WModalDialog.Create(AppObj : WApp ; Parent : WWindow ; ResID : String);
begin
    AutoClose := True;
    App := AppObj;
    ResourceID := ResID;

    if Parent <> nil then
        ParentHandle := Parent.Handle
    else
        ParentHandle := 0;
end;

constructor WModalDialog.Create(Parent : WHandle ; ResID : String);
begin
    AutoClose := True;
    ResourceID := ResID;
    ParentHandle := Parent;
end;

destructor WModalDialog.Destroy;
begin
    inherited Destroy;
end;

function WModalDialog.GetHandle(ID : Integer) : WHandle;
begin
    Result := GetDlgItem(Handle, ID);
end;

procedure WModalDialog.Close;
begin
    EndDialog(Handle, 0);
end;

procedure WModalDialog.DefaultHandler(var Message);
begin
    TMessage(Message).Result := 0;
end;

procedure WModalDialog.Show;
var
    Value : Integer;
begin
    WindowQue.Add(Integer(Pointer(Self)));
    Value := StrToInt(ResourceID);
    ReleaseCapture;

    if Value = 0 then
        DialogBox(hInstance, PChar(ResourceID), ParentHandle, @ModalDialogProc)
    else
        DialogBox(hInstance, MAKEINTRESOURCE(Value), ParentHandle, @ModalDialogProc);

    SetFocus(ParentHandle);
end;

procedure WModalDialog.ShowModal;
begin
    Show;
end;

//******************************************************************************
// WFrame

function WindowProc(hDlg : WHandle ; uMsg : Cardinal ; wParam : Integer
                                        ; lParam : Integer) : WResult; stdcall;
var
    Window : WFrame;
    Msg : TMessage;
    DoClose : Boolean;
begin
    Result := 0;
    Window := WFrame(ExtractWindow(hDlg));

    if Window = nil then
    begin
        Result := DefWindowProc(hDlg, uMsg, wParam, lParam);
        Exit;
    end;

    case uMsg of
        WM_CREATE:
            Window.OnInit(Window, Result);

        WM_KEYDOWN:
            Window.OnKeyDown(Window, wParam, Result);

        WM_KEYUP:
            Window.OnKeyUp(Window, wParam, Result);

        WM_LBUTTONDOWN, WM_RBUTTONDOWN:
            Window.OnMouseDown(Window, wParam, LOWORD(lParam)
                                    , HIWORD(lParam), Result);

        WM_MOUSEMOVE:
            Window.OnMouseMove(Window, wParam, LOWORD(lParam)
                                    , HIWORD(lParam), Result);

        WM_LBUTTONUP, WM_RBUTTONUP:
            Window.OnMouseUp(Window, wParam, LOWORD(lParam)
                                , HIWORD(lParam), Result);

        WM_SHOWWINDOW:
            Window.OnShow(Window, Result);

        WM_DESTROY:
            begin
                RemoveProp(hDlg, 'Handle');
                Window.OnDestroy(Window, Result);
            end;

        WM_COMMAND:
            begin
                Window.OnCommand(Window, wParam, lParam, Result);
            end;

        WM_CLOSE:
            begin
                DoClose := True;
                Window.OnClose(Window, DoClose);

                if DoClose then
                begin
                    Window.NowShown := False;
                end;
            end;

        WM_PAINT:
            begin
                Window.OnPaint(Window, Result);
            end;

        else
            begin
                Msg.Msg := uMsg;
                Msg.WParam := wParam;
                Msg.LParam := lParam;
                Msg.Result := HRESULT(False);
                Window.Dispatch(Msg);
                Result := Msg.Result;
            end;
    end;
end;

constructor WFrame.Create(AppObj : WApp ; Parent : WWindow);
const
    WINNAME     =   'TNakFrameWorks';
var
    ParentHandle : WHandle;
    WClass : WNDCLASS;
begin
    App := AppObj;
    NowShown := False;

    if Parent = nil then
        ParentHandle := 0
    else
        ParentHandle := Parent.Handle;

    if not GetClassInfo(hInstance, WINNAME, WClass) then
        UnregisterClass(WINNAME, hInstance);

    FillChar(WClass, SizeOf(WClass), #0);
    WClass.hInstance := hInstance;
    WClass.lpfnWndProc := @WindowProc;
    WClass.lpszClassName := WINNAME;
    WClass.hbrBackground := HBRUSH(COLOR_BTNFACE + 1);
    WClass.style := CS_OWNDC or CS_VREDRAW or CS_HREDRAW;
    WClass.hCursor := LoadCursor(0, IDC_ARROW);
    //WClass.cbClsExtra := 0;
    //WClass.cbWndExtra := 0;
    WClass.hIcon := AppObj.Icon;
    //WClass.lpszMenuName := nil;
    RegisterClass(WClass);

    Handle := CreateWindowEx(WS_EX_APPWINDOW or WS_EX_WINDOWEDGE
                                , WINNAME, nil
                                , WS_SYSMENU or WS_BORDER or WS_MAXIMIZEBOX
                                    or WS_MINIMIZEBOX or WS_SIZEBOX
                                , 0, 0, 0, 0, ParentHandle, 0, hInstance, nil);

    UnregisterClass(WINNAME, hInstance);
    SetProp(Handle, 'Handle', Cardinal(Self));
    ShowWindow(Handle, SW_HIDE);
    //SetWindowLong(Handle, GWL_WNDPROC, Longint(@WindowProc));
end;

constructor WFrame.Create(Parent : WHandle);
const
    WINNAME     =   'TNakFrameWorks';
var
    ParentHandle : WHandle;
    WClass : WNDCLASS;
    Style : Integer;
begin
    App := nil;
    NowShown := False;
    Style := WS_SYSMENU or WS_BORDER or WS_MAXIMIZEBOX
                or WS_MINIMIZEBOX or WS_SIZEBOX;

    if Parent = 0 then
        ParentHandle := 0
    else
    begin
        ParentHandle := Parent;
        Style := Style or WS_CHILD;
    end;

    if not GetClassInfo(hInstance, WINNAME, WClass) then
        UnregisterClass(WINNAME, hInstance);

    FillChar(WClass, SizeOf(WClass), #0);
    WClass.hInstance := hInstance;
    WClass.lpfnWndProc := @WindowProc;
    WClass.lpszClassName := WINNAME;
    WClass.hbrBackground := HBRUSH(COLOR_BTNFACE + 1);
    WClass.style := CS_OWNDC or CS_VREDRAW or CS_HREDRAW;
    WClass.hCursor := LoadCursor(0, IDC_ARROW);
    WClass.hIcon := 0;
    RegisterClass(WClass);

    Handle := CreateWindowEx(WS_EX_APPWINDOW or WS_EX_WINDOWEDGE
                                , WINNAME, nil, Style, 0, 0, 0, 0
                                , ParentHandle, 0, hInstance, nil);

    UnregisterClass(WINNAME, hInstance);
    SetProp(Handle, 'Handle', Cardinal(Self));
    ShowWindow(Handle, SW_HIDE);
    //SetWindowLong(Handle, GWL_WNDPROC, Longint(@WindowProc));
end;

destructor WFrame.Destroy;
begin
    if Handle <> 0 then
    begin
        RemoveProp(Handle, 'Handle');
        DestroyWindow(Handle);
        Handle := 0;
    end;

    inherited Destroy;
end;

procedure WFrame.Show;
begin
    if Handle = 0 then Exit;
    ShowWindow(Handle, SW_SHOW);
end;

procedure WFrame.ShowModal;
var
    msg : TMsg;
begin
    NowShown := True;
    ShowWindow(Handle, SW_SHOW);
    UpdateWindow(Handle);

    while GetMessage(msg, 0, 0, 0) and NowShown do
    begin
        TranslateMessage(msg);
        DispatchMessage(msg);
    end;

    //ReleaseCapture;
end;

procedure WFrame.Close;
begin
    if Handle = 0 then Exit;
    ShowWindow(Handle, SW_HIDE);
end;

//******************************************************************************

constructor WTrayIcon.Create(AppObj : WApp ; Parent : WWindow);
begin
    FTitle := '';
    FVisible := False;
    Icon := AppObj.Icon;

    inherited;
end;

destructor WTrayIcon.Destroy;
begin
    if FVisible then SetupIcon(NIM_DELETE);
    inherited;
end;

procedure WTrayIcon.PopupMenu(Menu : HMENU ; x : Integer ; y : Integer);
begin
    SetForegroundWindow(Handle);
    TrackPopupMenu(Menu, TPM_LEFTALIGN or TPM_RIGHTBUTTON
                        , x, y, 0, Handle, nil);
end;

procedure WTrayIcon.SetIcon(Value : HICON);
begin
    FIcon := Value;
    if FVisible then SetupIcon(NIM_MODIFY);
end;

procedure WTrayIcon.SetTitle(Value : String);
begin
    FTitle := Value;
    if FVisible then SetupIcon(NIM_MODIFY);
end;

procedure WTrayIcon.SetVisible(Value : Boolean);
begin
    if Value then
    begin
        if FVisible then Exit;
        SetupIcon(NIM_ADD);
        FVisible := True;
    end
    else
    begin
        if not FVisible then Exit;
        SetupIcon(NIM_DELETE);
        FVisible := False;
    end;
end;

procedure WTrayIcon.SetupIcon(Action : WTrayAction);
var
    nid : TNOTIFYICONDATA;
begin
    nid.cbSize := sizeof(TNOTIFYICONDATA);
    nid.uID := Handle;
    nid.Wnd := Handle;
    nid.uFlags := NIF_MESSAGE or NIF_ICON or NIF_TIP;
    nid.uCallbackMessage := WM_TRAY;
    nid.hIcon := FIcon;
    StrPCopy(nid.szTip, FTitle);
    Shell_NotifyIcon(Cardinal(Action), @nid);
end;

procedure WTrayIcon.OnTray(var Message: TMessage);
var
    Point : TPoint;
    Return : WResult;
begin
    GetCursorPos(Point);
    Return := TMessage(Message).Result;

    case TMessage(Message).LParam of
        WM_LBUTTONDOWN:
            OnMouseDown(Self, MK_LBUTTON, Point.X, Point.Y, Return);

        WM_LBUTTONUP:
            begin
                OnMouseUp(Self, MK_LBUTTON, Point.X, Point.Y, Return);
                OnClick(Self, Return);
            end;

        WM_RBUTTONDOWN:
            OnMouseDown(Self, MK_RBUTTON, Point.X, Point.Y, Return);

        WM_RBUTTONUP:
            OnMouseUp(Self, MK_RBUTTON, Point.X, Point.Y, Return);
    end;

    TMessage(Message).Result := Return;
end;

//******************************************************************************

function ControlProc(hDlg : WHandle ; uMsg : Cardinal ; wParam : Integer
                                        ; lParam : Integer) : WResult; stdcall;
var
    Control : WControl;
    Msg : TMessage;
begin
    Result := 0;
    Control := WControl(ExtractWindow(hDlg));

    if Control = nil then
    begin
        Exit;
    end;
    {
    Msg.Msg := uMsg;
    Msg.WParam := wParam;
    Msg.LParam := lParam;
    Msg.Result := HRESULT(False);
    Control.Dispatch(Msg);
    Result := Msg.Result;
    Exit;
    }
    case uMsg of
        WM_CREATE:
            Control.OnInit(Control, Result);

        WM_KEYDOWN:
            Control.OnKeyDown(Control, wParam, Result);

        WM_KEYUP:
            Control.OnKeyUp(Control, wParam, Result);

        WM_LBUTTONDOWN, WM_RBUTTONDOWN:
            Control.OnMouseDown(Control, wParam, LOWORD(lParam)
                                    , HIWORD(lParam), Result);

        WM_MOUSEMOVE:
            Control.OnMouseMove(Control, wParam, LOWORD(lParam)
                                    , HIWORD(lParam), Result);

        WM_LBUTTONUP, WM_RBUTTONUP:
            Control.OnMouseUp(Control, wParam, LOWORD(lParam)
                                , HIWORD(lParam), Result);

        WM_SHOWWINDOW:
            Control.OnShow(Control, Result);

        WM_DESTROY:
            begin
                RemoveProp(hDlg, 'Handle');
                Control.OnDestroy(Control, Result);
            end;

        WM_COMMAND:
            begin
                Control.OnCommand(Control, wParam, lParam, Result);
            end;

        WM_PAINT:
            begin
                Control.OnPaint(Control, Result);
            end;
    end;

    Msg.Msg := uMsg;
    Msg.WParam := wParam;
    Msg.LParam := lParam;
    Msg.Result := HRESULT(False);
    Control.Dispatch(Msg);
    Result := Msg.Result;
end;

constructor WControl.Create(Parent : WWindow);
begin
    Assigned := False;
end;

constructor WControl.Create(AssignHandle : WHandle);
begin
    if (AssignHandle = 0) or (Handle <> 0) then Exit;

    Assigned := True;
    Handle := AssignHandle;
    SetProp(Handle, 'Handle', Cardinal(Self));
    DefProc := Pointer(GetWindowLong(Handle, GWL_WNDPROC));
    SetWindowLong(Handle, GWL_WNDPROC, Integer(@ControlProc));
end;

destructor WControl.Destroy;
begin
    if Assigned then
    begin
        RemoveProp(Handle, 'Handle');
        SetWindowLong(Handle, GWL_WNDPROC, Integer(@DefProc));
        Handle := 0;
    end;
    
    inherited;
end;

procedure WControl.DefaultHandler(var Message);
begin
    if @DefProc <> nil then
        TMessage(Message).Result := DefProc(Handle, TMessage(Message).Msg
                        , TMessage(Message).WParam, TMessage(Message).LParam);
end;

procedure WControl.SetAlign(Value : WAlign);
begin

end;

procedure WControl.OnPaint(Sender : WObject ; var Return : WResult);
begin

end;

//******************************************************************************

constructor WImageList.Create(x : Integer ; y : Integer ; ColorType : Cardinal);
begin
    Width := x;
    Height := y;
    Handle := ImageList_Create(x, y, ColorType, 0, 0);

    MaskColor := RGB(255, 0, 255);
end;

destructor WImageList.Destroy;
begin
    ImageList_Destroy(Handle);
    inherited;
end;

procedure WImageList.LoadResBitmap(ResID : String);
var
    Value : Integer;
begin
    Value := StrToInt(ResID);

    if Value = 0 then
        ImageList_LoadBitmap(hInstance, PChar(ResID), Width, Height, MaskColor)
    else
        ImageList_LoadBitmap(hInstance, MAKEINTRESOURCE(Value)
                                        , Width, Height, MaskColor);
end;

//******************************************************************************

function ExtractWindow(Handle : WHandle) : WWindow;
begin
    Result := Pointer(GetProp(Handle, 'Handle'));
end;

//******************************************************************************

function Ctl_SetText(Handle : THandle ; Value : String) : Integer;
begin
    Result := SendMessage(Handle, WM_SETTEXT, 0, Integer(PChar(Value)));
end;

function Ctl_GetText(Handle : THandle) : String;
var
    Ln : Integer;
    Buf : String;
begin
    Ln := SendMessage(Handle, WM_GETTEXTLENGTH, 0, 0);
    SetLength(Buf, Ln + 1);
    SendMessage(Handle, WM_GETTEXT, Length(Buf), Integer(PChar(Buf)));
    Result := String(PChar(Buf));
end;

procedure Ctl_SetEnabled(Handle : THandle ; Value : Boolean);
begin
    //SendMessage(Handle, WM_ENABLE, Integer(Value), 0);
    EnableWindow(Handle, Value);
end;

//******************************************************************************

procedure LV_AddColumn(Handle : THandle ; Text : String ; Index, Size : Integer);
var
    Inf : LV_COLUMN;
begin
    FillChar(Inf, SizeOf(LV_COLUMN), #0);
    Inf.cchTextMax := Length(Text);
    Inf.cx := Size;
    Inf.fmt := LVCFMT_LEFT;
    Inf.iImage := 0;
    Inf.mask := LVCF_TEXT or LVCF_WIDTH or LVCF_FMT;
    Inf.pszText := PChar(Text);

    SendMessage(Handle, LVM_INSERTCOLUMN, Index, Integer(@Inf));
end;

procedure LV_SetImageList(Handle : THandle ; ImgList : HIMAGELIST);
begin
    SendMessage(Handle, LVM_SETIMAGELIST, LVSIL_SMALL, ImgList);
end;

procedure LV_AddItem(Handle : THandle ; var List : TNStringList ; ImgIndex : Integer);
var
    Item : LV_ITEM;
    Index : Integer;
    n : Integer;
begin
    Index := SendMessage(Handle, LVM_GETITEMCOUNT, 0, 0);

    for n := 0 to List.Count - 1 do
    begin
        FillChar(Item, SizeOf(LV_ITEM), #0);

        Item.cchTextMax := Length(List.Strings[n]);
        Item.iImage := ImgIndex;
        Item.iItem := Index;
        Item.iSubItem := n;
        Item.Mask := LVIF_TEXT or LVIF_IMAGE;
        Item.pszText := PChar(List.Strings[n]);

        if n = 0 then
            SendMessage(Handle, LVM_INSERTITEM, 0, Integer(@Item))
        else
            SendMessage(Handle, LVM_SETITEM, 0, Integer(@Item));
    end;
end;

procedure LV_SetStyle(Handle : THandle ; AdditionStyle : Cardinal);
var
    CurrentStyle : Cardinal;
begin
    CurrentStyle := ListView_GetExtendedListViewStyle(Handle) or AdditionStyle;
    ListView_SetExtendedListViewStyle(Handle, CurrentStyle);
end;

function LV_GetSelectedCount(Handle : THandle) : Integer;
begin
    Result := ListView_GetSelectedCount(Handle);
end;

function LV_GetIndex(Handle : THandle) : Integer;
begin
    Result := ListView_GetNextItem(Handle, -1, LVNI_SELECTED);
end;

function LV_GetItem(Handle : THandle ; Index : Integer ; SubIndex : Integer) : String;
var
    Size : Integer;
begin
    SetLength(Result, 4096);
    Size := ListView_GetItemText(Handle, Index, SubIndex, PChar(Result), 4096);
    SetLength(Result, Size);
end;

procedure LV_SetItem(Handle : THandle ; var List : TNStringList ; ImgIndex : Integer ; Index : Integer);
var
    Item : LV_ITEM;
    n : Integer;
begin
    for n := 0 to List.Count - 1 do
    begin
        FillChar(Item, SizeOf(LV_ITEM), #0);

        Item.cchTextMax := Length(List.Strings[n]);
        Item.iImage := ImgIndex;
        Item.iItem := Index;
        Item.iSubItem := n;
        Item.Mask := LVIF_TEXT or LVIF_IMAGE;
        Item.pszText := PChar(List.Strings[n]);

        SendMessage(Handle, LVM_SETITEM, 0, Integer(@Item))
    end;
end;

//******************************************************************************

initialization
    WindowQue := TNIntList.Create;

finalization
    WindowQue.Free;
end.
