unit tn_dns;

interface

uses
	windows, winsock, tn_socket, tn_utils, tn_classes;

type
    TDnsRequestHeader   =   packed record
        ID      : WORD;
        Flags   : WORD;
        QDCount : WORD;
        ANCount : WORD;
        NSCount : WORD;
        ARCount : WORD;
    end;

    PDnsRequestHeader   =   ^TDnsRequestHeader;

    TDnsItem            =   packed record
        QType : WORD;
        QClass : WORD;
        TTL : DWORD;
        RDataLen : WORD;
    end;

    PDnsItem            =   ^TDnsItem;

    TItemInfo           =   record
        IpAddr : array [0..7] of Byte;
        QType : Integer;
        QClass : Integer;
        TTL : LongInt;
        MxName : String;
        MXReference : Integer;
        PtrName : String;
    end;

    TMxInfo             =   record
        Count : Integer;
    end;

    TAInfo              =   record
        Count : Integer;
    end;

    TPtrInfo            =   record
        Count : Integer;
    end;

    TDnsInfo            =   record
        ResponseID : Integer;
        ResponseCode : Integer;
        ResponseOpCode : Integer;
        ResponseAuthoritative : Boolean;
        ResponseTruncation : Boolean;
        ResponseRecursionAvailable : Boolean;
        ResponseQDCount : Integer;
        ResponseANCount : Integer;
        ResponseNSCount : Integer;
        ResponseARCount : Integer;
        QuestionName : String;
        QuestionType : Integer;
        QuestionClass : Integer;
    end;

	TNDns	=	class(TNSocket)
        private
            FReqCount : Integer;
            FDnsServer : String;
            FItemCount : Integer;
            procedure MakeRequestHeader(Target : PDnsRequestHeader ; ID : WORD ; OPCode : BYTE ; Recursion : Boolean ; QDCount, ANCount, NSCount, ARCount : WORD);
            function MakeQuestion(Target : PChar; const QName : String ; QType, QClass : WORD) : Integer;
            procedure GetResponse;
        public
            AInfo : TAInfo;
            MxInfo : TMxInfo;
            PtrInfo : TPtrInfo;
            DnsInfo : TDnsInfo;
            Items : array of TItemInfo;
            constructor Create;
            function AskA(Domain : String) : Boolean;
            function AskMx(Domain : String) : Boolean;
            function AskPTR(IpAddr : String) : Boolean;
        published
            property DnsServer : String read FDnsServer write FDnsServer;
            property Count : Integer read FItemCount;
	end;

const
    DnsQueryVersion    = 102;

    { Maximum answers (responses) count }
    MAX_ANCOUNT     = 50;
    { Maximum number of MX records taken into account in responses }
    MAX_MX_RECORDS  = 50;
    MAX_A_RECORDS   = 50;
    MAX_PTR_RECORDS = 10;

    { DNS Classes }
    DnsClassIN      = 1;   { The internet                                      }
    DnsClassCS      = 2;   { The CSNET class (obsolete, used only for examples)}
    DnsClassCH      = 3;   { The CHAOS class                                   }
    DnsClassHS      = 4;   { Hesiod name service                               }
    DnsClassALL     = 255; { Any class                                         }

    { Type of query/response a DNS can handle }
    DnsQueryA       = 1;  { A     HostAddress                                  }
    DnsQueryNS      = 2;  { NS    Authoritative name server                    }
    DnsQueryMD      = 3;  { MD    MailDestination, obsolete, use Mail Exchange }
    DnsQueryMF      = 4;  { MF    MailForwarder, obsolete, use Mail Exchange   }
    DnsQueryCNAME   = 5;  { CNAME CanonicalName                                }
    DnsQuerySOA     = 6;  { SOA   Start of a Zone of Authority                 }
    DnsQueryMB      = 7;  { MB    MailBox, experimental                        }
    DnsQueryMG      = 8;  { MG    MailGroup, experimental                      }
    DnsQueryMR      = 9;  { MR    MailRename, experimental                     }
    DnsQueryNULL    = 10; { NULL  Experimental                                 }
    DnsQueryWKS     = 11; { WKS   Well Known Service Description               }
    DnsQueryPTR     = 12; { PTR   Domain Name Pointer                          }
    DnsQueryHINFO   = 13; { HINFO Host Information                             }
    DnsQueryMINFO   = 14; { MINFO Mailbox information                          }
    DnsQueryMX      = 15; { MX    Mail Exchange                                }
    DnsQueryTXT     = 16; { TXT   Text Strings                                 }

    { Some additional type only allowed in queries }
    DnsQueryAXFR    = 252; { Transfer for an entire zone                       }
    DnsQueryMAILB   = 253; { Mailbox related records (MB, MG or MR)            }
    DnsQueryMAILA   = 254; { MailAgent, obsolete, use MX instead               }
    DnsQueryALL     = 255; { Request ALL records                               }

    { Opcode field in query flags }
    DnsOpCodeQUERY  = 0;
    DnsOpCodeIQUERY = 1;
    DnsOpCodeSTATUS = 2;

    DNS_PORT        =   53;

implementation

//******************************************************************************
// TNDns

constructor TNDns.Create;
begin
    TimeOut := 10;
    FReqCount := 1;
    inherited Create(SOCK_DGRAM);
end;

procedure TNDns.MakeRequestHeader(Target : PDnsRequestHeader ; ID : WORD
                                    ; OPCode : BYTE ; Recursion : Boolean
                                    ; QDCount, ANCount, NSCount, ARCount : WORD);
begin
    if Target = nil then Exit;
    Target^.ID      := htons(ID);
    Target^.Flags   := htons((OpCode shl 11) + (Ord(Recursion) shl 8));
    Target^.QDCount := htons(QDCount);
    Target^.ANCount := htons(ANCount);
    Target^.NSCount := htons(NSCount);
    Target^.ARCount := htons(ARCount);
end;

function TNDns.MakeQuestion(Target : PChar; const QName : String
                                            ; QType, QClass : WORD) : Integer;
var
    p, qp : PChar;
    n : Integer;
    temp, stemp : String;
begin
    Result := 0;
    if Target = nil then Exit;
    if Length(QName) > 255 then Exit;

    p := Target + SizeOf(TDnsRequestHeader);
    qp := PChar(QName);
    stemp := '';
    n := 0;

    while n < Length(QName) do
    begin
        temp := '';

        while (n < Length(QName)) and (qp^ <> '.') do
        begin
            temp := temp + qp^;
            Inc(n);
            Inc(qp);
        end;

        stemp := stemp + Chr(Length(temp)) + temp;
        temp := '';
        Inc(n);
        Inc(qp);
    end;

    //Rs[
    Move(Pointer(PChar(stemp))^, Pointer(p)^, Length(stemp));
    Inc(p, Length(stemp));

    p^ := #0;
    Inc(p);
    PWORD(p)^ := htons(QType);
    Inc(p, 2);
    PWORD(p)^ := htons(QClass);

    Result := SizeOf(TDnsRequestHeader) + Length(stemp) + 5;
end;

procedure TNDns.GetResponse;
var
    buf : array [0..511] of Char;
    pdns : PDnsRequestHeader;
    p : PChar;
    ln : Integer;
    m, n : Integer;
    temp : String;
    chi : Integer;
    pItem : PDnsItem;
    Terminated : Boolean;
begin
    //Mf[^M
    FillChar(buf, SizeOf(buf), #0);
    Read(@buf, SizeOf(buf));

    //\Pbg
    Close;

    pdns := @buf;
    DnsInfo.ResponseID := ntohs(pdns.ID);
    DnsInfo.ResponseCode := pdns.Flags and $000F;
    DnsInfo.ResponseOpCode := (pdns.Flags shr 11) and $000F;
    DnsInfo.ResponseAuthoritative := (pdns.Flags and $0400) = $0400;
    DnsInfo.ResponseTruncation := (pdns.Flags and $200) = $200;
    DnsInfo.ResponseRecursionAvailable := (pdns.Flags and $0080) = $0080;
    DnsInfo.ResponseQDCount := ntohs(pdns.QDCount);
    DnsInfo.ResponseANCount := ntohs(pdns.ANCount);
    DnsInfo.ResponseNSCount := ntohs(pdns.NSCount);
    DnsInfo.ResponseARCount := ntohs(pdns.ARCount);

    if DnsInfo.ResponseQDCount = 0 then
    begin
        DnsInfo.QuestionName  := '';
        DnsInfo.QuestionType  := 0;
        DnsInfo.QuestionClass := 0;
        Exit;
    end;

    p := @buf;
    temp := '';
    Inc(p, SizeOf(TDnsRequestHeader));

    while True do
    begin
        ln := Byte(p^);
        if ln = 0 then Break;
        if temp <> '' then temp := temp + '.';
        Inc(p);
        n := 0;

        while n < ln do
        begin
            temp := temp + p^;
            Inc(p);
            Inc(n);
        end;
    end;

    Inc(p);
    DnsInfo.QuestionName := temp;
    DnsInfo.QuestionType  := ntohs(PWORD(p)^);
    Inc(p, 2);
    DnsInfo.QuestionClass := ntohs(PWORD(p)^);
    Inc(p, 2);

    if DnsInfo.ResponseANCount = 0 then
    begin
        MxInfo.Count := 0;
        AInfo.Count := 0;
        PtrInfo.Count := 0;
        FItemCount := 0;
        SetLength(Items, 0);
        Exit;
    end;

    FItemCount := DnsInfo.ResponseANCount;
    SetLength(Items, FItemCount);

    for n := 0 to DnsInfo.ResponseANCount - 1 do
    begin
        chi := Byte(p^);

        if (chi and $C0) = $C0 then
        begin
            Inc(p);
            //chi := ((chi and $3F) shl 8) + Byte(p^);
            Inc(p);

            pItem := Pointer(p);
            Items[n].QType := ntohs(pItem.QType);
            Items[n].QClass := ntohs(pItem.QClass);
            Items[n].TTL := ntohs(pItem.TTL);
            Inc(p, SizeOf(TDnsItem));

            case Items[n].QType of
                DnsQueryA:
                    begin
                        Items[n].IpAddr[0] := Byte(p^);
                        Inc(p);
                        Items[n].IpAddr[1] := Byte(p^);
                        Inc(p);
                        Items[n].IpAddr[2] := Byte(p^);
                        Inc(p);
                        Items[n].IpAddr[3] := Byte(p^);
                        Inc(p);
                    end;

                DnsQueryMX:
                    begin
                        Items[n].MXReference := ntohs(PWORD(p)^);
                        Inc(p, 2);

                        temp := '';
                        Terminated := False;

                        repeat
                            //MXGg̒
                            ln := Byte(p^);
                            if ln = 0 then
                            begin
                                Terminated := True;
                                Inc(p);
                                Break;
                            end;

                            if (temp <> '') and (ln <> 192) then
                                temp := temp + '.';

                            if ln <> 192 then
                            begin
                                Inc(p);
                                for m := 1 to ln do
                                begin
                                    temp := temp + p^;
                                    Inc(p);
                                end;
                            end;
                        until p^ = #192;

                        if temp = '' then
                            Items[n].MxName := DnsInfo.QuestionName
                        else if Terminated then
                        begin
                            Items[n].MxName := temp;
                            Continue;
                        end
                        else
                            Items[n].MxName := temp + '.' + DnsInfo.QuestionName;

                        Inc(p, 2);
                    end;

                DnsQueryPTR:
                    begin
                        temp := '';

                        repeat
                            ln := Byte(p^);
                            Inc(p);
                            if temp <> '' then temp := temp + '.';

                            for m := 1 to ln do
                            begin
                                temp := temp + p^;
                                Inc(p);
                            end;

                        until p^ = #0;

                        Items[n].PtrName := temp;
                    end;
            end;
        end
        else
            Break;
    end;
end;

function TNDns.AskA(Domain : String) : Boolean;
var
    Buf : array [0..511] of Char;
begin
    Result := False;
    FillChar(Buf, SizeOf(Buf), #0);
    MakeRequestHeader(@Buf, FReqCount, DnsOpCodeQuery, True, 1, 0, 0, 0);
    MakeQuestion(@Buf, Domain, DnsQueryA, DnsClassIN);
    if FReqCount > $FFFFFFF then FReqCount := 1;

    //ڑ
    if Connect(DnsServer, DNS_PORT) then
    begin
        Result := True;
        Write(@Buf, SizeOf(Buf));
        GetResponse;
    end
    else
        Close;
end;

function TNDns.AskMx(Domain : String) : Boolean;
var
    Buf : array [0..511] of Char;
begin
    Result := False;
    FillChar(Buf, SizeOf(Buf), #0);
    MakeRequestHeader(@Buf, FReqCount, DnsOpCodeQuery, True, 1, 0, 0, 0);
    MakeQuestion(@Buf, Domain, DnsQueryMX, DnsClassIN);
    if FReqCount > $FFFFFFF then FReqCount := 1;

    //ڑ
    if Connect(DnsServer, DNS_PORT) then
    begin
        Result := True;
        Write(@Buf, SizeOf(Buf));
        GetResponse;
    end
    else
        Close;
end;

function TNDns.AskPTR(IpAddr : String) : Boolean;
var
    Buf : array [0..511] of Char;

    function ReverseAddr : String;
    var
        List : TNStringList;
        n : Integer;
        temp : String;
    begin
        Result := '';
        List := TNStringList.Create;

        for n := 1 to Length(IpAddr) do
        begin
            if IpAddr[n] = '.' then
            begin
                List.Add(temp);
                temp := '';
            end
            else
                temp := temp + IpAddr[n];
        end;

        List.Add(temp);

        for n := List.Count - 1 downto 0 do
        begin
            if Result <> '' then Result := Result + '.';
            Result := Result + List.Strings[n];
        end;

        List.Free;
    end;

begin
    Result := False;
    FillChar(Buf, SizeOf(Buf), #0);
    MakeRequestHeader(@Buf, FReqCount, DnsOpCodeQuery, True, 1, 0, 0, 0);
    MakeQuestion(@Buf, ReverseAddr + '.in-addr.arpa', DnsQueryPTR, DnsClassIN);
    Inc(FReqCount);
    if FReqCount > $FFFFFFF then FReqCount := 1;

    //ڑ
    if Connect(DnsServer, DNS_PORT) then
    begin
        Result := True;
        Write(@Buf, SizeOf(Buf));
        GetResponse;
    end
    else
        Close;
end;

end.