/**
 * JPicosheet: Spreadsheet engine for Java Applications
 * Copyright (C) 2011 yusuke nishikawa
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */
package test;

import java.math.BigDecimal;

import jp.co.nissy.jpicosheet.core.Book;
import jp.co.nissy.jpicosheet.core.Cell;
import jp.co.nissy.jpicosheet.core.Element;
import jp.co.nissy.jpicosheet.core.Sheet;
import jp.co.nissy.jpicosheet.core.Element.ElementType;
import jp.co.nissy.jpicosheet.core.Element.ErrorType;
import junit.framework.TestCase;

public class FunctionTest extends TestCase {

	Book book;
	Sheet sheet;
	Cell cell;

	protected void setUp() throws Exception {
		super.setUp();
		book= new Book("myBook");
		sheet = book.addSheet("mySheet");
		cell = sheet.addCell("targetCell");
	}

	public void testNonexistenceFunction() {
		cell.setFormula("nonexistence(1,2,3");
		assertEquals(Cell.CellStatus.ERROR, cell.getStatus());
		assertEquals(Element.ElementType.ERROR, cell.getValue().getType());
		assertEquals(Element.ErrorType.INVALID_FORMULA, cell.getValue().getErrorType());
	}

	public void testFunctionNameCharacterCase() {
		cell.setFormula("sum(1, 2, 3)");
		assertEquals(new BigDecimal(1 + 2 + 3), cell.getValue().getNumber());
		cell.setFormula("SUM(1, 2, 3)");
		assertEquals(new BigDecimal(1 + 2 + 3), cell.getValue().getNumber());
		cell.setFormula("SUM(1, 2, 3)");
		assertEquals(new BigDecimal(1 + 2 + 3), cell.getValue().getNumber());

	}

	public void testFunctionInFunction() {
		cell.setFormula("sum(sum(1,1,1), 2, 3)");
		assertEquals(new BigDecimal("8"), cell.getValue().getNumber());
		cell.setFormula("sum(2,sum(1,1,1), 3)");
		assertEquals(new BigDecimal("8"), cell.getValue().getNumber());
		cell.setFormula("sum(2, 3, sum(1,1,1))");
		assertEquals(new BigDecimal("8"), cell.getValue().getNumber());

		cell.setFormula("sum(sum(), 2, 3)");
		assertEquals(new BigDecimal("5"), cell.getValue().getNumber());
		cell.setFormula("sum(2, sum(), 3)");
		assertEquals(new BigDecimal("5"), cell.getValue().getNumber());
		cell.setFormula("sum(2, 3, sum())");
		assertEquals(new BigDecimal("5"), cell.getValue().getNumber());

		cell.setFormula("sum(sum(sum(1),sum(1,2,3),sum(1,2,3)), 2, 3)");
		assertEquals(new BigDecimal("18"), cell.getValue().getNumber());
		cell.setFormula("(sum(sum((sum(1)),((sum((1),2,3))),sum(1,2,3)), 2, 3))");
		assertEquals(new BigDecimal("18"), cell.getValue().getNumber());
		cell.setFormula("( sum(sum(  ( sum( 1	 )	 ) ,(	  ( 	sum( ( 	1),	2 	 ,3))	),sum( 1 , 2  , 3 ) 	) , 2 , 3) )    ");
		assertEquals(new BigDecimal("18"), cell.getValue().getNumber());

	}

	public void testSum() {
		cell.setFormula("sum(1, 2, 3)");
		assertEquals(new BigDecimal(1 + 2 + 3), cell.getValue().getNumber());
	}


	public void testAverage() {
		cell.setFormula("average(2, 4, 10, 4)");
		assertEquals(new BigDecimal(5), cell.getValue().getNumber());
	}

	public void testABS() {
		cell.setFormula("abs(10)");
		assertEquals(new BigDecimal("10"), cell.getValue().getNumber());
		cell.setFormula("abs(-10)");
		assertEquals(new BigDecimal("10"), cell.getValue().getNumber());
		cell.setFormula("abs(0.00001)");
		assertEquals(new BigDecimal("0.00001"), cell.getValue().getNumber());
		cell.setFormula("abs(-0.00001)");
		assertEquals(new BigDecimal("0.00001"), cell.getValue().getNumber());
		cell.setFormula("abs(10-100)");
		assertEquals(new BigDecimal("90"), cell.getValue().getNumber());
	}

	public void testMOD() {
		cell.setFormula("mod(10, 3)");
		assertEquals(new BigDecimal("1"), cell.getValue().getNumber());
		cell.setFormula("mod(10, -3)");
		assertEquals(new BigDecimal("-2"), cell.getValue().getNumber());
	}


	public void testINT() {
		cell.setFormula("int(1.3)");
		assertEquals(new BigDecimal("1"), cell.getValue().getNumber());
		cell.setFormula("int(-1.3)");
		assertEquals(new BigDecimal("-2"), cell.getValue().getNumber());
	}

	public void testIF() {
		cell.setFormula("if(true, 10, 20)");
		assertEquals(new BigDecimal("10"), cell.getValue().getNumber());
		cell.setFormula("if(false, 10, 20)");
		assertEquals(new BigDecimal("20"), cell.getValue().getNumber());

		cell.setFormula("if(true = true, 10, 20)");
		assertEquals(new BigDecimal("10"), cell.getValue().getNumber());
		cell.setFormula("if(true <> true, 10, 20)");
		assertEquals(new BigDecimal("20"), cell.getValue().getNumber());

		Cell c1 = sheet.addCell("cell1").setNumberValue("100");
		Cell c2 = sheet.addCell("cell2").setFormula("cell1 < 150");
		cell.setFormula("if(cell2, 10, 20)");
		assertEquals(new BigDecimal("10"), cell.getValue().getNumber());
		c1.setNumberValue("200");
		cell.setFormula("if(cell2, 10, 20)");
		assertEquals(new BigDecimal("20"), cell.getValue().getNumber());

		cell.setFormula("if(cell2, \"small\", \"large\")");
		assertEquals("large", cell.getString());

	}

	public void testAND() {
		cell.setFormula("and(true)");
		assertEquals(true, cell.getValue().getBoolean().booleanValue());
		cell.setFormula("and(true,true)");
		assertEquals(true, cell.getValue().getBoolean().booleanValue());
		cell.setFormula("and(true,true,true,true,true,true)");
		assertEquals(true, cell.getValue().getBoolean().booleanValue());
		cell.setFormula("and(false)");
		assertEquals(false, cell.getValue().getBoolean().booleanValue());
		cell.setFormula("and(false,false)");
		assertEquals(false, cell.getValue().getBoolean().booleanValue());
		cell.setFormula("and(false,false,false,false,false)");
		assertEquals(false, cell.getValue().getBoolean().booleanValue());
		cell.setFormula("and(true,false)");
		assertEquals(false, cell.getValue().getBoolean().booleanValue());
		cell.setFormula("and(false,true)");
		assertEquals(false, cell.getValue().getBoolean().booleanValue());
		cell.setFormula("and(true,false,true)");
		assertEquals(false, cell.getValue().getBoolean().booleanValue());

		cell.setFormula("and(10=10,1<5,-4<>4)");
		assertEquals(true, cell.getValue().getBoolean().booleanValue());

		cell.setFormula("and(123)");
		assertEquals(ElementType.ERROR, cell.getValue().getType());
		assertEquals(ErrorType.INVALID_PARAMETER, cell.getValue().getErrorType());
		cell.setFormula("and(1)");
		assertEquals(ElementType.ERROR, cell.getValue().getType());
		assertEquals(ErrorType.INVALID_PARAMETER, cell.getValue().getErrorType());
	}

	public void testOR() {
		cell.setFormula("or(true)");
		assertEquals(true, cell.getValue().getBoolean().booleanValue());
		cell.setFormula("or(true,true)");
		assertEquals(true, cell.getValue().getBoolean().booleanValue());
		cell.setFormula("or(true,true,true,true,true,true)");
		assertEquals(true, cell.getValue().getBoolean().booleanValue());
		cell.setFormula("or(false)");
		assertEquals(false, cell.getValue().getBoolean().booleanValue());
		cell.setFormula("or(false,false)");
		assertEquals(false, cell.getValue().getBoolean().booleanValue());
		cell.setFormula("or(false,false,false,false,false)");
		assertEquals(false, cell.getValue().getBoolean().booleanValue());
		cell.setFormula("or(true,false)");
		assertEquals(true, cell.getValue().getBoolean().booleanValue());
		cell.setFormula("or(false,true)");
		assertEquals(true, cell.getValue().getBoolean().booleanValue());
		cell.setFormula("or(true,false,true)");
		assertEquals(true, cell.getValue().getBoolean().booleanValue());

		cell.setFormula("or(10=10,1>5,-4=4)");
		assertEquals(true, cell.getValue().getBoolean().booleanValue());

		cell.setFormula("or(123)");
		assertEquals(ElementType.ERROR, cell.getValue().getType());
		assertEquals(ErrorType.INVALID_PARAMETER, cell.getValue().getErrorType());
		cell.setFormula("or(1)");
		assertEquals(ElementType.ERROR, cell.getValue().getType());
		assertEquals(ErrorType.INVALID_PARAMETER, cell.getValue().getErrorType());
	}

	public void testNOT() {
		cell.setFormula("not(true)");
		assertEquals(false, cell.getValue().getBoolean().booleanValue());
		cell.setFormula("not(false)");
		assertEquals(true, cell.getValue().getBoolean().booleanValue());
	}

}
