<?php
/*
 * This file is part of INQMAN
 *
 * Copyright(c) 2008 BULLHEAD,INC. ALL RIGHTS RESERVED.
 *
 * http://www.bullhead.co.jp/
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation;
 * either version 3 of the License, or (at your option) any later
 * version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

class Inqman_MailSender
{
    
    /**
     * ロギングオブジェクト
     * @var Bullhead_Log
     */
    protected $_logger;
    
    /**
     * Configオブジェクト
     */
    protected $_config;
    
    /**
     * コンストラクタ
     */
    public function __construct()
    {
        $this->_setupLogger();
        
        $hostname   = $this->_config->smtp->host;
        $portnumber = $this->_config->smtp->port;
        
        if (empty($hostname)) {
            throw new Exception("smtp host not config");
        }
        
        if (empty($portnumber)) {
            $portnumber = 25;
        }
        
        $tr = new Zend_Mail_Transport_Smtp($hostname, array('port' => $portnumber));        
        Zend_Mail::setDefaultTransport($tr);
    }
        
    /**
     * 同報メールを送信する。
     * 
     * 配列で指定されたメールアドレスに対して、同一内容のメールを一斉配信する。
     * 
     * @param array $toList 宛先メールアドレスの配列
     * @param array $headers To以外のヘッダ情報（Subject, From, Priority）
     * 
     */
    public function sendBroadcastMail($to_list, $headers, $body)
    {
        $charset  = 'UTF-8';//'ISO-2022-JP';
        $encoding = 'base64';//'quoted-printable';

        $from      = $headers['from'];
        //$from_name = mb_encode_mimeheader($headers['from_name'], $charset);
        $from_name = strtr(mb_encode_mimeheader($headers['from_name'], $charset), array("\r" => '', "\n" => ''));
        $subject   = strtr(mb_encode_mimeheader($headers['subject'], $charset), array("\r"=>'', "\n"=>''));
        
        foreach ((array) $to_list as $to) {
            $mail = new Zend_Mail($charset);
            $mail->setFrom($from, $from_name);
            $mail->setSubject($subject);
            $mail->setBodyText(mb_convert_encoding($body, $charset, mb_detect_encoding($body)), null, $encoding);
            $mail->addTo($to);
            $this->_logger->info('Inqman_MailSender->sendBroadcastMail[To:'.$to.'] [FROM:'.$from.'] [SUBJECT:'.$subject.']');

            try {
                $mail->send();
            }catch (Exception $e) {
                $this->_logger->warn($e->getMessage());
            }
        }
    }
    
    /**
     * Configオブジェクトを設定する
     */
    protected function _setupConfig()
    {
        $config = new Zend_Config_Ini(MOD_INQMAN_CONFIGS_DIR . '/config.ini.php', APPLICATION_RUNNING_MODE);
        $this->_config = $config;
    }
    
	/**
	 * ロギングオブジェクトを設定する
	 * 
	 */
    protected function _setupLogger()
    {
        $this->_setupConfig();
        $log_config   = $this->_config->log_files->mail_log;
        $save_dir     = $log_config->save_dir;
        $output_level = intval($log_config->output_level);
        
        $logger = new Bullhead_Log();
        $logger->addWriter(new Zend_Log_Writer_Stream(ROOT_DIR . $save_dir . 'mail.log'));
        $logger->addFilter(new Zend_Log_Filter_Priority($output_level));
        $this->_logger = $logger;
    }
    
}
