<?php
/*
 * This file is part of INQMAN
 *
 * Copyright(c) 2008 BULLHEAD,INC. ALL RIGHTS RESERVED.
 *
 * http://www.bullhead.co.jp/
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation;
 * either version 3 of the License, or (at your option) any later
 * version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

require_once 'Zend/Validate.php';
require_once 'Zend/Validate/StringLength.php';
require_once 'Zend/Validate/Regex.php';
require_once 'Zend/Validate/NotEmpty.php';
require_once 'Zend/Validate/Int.php';
require_once 'Zend/Validate/EmailAddress.php';
require_once 'Zend/Validate/Alnum.php';
require_once 'Bullhead/Validate/StringEquals.php';

require_once 'Zend/Filter/StringToLower.php';
require_once 'Zend/Filter/StringToUpper.php';
require_once 'Zend/Filter/StringTrim.php';
require_once 'Bullhead/Filter/ZenToHan.php';

/**
 * 
 * 
 */
abstract class AbstractForm
{
    /**
     * フィルタチェイン
     */
    protected $_filters;
    
    /**
     * バリデータチェイン
     */
    protected $_validators;
    
    /**
     * フィルタ＆バリデータチェイン
     */
    protected $_filter_input;
    
    /**
     * バリデータ＆フィルタ済みの値
     */
    protected $_values;
    
    /**
     * フィルタ未実施の値
     */
    protected $_unfiltered_values;
    
    /**
     * メッセージオブジェクト
     */
    protected $_messages;
    
    /**
     * 翻訳情報オブジェクト
     */
    protected $_translator;
    
    /**
     * 設定情報オブジェクト
     */
    protected $_config;
    
    /**
     * コンストラクタ
     */
    public function __construct()
    {
        //設定ファイルを読み込む
        $this->_config     = $this->_prepareConfig();
        
        //フィルタ＆バリデータチェインを設定する
        $this->_filters    = $this->_prepareFilters();
        $this->_validators = $this->_prepareValidators();
    }
    
    /**
     * 入力値の検証（Validate）を行う
     */
    public function isValid($data = null)
    {
        $valid = false;
        $this->_prepare($data);
        
        if ($this->_filter_input->isValid()) {
            $this->_values = $this->_filter_input->getUnescaped();
            $valid = true;
        } else {
            $this->_messages = $this->_filter_input->getMessages();
        }
        return $valid;
    }
    
    /**
     * メッセージを取得する
     * 
     * @access public
     * @return array メッセージを配列で返す
     */
    public function getMessages()
    {
        return $this->_messages;
    }
    
    /**
     * 指定されたフィールドのフィルタ未実施の値を返す
     * 
     * @access public
     * @param string $field_name 取得する値のフィールド名
     * @return 指定されたフィールドの値を返す
     */
    public function getUnfilteredValue($field_name)
    {
        if (isset($this->_unfiltered_values[$field_name])) {
            return $this->_unfiltered_values[$field_name]; 
        }
        return null;
    }
    
    /**
     * 全てのフィールドのフィルタ未実施の値を返す
     * 
     * @access public
     * @return array フィールドの値を配列で返す
     */
    public function getUnfilteredValues()
    {
        return $this->_unfiltered_values;
    }
    
    /**
     * 指定されたフィールドのフィルタ済みの値を返す
     * 
     * @access public
     * @param string $field_name
     * @return 指定されたフィールドの値を返す
     */
    public function getValue($field_name)
    {
        if (isset($this->_values[$field_name])) {
            return $this->_values[$field_name]; 
        }
        return null;
    }
    
    /**
     * 全てのフィールドのフィルタ済みの値を返す
     * 
     * @access public
     * @return array 全てのフィールドの値を配列で返す
     */
    public function getValues()
    {
        return $this->_values;
    }
    
    /**
     * 指定のフィールドに値を格納する
     * 
     * @access public
     * @param string $field_name フィールド名
     * @param mixed $value 値
     */
    public function setValue($field_name, $value)
    {
        $this->_values[$field_name] = $value;
    }
    
    /**
     * 翻訳オブジェクトを設定する
     */
    public function setTranslator($translator)
    {
        $this->_translator = $translator;
    }
    
    /**
     * バリデータチェインオブジェクトを取得する
     * 
     * @access public
     * @return array バリデータチェインオブジェクト
     */
    public function getValidators() {
        return $this->_validators;
    }
    
    /**
     * バリデータチェインオブジェクトを格納する
     */
    public function setValidators($validators) {
        $this->_validators = $validators;
    }
    
    /**
     * フィルタチェインオブジェクトを取得する
     */
    public function getFilters() {
        return $this->_filters;
    }
    
    /**
     * フィルタチェインオブジェクトを格納する
     */
    public function setFilters($filters) {
        $this->_filters = $filters;
    }
    
    public function prepareModifyForm() {/**/}
    
    /** Local Method **/
    
    /**
     * 入力値を格納し、フィルタ＆バリデータチェインに設定する
     * 
     * @access protected
     * @param array $data フォーム入力値
     */
    protected function _prepare($data)
    {
        $this->_unfiltered_values = $data;      
        $this->_filter_input = new Zend_Filter_Input($this->_filters, $this->_validators);
        $this->_filter_input->setData($data);
    }
    
    /**
     * 設定ファイルを読み込む
     */
    protected function _prepareConfig()
    {
        $config = new Zend_Config_Ini(MOD_INQMAN_CONFIGS_DIR . '/validate.ini.php', 'form');
        return $config;
    }
    
    /**
     * バリデータチェインを設定する
     * 
     * @access protected
     */
    abstract protected function _prepareValidators();
    
    /**
     * フィルタチェインを設定する
     * 
     * @access protected
     */
    abstract protected function _prepareFilters();

}

?>
