<?php
/*
 * This file is part of INQMAN
 *
 * Copyright(c) 2008 BULLHEAD,INC. ALL RIGHTS RESERVED.
 *
 * http://www.bullhead.co.jp/
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation;
 * either version 3 of the License, or (at your option) any later
 * version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

define('ACL_KEY_INQUIRY_VIEW',            'INQUIRY_VIEW');
define('ACL_KEY_COMMENT_VIEW',            'COMMENT_VIEW');
define('ACL_KEY_INQUIRY_COMMENT',         'INQUIRY_COMMENT');
define('ACL_KEY_REPLY_VIEW',              'REPLY_VIEW');
define('ACL_KEY_INQUIRY_TAKECHARGE',      'INQUIRY_TAKECHARGE');
define('ACL_KEY_INQUIRY_REPLY',           'INQUIRY_REPLY');
define('ACL_KEY_INQUIRY_DERIVATION',      'INQUIRY_DERIVATION');
define('ACL_KEY_INQUIRY_REFERENCE',       'INQUIRY_REFERENCE');
define('ACL_KEY_UNACCEPT_VIEW',           'UNACCEPT_VIEW');
define('ACL_KEY_UNACCEPT_ACCEPT',         'UNACCEPT_ACCEPT');
define('ACL_KEY_MAILTEMPLATE_MANAGEMENT', 'MAILTEMPLATE_MANAGEMENT');
define('ACL_KEY_CUSTOMER_MANAGEMENT',     'CUSTOMER_MANAGEMENT');
define('ACL_KEY_UNAPPROVAL_VIEW',         'UNAPPROVAL_VIEW');
define('ACL_KEY_UNAPPROVAL_APPROVE',      'UNAPPROVAL_APPROVE');
define('ACL_KEY_EVENT_MANAGEMENT',        'EVENT_MANAGEMENT');
define('ACL_KEY_HISTORY_MANAGEMENT',      'HISTORY_MANAGEMENT');

define('REQUEST_KEY_SERVICE_ID',          'service_id');
define('REQUEST_KEY_INQUIRY_ID',          'inquiry_id');
define('REQUEST_KEY_REPLY_ID',            'reply_id');
define('REQUEST_KEY_PARENT_INQUIRY_ID',   'parent_inquiry_id');
define('REQUEST_KEY_PARENT_REPLY_ID',     'parent_id');
define('REQUEST_KEY_ATTACH_FILES',        'attach_files');

define('VIEW_KEY_STATUS_UNACCEPT',        'unaccept');
define('VIEW_KEY_STATUS_UNAPPROVAL',      'unapproval');
define('VIEW_KEY_STATUS_UNSOLVED',        'unsolved');
define('VIEW_KEY_STATUS_REJECT',          'reject');
define('VIEW_KEY_STATUS_UNASSIGN',        'unassign');
define('VIEW_KEY_STATUS_PROCESS',         'process');
define('VIEW_KEY_STATUS_SUSPEND',         'suspend');
define('VIEW_KEY_STATUS_SOLVED',          'solved');

define('VIEW_KEY_SERVICES',               'services');
define('VIEW_KEY_INQUIRIES',              'inquiries');
define('VIEW_KEY_REPLIES',                'replies');
define('VIEW_KEY_REPLY_COUNT',            'reply_count');
define('VIEW_KEY_COMMENT_COUNT',          'comment_count');
define('VIEW_KEY_OPTIONS_SERVICE',        'service_options');
define('VIEW_KEY_OPTIONS_STATUS',         'status_options');
define('VIEW_KEY_OPTIONS_PRIORITY',       'priority_options');
define('VIEW_KEY_PARENT_DATA',            'parent_data');



/**
 * 問い合わせ管理画面のコントローラークラス
 *
 * @author 
 * @copyright COPYRIGHT &copy 2008 BULLHEAD,INC. ALL RIGHTS RESERVED.
 * @version 
 * @package 
 * @see AbstractController
 */
class Inqman_InquiryController extends AbstractController
{
    /** Public Methods **/
    
    /**
     * インデックスページ（マイページ）を表示する
     * 
     * @access public
     *
     */
    public function indexAction()
    {
        $this->_redirect('/');
    }
    
    /**
     * POP3サーバからメールを受信して登録する
     * 
     */
    public function receiveAction()
    {
        //ページタイトルの設定
        $this->_setupPageTitle('receive', 'inquiry');
        
        //リクエストパラメータからサービスIDを取得
        $service_id = $this->getRequest()->getParam(Inqman_InquiryForm::SERVICE_ID);
        
        //サービスIDからサービス情報を取得
        $service_model = new Inqman_ServiceModel();
        $service = $service_model->getOneService($service_id);
        $this->view->assign('service', $service);
        
        //アクセス権限のチェック
        $this->_checkAcl(ACL_RESOURCE_KEY_INQUIRY, ACL_PRIVILEGE_KEY_ACCEPT, $service_id);
        
        $pop_server = new Inqman_PopServer($service_id);
        $result = $pop_server->perform();
        
        $this->view->assign('inquiry_count', $result['inquiry_count']);
        $this->view->assign('reply_count', $result['reply_count']);
        $this->view->assign('comment_count', $result['comment_count']);
    }  
    
    /**
     * 問い合わせ情報入力（起票）画面を表示する。
     * 
     * @access public
     */
    public function createinputAction()
    {
        //ロジックインスタンスの生成
        $inquiry_model = new Inqman_InquiryModel();
        $service_model = new Inqman_ServiceModel();
        
        //ページタイトルの設定
        $this->_setupPageTitle('create_input', 'inquiry');
        
        //リクエストパラメータからサービスIDを取得し、サービス情報を取得
        $service_id = $this->getRequest()->getParam(Inqman_InquiryForm::SERVICE_ID);
        $service = $service_model->getOneService($service_id);
        
        //アクセス権限のチェック
        $this->_checkAcl(ACL_RESOURCE_KEY_INQUIRY, ACL_PRIVILEGE_KEY_ACCEPT, $service_id);
        
        //フォームの初期入力値を設定
        if (!empty($_POST)) {
            $form_input = $_POST;
        } else {
            $form_input = array(
                Inqman_InquiryForm::ID            => 0,
                Inqman_InquiryForm::SERVICE_ID    => $service_id,
                Inqman_InquiryForm::PRIORITY      => Inqman_InquiryModel::PRIORITY_CODE_MIDDLE,
                Inqman_InquiryForm::SUBJECT       => "",
                Inqman_InquiryForm::RECEIVED_FROM => "",
                Inqman_InquiryForm::CONTENT       => "",
                Inqman_InquiryForm::PARENT_INQ_ID => 0,
            );
        }
        
        //OPTION情報を取得
        $priority_options = $inquiry_model->getPriorityOptions();
        
        //登録機能の固有情報をVIEWに設定する
        $this->_assignCreateActionValue();
        
        //表示情報をVIEWに設定する
        $this->view->assign(VIEW_KEY_FORM_INPUT, $form_input);
        $this->view->assign('service', $service);
        $this->view->assign('priority_options', $priority_options);

        $this->_helper->viewRenderer->setRender('input');
    }
    
    /**
     * 問い合わせ情報入力（起票）確認画面を表示する。
     * 
     * @access public
     */
    public function createconfirmAction()
    {
        //ロジックインスタンスの初期化
        $inquiry_model = new Inqman_InquiryModel();
        $service_model = new Inqman_ServiceModel();
        
        //ページタイトルの設定
        $this->_setupPageTitle('create_confirm', 'inquiry');
        
        //リクエストパラメータからサービスIDを取得しサービス情報を取得
        $service_id = $this->getRequest()->getParam(Inqman_InquiryForm::SERVICE_ID);
        $service = $service_model->getOneService($service_id);
        
        //アクセス権限のチェック
        $this->_checkAcl(ACL_RESOURCE_KEY_INQUIRY, ACL_PRIVILEGE_KEY_ACCEPT, $service_id);        
        
        //フォーム入力値の検証
        $form = new Inqman_InquiryForm();
        $error_messages = null;
        $tmp_names = null;
        if(!$form->isValid($_POST)) {
            $error_messages = $form->getMessages();
        }
        
        //添付ファイルの検証と保存
        $attach_files = null;
        $attach_error_messages = $this->_validateUploadFiles($_FILES, REQUEST_KEY_ATTACH_FILES);
        if (!empty($attach_error_messages)) {
            $error_messages = array_merge($error_messages, $attach_error_messages);
        } else {
            $attach_files = $this->_saveUploadFiles($_FILES, REQUEST_KEY_ATTACH_FILES);
        }
        
        //エラーメッセージの有無で表示するページを振り分け
        if(!empty($error_messages)) {
            $form_input = $form->getUnfilteredValues();
            $this->_helper->viewRenderer->setRender('input');
        } else {
            $form_input = $form->getValues();
            $form_input[REQUEST_KEY_ATTACH_FILES] = $attach_files;
            $this->_helper->viewRenderer->setRender('confirm');
        }
        
        //OPTION情報を取得
        $priority_options = $inquiry_model->getPriorityOptions();
        
        //登録時特有の表示情報をVIEWに設定
        $this->_assignCreateActionValue();
        
        //表示情報をVIEWに設定
        $this->view->assign(VIEW_KEY_FORM_INPUT, $form_input);
        $this->view->assign('service', $service);
        $this->view->assign('priority_options', $priority_options);
        $this->view->assign(VIEW_KEY_ERROR_MESSAGES, $error_messages);
    }
    
    /**
     * 問い合わせ情報入力（起票）処理を実行する。
     * 
     */
    public function createcommitAction()
    {
        //ロジックインスタンスの生成
        $inquiry_model = new Inqman_InquiryModel();
        $service_model = new Inqman_ServiceModel();
        
        //ページタイトルの設定
        $this->_setupPageTitle('create_commit', 'inquiry');
        
        //リクエストパラメータからサービスIDを取得
        $service_id = $this->getRequest()->getParam(Inqman_InquiryForm::SERVICE_ID);
        $service = $service_model->getOneService($service_id);
        
        //アクセス権限のチェック
        $this->_checkAcl(ACL_RESOURCE_KEY_INQUIRY, ACL_PRIVILEGE_KEY_ACCEPT, $service_id);
        
        //フォーム入力値の検証
        $form = new Inqman_InquiryForm();
        $error_messages = null;
        if(!$form->isValid($_POST)) {
            $error_messages = $form->getMessages();
        }
        
        if(empty($error_messages)) {  
            //問い合わせ情報の登録
            $attach_files = !empty($_POST[REQUEST_KEY_ATTACH_FILES]) ? (array) $_POST[REQUEST_KEY_ATTACH_FILES] : array();
            $enced_attach_files = $this->_encodeUploadFiles($attach_files);
            
            $status = Inqman_InquiryModel::STATUS_CODE_UNAPPROVAL;
            //承認フローが無効の場合は、状態を「割当待ち」にする
            if (intval($this->_config->workflow->approve->inquiry->accept) != 1) {
                $status = Inqman_InquiryModel::STATUS_CODE_UNASSIGN;
            }
            
            $now = date("Y-m-d H:i:s", time());
            $values = array(
                    Inqman_InquiryModel::SUBJECT         => $form->getValue(Inqman_InquiryForm::SUBJECT),
                    Inqman_InquiryModel::RECEIVED_FROM   => $form->getValue(Inqman_InquiryForm::RECEIVED_FROM),
                    Inqman_InquiryModel::CONTENT         => $form->getValue(Inqman_InquiryForm::CONTENT),
                    Inqman_InquiryModel::SERVICE_ID      => $form->getValue(Inqman_InquiryForm::SERVICE_ID),
                    Inqman_InquiryModel::PRIORITY        => $form->getValue(Inqman_InquiryForm::PRIORITY),
                    Inqman_InquiryModel::STATUS          => $status,
                    Inqman_InquiryModel::REGISTRANT_ID   => $this->_getUserInfo()->id,
                    Inqman_InquiryModel::ACCEPTER_ID     => $this->_getUserInfo()->id,
                    Inqman_InquiryModel::CREATE_DATETIME => $now,
                    Inqman_InquiryModel::ACCEPT_DATETIME => $now,
            );
            
            $inquiry_id = $inquiry_model->createInquiry($values, $enced_attach_files);
                
            //承認フローが無効の場合は、同時に承認処理も行う
            if (intval($this->_config->workflow->approve->inquiry->accept) != 1) {
                $values['approver_id'] = $this->_getUserInfo()->id;
                $inquiry_model->approveInquiry($inquiry_id, $values);
            }
            
            $event_action_id = 4;
            $criteria = array('service_id'=>$service_id, 'inquiry_id'=>$inquiry_id);
            $inquiry_model->sendMail($event_action_id, $criteria);
            
            $this->view->assign('service', $service);
            $this->_helper->viewRenderer->setRender('commit');
        }
        //エラーがあった場合はメッセージをセットして、入力画面を再表示
        else {
            $this->view->assign(VIEW_KEY_ERROR_MESSAGES, $error_messages);
            $this->view->assign(VIEW_KEY_FORM_INPUT, $form->getUnfilteredValues());
            $this->_helper->viewRenderer->setRender('input');
        }
        
        //登録時特有の表示情報をVIEWに設定
        $this->_assignCreateActionValue();
    }
    
    /**
     * 全てのサービスの問い合わせ一覧画面を表示する
     * 
     * @access public
     * 
     */
    public function viewallAction()
    {
        //ページタイトルを設定する
        $this->_setupPageTitle('view_all', 'inquiry');
        
        $service_model = new Inqman_ServiceModel();
        $service_options = $service_model->getEnableServiceOptions($this->_getUserInfo()->id);
        
        //リクエストパラメータからサービスIDを取得
        $service_id = $this->getRequest()->getParam(Inqman_InquiryForm::SERVICE_ID);
        
        //サービスIDからサービス情報を取得
        $service = $service_model->getOneService($service_id);
        $this->view->assign('service', $service);
    
        //アクセス権限のチェック
        $this->_checkAcl(ACL_RESOURCE_KEY_INQUIRY, ACL_PRIVILEGE_KEY_READ, $service_id);
        
        //リクエストパラメータから検索条件を取得
        $criteria = $this->getRequest()->getParam('crtr');
        $this->view->assign('crtr', $criteria);
        
        $pname = "crtr";
        $paramStr = "";
        foreach ((array)$criteria as $key => $value) {
            if (is_array($value)) {
                foreach ($value as $k => $v) {
                    $k = urlencode("{$pname}[{$key}][{$k}]");
                    $v = urlencode($v);
                    
                    $paramStr .= "&{$k}={$v}";
                }
            } else {
                $key = urlencode("{$pname}[{$key}]");
                $value = urlencode($value);
                $paramStr .= "&{$key}={$value}";
            }
        }
        
        $inquiry_model    = new Inqman_InquiryModel();
        $status_options   = $inquiry_model->getStatusOptions();
        $priority_options = $inquiry_model->getPriorityOptions();
        
        ksort($service_options);
        ksort($status_options);
        ksort($priority_options);
        $this->view->assign('request_param',    $paramStr);
        $this->view->assign('service_options',  $service_options);
        $this->view->assign('status_options',   $status_options);
        $this->view->assign('priority_options', $priority_options);
    }
    
    /**
     * 指定された問い合わせの詳細情報及び回答・コメントの一覧を表示する。
     * 
     * @access public
     *
     */
    public function viewdetailAction()
    {
        //ページタイトルを設定する
        $this->_setupPageTitle('view_detail', 'inquiry');
        
        //リクエストパラメータから問い合わせIDを取得
        $inquiry_id = $this->getRequest()->getParam(REQUEST_KEY_INQUIRY_ID);
        
        //問い合わせ情報を取得
        $inquiry_model = new Inqman_InquiryModel();
        $data = $inquiry_model->getOneInquiry($inquiry_id);
        
        if (empty($_POST['replylimit_date']) && !empty($data['replylimit_datetime'])) {
            $data['replylimit_date'] = date('Y-m-d', strtotime($data['replylimit_datetime']));
        } elseif (!empty($_POST['replylimit_date'])) {
            $data['replylimit_date'] = $_POST['replylimit_date'];
        } else {
            $data['replylimit_date'] = "";
        }
        
        if (empty($_POST['replylimit_time']) && !empty($data['replylimit_datetime'])) {
            $data['replylimit_time'] = date('H:i', strtotime($data['replylimit_datetime']));
        } elseif (!empty($_POST['replylimit_time'])) {
            $data['replylimit_time'] = $_POST['replylimit_time'];
        } else {
            $data['replylimit_time'] = "";
        }
        
        $this->view->assign(VIEW_KEY_FORM_INPUT, $data);
        
        //問い合わせ情報からサービスIDを取得
        $service_id = $data[Inqman_InquiryModel::SERVICE_ID];
        
        //サービス情報を取得
        $service_model = new Inqman_ServiceModel();
        $service = $service_model->getOneService($service_id);
        $this->view->assign('service', $service);
        
        //アクセス権限のチェック
        $this->_checkAcl(ACL_RESOURCE_KEY_INQUIRY, ACL_PRIVILEGE_KEY_READ, $service_id);
        
        //回答・コメント情報を取得
        $reply_model = new Inqman_ReplyModel();
        $replies = $reply_model->getRepliesByInquiryId($inquiry_id);
        $this->view->assign(VIEW_KEY_REPLIES, $replies);
        
        //親問い合わせ情報を取得
        $parent_inquiry = null;
        if (!empty($data['parent_inquiry_id'])) {
            $parent_inquiry = $inquiry_model->getOneInquiry($data['parent_inquiry_id']);
        }
        $this->view->assign('parent_inquiry', $parent_inquiry);
        
        //派生した問い合わせ情報を取得
        $derivated_inquiries = (array) $inquiry_model->getDerivatedInquiries($inquiry_id);
        $this->view->assign('derivated_inquiries', $derivated_inquiries);
        
        $status_options   = $inquiry_model->getStatusOptions();
        $priority_options = $inquiry_model->getPriorityOptions();
        $service_options  = $service_model->getAllServiceOptions();
        $user_model = new Inqman_UserModel();
        $user_options     = $user_model->getUserOptions();
        
        $status_codes = array(Inqman_InquiryModel::STATUS_CODE_UNASSIGN, Inqman_InquiryModel::STATUS_CODE_PROCESS, Inqman_InquiryModel::STATUS_CODE_SUSPEND, Inqman_InquiryModel::STATUS_CODE_SOLVED);
        $inquiries  = $inquiry_model->getInquiriesByServiceId($service_id, $status_codes);
        $inquiry_options = array();
        foreach((array)$inquiries as $inquiry) {
            $inquiry_options[$inquiry['id']] = mb_strcut($inquiry['subject'], 0, 20) . '...';
        }
        
        $referenced_inquiries = $inquiry_model->getReferenceInquiries($inquiry_id);
        
        $this->view->assign('referenced_inquiries', $referenced_inquiries);
        $this->view->assign('inquiry_options', $inquiry_options);
        $this->view->assign('status_options', $status_options);
        $this->view->assign('priority_options', $priority_options);
        $this->view->assign('service_options', $service_options);
        $this->view->assign('user_options', $user_options);
        $this->view->assign('workflow_approve', intval($this->_config->workflow->approve->inquiry->accept));
        
        //アクセス権限情報を取得
        $this->_assignAclFlag($service_id, $inquiry_id);
    }
    
    /**
     * 問い合わせ情報を取得して、JSON形式で返す
     * 
     */
    public function jsonAction()
    {
        //ロジックインスタンスの生成
        $service_model = new Inqman_ServiceModel();
        $inquiry_model = new Inqman_InquiryModel();
        $acl_model     = new Inqman_AclModel();
        
        // 出力にテンプレートを使わないので、NoRender
        $this->_helper->viewRenderer->setNoRender();
        
        // リクエストのフィルタリング。
        $f = new Zend_Filter_StripTags();
        $qtype      = $f->filter($this->_request->getParam('qtype'));
        $query      = $f->filter($this->_request->getParam('query'));
        $page       = $f->filter($this->_request->getParam('page',1));
        $rp         = $f->filter($this->_request->getParam('rp',10));
        $sortname   = $f->filter($this->_request->getParam('sortname', 'id'));
        $sortorder  = $f->filter($this->_request->getParam('sortorder', 'desc'));
        $service_id = $f->filter($this->_request->getParam('service_id'));
        $criteria   = $this->_request->getParam('crtr');
        $this->_logger->debug("service_id:{$service_id}, qtype:{$qtype}, query:{$query}, page:{$page}, rp:{$rp}, sortname:{$sortname}, sortorder:{$sortorder}");

        if (!empty($service_id)) {
            //サービス情報を取得
            $service = $service_model->getOneService($service_id);
            
            //権限情報を取得
            $userInfo = $this->_getUserInfo();
        } else {
            
            $services = $service_model->getEnableServices($this->_getUserInfo()->id);
            $service_id = array();
            foreach ($services as $service) {
                $service_id[] = $service['service_id'];
            }
        }
        
        //取得する問い合わせ情報のステータスコードを設定
        $statusCode = Inqman_InquiryModel::getStatusCodes();
        $order = "{$sortname}_{$sortorder}";
        
        /**
         * 問い合わせ情報取得処理
         * 権限ごとに参照できる問い合わせ情報を取得
         */
        $inquiries = array();
        $count = 0;
        $statusCode = array();
        $authorities = array_keys($this->_getUserInfo()->roles);
        
        //未起票（起票待ち＆却下）
        if ($acl_model->checkAcl(ACL_RESOURCE_KEY_INQUIRY, ACL_PRIVILEGE_KEY_ACCEPT, $service_id, $authorities)) {
            $statusCode[] = Inqman_InquiryModel::STATUS_CODE_UNACCEPT;
            $statusCode[] = Inqman_InquiryModel::STATUS_CODE_REJECT;
        }
        
        //未解決（割当待ち＆処理中＆保留中）
        $statusCode[] = Inqman_InquiryModel::STATUS_CODE_UNASSIGN;
        $statusCode[] = Inqman_InquiryModel::STATUS_CODE_PROCESS;
        $statusCode[] = Inqman_InquiryModel::STATUS_CODE_SUSPEND;

        //未承認（承認待ち）
        if ($acl_model->checkAcl(ACL_RESOURCE_KEY_INQUIRY, ACL_PRIVILEGE_KEY_APPROVE, $service_id, $authorities)) {
            $statusCode[] = Inqman_InquiryModel::STATUS_CODE_UNAPPROVAL;
        }

        //解決済み
        $statusCode[] = Inqman_InquiryModel::STATUS_CODE_SOLVED;
        
        $rows  = (array) $inquiry_model->getInquiriesByServiceId($service_id, $statusCode, $order, $page, $rp, $criteria);
        $count = $inquiry_model->getInquiriesDataSizeByserviceId($service_id, $statusCode, $criteria);
        
        // jsonの作成。
        $json = array(
                'page' => (int) $page,
                'total' => (int) $count,
        );

        foreach ($rows as $row) {
            $ctrlName = $this->_getControllerByStatus($row['status']);
            $tmp = $this->_prepareJson($row, $ctrlName);
            $json['rows'][] = array('cell' => $tmp);
        }
        $this->_helper->json->sendJson($json);
    }

    /**
     * 担当している問い合わせ情報をJSON形式で返す
     * 
     * @access public
     */
    public function jsontakechargeAction()
    {
        //ロジックインスタンスを生成
        $service_model = new Inqman_ServiceModel();
        $inquiry_model = new Inqman_InquiryModel();
        
        // 出力にテンプレートを使わないので、NoRender。
        $this->_helper->viewRenderer->setNoRender();
        
        // リクエストのフィルタリング。
        $f = new Zend_Filter_StripTags();
        $qtype      = $f->filter($this->_request->getParam('qtype'));
        $query      = $f->filter($this->_request->getParam('query'));
        $page       = $f->filter($this->_request->getParam('page',1));
        $rp         = $f->filter($this->_request->getParam('rp',10));
        $sortname   = $f->filter($this->_request->getParam('sortname', 'id'));
        $sortorder  = $f->filter($this->_request->getParam('sortorder', 'desc'));
        $service_id = $f->filter($this->_request->getParam('service_id'));
        $criteria   = $this->_request->getParam('crtr');
        $this->_logger->debug("service_id:{$service_id}, qtype:{$qtype}, query:{$query}, page:{$page}, rp:{$rp}, sortname:{$sortname}, sortorder:{$sortorder}");
        
        if (!empty($service_id)) {
            $service = $service_model->getOneService($service_id);
            $userInfo = $this->_getUserInfo();
        } else {
            $services = $service_model->getEnableServiceOptions($this->_getUserInfo()->id);
            $service_id = array_keys($services);
        }
        
        $order = "{$sortname}_{$sortorder}";
        $rows  = (array) $inquiry_model->getTakesChargeInquiries($service_id, $this->_getUserInfo()->id, $order, $page, $rp, $criteria);
        
        $count = 0;
        $count = $inquiry_model->getTakesChargeInquiriesDataSize($service_id, $this->_getUserInfo()->id, $criteria);
        
        // jsonの作成。
        $json = array(
                'page' => (int) $page,
                'total' => (int) $count,
        );

        foreach ($rows as $row) {
            $ctrlName = $this->_getControllerByStatus($row['status']);
            $tmp = $this->_prepareJson($row, $ctrlName);
            $json['rows'][] = array('cell' => $tmp);
        }
        $this->_helper->json->sendJson($json);
    }
    
    /**
     * 起票した問い合わせ情報をJSON形式で返す
     */
    public function jsonacceptedAction()
    {
        //ロジックインスタンスを生成
        $service_model = new Inqman_ServiceModel();
        $inquiry_model = new Inqman_InquiryModel();
        
        // 出力にテンプレートを使わないので、NoRender。
        $this->_helper->viewRenderer->setNoRender();
        
        // リクエストのフィルタリング。
        $f = new Zend_Filter_StripTags();
        $qtype      = $f->filter($this->_request->getParam('qtype'));
        $query      = $f->filter($this->_request->getParam('query'));
        $page       = $f->filter($this->_request->getParam('page',1));
        $rp         = $f->filter($this->_request->getParam('rp',10));
        $sortname   = $f->filter($this->_request->getParam('sortname', 'id'));
        $sortorder  = $f->filter($this->_request->getParam('sortorder', 'desc'));
        $service_id = $f->filter($this->_request->getParam('service_id'));
        $criteria   = $this->_request->getParam('crtr');        
        $this->_logger->debug("service_id:{$service_id}, qtype:{$qtype}, query:{$query}, page:{$page}, rp:{$rp}, sortname:{$sortname}, sortorder:{$sortorder}");
        
        $service_model = new Inqman_ServiceModel();
        $inquiry_model = new Inqman_InquiryModel();
        
        if (!empty($service_id)) {
            $service = $service_model->getOneService($service_id);
            $userInfo = $this->_getUserInfo();
        } else {
            $services = $service_model->getEnableServiceOptions($this->_getUserInfo()->id);
            $service_id = array_keys($services);
        }
        
        $order = "{$sortname}_{$sortorder}";
        $rows  = (array) $inquiry_model->getAcceptedInquiries($service_id, $this->_getUserInfo()->id, $order, $page, $rp, $criteria);

        $count = 0;
        $count = $inquiry_model->getAcceptedInquiriesDataSize($service_id, $this->_getUserInfo()->id, $criteria);
        
        // jsonの作成。
        $json = array(
                'page' => (int) $page,
                'total' => (int) $count,
        );

        foreach ($rows as $row) {
            $ctrlName = $this->_getControllerByStatus($row['status']);
            $tmp = $this->_prepareJson($row, $ctrlName);
            $json['rows'][] = array('cell' => $tmp);
        }
        $this->_helper->json->sendJson($json);
    }
    
    /**
     * 担当者を変更する
     */
    public function changeoperatorAction()
    {
        $inquiry_id  = $this->getRequest()->getParam('inquiry_id');
        $operator_id = $this->getRequest()->getParam('operator_id');
        
        $inquiry_model = new Inqman_InquiryModel();
        $inquiry_model->changeOperator($inquiry_id, $operator_id);
        
        $this->_redirect("/inquiry/view_detail/inquiry_id/{$inquiry_id}/");
    }
    
    /**
     * 回答期限、保留開始日、解決日などの日時情報を設定する。
     * 
     */
    public function setupdatetimeAction()
    {
        $inquiry_id  = $this->getRequest()->getParam('inquiry_id');
        $field       = $this->getRequest()->getParam('field');
        $str_date    = $this->getRequest()->getParam($field . '_date');
        $str_time    = $this->getRequest()->getParam($field . '_time');
        
        //文字列で指定された日時をdate型に変換
        $date = null;
        $error_messages = array();
        if (!empty($str_date)) {
            if (empty($str_time)) $str_time = "00:00";
            
            if (!preg_match('/[0-9]{2,4}-[0-9]{1,2}-[0-9]{1,2}/', $str_date)) {
                $error_messages[] = array($this->view->translate('dateFormatInvalid'));
            }            
            if (!preg_match('/[0-9]{1,2}:[0-9]{1,2}/', $str_time)) {
                $error_messages[] = array($this->view->translate('timeFormatInvalid'));
            }
            
            if (empty($error_messages)) {
                $date = strtotime("{$str_date} {$str_time}");
            }
        }
        
        if (!empty($field)) {
            $inquiry_model = new Inqman_InquiryModel();
            
            //問い合わせ情報からサービスIDを取得
            $inquiry = $inquiry_model->getOneInquiry($inquiry_id);
            $service_id = $inquiry[Inqman_InquiryModel::SERVICE_ID];
            
            $this->_assignAclFlag($service_id, $inquiry_id);
            switch ($field) {
                case 'replylimit' :
                    $this->_checkAcl(ACL_RESOURCE_KEY_INQUIRY, ACL_PRIVILEGE_KEY_SETLIMIT, $service_id);
                    $inquiry_model->setReplyLimitDatetime($inquiry_id, $date);
                    break;
                case 'suspend' :
                    if ($this->view->can_suspend) {
                        $inquiry_model->suspendInquiry($inquiry_id);
                    }
                    break;
                case 'resume' :
                    if ($this->view->can_resume) {
                        $inquiry_model->resumeInquiry($inquiry_id);
                    }
                    break;
                case 'solution' :
                    if ($this->view->can_solution) {
                        $inquiry_model->solutionInqiury($inquiry_id);
                    }
                    break;
                default :
                    break;
            }
        }
        
        if (empty($error_messages)) {
            $this->_redirect("inquiry/view_detail/inquiry_id/{$inquiry_id}/");
        } else {
            $this->view->assign(VIEW_KEY_ERROR_MESSAGES, $error_messages);
            $this->_forward('view_detail', 'inquiry', array('inquiry_id', $inquiry_id));
        }
    }
    
    /**
     * 添付ファイルをダウンロードする
     */
    public function downloadfileAction()
    {
        $this->_helper->viewRenderer->setNoRender();
        $this->_helper->layout->disableLayout();
        
        $inquiry_id = $this->getRequest()->get('inquiry_id');
        $attach_file_id = $this->getRequest()->get('attach_file_id');
        
        $inquiry_model = new Inqman_InquiryModel();
        $inquiry = $inquiry_model->getOneInquiry($inquiry_id);
        
        $attach_files = (array) $inquiry['attach_files'];
        $response_data = null;
        foreach ($attach_files as $attach_file) {
            if ($attach_file['id'] == $attach_file_id) {
                $response_data = $attach_file;
            }
        }
        
        if ($response_data != null) {
            $file_name = $response_data['name'];
            $output_encode = mb_http_output();
            $file_name = mb_convert_encoding($file_name, 'Shift-JIS', mb_detect_encoding($file_name));
            $mime_type = $response_data['mime_type'];
            $content = base64_decode($response_data['enced_content']);
            $length = strlen($content);
            
            $response = $this->getResponse()
                 ->setHeader('Content-Disposition', "attachment;filename={$file_name}\n" )
                 ->setHeader('Content-type', "{$mime_type}\n")
                 ->setHeader('Content-Length', "{$length}\n")
                 ->appendBody($content)
            ;
            $response->sendHeaders();
            $response->outputBody();
            
        }
    }
    
    /**
     * 派生問い合わせの入力
     * 
     */
    public function derivateinputAction()
    {
        //ロジックインスタンスの生成
        $inquiry_model = new Inqman_InquiryModel();
        $service_model = new Inqman_ServiceModel();
        
        //ページタイトルの設定
        $this->_setupPageTitle('derivate_input', 'inquiry');
        
        //リクエストパラメータから問い合わせIDを取得し、問い合わせ情報を取得
        $inquiry_id = $this->getRequest()->getParam('parent_inquiry_id');
        $inquiry = $inquiry_model->getOneInquiry($inquiry_id);
        
        //問い合わせ情報からサービスIDを取得
        $service_id = $inquiry['service_id'];
        $service = $service_model->getOneService($service_id);
        
        //アクセス権限のチェック
        $this->_checkAcl(ACL_RESOURCE_KEY_INQUIRY, ACL_PRIVILEGE_KEY_DERIVATION, $service_id);
        
            //フォームの初期入力値を設定
        if (!empty($_POST)) {
            $form_input = $_POST;
        } else {
            $form_input = array(
                Inqman_InquiryForm::ID            => 0,
                Inqman_InquiryForm::PARENT_INQ_ID => $inquiry_id,
                Inqman_InquiryForm::SERVICE_ID    => $service_id,
                Inqman_InquiryForm::PRIORITY      => Inqman_InquiryModel::PRIORITY_CODE_MIDDLE,
                Inqman_InquiryForm::SUBJECT       => $inquiry['subject'],
                Inqman_InquiryForm::RECEIVED_FROM => $inquiry['received_from'],
                Inqman_InquiryForm::CONTENT       => $inquiry['content'],
            );
        }
        
        //OPTION情報を取得
        $priority_options = $inquiry_model->getPriorityOptions();
        
        //派生機能の固有情報をVIEWに設定する
        $this->view->assign(ACTION_PREFIX_KEY, ACTION_PREFIX_DERIVATE);
        $this->view->assign(ACTION_PREFIX_NAME, $this->view->translate('inquiry.label.derivate'));
        
        //表示情報をVIEWに設定する
        $this->view->assign(VIEW_KEY_FORM_INPUT, $form_input);
        $this->view->assign('service', $service);
        $this->view->assign('priority_options', $priority_options);

        $this->_helper->viewRenderer->setRender('input');
    }
    
    /**
     * 派生問い合わせの入力確認
     * 
     */
    public function derivateconfirmAction()
    {
        //ロジックインスタンスの初期化
        $inquiry_model = new Inqman_InquiryModel();
        $service_model = new Inqman_ServiceModel();
        
        //ページタイトルの設定
        $this->_setupPageTitle('derivate_confirm', 'inquiry');
        
        //リクエストパラメータから問い合わせIDを取得し、問い合わせ情報を取得
        $inquiry_id = $this->getRequest()->getParam('parent_inquiry_id');
        $inquiry = $inquiry_model->getOneInquiry($inquiry_id);
        
        //問い合わせ情報からサービスIDを取得し、サービス情報を取得
        $service_id = $inquiry['service_id'];
        $service = $service_model->getOneService($service_id);
        
        //アクセス権限のチェック
        $this->_checkAcl(ACL_RESOURCE_KEY_INQUIRY, ACL_PRIVILEGE_KEY_DERIVATION, $service_id);     
        
        //フォーム入力値の検証
        $form = new Inqman_InquiryForm();
        $error_messages = null;
        $tmp_names = null;
        if(!$form->isValid($_POST)) {
            $error_messages = $form->getMessages();
        }
        
        //添付ファイルの検証と保存
        $attach_files = null;
        $attach_error_messages = $this->_validateUploadFiles($_FILES, REQUEST_KEY_ATTACH_FILES);
        if (!empty($attach_error_messages)) {
            $error_messages = array_merge($error_messages, $attach_error_messages);
        } else {
            $attach_files = $this->_saveUploadFiles($_FILES, REQUEST_KEY_ATTACH_FILES);
        }
        
        //エラーメッセージの有無で表示するページを振り分け
        if(!empty($error_messages)) {
            $form_input = $form->getUnfilteredValues();
            $this->_helper->viewRenderer->setRender('input');
        } else {
            $form_input = $form->getValues();
            $form_input[REQUEST_KEY_ATTACH_FILES] = $attach_files;
            $this->_helper->viewRenderer->setRender('confirm');
        }
        
        //OPTION情報を取得
        $priority_options = $inquiry_model->getPriorityOptions();
        
        //派生機能の固有情報をVIEWに設定する
        $this->view->assign(ACTION_PREFIX_KEY, ACTION_PREFIX_DERIVATE);
        $this->view->assign(ACTION_PREFIX_NAME, $this->view->translate('inquiry.label.derivate'));
        
        //表示情報をVIEWに設定
        $this->view->assign(VIEW_KEY_FORM_INPUT, $form_input);
        $this->view->assign('service', $service);
        $this->view->assign('priority_options', $priority_options);
        $this->view->assign(VIEW_KEY_ERROR_MESSAGES, $error_messages);
    }
    
    /**
     * 派生問い合わせの登録実行
     * 
     */
    public function derivatecommitAction()
    {
        //ロジックインスタンスの生成
        $inquiry_model = new Inqman_InquiryModel();
        $service_model = new Inqman_ServiceModel();
        
        //ページタイトルの設定
        $this->_setupPageTitle('derivate_commit', 'inquiry');
        
        //リクエストパラメータからサービスIDを取得
        $service_id = $this->getRequest()->getParam(Inqman_InquiryForm::SERVICE_ID);
        $service = $service_model->getOneService($service_id);
        
        //アクセス権限のチェック
        $this->_checkAcl(ACL_RESOURCE_KEY_INQUIRY, ACL_PRIVILEGE_KEY_DERIVATION, $service_id); 
        
        //フォーム入力値の検証
        $form = new Inqman_InquiryForm();
        $error_messages = null;
        if(!$form->isValid($_POST)) {
            $error_messages = $form->getMessages();
        }
        
        if(empty($error_messages)) {  
            //問い合わせ情報の登録
            $attach_files = !empty($_POST[REQUEST_KEY_ATTACH_FILES]) ? (array) $_POST[REQUEST_KEY_ATTACH_FILES] : array();
            $enced_attach_files = $this->_encodeUploadFiles($attach_files);
            
            $status = Inqman_InquiryModel::STATUS_CODE_UNAPPROVAL;
            //承認フローが無効の場合は、状態を「割当待ち」にする
            if (intval($this->_config->workflow->approve->inquiry->accept) != 1) {
                $status = Inqman_InquiryModel::STATUS_CODE_UNASSIGN;
            }
            
            $now = date("Y-m-d H:i:s", time());
            $values = array(
                    Inqman_InquiryModel::SUBJECT         => $form->getValue(Inqman_InquiryForm::SUBJECT),
                    Inqman_InquiryModel::RECEIVED_FROM   => $form->getValue(Inqman_InquiryForm::RECEIVED_FROM),
                    Inqman_InquiryModel::CONTENT         => $form->getValue(Inqman_InquiryForm::CONTENT),
                    Inqman_InquiryModel::SERVICE_ID      => $form->getValue(Inqman_InquiryForm::SERVICE_ID),
                    Inqman_InquiryModel::PRIORITY        => $form->getValue(Inqman_InquiryForm::PRIORITY),
                    Inqman_InquiryModel::STATUS          => $status,
                    Inqman_InquiryModel::REGISTRANT_ID   => $this->_getUserInfo()->id,
                    Inqman_InquiryModel::CREATE_DATETIME => $now,
                    Inqman_InquiryModel::ACCEPTER_ID     => $this->_getUserInfo()->id,
                    Inqman_InquiryModel::ACCEPT_DATETIME => $now,
                    Inqman_InquiryModel::PARENT_INQUIRY_ID => $form->getValue(Inqman_InquiryForm::PARENT_INQ_ID),
            );
            
            $inquiry_id = $inquiry_model->createInquiry($values, $enced_attach_files);
                
            //承認フローが無効の場合は、同時に承認処理も行う
            if (intval($this->_config->workflow->approve->inquiry->accept) != 1) {
                $values['approver_id'] = $this->_getUserInfo()->id;
                $inquiry_model->approveInquiry($inquiry_id, $values);
            }
            
            $event_action_id = 4;
            $criteria = array('service_id'=>$service_id, 'inquiry_id'=>$inquiry_id);
            $inquiry_model->sendMail($event_action_id, $criteria);
            
            $this->view->assign(ACTION_PREFIX_KEY, ACTION_PREFIX_DERIVATE);
            $this->view->assign(ACTION_PREFIX_NAME, $this->view->translate('inquiry.label.derivate'));
            $this->view->assign('service', $service);
            $this->_helper->viewRenderer->setRender('commit');
        }
        //エラーがあった場合はメッセージをセットして、入力画面を再表示
        else {
            $this->view->assign(VIEW_KEY_ERROR_MESSAGES, $error_messages);
            $this->view->assign(VIEW_KEY_FORM_INPUT, $form->getUnfilteredValues());
            $this->_helper->viewRenderer->setRender('input');
        }
    }
    
    /**
     * 派生関係の削除
     */
    public function removederivationAction()
    {
        $inquiry_id  = $this->getRequest()->getParam('inquiry_id');
        $derivated_inquiry_id = (array) $this->getRequest()->getParam('derivated_inquiry_id');
        
        $inquiry_model = new Inqman_InquiryModel();
        foreach ($derivated_inquiry_id as $id) {
            $inquiry_model->removeDerivation($id);
        }
        
        $this->_redirect("inquiry/view_detail/inquiry_id/{$inquiry_id}/");
    }
    
    /**
     * 参照問い合わせの追加
     */
    public function addreferenceAction()
    {
        $inquiry_id  = $this->getRequest()->getParam('inquiry_id');
        $referenced_inquiry_id = (array)$this->getRequest()->getParam('referenced_inquiry_id');
        
        $inquiry_model = new Inqman_InquiryModel();
        foreach ($referenced_inquiry_id as $ref_id) {
            $inquiry_model->addReference($inquiry_id, $ref_id);
        }
        
        $this->_redirect("inquiry/view_detail/inquiry_id/{$inquiry_id}/");
    }
    
    /**
     * 参照関係の削除
     */
    public function removereferenceAction()
    {
        $inquiry_id  = $this->getRequest()->getParam('inquiry_id');
        $referenced_inquiry_id = (array) $this->getRequest()->getParam('referenced_inquiry_id');
        
        $inquiry_model = new Inqman_InquiryModel();
        foreach ($referenced_inquiry_id as $ref_id) {
            $inquiry_model->removeReference($inquiry_id, $ref_id);
        }
        
        $this->_redirect("inquiry/view_detail/inquiry_id/{$inquiry_id}/");
    }
    
    
    /** Local Methods **/
    
    /**
     * 各種権限を取得し、VIEWに割り付ける
     * 
     */
    protected function _assignAclFlag($service_id, $inquiry_id=null)
    {
        $inquiry_model = new Inqman_InquiryModel();
        $user_id = $this->_getUserInfo()->id;
        
        //作成者か否か
        $is_registrant = false;
        if (!empty($inquiry_id)) {
            $is_registrant = $inquiry_model->isRegistrant($inquiry_id, $user_id);
            $this->view->assign('is_registrant', $is_registrant);
        }
        
        //起票者か否か
        $is_accepter = false;
        if (!empty($inquiry_id)) {
            $is_accepter = $inquiry_model->isAccepter($inquiry_id, $user_id);
            $this->view->assign('is_accepter', $is_accepter);
        }
        
        //承認者か否か
        $is_approver = false;
        if (!empty($inquiry_id)) {
            $is_approver = $inquiry_model->isApprover($inquiry_id, $user_id);
            $this->view->assign('is_approver', $is_approver);
        }
        
        //担当者か否か
        $is_operator = false;
        if (!empty($inquiry_id)) {
            $is_operator = $inquiry_model->isOperator($inquiry_id, $user_id);
            $this->view->assign('is_operator', $is_operator);
        }

        $acl_model = new Inqman_AclModel();
        $authorities = array_keys($this->_getUserInfo()->roles);
        $inquiry = $inquiry_model->getOneInquiry($inquiry_id);
        $status = $inquiry['status'];
        
        //コメントする権限
        $can_comment_create = false;
        if ($status == Inqman_InquiryModel::STATUS_CODE_UNASSIGN || $status == Inqman_InquiryModel::STATUS_CODE_PROCESS || $status == Inqman_InquiryModel::STATUS_CODE_SUSPEND) {
            $can_comment_create = $acl_model->checkAcl(ACL_RESOURCE_KEY_COMMENT, ACL_PRIVILEGE_KEY_CREATE, $service_id, $authorities);
        }
        $this->view->assign('can_comment_create', $can_comment_create);
        
        //回答する権限
        $can_reply_create = false;
        if ($status == Inqman_InquiryModel::STATUS_CODE_UNASSIGN || $status == Inqman_InquiryModel::STATUS_CODE_PROCESS || $status == Inqman_InquiryModel::STATUS_CODE_SUSPEND) {
            $can_reply_create   = $acl_model->checkAcl(ACL_RESOURCE_KEY_REPLY, ACL_PRIVILEGE_KEY_CREATE, $service_id, $authorities);
        }
        $this->view->assign('can_reply_create',   $can_reply_create);
        
        //起票する権限
        $can_inquiry_accept = false;
        if ($status == Inqman_InquiryModel::STATUS_CODE_UNACCEPT || $status == Inqman_InquiryModel::STATUS_CODE_REJECT) {
            $can_inquiry_accept   = $acl_model->checkAcl(ACL_RESOURCE_KEY_INQUIRY, ACL_PRIVILEGE_KEY_ACCEPT, $service_id, $authorities);
        }
        $this->view->assign('can_inquiry_accept',   $can_inquiry_accept);
        
        //承認する権限
        $can_inquiry_approve = false;
        if ($status == Inqman_InquiryModel::STATUS_CODE_UNAPPROVAL) {
            $can_inquiry_approve   = $acl_model->checkAcl(ACL_RESOURCE_KEY_INQUIRY, ACL_PRIVILEGE_KEY_APPROVE, $service_id, $authorities);
        }
        $this->view->assign('can_inquiry_approve',   $can_inquiry_approve);
        
        //期限を設定する権限
        $can_setlimit = false;
        if ($status == Inqman_InquiryModel::STATUS_CODE_UNASSIGN || $status == Inqman_InquiryModel::STATUS_CODE_PROCESS || $status == Inqman_InquiryModel::STATUS_CODE_SUSPEND) {
            $can_setlimit = $acl_model->checkAcl(ACL_RESOURCE_KEY_INQUIRY, ACL_PRIVILEGE_KEY_SETLIMIT, $service_id, $authorities);
        }
        $this->view->assign('can_setlimit', $can_setlimit);
        
        //保留にする権限
        $can_suspend = false;
        if ($status == Inqman_InquiryModel::STATUS_CODE_PROCESS) {
            if ($inquiry_model->isOperator($inquiry_id, $user_id) || 
                    $acl_model->checkAcl(ACL_RESOURCE_KEY_INQUIRY, ACL_PRIVILEGE_KEY_SUSPEND, $service_id, $authorities))
            {
                $can_suspend = true;
            }
        }
        $this->view->assign('can_suspend', $can_suspend);
        
        //保留解除権限
        $can_resume = false;
        if ($status == Inqman_InquiryModel::STATUS_CODE_SUSPEND) {
            if ($inquiry_model->isOperator($inquiry_id, $user_id) || 
                    $acl_model->checkAcl(ACL_RESOURCE_KEY_INQUIRY, ACL_PRIVILEGE_KEY_SUSPEND, $service_id, $authorities))
            {
                $can_resume = true;
            }
        }
        $this->view->assign('can_resume', $can_resume);
        
        //解決にする権限
        $can_solution = false;
        if ($status == Inqman_InquiryModel::STATUS_CODE_PROCESS) {
            if ($inquiry_model->isOperator($inquiry_id, $user_id) ||
                    $acl_model->checkAcl(ACL_RESOURCE_KEY_INQUIRY, ACL_PRIVILEGE_KEY_SOLUTION, $service_id, $authorities))
            {
                $can_solution = true;
            }
        }
        $this->view->assign('can_solution', $can_solution);
        
        //参照を登録する権限
        $can_reference = $acl_model->checkAcl(ACL_RESOURCE_KEY_INQUIRY, ACL_PRIVILEGE_KEY_REFERENCE, $service_id, $authorities);
        $this->view->assign('can_reference', $can_reference);

        //派生登録する権限
        $can_derivation = $acl_model->checkAcl(ACL_RESOURCE_KEY_INQUIRY, ACL_PRIVILEGE_KEY_DERIVATION, $service_id, $authorities);
        $this->view->assign('can_derivation', $can_derivation);
        
    }
    
    /**
     * アップロードされたファイルを検証する
     * 
     * @param array $_FILES
     * @param string $field_name
     * @return array エラーメッセージを配列で返す。エラーが無かった場合はnullを返す。
     */
    protected function _validateUploadFiles($_FILES, $field_name)
    {
        
        $max_size  = intval($this->_config->attach_files->max_size);
        $max_count = intval($this->_config->attach_files->max_count);
        
        $file_names = $_FILES[$field_name]['name'];
        $file_sizes = $_FILES[$field_name]['size'];
        
        $error_messages = array();
        for ($i=0; $i<count($file_sizes); $i++) {
            if ($file_sizes[$i] > $max_size) {
                $error_messages[] = array($this->view->translate('attachFileSizeOver'). ":" . $file_names[$i]);
            }
        }
        return $error_messages;
    }
    
    /**
     * アップロードされたファイルを保存する
     * 
     * @param array $_FILES
     * @param string $field_name
     * @return array アップロードされたファイル名と保存先ファイル名を配列で返す
     * 
     */
    protected function _saveUploadFiles($_FILES, $field_name)
    {
        //TODO ごみ処理
        $file_names = $_FILES[$field_name]['name'];
        $tmp_names  = $_FILES[$field_name]['tmp_name'];
        $file_sizes = $_FILES[$field_name]['size'];
        $mime_types = $_FILES[$field_name]['type'];
        
        $attach_files = array();
        for ($i=0; $i<count($tmp_names); $i++) {
            if ($file_sizes[$i] > 0 && !empty($file_names[$i])) {
                $save_file_name = sha1($file_names[$i] . date('YmdHis'));
                $save_file_dir = ROOT_DIR . $this->_config->attach_files->save_dir;
                move_uploaded_file($tmp_names[$i], "{$save_file_dir}/{$save_file_name}");
                
                $attach_files[] = array(
                        'name' => $file_names[$i],
                        'type' => $mime_types[$i],
                        'tmp_name' => $save_file_name
                );
            }
        }
        return $attach_files;
    }
    
    /**
     * アップロードされたファイルのBase64エンコードを行う
     */
    protected function _encodeUploadFiles($attach_files)
    {
        $tmp_file_dir = ROOT_DIR . $this->_config->attach_files->save_dir;
        $enced_attach_files = array();
        
        foreach ($attach_files as $attach_file) {
            $tmp_file_path = "{$tmp_file_dir}/{$attach_file['tmp_name']}";
            
            if (file_exists($tmp_file_path)) {
                $enced_attach_files[] = array(
                    'name'          => $attach_file['name'],
                    'type'          => $attach_file['type'],
                    'enced_content' => base64_encode(file_get_contents($tmp_file_path)),
                );
            }
        }
        return $enced_attach_files;
    }
    
    /**
     * JSON返却用データを生成する
     */
    protected function _prepareJson($row, $ctrlName)
    {
        $tmp = array();
        $tmp[] = $row['id'];
        $tmp[] = empty($row['service_name']) ? "--" : $this->view->escape($row['service_name']);
        $tmp[] = '<a href="' . $ctrlName . '/view_detail/inquiry_id/' . $row['id'] . '/">' . $this->view->escape($row['subject']) . '</a>';
        $tmp[] = empty($row['received_from']) ? "--" : $this->view->escape($row['received_from']);
        $tmp[] = empty($row['content']) ? "--" : $this->view->escape(mb_strcut($row['content'], 0, 200, 'UTF-8').'...');
        $tmp[] = empty($row['create_datetime']) ? "--" : $row['create_datetime'];
        $tmp[] = empty($row['registrant_name']) ? "--" : $row['registrant_name'];
        $tmp[] = empty($row['accept_datetime']) ? "--" : $row['accept_datetime'];
        $tmp[] = empty($row['accepter_name']) ? "--" : $row['accepter_name'];
        $tmp[] = empty($row['operator_name']) ? "--" : $row['operator_name'];
        $tmp[] = '<div style="width:100%;padding:0;margin:0;background-color:'. $this->_config->view->bgcolor->status->get($row['status']) .';border:solid 1px #999">' . $this->view->translate("inquiry.status.label." . $row['status']) . '</div>';
        $tmp[] = $this->view->translate("inquiry.priority.label." . $row['priority']);
        
        return $tmp;
    }
    
    /**
     * ステータスから適切なコントローラー名を取得する
     * 
     */
    protected function _getControllerByStatus($status)
    {
        $ctrlName = "";
        switch ($status) {
            case Inqman_InquiryModel::STATUS_CODE_UNACCEPT:
            case Inqman_InquiryModel::STATUS_CODE_REJECT:
                $ctrlName = "unaccept";
                break;
            case Inqman_InquiryModel::STATUS_CODE_UNAPPROVAL:
                $ctrlName = "unapproval";
                break;
            case Inqman_InquiryModel::STATUS_CODE_UNASSIGN:
            case Inqman_InquiryModel::STATUS_CODE_PROCESS:
            case Inqman_InquiryModel::STATUS_CODE_SUSPEND:
            case Inqman_InquiryModel::STATUS_CODE_SOLVED:
                $ctrlName = "inquiry";
                break;
            default:
                break;
        }
        return $ctrlName;
    }
}
