/*************************************************************************************************/
/*!
   	@file		PathSegmentSampler.h
	@author 	Fanzo
 	@date 		2008/5/21
*/
/*************************************************************************************************/
#pragma		once

///////////////////////////////////////////////////////////////////////////////////////////////////
//include files
#include	<signal.h>

#pragma pack( push , 8 )		//set align

namespace icubic
{

///////////////////////////////////////////////////////////////////////////////////////////////////
// preprocessor deifne

///////////////////////////////////////////////////////////////////////////////////////////////////
// type define

enum InnerBridgeType
{
	Miter_InnerBridgeType ,
	Jag_InnerBridgeType ,  
	Round_InnerBridgeType , 
};

///////////////////////////////////////////////////////////////////////////////////////////////////
// global functions
//=================================================================================================
//!	insert edge node
//!	@retval			---
//-------------------------------------------------------------------------------------------------
cb_inline
OutlineNodePtr* InsertStrokeNode
		(
		iOutline		&outline ,
		OutlineNodePtr	*node , 
		IPathSegment*	path , 
		float			t , 
		float			w_r2 , 		// if right , set minus
		fvector2*		v		= 0 , 
		float*			v_abs	= 0 , 
		fvector2*		t_pos	= 0 , 
		fvector2*		t_vec	= 0 , 
		fvector2*		t_w0	= 0 , 
		fvector2*		t_w1	= 0 
		)
{
	fvector2	pos		= store( t_pos , path->Value( t ) );
	fvector2	diff	= store( v , path->Differential( t ) );
	float		diff_len= store( v_abs , diff.Length() );
	
	fvector2	vec		= store( t_vec , diff / ( diff_len == 0.0f ? 0.001f : diff_len ) );
	fvector2	rv		= vec.GetRotate90();
	rv = rv * w_r2;
		
	outline->InsertNode( node , store( t_w1 , pos + rv ) );
	return outline->InsertNode( node , store( t_w0 , pos - rv ) );
}
//=================================================================================================
//!	insert edge node
//!	@retval			---
//-------------------------------------------------------------------------------------------------
cb_inline
void GetStrokeEdge
		(
		const fvector2&		pos , 
		const fvector2&		vec , 
		float				w_r2 , 
		fvector2*			w0 , 
		fvector2*			w1
		)
{
	fvector2	rv	= vec.GetRotate90();
	rv = rv * w_r2;
	*w0	= pos - rv;
	*w1	= pos + rv;
}
//=================================================================================================
//!	insert edge node
//!	@retval			---
//-------------------------------------------------------------------------------------------------
cb_inline
void GetStrokeEdge
		(
		IPathSegment*	path , 
		float			t , 
		float			w_r2 , 
		fvector2		*t_pos , 
		fvector2		*t_vec , 
		fvector2		*t_w0 , 
		fvector2		*t_w1
		)
{
	fvector2	pos = store( t_pos , path->Value( t ) );
	fvector2	vec = store( t_vec , path->Differential( t ).GetUnit() );
	fvector2	rv	= vec.GetRotate90();
	rv = rv * w_r2;
	
	store( t_w0 , pos - rv );
	store( t_w1 , pos + rv );
}
//=================================================================================================
//!	normalize sampling order
//!	@retval			---
//-------------------------------------------------------------------------------------------------
cb_inline
void NormalizeSampleOrder
		(
		fvector2		pnt[3] , 
		bool			invert
		)
{
	if( invert == false )
	{
		if( ( pnt[1] - pnt[0] ).Outer( pnt[2] - pnt[0] ) < 0.0f )
			swap( &pnt[1] , &pnt[2] );
	}
	else
	{
		if( ( pnt[1] - pnt[0] ).Outer( pnt[2] - pnt[0] ) > 0.0f )
			swap( &pnt[1] , &pnt[2] );
	}
}
///////////////////////////////////////////////////////////////////////////////////////////////////
// classes define

/**************************************************************************************************
"PathLength" class 
**************************************************************************************************/
class PathLengthTbl
{
// variable member
private:
	Array<float>	m_lengthtbl;
	float			m_dt;
	int				m_searchpos;
		
// public functions
public:
//=================================================================================================
//!	construct
//-------------------------------------------------------------------------------------------------
PathLengthTbl() : m_lengthtbl( Expand_ArrayCashType , 32 ) , m_dt( 0.0f ) , m_searchpos( 0 )
{
}
//=================================================================================================
//!	set path
//!	@retval			---
//-------------------------------------------------------------------------------------------------
void SetPath
		(
		IPathSegment*	ps
		)
{
	m_searchpos	= 0;
	
	int		off , num = ps->GetOptimumLengthSample();
	m_dt	= 1.0f / ( num - 1 );
	m_lengthtbl.Resize( num );

	m_lengthtbl[ 0 ]	= 0.0f;
	float		t	= 0.0f;
	fvector2	sp	= ps->Value( t );
	t += m_dt;

	for( off = 1 ; off < num ; off++ )
	{
		fvector2	tp	= ps->Value( t );
		m_lengthtbl[ off ]	= ( tp - sp ).Length() + m_lengthtbl[ off - 1 ];
		sp	= tp;
		t += m_dt;
	}
}
//=================================================================================================
//!	t to length
//!	@retval			---
//-------------------------------------------------------------------------------------------------
float ToLength
		(
		float		t
		)
{
	int		tbloff	= ( int )floor( t / m_dt );
	if( tbloff < 0 )
		return m_lengthtbl[ 0 ];
	if( tbloff >= m_lengthtbl.GetDatanum() - 1 )
		return m_lengthtbl[ m_lengthtbl.GetDatanum() - 1 ];

	float	rate		= ( t - tbloff * m_dt ) / m_dt;
	return ( m_lengthtbl[ tbloff + 1 ] - m_lengthtbl[ tbloff ] ) * rate + m_lengthtbl[ tbloff ];
}
//=================================================================================================
//!	length to t
//!	@retval			---
//-------------------------------------------------------------------------------------------------
float ToParam
		(
		float		length
		)
{
	if( length <= 0.0f )
		return 0.0f;
	int		tblnum = m_lengthtbl.GetDatanum();
	if( length >= m_lengthtbl[ tblnum - 1 ] )
		return 1.0f;
	
	const float	*ptbl	= m_lengthtbl.GetConstPtr();
	int		tbloff;
	for( tbloff = m_searchpos ; tbloff < tblnum - 1 ; tbloff++ )
	{
		if( ptbl[ tbloff ] <= length && length <= ptbl[ tbloff + 1 ] )
		{
			m_searchpos	= tbloff;
			float	t = m_dt * tbloff + m_dt * ( length - ptbl[ tbloff ] ) / ( ptbl[ tbloff + 1 ] - ptbl[ tbloff ] );
			return t;
		}
	}
	for( tbloff = 0 ; tbloff < m_searchpos ; tbloff++ )
	{
		if( ptbl[ tbloff ] <= length && length <= ptbl[ tbloff + 1 ] )
		{
			m_searchpos	= tbloff;
			float	t = m_dt * tbloff + m_dt * ( length - ptbl[ tbloff ] ) / ( ptbl[ tbloff + 1 ] - ptbl[ tbloff ] );
			return t;
		}
	}
	cb_assert( false , L"ToParam algorithm error." );
	return 0.0f;
}
};

/**************************************************************************************************
"PathSegmentSamplerPaint" class 
**************************************************************************************************/
class PathSegmentSamplerPaint
{
// public functions
public:
//=================================================================================================
//!	construct
//-------------------------------------------------------------------------------------------------
PathSegmentSamplerPaint()
{
}
//=================================================================================================
//!	Sampling
//!	@retval			---
//-------------------------------------------------------------------------------------------------
OutlineNodePtr* cb_call Sampling
		(
		iOutline&			outline ,
		OutlineNodePtr*		node , 
		IPathSegment*		ps , 
		float				st , 
		float				tt , 
		float				samplescale
		)
{
	if( ps->IsExist() == false )
		return node;
	if( st == tt )
		return node;
	
	int		spnum	= ps->GetOptimumSample( samplescale );
	spnum	= ( int )( ( tt - st ) * spnum );
	spnum	= spnum < 2 ? 2 : spnum;
	
	float	dt	= ( tt - st ) / ( float )( spnum - 1 );
	int		spoff;
	for( spoff = 0 ; spoff < spnum ; spoff++ )
	{
		node	= outline->InsertNode( node , ps->Value( st ) );
		st += dt;
	}
	return node;
}
};

/**************************************************************************************************
"PathSegmentSamplerStroke" class 
**************************************************************************************************/
class PathSegmentSamplerStroke
{
// variable member
private:
	float					m_w_r2;
	InnerBridgeType			m_bridgetype;
	float					m_da;

	typedef 
	OutlineNodePtr* BridgeFunc
		(
		iOutline& ,
		OutlineNodePtr* , 
		float , 
		float , 
		fvector2& , 
		fvector2& ,  
		fvector2& , 
		fvector2& ,  
		fvector2& , 
		fvector2&
		);
	
// public functions
public:

// private functions
private:
//=================================================================================================
//!	UpdateParam
//!	@retval			---
//-------------------------------------------------------------------------------------------------
void UpdateParam()
{
	if( -0.0001f <= m_w_r2 && m_w_r2 <= 0.0001f )
		m_da	= PI_f;
	else
		m_da	= sqrtf( 8.0f * cb_pathsegment_max_accident_error / fabs(m_w_r2) );
}
//=================================================================================================
//!	mitter
//!	@retval			---
//-------------------------------------------------------------------------------------------------
static
OutlineNodePtr* Miter
		(
		iOutline&			outline ,
		OutlineNodePtr*		node , 
		float				a , 
		float				w_r2 , 
		fvector2&			b_pos , 
		fvector2&			b_w0 , 
		fvector2&			b_w1 ,  
		fvector2&			pos , 
		fvector2&			w0 , 
		fvector2&			w1
		)
{
	outline->InsertNode( node , w1 );
	node = outline->InsertNode( node , w0 );
	return node;
}
//=================================================================================================
//!	jag
//!	@retval			---
//-------------------------------------------------------------------------------------------------
static
OutlineNodePtr* Jag
		(
		iOutline&			outline ,
		OutlineNodePtr*		node , 
		float				a , 
		float				w_r2 , 
		fvector2&			b_pos , 
		fvector2&			b_w0 , 
		fvector2&			b_w1 ,  
		fvector2&			pos , 
		fvector2&			w0 , 
		fvector2&			w1
		)
{
	fvector2	v0	= w0 - b_w0;
	fvector2	v1	= w1 - b_w1;
	
	if( v0.Inner( v1 ) < 0.0f )
	{
		if( a >= 0.0f )
		{
			outline->InsertNode( node , b_pos );
			outline->InsertNode( node , w1 );
			node = outline->InsertNode( node , w0 );
		}
		else
		{
			outline->InsertNode( node , w1 );
			node = outline->InsertNode( node , b_pos );
			node = outline->InsertNode( node , w0 );
		}
	}
	else
	{
		outline->InsertNode( node , w1 );
		node = outline->InsertNode( node , w0 );
	}
	return node;
}
//=================================================================================================
//!	round
//!	@retval			---
//-------------------------------------------------------------------------------------------------
static
OutlineNodePtr* Round
		(
		iOutline&			outline ,
		OutlineNodePtr*		node , 
		float				a , 
		float				w_r2 , 
		fvector2&			b_pos , 
		fvector2&			b_w0 , 
		fvector2&			b_w1 ,  
		fvector2&			pos , 
		fvector2&			w0 , 
		fvector2&			w1
		)
{
	fvector2	v0	= w0 - b_w0;
	fvector2	v1	= w1 - b_w1;

	if( v0.Inner( v1 ) < 0.0f )
	{
		if( a >= 0.0f )
		{
			node = outline->InsertNode( node , w0 );
			outline->InsertNode( node , b_pos );
			outline->InsertNode( node , pos );
			outline->InsertNode( node , w1 );
			{
				fvector2	pnt[3]	= { b_pos , pos , w1 };
				NormalizeSampleOrder( pnt , w_r2 < 0.0f );
				OutlineNodePtr*	tnode	= &outline->CreateOutline()->m_node;
				outline->InsertNode( tnode , pnt , 3 );
			}
			{
				fvector2	pnt[3]	= { b_pos , w1 , b_w1 };
				NormalizeSampleOrder( pnt , w_r2 < 0.0f );
				OutlineNodePtr*	tnode	= &outline->CreateOutline()->m_node;
				outline->InsertNode( tnode , pnt , 3 );
			}
		}
		else
		{
			node = outline->InsertNode( node , b_pos );
			node = outline->InsertNode( node , pos );
			node = outline->InsertNode( node , w0 );
			outline->InsertNode( node , w1 );
			{
				fvector2	pnt[3]	= { b_pos , pos , w0 };
				NormalizeSampleOrder( pnt , w_r2 < 0.0f );
				OutlineNodePtr*	tnode	= &outline->CreateOutline()->m_node;
				outline->InsertNode( tnode , pnt , 3 );
			}
			{
				fvector2	pnt[3]	= { b_pos , w0 , b_w0 };
				NormalizeSampleOrder( pnt , w_r2 < 0.0f );
				OutlineNodePtr*	tnode	= &outline->CreateOutline()->m_node;
				outline->InsertNode( tnode , pnt , 3 );
			}
		}
	}
	else
	{
		node = outline->InsertNode( node , w0 );
		outline->InsertNode( node , w1 );
	}
	return node;
}
//=================================================================================================
//!	sub sampling mitter
//!	@retval			---
//-------------------------------------------------------------------------------------------------
template< BridgeFunc t_bridge_func >
OutlineNodePtr* SubSampling
		(
		iOutline&			outline ,
		OutlineNodePtr*		node , 
		IPathSegment*		ps , 
		float				st , 
		float				tt , 
		fvector2*			b_pos , 
		fvector2*			b_vec , 
		fvector2*			b_w0 , 
		fvector2*			b_w1 , 
		float				samplescale
		)
{
	fvector2	tp	= ps->Value( tt );
	fvector2	tuv	= ps->Differential( tt ).GetUnit();	

	// angle
	float	a;
	{	
		float	inr	= b_vec->Inner( tuv );
		a	= acosf( clip( inr , -1.0f , 1.0f ) );
	}
	// sub sampling
	int		spnum = ( int )floorf( samplescale * a  / m_da + 0.5f );
	if( spnum > 0 )
	{
		a		= b_vec->Outer( tuv ) < 0.0f ? -a : a;
		float		da		= a / ( float )( spnum + 1 );
		float		sin_da	= sinf( da );
		float		cos_da	= cosf( da );
		fvector2	r		= *b_w0 - *b_pos;
		float		dt		= ( tt - st ) / ( float )( spnum + 1 );
		float		t		= st;
		
		int			spoff;
		for( spoff = 0 ; spoff < spnum ; spoff++ )
		{
			// update
			float	rx = r.x;
			r.x = cos_da * rx - sin_da * r.y;
			r.y = sin_da * rx + cos_da * r.y;
			t += dt;
			
			// value
			fvector2	pos	= ps->Value( t );
			fvector2	w0	= pos + r;
			fvector2	w1	= pos - r;
			node = t_bridge_func( outline , node , a , m_w_r2 , *b_pos , *b_w0 , *b_w1 , pos , w0 , w1 );
			
			// store
			store( b_pos , pos );
			store( b_w0 , w0 );
			store( b_w1 , w1 );
		}
	}
	// last sampling
	{
		// value
		fvector2	n	= tuv.GetRotate90() * m_w_r2;
		fvector2	w0	= tp - n;
		fvector2	w1	= tp + n;
		node = t_bridge_func( outline , node , a , m_w_r2 , *b_pos , *b_w0 , *b_w1 , tp , w0 , w1 );
		
		// store
		store( b_pos , tp );
		store( b_vec , tuv );
		store( b_w0 , w0 );
		store( b_w1 , w1 );
	}	
	return node;
}
//=================================================================================================
//!	Sampling
//!	@retval			---
//-------------------------------------------------------------------------------------------------
template< BridgeFunc t_bridge_func >
OutlineNodePtr* cb_call SamplingT
		(
		iOutline&			outline ,
		OutlineNodePtr*		node , 
		IPathSegment*		ps , 
		float				st , 
		float				tt , 
		float				samplescale
		)
{
	if( ps->IsExist() == false )
		return node;
	if( st == tt )
		return node;

	int		spnum	= ps->GetOptimumSample( samplescale );
	spnum	= ( int )( ( tt - st ) * spnum );
	spnum	= spnum < 2 ? 2 : spnum;

	fvector2	pos , vec , w0 , w1;
	node = InsertStrokeNode( outline , node , ps , st , m_w_r2 , 0 , 0 , &pos , &vec , &w0 , &w1 );

	float	sst = st;
	int		spoff;
	for( spoff = 0 ; spoff < spnum - 1 ; spoff++ )
	{
		float	t = ( tt - st ) * ( spoff + 1 ) / ( float )( spnum - 1 ) + st;
		node	= SubSampling<t_bridge_func>( outline , node , ps , sst , t , &pos , &vec , &w0 , &w1 , samplescale );
		sst = t;
	}
	return node;
}
// public functions
public:
//=================================================================================================
//!	construct
//-------------------------------------------------------------------------------------------------
PathSegmentSamplerStroke() : 
		m_w_r2( 0.5f ) , 
		m_da( 1.0f ) , 
		m_bridgetype( Round_InnerBridgeType )
{
}
//=================================================================================================
//!	set width
//!	@retval			---
//-------------------------------------------------------------------------------------------------
void cb_call SetWidth
		(
		float		w_r2
		)
{
	m_w_r2			= w_r2;
	UpdateParam();
}
//=================================================================================================
//!	set inner bridge type
//!	@retval			---
//-------------------------------------------------------------------------------------------------
void cb_call SetInnerBridgeType
		(
		InnerBridgeType	type
		)
{
	m_bridgetype	= type;
}
//=================================================================================================
//!	Sampling
//!	@retval			---
//-------------------------------------------------------------------------------------------------
OutlineNodePtr* cb_call Sampling
		(
		iOutline&			outline ,
		OutlineNodePtr*		node , 
		IPathSegment*		ps , 
		float				st , 
		float				tt , 
		float				samplescale
		)
{
	if( m_bridgetype == Miter_InnerBridgeType )
		return SamplingT<PathSegmentSamplerStroke::Miter>( outline , node , ps , st , tt , samplescale );
	else if( m_bridgetype == Jag_InnerBridgeType )
		return SamplingT<PathSegmentSamplerStroke::Jag>( outline , node , ps , st , tt , samplescale );
	else
		return SamplingT<PathSegmentSamplerStroke::Round>( outline , node , ps , st , tt , samplescale );
}
};

/**************************************************************************************************
"PathSegmentSamplerDash" class 
**************************************************************************************************/
class PathSegmentSamplerDash
{
// variable member
private:
	PathLengthTbl					m_lengthtbl;
	PathSegmentSamplerStroke		m_strokesampler;
	float							m_w_r2;
	
// public functions
public:
//=================================================================================================
//!	construct
//-------------------------------------------------------------------------------------------------
PathSegmentSamplerDash() : 
		m_w_r2( 0.5f )
{
	m_strokesampler.SetWidth( m_w_r2 );
}
//=================================================================================================
//!	set width
//!	@retval			---
//-------------------------------------------------------------------------------------------------
void cb_call SetWidth
		(
		float		w_r2
		)
{
	m_w_r2			= w_r2;
	m_strokesampler.SetWidth( w_r2 );
}
//=================================================================================================
//!	set inner bridge type
//!	@retval			---
//-------------------------------------------------------------------------------------------------
void cb_call SetInnerBridgeType
		(
		InnerBridgeType	type
		)
{
	m_strokesampler.SetInnerBridgeType( type );
}
//=================================================================================================
//!	Sampling
//!	@retval			---
//-------------------------------------------------------------------------------------------------
OutlineNodePtr* cb_call Sampling
		(
		iOutline&			outline ,
		OutlineNodePtr*		node , 
		IPathSegment*		ps , 
		float				st , 
		float				tt , 
		const float*		dash , 
		int					dashnum , 
		int*				dashoff , 
		float*				dash_dl , 
		iCap&				cap ,
		float				samplescale
		)
{
	if( ps->IsExist() == false )
		return node;
	if( st == tt )
		return node;

	m_lengthtbl.SetPath( ps );
	float	dsl		= -( *dash_dl );
	float	dtl		= dash[ *dashoff ] + dsl;
	float	dtt		= m_lengthtbl.ToParam( dtl );

	fvector2	t1 , t2;
	while( dtt < tt )
	{
		if( node != 0 /*( ( *dashoff ) & 1 ) == 0*/ )
		{
			// draw st to dtt and end cap
			node	= m_strokesampler.Sampling( outline , node , ps , st , dtt , samplescale );
			fvector2	vec , w0 , w1;
			GetStrokeEdge( ps , dtt , m_w_r2 , 0 , &vec , &w0 , &w1 );
			node	= cap->InsertCap( outline , node , vec , w0 , w1 , samplescale );
			node	= 0;
		}
		else
		{
			// start cap dtt
			OutlineSeg*	outlineinfo	= outline->CreateOutline();
			node	= &outlineinfo->m_node;

			fvector2	vec , w0 , w1;
			GetStrokeEdge( ps , dtt , m_w_r2 , 0 , &vec , &w0 , &w1 );
			node = cap->InsertCap( outline , node , -vec , w1 , w0 , samplescale );
		}
		// update
		*dashoff	= ( *dashoff + 1 ) == dashnum ? 0 : ( *dashoff + 1 );
		dsl	= dtl;
		dtl	+= dash[ *dashoff ];
		st	= dtt;
		dtt	= m_lengthtbl.ToParam( dtl );
	}
	if( node != 0 && st < tt )
	{
		node	= m_strokesampler.Sampling( outline , node , ps , st , tt , samplescale );
	}
	*dash_dl	= m_lengthtbl.ToLength( tt ) - dsl;
	return node;
}
};

};	//namespace

//using namespace icubic;		

#pragma pack( pop )			//release align
