#include <kernel.h>
#include <task.h>
#include <intmgr.h>
#include <gdt.h>

TaskScheduler* TaskScheduler::Instance = 0;
TCB* TaskScheduler::CurrentTcb;

extern "C" void task_dispatch(TCB* tcb);

void TaskEntryPoint(Task* task) {
    task->run();
    while(true) {
        __asm__ __volatile__("hlt");
    }
}

//-------------------------------------------------------------------
// Task Class Method
//-------------------------------------------------------------------
Task::Task(size_t stack_size) :
    TimerJob(0), StackSize(stack_size), Priority(NORMAL_PRIORITY), 
    SubPriority(TaskPriority(2)), isStackInMemPool(true) {

    // set return address(dummy) & instance pointer
    Stack = new uint8[StackSize];
    *(reinterpret_cast<uint32*>(&Stack[StackSize - 8])) = 0; // dummy
    *(reinterpret_cast<uint32*>(&Stack[StackSize - 4])) = 
        reinterpret_cast<uint32>(this);

    // set task context
    Tcb.Context = reinterpret_cast<TaskContext*>(
        &Stack[StackSize - sizeof(TaskContext) - 8]);
    Tcb.Context->eflags = 0x0202;
    Tcb.Context->cs  = Gdt::KERNEL_CS * sizeof(GdtEntry);
    Tcb.Context->eip = reinterpret_cast<uint32>(TaskEntryPoint);
    Tcb.Context->ds  = Gdt::KERNEL_DS * sizeof(GdtEntry);
    Tcb.Context->es  = Tcb.Context->ds;
    Tcb.StackSegment = Tcb.Context->ds;
    Status = WAIT;
}

void Task::sendEvent(Event& evt) {
    EventQueue.enqueue(evt);
    TaskScheduler::getInstance()->wakeup(*this);
}

Event* Task::waitForEvent(void) {
    while (EventQueue.isEmpty()) {
        TaskScheduler::getInstance()->sleep(*this);
    }
    return (Event*)EventQueue.dequeue();
}

void Task::onExpire(void) {
    TaskScheduler::getInstance()->wakeup(*this);
}

//-------------------------------------------------------------------
// TaskScheduler Class Method
//-------------------------------------------------------------------
void TaskScheduler::dispatch(bool dispatch_immd) {
    Task* next_task;
    if (((next_task = (Task*)ReadyQueue.dequeue()) != 0) &&
        (CurrentTask->getPriority() <= next_task->getPriority())) {
        addReadyQueue(*CurrentTask);
        CurrentTask = next_task;
        CurrentTask->setStatus(Task::RUNNING);
        TimeSlice = NORMAL_TIME_SLICE;
        if (dispatch_immd) {
            task_dispatch(&CurrentTask->getTcb());
        } else {
            CurrentTcb = &CurrentTask->getTcb();
        }
    }
}

void TaskScheduler::setInitialTask(Task& cur_task) {
    CurrentTask = &cur_task;
    CurrentTcb  = &cur_task.getTcb();
    CurrentTask->setStatus(Task::RUNNING);
    TimeSlice = NORMAL_TIME_SLICE;
}

void TaskScheduler::sleep(Task& task, time_t duration) {
    IntMgr* int_mgr = IntMgr::getInstance();
    int_mgr->disable();
    if (CurrentTask == &task) {
       Task* next_task;
       if ((next_task = reinterpret_cast<Task*>(ReadyQueue.dequeue())) != 0) {
            CurrentTask->setStatus(Task::WAIT);
            if (duration != 0) {
                CurrentTask->setSleepTime(duration);
                Timer::getInstance()->add(*CurrentTask);
            }

            CurrentTask = next_task;
            CurrentTask->setStatus(Task::RUNNING);
            TimeSlice = NORMAL_TIME_SLICE;
            task_dispatch(&CurrentTask->getTcb());
        }
    } else if (task.getStatus() == Task::READY) {
        ReadyQueue.remove(task);
        task.setStatus(Task::WAIT);
    }
    int_mgr->enable();
}

void TaskScheduler::wakeup(Task& task) {
    if (task.getStatus() == Task::WAIT) addReadyQueue(task);
}

TCB* TaskScheduler::getCurrentTcb(void) {
    return &CurrentTask->getTcb();
}

