
#ifndef _TARRAY_H_
#define _TARRAY_H_

#include <string.h>
#include <assert.h>

namespace Houken {

    // simple array of T
    template <typename T> class TArray {
    protected:
        int     _capacity;
        int     _size;
        T*      _array;
    public:
        TArray(int initCapacity = 1)
            : _size(0)
        {
            if (initCapacity > 0)
                _capacity = initCapacity;
            else
                _capacity = 1;
            _array = new T[_capacity];
        }

        TArray(const T& t)
            : _capacity(1), _size(1)
        {
            _array = new T[_capacity];
            _array[0] = t;
        }

        ~TArray() {
            delete [] _array;
        }

        int size(void) { return _size; }

        int remain(void) { return _capacity - _size; }

        void clean(void) { _size = 0; }

        T& get(int idx) {
            assert((idx >= 0) && (idx < _size));
            return _array[idx];
        }

        void set(int idx, const T& x) {
            assert((idx >= 0) && (idx < _size));
            _array[idx] = x;
        }

        TArray* add(const T& t) {
            expand();
            _array[_size++] = t;
            return this;
        }

        T* keep(void) {
            expand();
            return &_array[_size++];
        }
        
        TArray* remove(int idx) {
            assert((idx >= 0) && (idx < _size));
            for (int i = idx+1; i < _size; ++idx,++i)
                _array[idx] = _array[i];
            -- _size;
            return this;
        }

        TArray* insertAt(int idx, const T& t) {
            expand();
            for (int i = _size; i > idx; --i) {
                _array[i] = _array[i-1];
            }
            ++_size;
            _array[idx] = t;
            return this;
        }

        void expand(void) {
            if (_size >= _capacity)
                resize( ((_capacity * 3 + 3) >> 2) * 2 );
        }

        void resize(int newCapacity) {
            if (newCapacity <= _capacity)
                return;
            T* newArr = new T[newCapacity];
            assert(newArr != NULL);
            memcpy(newArr, _array, _capacity * sizeof(T));
            delete [] _array;
            _array = newArr;
            _capacity = newCapacity;
        }

        void chop(int idx) {
            assert((idx >= 0) && (idx <= _size));
            _size = idx;
        }

    };


    template <typename T> class TArrayIterator {
    protected:
        TArray<T>*      _arr;
        int             _idx;
        int             _size;
    public:
        TArrayIterator(TArray<T>* arr)
            : _arr(arr)
        {
            _idx = 0;
            _size = arr->size();
        }

        bool hasMore(void)
        {
            return _idx < _size;
        }

        const T& next(void)
        {
            assert(_idx < _size);
            return _arr->get(_idx++);
        }
            
        void rewind(void)
        {
            _idx = 0;
        }
    };

}


#endif /* _TARRAY_H_ */
