
#include "PegParser.h"
#include "../etc/printSyntaxTree.h"
#include <stdio.h>
#include <string.h>


#define THIS_NAME       "houken"
#define VERSION         "version 0.1"
#ifdef __CYGWIN__
#define EXE_FILENAME    "houken.exe"
#else
#define EXE_FILENAME    "houken"
#endif

char* tab_basename;     // export to PegParserAction.cpp

using namespace Houken;


void printError(InputBuffer* inp)
{
    int n = Parser::numErrors();
    for (--n; n >= 0; --n) {
        Parser::Error_t err = Parser::getError(n);
        char pbuf[128];
        inp->posInfo(pbuf, 128, err.pos);
        const char* nam = err.parser->name();
        if ((nam != NULL) && (*nam != '('))
            printf("parser=%s, %s\n", nam, pbuf);
    }
}

void usage(void)
{
    printf("usage: " EXE_FILENAME " [options] PEGfile\n");
    printf("option:\n");
    printf("  -m : not use memoization\n");
    printf("  -s : print syntax tree\n");
    printf("  -d : print intermediate parsing result\n");
    printf("  -d2 : print detail intermediate parsing result\n");
}

int main(int argc, const char* argv[])
{
    const char* pegFileName = NULL;
    bool bPrintSyntaxTree = false;
    int  printIntermediateLevel = 0;
    bool bUseMemoize = true;

    printf("PEG parser generator " THIS_NAME " " VERSION "\n");
    for (int i = 1; i < argc; i++) {
        if (argv[i][0] == '-')
            for (const char* p = &argv[i][1]; *p != '\0'; ++p) {
                switch (*p) {
                case 's':
                    bPrintSyntaxTree = true;
                    break;
                case 'd':
                    if (p[1] >= '0' && p[1] <= '9')
                        printIntermediateLevel = *++p - '0';
                    else
                        printIntermediateLevel = 1;
                    break;
                case 'm':
                    bUseMemoize = false;
                    break;
                default:
                    printf("unknown option: %s\n", argv[i]);
                    usage();
                    return 1;
                }
            }
        else {
            if (pegFileName != NULL) {
                printf("error: only 1 PEGfile required\n");
                usage();
                return 1;
            }
            pegFileName = argv[i];
        }
    }

    if (pegFileName == NULL) {
        printf("error: PEGfile required\n");
        usage();
        return 1;
    }

    const char* p = strchr(pegFileName, '.');
    size_t len = (p == NULL) ? strlen(pegFileName) : p - pegFileName;
    tab_basename = new char[len+1];
    strncpy(tab_basename, pegFileName, len);
    tab_basename[len] = '\0';

    printf("generating %s.tab.* from %s\n", tab_basename, pegFileName);

    FileInputBuffer inp(pegFileName);
    Parser::initialize(&inp, 40);
    Parser::setUseMemoize(bUseMemoize);
    Parser::setPrintIntermediateLevel(printIntermediateLevel);
    SyntaxTree* st = p_Start->parse();
    if (st->isValidTree()) {
        if (bPrintSyntaxTree)
            printSyntaxTree(st, &inp, 0);
        p_Start->action(st, NULL);
        printf("done.\n");
        delete st;
    } else {
        printf("parse error\n");
        printError(&inp);
    }

    delete[] tab_basename;

    Parser::finalize();
    SyntaxTree::finalize();
 }
