/** 
 *  Hyper Operating System V4 Advance
 *
 * @file  tget_mpf.c
 * @brief %jp{Œ蒷v[̊l(^CAEg)}%en{Acquired Fixed-Sized Memory Block(with Timeout)}
 *
 * Copyright (C) 1998-2007 by Project HOS
 * http://sourceforge.jp/projects/hos/
 */



#include "core/core.h"
#include "object/mpfobj.h"



#if _KERNEL_SPT_TGET_MPF


#if _KERNEL_SPT_TGET_MPF && (_KERNEL_OPT_CODE_SIZE <= _KERNEL_OPT_SPEED)	/* %jp{tget_mpfŁATCYDȂ} */

/** %jp{Œ蒷v[̊l(^CAEg)}%en{Acquired Fixed-Sized Memory Block(with Timeout)}
 * @param  mpfid	%jp{ubNlΏۂ̌Œ蒷v[IDԍ}%en{ID number of the fixed-sized memory pool from which resource is acquired}
 * @param  p_blkf	%jp{Œ蒷v[ꂽpPbgւ̃|C^}%en{Start address of the acquired memory block}
 * @retval E_OK     %jp{I}%en{Normal completion}
 * @retval E_ID     %jp{sIDԍ(mpfids邢͎gpłȂ)}%en{Invalid ID number(mpfid is invalid or unusable)}
 * @retval E_NOEXS  %jp{IuWFNg(ΏیŒ蒷v[o^)}%en{Non-existant object(specified fixed-sized memory pool is not registerd)}
 * @retval E_PAR    %jp{p[^G[(p_blk, tmouts)}%en{Forced release from waiting(accept rel_wai while waiting)}
 * @retval E_RLWAI  %jp{҂Ԃ̋(҂Ԃ̊Ԃrel_wait)}%en{Forced release from waiting(accept rel_wai while waiting)}
 * @retval E_DLT    %jp{҂IuWFNg̍폜(҂Ԃ̊ԂɌŒ蒷v[폜)}%en{Waiting object deleted(fixed-sized memory pool is deleted waiting)}
 */
ER tget_tmpf(ID mpfid, VP *p_blk, TMO tmout)
{
	_KERNEL_T_MPFCB_PTR  mpfcb;
	_KERNEL_T_TSKHDL     tskhdl;
	_KERNEL_T_TCB        *tcb;
	_KERNEL_MPF_T_BLKHDL blkhdl;
	VP                   blkptr;
	ER                   ercd;
	
	/* %jp{ReLXg`FbN} */
#if _KERNEL_SPT_GET_MPF_E_CTX
	if ( _KERNEL_SYS_SNS_DPN() )
	{
		return E_CTX;			/* %jp{ReLXgG[}%en{Context error} */
	}
#endif
	
	/* %jp{ID ̃`FbN} */
#if _KERNEL_SPT_GET_MPF_E_ID
	if ( !_KERNEL_MPF_CHECK_MPFID(mpfid) )
	{
		return E_ID;	/* %jp{sIDԍ}%en{Invalid ID number} */
	}
#endif
	
	_KERNEL_ENTER_SVC();		/* %jp{T[rXR[ɓ}%en{enter service-call} */
	
	/* %jp{IuWFNg݃`FbN} */
#if _KERNEL_SPT_GET_MPF_E_NOEXS
	if ( !_KERNEL_MPF_CHECK_EXS(mpfid) )
	{
		_KERNEL_LEAVE_SVC();	/* %jp{T[rXR[o}%en{leave service-call} */
		return E_NOEXS;			/* %jp{IuWFNg}%en{Non-existant object} */
	}
#endif

	/* %jp{Rg[ubN擾} */
	mpfcb = _KERNEL_MPF_ID2MPFCB(mpfid);
	
	/* %jp{Œ蒷v[JE^擾} */
	blkhdl = _KERNEL_MPF_GET_FREBLK(mpfcb);
	
	if ( blkhdl != _KERNEL_MPF_BLKHDL_NULL )
	{
		/* %jp{Œ蒷v[lłΐ} */
		blkptr = _KERNEL_MPF_BLKHDL2PTR(mpfcb, blkhdl);
		*p_blk = blkptr;
		_KERNEL_MPF_SET_FREBLK(mpfcb, *(_KERNEL_MPF_T_BLKHDL *)blkptr);
		_KERNEL_MPF_SET_FBLKCNT(mpfcb, _KERNEL_MPF_GET_FBLKCNT(mpfhdl) - 1);
		ercd = E_OK;
	}
	else
	{
#if _KERNEL_SPT_TGET_MPF || _KERNEL_SPT_PGET_MPF
		if ( tmout == TMO_POL )
		{
			ercd = E_TMOUT;  /* %jp{^CAEg}%en{Timeout} */
		}
		else
#endif
		{
				/* %jp{^XN҂Ԃɂ} */
			tskhdl = _KERNEL_SYS_GET_RUNTSK();
			tcb = _KERNEL_TSK_TSKHDL2TCB(tskhdl);				/* %jp{TCB擾} */
			_KERNEL_TSK_SET_TSKWAIT(tcb, _KERNEL_TTW_MPF);
			_KERNEL_TSK_SET_WOBJID(tcb, mpfid);
			_KERNEL_TSK_SET_DATA(tcb, (VP_INT)p_blk);
			_KERNEL_DSP_WAI_TSK(tskhdl);
			_KERNEL_MPF_ADD_QUE(mpfcb, _KERNEL_MPF_GET_MPFCB_RO(mpfid, mpfcb), tskhdl);		/* %jp{҂sɒǉ} */

#if _KERNEL_SPT_TGET_MPF
			if ( tmout != TMO_FEVR )
			{
				_KERNEL_SEM_ADD_TOQ(tskhdl, tmout);				/* %jp{^CAEgL[ɒǉ} */
			}
#endif

			/* %jp{^XNfBXpb`̎s} */
			_KERNEL_DSP_TSK();

			/* %jp{G[R[h̎擾} */
			ercd = _KERNEL_TSK_GET_ERCD(tcb);
		}
	}
	
	_KERNEL_LEAVE_SVC();	/* %jp{IuWFNg}%en{Non-existant object} */
	
	return ercd;
}


#endif


#else	/* _KERNEL_SPT_PGET_MPF */


#if _KERNEL_SPT_PGET_MPF_E_NOSPT

/** %jp{Œ蒷v[̊l(^CAEg)}%en{Acquired Fixed-Sized Memory Block(with Timeout)}
 * @param  mpfid	%jp{ubNlΏۂ̌Œ蒷v[IDԍ}%en{ID number of the fixed-sized memory pool from which resource is acquired}
 * @param  p_blkf	%jp{Œ蒷v[ꂽpPbgւ̃|C^}%en{Start address of the acquired memory block}
 * @retval E_NOSPT  %jp{T|[g@\}%en{Unsupported function}
 */
ER tget_mpf(ID mpfid)
{
	return E_NOSPT;
}

#endif


#endif	/* _KERNEL_SPT_PGET_MPF */



/* end of file */
