/** 
 *  Hyper Operating System V4 Advance
 *
 * @file  tsnd_dtq.c
 * @brief %jp{f[^L[ւ̑M(^CAEg)}%en{Send to Data Queue(with Timeout)}
 *
 * Copyright (C) 1998-2009 by Project HOS
 * http://sourceforge.jp/projects/hos/
 */



#include "core/core.h"
#include "object/dtqobj.h"



#if _KERNEL_SPT_TSND_DTQ


/** %jp{f[^L[ւ̑M(^CAEg)}%en{Send to Data Queue(with Timeout)}
 * @param  dtqid    %jp{MΏۂ̃f[^L[IDԍ}%en{ID number of the data queue to which the data element is sent}
 * @param  data     %jp{f[^L[֑Mf[^}%en{Data element to be sent}
 * @param  tmout    %jp{^CAEgw}%en{Specified timeout}
 * @retval E_OK     %jp{I}%en{Normal completion}
 * @retval E_ID     %jp{sIDԍ(dtqids邢͎gpłȂ)}%en{Invalid ID number(dtqid is invalid or unusable)}
 * @retval E_CTX    %jp{ReLXgG[}%en{Context error}
 * @retval E_NOEXS  %jp{IuWFNg(Ώۃf[^L[o^)}%en{Non-existant object(specified data queue is not registerd)}
 * @retval E_PAR    %jp{p[^G[(tmouts)}%en{Parameter error(tmout is invalid)}
 * @retval E_RLWAI  %jp{҂Ԃ̋(҂Ԃ̊Ԃrel_wait)}%en{Forced release from waiting(accept rel_wai while waiting)}
 * @retval E_TMOUT  %jp{^CAEg}%en{Timeout}
 * @retval E_DLT    %jp{҂IuWFNg̍폜(҂Ԃ̊ԂɑΏۃZ}tH폜)}%en{Waiting object deleted(semaphore is deleted waiting)}
 */
ER tsnd_dtq(ID dtqid, VP_INT data, TMO tmout)
{
	return _kernel_snd_dtq(dtqid, data, tmout);
}


#else	/* _KERNEL_SPT_TSND_DTQ */


#if _KERNEL_SPT_TSND_DTQ_E_NOSPT

/** %jp{f[^L[ւ̑M(^CAEg)}%en{Send to Data Queue(with Timeout)}
 * @param  dtqid    %jp{MΏۂ̃f[^L[IDԍ}%en{ID number of the data queue to which the data element is sent}
 * @param  data     %jp{f[^L[֑Mf[^}%en{Data element to be sent}
 * @param  tmout    %jp{^CAEgw}%en{Specified timeout}
 * @retval E_NOSPT  %jp{T|[g@\}%en{Unsupported function}
 */
ER tsnd_dtq(ID dtqid, VP_INT data, TMO tmout)
{
	return E_NOSPT;
}

#endif


#endif	/* _KERNEL_SPT_TSND_DTQ */



#if _KERNEL_SPT_KSND_DTQ

ER _kernel_snd_dtq(ID dtqid, VP_INT data, TMO tmout)
{
	_KERNEL_T_DTQCB  *dtqcb;
	_KERNEL_T_TSKHDL tskhdl;
	_KERNEL_T_TCB    *tcb;
	ER               ercd;
	
	/* %jp{ReLXg`FbN} */
#if _KERNEL_SPT_TSND_DTQ_E_CTX
	if ( _KERNEL_SYS_SNS_DPN() )
	{
		return E_CTX;			/* %jp{ReLXgG[}%en{Context error} */
	}
#endif
	
	/* %jp{ID ̃`FbN} */
#if _KERNEL_SPT_TSND_DTQ_E_ID
	if ( !_KERNEL_DTQ_CHECK_DTQID(dtqid) )
	{
		return E_ID;	/* %jp{sIDԍ}%en{Invalid ID number} */
	}
#endif

	/* %jp{p[^̃`FbN} */
#if _KERNEL_SPT_TSND_DTQ_E_PAR
	if ( tmout < 0 && tmout != TMO_FEVR && tmout != TMO_POL)
	{
		return E_PAR;	/* %jp{p[^G[(tmouts)}%en{Parameter error(tmout is invalid)} */
	}
#endif
	
	_KERNEL_ENTER_SVC();	/* %jp{T[rXR[ɓ}%en{enter service-call} */
	
	/* %jp{IuWFNg݃`FbN} */
#if _KERNEL_SPT_TSND_DTQ_E_NOEXS
	if ( !_KERNEL_DTQ_CHECK_EXS(dtqid) )
	{
		_KERNEL_LEAVE_SVC();	/* %jp{T[rXR[I} */
		return E_NOEXS;			/* %jp{IuWFNg} */
	}
#endif
	
	/* %jp{f[^L[Rg[ubN擾} */
	dtqcb = _KERNEL_DTQ_ID2DTQCB(dtqid);
	
	/* %jp{M҂s擪^XNo} */
	tskhdl = _KERNEL_DTQ_RMH_RQUE(dtqcb);
	if ( tskhdl != _KERNEL_TSKHDL_NULL )
	{
		VP_INT *p_data;

		/* %jp{҂^XNΑ҂} */
		tcb = _KERNEL_TSK_TSKHDL2TCB(tskhdl);		/* %jp{TCB擾} */
		_KERNEL_TSK_SET_ERCD(tcb, E_OK);			/* %jp{G[R[hݒ} */
		p_data = (VP_INT *)_KERNEL_TSK_GET_DATA(tcb);
		*p_data = data;
		_KERNEL_DSP_WUP_TSK(tskhdl);				/* %jp{^XN̑҂} */
		_KERNEL_DTQ_RMV_RTOQ(tskhdl);
		
		/* %jp{^XNfBXpb`̎s} */
		_KERNEL_DSP_TSK();
		
		ercd = E_OK;	/* %jp{I}%en{Normal completion} */
	}
	else
	{
		const _KERNEL_T_DTQCB_RO *dtqcb_ro;
		_KERNEL_DTQ_T_DTQCNT     sdtqcnt;
		_KERNEL_DTQ_T_DTQCNT     dtqcnt;
		
		/* %jp{RO擾} */
		dtqcb_ro = _KERNEL_DTQ_GET_DTQCB_RO(dtqid, dtqcb);

		/* %jp{f[^L[JE^擾} */
		sdtqcnt = _KERNEL_DTQ_GET_SDTQCNT(dtqcb);
		dtqcnt  = _KERNEL_DTQ_GET_DTQCNT(dtqcb_ro);

		if ( sdtqcnt < dtqcnt )		/* %jp{L[ɋ󂫂͂邩H} */
		{
			_KERNEL_DTQ_T_DTQCNT head;
			VP_INT               *dtq;

			/* %jp{f[^L[擾} */
			head = _KERNEL_DTQ_GET_HEAD(dtqcb);
			dtq  = _KERNEL_DTQ_GET_DTQ(dtqcb_ro);
			
			/* %jp{f[^L[ɒǉ} */
			if ( head < dtqcnt - sdtqcnt - 1 )
			{
				dtq[head + sdtqcnt] = data;
			}
			else
			{
				dtq[head + sdtqcnt - dtqcnt] = data;
			}
			
			/* %jp{f[^Z} */
			sdtqcnt++;
			_KERNEL_DTQ_SET_SDTQCNT(dtqcb, sdtqcnt);

			ercd = E_OK;	/* %jp{I}%en{Normal completion} */
		}
		else
		{
#if _KERNEL_SPT_TSND_DTQ || _KERNEL_SPT_PSND_DTQ
			if ( tmout == TMO_POL )
			{
				ercd = E_TMOUT;  /* %jp{^CAEg}%en{Timeout} */
			}
			else
#endif
			{
				/* %jp{^XN҂Ԃɂ} */
				tskhdl = _KERNEL_SYS_GET_RUNTSK();
				tcb    = _KERNEL_TSK_TSKHDL2TCB(tskhdl);			/* %jp{TCB擾} */
				_KERNEL_TSK_SET_TSKWAIT(tcb, _KERNEL_TTW_SDTQ);
				_KERNEL_TSK_SET_WOBJID(tcb, dtqid);
				_KERNEL_TSK_SET_DATA(tcb, data);
				_KERNEL_DSP_WAI_TSK(tskhdl);
				_KERNEL_DTQ_ADD_SQUE(dtqcb, dtqcb_ro, tskhdl);		/* %jp{҂sɒǉ} */

#if _KERNEL_SPT_TSND_DTQ
			if ( tmout != TMO_FEVR )
			{
				_KERNEL_DTQ_ADD_STOQ(tskhdl, tmout);				/* %jp{^CAEgL[ɒǉ} */
			}
#endif
				/* %jp{^XNfBXpb`̎s} */
				_KERNEL_DSP_TSK();
				
				/* %jp{G[R[h̎擾} */
				ercd = _KERNEL_TSK_GET_ERCD(tcb);
			}
		}		
	}
	
	_KERNEL_LEAVE_SVC();	/* %jp{T[rXR[o}%en{leave service-call} */
	
	return ercd;
}

#endif	/* _KERNEL_SPT_KSND_DTQ */



/* end of file */
