#!/bin/bash

# Copyright (C) 2013 NTT DATA Corporation
# 
# This program is free software; you can redistribute it and/or
# Modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation, version 2.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details

FILENAME_POSTFIX=`date '+%Y-%m-%d_%H-%M-%S'`
SUMMARY_FILE="/opt/hinemos/var/log/hinemos_manager_summary.${FILENAME_POSTFIX}"

echo ""
echo ""
echo "#################################################"
echo "### Hinemos   stats collect script (20120401) ###"
echo "###    Script start at `date '+%Y-%m-%d %T'`    ###"
echo "#################################################"
echo ""

function usage {
        echo "usage: "
        echo "$0 -v   : Get information about HINEMOS SUMMARY"
        echo "$0 -vv  : Get information about HINEMOS SUMMARY + OS INFORMATION"
        echo "$0 -vvv : Get information about DETAIL"
        echo "$0 -o   : Get information about DETAIL + Log about HINEMOS MANAGER"
        echo "$0 -t   : Get thread dump"
        return 0
}

function init {
## Hinemos env
SCRIPT_DIR=$(cd $(dirname $0);pwd)
. ${SCRIPT_DIR}/../../hinemos.cfg

}

function hinemos {
## Version
VER=`cat ${HINEMOS_HOME}/_version`
VER_MJ=`echo ${VER} | cut -c1-3`

echo "Installed Hinemos ver : ${VER}" | tee -a ${SUMMARY_FILE}
echo "" | tee -a ${SUMMARY_FILE}

## init valiable
HOST="localhost"
PGPORT=24001
LDAPPORT=24000
case ${VER_MJ} in
    '2.4')
    PSQLBIN="${PG}/bin/psql"
    TABLE_COLLECTOR_PROFILE="collector_profile"
    TABLE_CALCULATED_DATA="calculated_data"
    COLUMN_COLLECTOR_PROFILE_INTERVAL="interval"
    VALUE_TRIGGER_NAME_PROCESS="PROCESS"
    ;;
    '3.0' | '3.1')
    PSQLBIN="${PG}/bin/psql"
    PG_FACILITY="false"
    if [ `echo ${VER} | grep -E "3\.1\.[0-9]+\.L" | wc -l` -ne 0 ]
    then
        # for Light
        PG_FACILITY="true"
    fi
    TABLE_COLLECTOR_PROFILE="cc_collector_profile"
    TABLE_CALCULATED_DATA="cc_calculated_data"
    COLUMN_COLLECTOR_PROFILE_INTERVAL="interval_sec"
    VALUE_TRIGGER_NAME_PROCESS="MON_PRC"
    ;;
    '3.2')
    PSQLBIN="${PG}/bin/psql"
    PG_FACILITY="true"
    TABLE_COLLECTOR_PROFILE="cc_collector_profile"
    TABLE_CALCULATED_DATA="cc_calculated_data"
    COLUMN_COLLECTOR_PROFILE_INTERVAL="interval_sec"
    VALUE_TRIGGER_NAME_PROCESS="MON_PRC"
    ;;
    '4.0' | '4.1')
    PSQLBIN="${PG_HOME}/bin/psql"
    PG_FACILITY="true"
    TABLE_CALCULATED_DATA="cc_calculated_data"
    COLUMN_COLLECTOR_PROFILE_INTERVAL="interval_sec"
    VALUE_TRIGGER_NAME_PROCESS="MON_PRC"
    ;;
    '2.0' | '2.1' | '2.2' | '2.3')
    echo "This script does not support Hinemos ${VER}" 1>&2
    echo "VER=${VER}, VER_MJ=${VER_MJ}"
    exit 1
    ;;
esac
LDAPBIN="${LDAP}/bin/ldapsearch"
DBNAME=hinemos
USER=hinemos
FILENAME="${HINEMOS_HOME}/var/log/db-summary.`date '+%Y-%m-%d_%T'`"
TMPFILE="${HINEMOS_HOME}/var/log/db-summary.tmp"

PSQLCMD="${PSQLBIN} -p ${PGPORT} -U ${USER} -d ${DBNAME} "
PSQLCMD_NA="${PSQLBIN} -p ${PGPORT} -U ${USER} -d ${DBNAME} -A -t "

## postgresql password
## postgresql connect check
export PGPASSWORD="hinemos"
stty -echo
read -p "PostgreSQL Password (default hinemos) : " PASSWD
stty echo
echo
if [ "${PASSWD}" != "" ] ; then
    export PGPASSWORD=${PASSWD}
fi
RESULT=`${PSQLCMD_NA} -c "SELECT 1;"`
EXIT_CODE=$?
if [ ${EXIT_CODE} -ne 0 ] ; then
    echo "Failed to access PostgreSQL" 1>&2
    echo "exit code ${EXIT_CODE}" 1>&2

    rm ${SUMMARY_FILE}

    exit ${EXIT_CODE}
fi

if [ "x${PG_FACILITY}" = "xfalse" ]
then
    ## ldap password
    ## openldap connect check
    stty -echo
    read -p "OpenLDAP Password (default hinemos) : " PASSWD
    stty echo
    echo
    if [ "${PASSWD}" != "" ] ; then
        export LDAPPASSWD=${PASSWD}
    else
        export LDAPPASSWD="hinemos"
    fi
    LDAPCMD="${LDAPBIN} -h ${HOST} -p ${LDAPPORT} -LLL -D "cn=Manager,dc=hinemos,dc=com" -x -w ${LDAPPASSWD}"
    RESULT=`${LDAPCMD} -b "dc=hinemos,dc=com"`
    EXIT_CODE=$?
    if [ ${EXIT_CODE} -ne 0 ] ; then
        echo "Failed to access OpenLDAP" 1>&2
        echo "exit code ${EXIT_CODE}" 1>&2

        rm ${SUMMARY_FILE}

        exit ${EXIT_CODE}
    fi
fi

# if jboss is installed
if [ -e ${JBOSS_HOME}/bin/twiddle.sh ]
then
	echo -n "clustercontrolRepositoryUpdate: " | tee -a ${SUMMARY_FILE}
	${JBOSS_HOME}/bin/twiddle.sh -s jnp://127.0.0.1:1099 get "jboss.messaging.destination:service=Topic,name=clustercontrolRepositoryUpdate" AllSubscriptionsCount | tee -a ${SUMMARY_FILE}
	
	echo -n "clustercontrolJobManagementExecute: " | tee -a ${SUMMARY_FILE}
	${JBOSS_HOME}/bin/twiddle.sh -s jnp://127.0.0.1:1099 get "jboss.messaging.destination:service=Topic,name=clustercontrolJobManagementExecute" AllSubscriptionsCount | tee -a ${SUMMARY_FILE}
	
	echo -n "clustercontrolLogtransferUpdate: " | tee -a ${SUMMARY_FILE}
	${JBOSS_HOME}/bin/twiddle.sh -s jnp://127.0.0.1:1099 get "jboss.messaging.destination:service=Topic,name=clustercontrolLogtransferUpdate" AllSubscriptionsCount | tee -a ${SUMMARY_FILE}
fi

echo "" | tee -a ${SUMMARY_FILE}

## qrtz_simple_trigger summary
echo "### Simple trigger summary ###" | tee -a ${SUMMARY_FILE}
${PSQLCMD} -c "
    SELECT
        a.trigger_state,
        a.trigger_group,
        b.repeat_interval,
        count(*)
    FROM
        qrtz_triggers as a,
        qrtz_simple_triggers as b
    WHERE
        a.trigger_name = b.trigger_name
    GROUP BY
        a.trigger_state,
        a.trigger_group,
        b.repeat_interval
    ORDER BY
        a.trigger_state,
        a.trigger_group,
        b.repeat_interval
" | tee -a ${SUMMARY_FILE}

## qrtz_cron_trigger summary
echo "### Cron trigger summary ###" | tee -a ${SUMMARY_FILE}
${PSQLCMD} -c "
    SELECT
        a.trigger_state,
        a.trigger_group,
        b.cron_expression,
        count(*)
    FROM
        qrtz_triggers as a,
        qrtz_cron_triggers as b
    WHERE
        a.trigger_name = b.trigger_name
    GROUP BY
        a.trigger_state,
        a.trigger_group,
        b.cron_expression -- fire_duration_minutes
    ORDER BY
        a.trigger_state,
        a.trigger_group
" | tee -a ${SUMMARY_FILE}

## "resource management" summary (collector_profile)
if [ ${VER_MJ} != '4.0' ] && [ ${VER_MJ} != '4.1' ]; then
echo "### collector_profile summary ###" | tee -a ${SUMMARY_FILE}
${PSQLCMD} -c "
    SELECT
        CASE collector_type
            WHEN 1 THEN 'REALTIME'
            WHEN 2 THEN
                CASE
                    WHEN stop_date is null THEN 'RECORD_NO_TIMELIMIT'
                    WHEN stop_date - now() > interval '0' THEN 'RECORD_TIMELIMIT'
                    ELSE 'RECORD_FINISHED'
                END
            WHEN 3 THEN 'MONITOR'
            ELSE 'UNKNOWN'
        END AS type,
        ${COLUMN_COLLECTOR_PROFILE_INTERVAL},
        count(*)
    FROM
        ${TABLE_COLLECTOR_PROFILE}
    GROUP BY
        type,
        ${COLUMN_COLLECTOR_PROFILE_INTERVAL}
" | tee -a ${SUMMARY_FILE}
fi

## Syslog-ng/SNMPTrap filter count
if [ ${VER_MJ} != '4.0' ] && [ ${VER_MJ} != '4.1' ]; then
echo "### Syslog-ng/SNMP-TRAP filter summary ###" | tee -a ${SUMMARY_FILE}
${PSQLCMD} -c "
    SELECT
        'syslog-ng' as type,
        valid_flg,
        count(*)
    FROM
        cc_syslog_monitor_rule
    GROUP BY
        valid_flg
    UNION
    SELECT
        'snmp-trap' as type,
        valid_flg,
        count(*)
    FROM
        cc_snmp_trap_oid_info
    GROUP BY
        valid_flg
" | tee -a ${SUMMARY_FILE}
fi

## JOB-net
echo "### Job-net summary ###" | tee -a ${SUMMARY_FILE}
${PSQLCMD} -c "
    SELECT
        CASE job_type
            WHEN 0 THEN 'JU'
            WHEN 1 THEN 'JN'
            WHEN 2 THEN 'JB'
            WHEN 3 THEN 'FT'
            ELSE        '??'
        END AS type,
        count(*)
    FROM
        cc_job_mst
    GROUP BY
        job_type
    ORDER BY
        job_type
" | tee -a ${SUMMARY_FILE}

## running job
echo "### Running Job count ###" | tee -a ${SUMMARY_FILE}
${PSQLCMD} -c "
    SELECT count(*)
    FROM
        cc_job_session s, cc_job_session_job j
    WHERE
        s.session_id = j.session_id AND s.jobunit_id = j.jobunit_id AND
         s.job_id = j.job_id AND status / 100 != 3
" | tee -a ${SUMMARY_FILE}


## log table size
echo "### log table summary ###" | tee -a ${SUMMARY_FILE}
${PSQLCMD} -c "
    SELECT
        relname,
        reltuples,
        pg_relation_size(relname::regclass)
    FROM
        pg_class
    WHERE
        relname IN (
            '${TABLE_CALCULATED_DATA}',
            'cc_event_log',
            'cc_crun_session',
            'cc_crun_session_detail',
            'cc_job_session',
            'cc_job_session_job',
            'cc_job_session_node'
        )
" | tee -a ${SUMMARY_FILE}

## large table
echo "### more than 100 record table ###" | tee -a ${SUMMARY_FILE}
${PSQLCMD} -c "
    SELECT
        relname,
        reltuples,
        pg_relation_size(relname::regclass)
    FROM
        pg_class
    WHERE
        relkind = 'r' AND
        relnamespace = (
            SELECT
                oid
            FROM
                pg_namespace
            WHERE
                nspname = 'public'
            limit 1
        ) AND
        reltuples > 100
    ORDER BY
        reltuples DESC
" | tee -a ${SUMMARY_FILE}

if [ "x${PG_FACILITY}" = "xfalse" ]
then
    ## LDAP info
    echo "### ldap repository info ###" | tee -a ${SUMMARY_FILE}
    
    SCOPE=`${LDAPCMD} -b "ou=facilitytree,dc=hinemos,dc=com" "(ccTreeType=0)" ccFacilityId | grep "ccFacilityId:" | wc -l`
    REALNODE=`${LDAPCMD} -b "ou=facilitytree,dc=hinemos,dc=com" "(ccTreeType=1)" ccFacilityId | grep "ccFacilityId:" | sort | uniq | wc -l`
    VIRTNODE=`${LDAPCMD} -b "ou=facilitytree,dc=hinemos,dc=com" "(ccTreeType=1)" ccFacilityId | grep "ccFacilityId:" | wc -l`
    DISK=`${LDAPCMD} -b "ou=facility,dc=hinemos,dc=com" "(ccDeviceType=disk)" cn | grep "cn: " | wc -l`
    NIC=`${LDAPCMD} -b "ou=facility,dc=hinemos,dc=com" "(ccDeviceType=nic)" cn | grep "cn: " | wc -l`
    FS=`${LDAPCMD} -b "ou=facility,dc=hinemos,dc=com" "(objectClass=ccFileSystemInfo)" cn | grep "cn: " | wc -l`
else
    ## PostgreSQL info
    echo "### rdbms repository info ###" | tee -a ${SUMMARY_FILE}
    
    SCOPE=`${PSQLCMD_NA} -c "SELECT count(*) FROM cc_cfg_facility WHERE facility_type = 0"`
    REALNODE=`${PSQLCMD_NA} -c "SELECT count(*) FROM cc_cfg_facility WHERE facility_type = 1"`
    VIRTNODE=`${PSQLCMD_NA} -c " SELECT count(*) FROM cc_cfg_facility_relation WHERE parent_facility_id IN ( SELECT facility_id FROM cc_cfg_facility WHERE facility_type = 0 )"`
    DISK=`${PSQLCMD_NA} -c "SELECT count(*) FROM cc_cfg_node_device WHERE device_type = 'disk'"`
    NIC=`${PSQLCMD_NA} -c "SELECT count(*) FROM cc_cfg_node_device WHERE device_type = 'nic'"`
    FS=`${PSQLCMD_NA} -c "SELECT count(*) FROM cc_cfg_node_filesystem"`
fi

echo "Scope      : ${SCOPE}" | tee -a ${SUMMARY_FILE}
echo "Node       : ${VIRTNODE} (RealNode : ${REALNODE})" | tee -a ${SUMMARY_FILE}
echo "DISK       : ${DISK}" | tee -a ${SUMMARY_FILE}
echo "NIC        : ${NIC}" | tee -a ${SUMMARY_FILE}
echo "FileSystem : ${FS}" | tee -a ${SUMMARY_FILE}
echo "" | tee -a ${SUMMARY_FILE}

if [ "x${PG_FACILITY}" = "xtrue" ]
then
    echo "### Node Platform ###" | tee -a ${SUMMARY_FILE}
    ${PSQLCMD} -c "
        SELECT
            platform_family,
            sub_platform_family,
            vm_node_type,
            cloud_node_type,
            count(*) count
        FROM
            cc_cfg_node
        GROUP BY
            platform_family,
            sub_platform_family,
            vm_node_type,
            cloud_node_type
        ORDER BY
            count DESC;
    " | tee -a ${SUMMARY_FILE}
fi

## pg_stat_database
echo "### pg_stat_database ###" | tee -a ${SUMMARY_FILE}
${PSQLCMD} -c "
    SELECT
        *
    FROM
        pg_stat_database
" | tee -a ${SUMMARY_FILE}

## pg_stat_user_tables
echo "### pg_stat_user_tables ###" | tee -a ${SUMMARY_FILE}
${PSQLCMD} -c "
    SELECT
        *
    FROM
        pg_stat_user_tables
    ORDER BY
        seq_scan + seq_tup_read + idx_scan + idx_tup_fetch + n_tup_ins +
        n_tup_upd + n_tup_del + n_tup_hot_upd + n_live_tup + n_dead_tup DESC
" | tee -a ${SUMMARY_FILE}

## pg_stat_user_indexes
echo "### pg_stat_user_indexes ###" | tee -a ${SUMMARY_FILE}
${PSQLCMD} -c "
    SELECT
        *
    FROM
        pg_stat_user_indexes
    ORDER BY
        idx_scan + idx_tup_read + idx_tup_fetch DESC
" | tee -a ${SUMMARY_FILE}

## pg_staiot_user_tables
echo "### pg_statio_user_tables ###" | tee -a ${SUMMARY_FILE}
${PSQLCMD} -c "
    SELECT
        *
    FROM
        pg_statio_user_tables
    ORDER BY
        heap_blks_read + heap_blks_hit + idx_blks_read + idx_blks_hit DESC
" | tee -a ${SUMMARY_FILE}

## pg_statio_user_indexes
echo "### pg_statio_user_indexes ###" | tee -a ${SUMMARY_FILE}
${PSQLCMD} -c "
    SELECT
        *
    FROM
        pg_statio_user_indexes
    ORDER BY
        idx_blks_read + idx_blks_hit DESC
" | tee -a ${SUMMARY_FILE}



}

function os {
## cat /etc/lsb-release
echo "### cat /etc/lsb-release ###" | tee -a ${SUMMARY_FILE}
cat /etc/lsb-release | tee -a ${SUMMARY_FILE}
echo -e "\n" | tee -a ${SUMMARY_FILE}

## chkconfig --list | grep hinemos
#echo "###  chkconfig --list | grep hinemos ###" | tee -a ${SUMMARY_FILE}
#chkconfig --list | grep hinemos | tee -a ${SUMMARY_FILE}
#echo -e "\n" | tee -a ${SUMMARY_FILE}

## dpkg -l
echo "### dpkg -l ###" | tee -a ${SUMMARY_FILE}
dpkg -l | tee -a ${SUMMARY_FILE}
echo -e "\n" | tee -a ${SUMMARY_FILE}

## cpuinfo
echo "### /proc/cpuinfo ###" | tee -a ${SUMMARY_FILE}
cat /proc/cpuinfo | tee -a ${SUMMARY_FILE}
echo -e "" | tee -a ${SUMMARY_FILE}

## meminfo
echo "### /proc/meminfo ###" | tee -a ${SUMMARY_FILE}
cat /proc/meminfo | tee -a ${SUMMARY_FILE}
echo -e "\n" | tee -a ${SUMMARY_FILE}

if [ -x sar ]; then
    ## sar -P ALL
    echo "### sar -P ALL ###" | tee -a ${SUMMARY_FILE}
    sar -P ALL | tee -a ${SUMMARY_FILE}
    echo -e "\n" | tee -a ${SUMMARY_FILE}

    ## sar -r
    echo "### sar -r ###" | tee -a ${SUMMARY_FILE}
    sar -r | tee -a ${SUMMARY_FILE}
    echo -e "\n" | tee -a ${SUMMARY_FILE}

    ## sar -n DEV
    echo "### sar -n DEV ###" | tee -a ${SUMMARY_FILE}
    sar -n DEV | tee -a ${SUMMARY_FILE}
    echo -e "\n" | tee -a ${SUMMARY_FILE}

    ## sar -n EDEV
    echo "### sar -n EDEV ###" | tee -a ${SUMMARY_FILE}
    sar -n EDEV | tee -a ${SUMMARY_FILE}
    echo -e "\n" | tee -a ${SUMMARY_FILE}
else
    echo -e "### command 'sar' not installed ###\n" | tee -a ${SUMMARY_FILE}
fi

## vmstat -n
if [ -x vmstat ]; then
    echo "### vmstat -n ###" | tee -a ${SUMMARY_FILE}
    vmstat -n | tee -a ${SUMMARY_FILE}
    echo -e "\n" | tee -a ${SUMMARY_FILE}
else
    echo -e "### command 'vmstat' not installed ###\n" | tee -a ${SUMMARY_FILE}
fi

## iostat -d -x
if [ -x iostat ]; then
    echo "### iostat -d -x ###" | tee -a ${SUMMARY_FILE}
    iostat -d -x | tee -a ${SUMMARY_FILE}
    echo -e "" | tee -a ${SUMMARY_FILE}
else
    echo -e "### command 'iostat' not installed ###\n" | tee -a ${SUMMARY_FILE}
fi

## ps -H auxwww
echo "### ps -H auxwww ###" | tee -a ${SUMMARY_FILE}
ps -H auxwww | tee -a ${SUMMARY_FILE}
echo -e "\n" | tee -a ${SUMMARY_FILE}

## netstat -s
echo "### netstat -s ###" | tee -a ${SUMMARY_FILE}
netstat -s | tee -a ${SUMMARY_FILE}
echo -e "\n" | tee -a ${SUMMARY_FILE}

## netstat -anpo
echo "### netstat -anpo ###" | tee -a ${SUMMARY_FILE}
netstat -anpo | tee -a ${SUMMARY_FILE}
echo -e "\n" | tee -a ${SUMMARY_FILE}

## df
echo "### df ###" | tee -a ${SUMMARY_FILE}
df | tee -a ${SUMMARY_FILE}
echo -e "\n" | tee -a ${SUMMARY_FILE}

## df -i
echo "### df -i ###" | tee -a ${SUMMARY_FILE}
df -i | tee -a ${SUMMARY_FILE}
echo -e "\n" | tee -a ${SUMMARY_FILE}

## uptime
echo "### uptime ###" | tee -a ${SUMMARY_FILE}
uptime | tee -a ${SUMMARY_FILE}
echo -e "\n" | tee -a ${SUMMARY_FILE}

## /proc/slabinfo
echo "### /proc/slabinfo ###" | tee -a ${SUMMARY_FILE}
cat /proc/slabinfo | tee -a ${SUMMARY_FILE}
echo -e "\n" | tee -a ${SUMMARY_FILE}

## /proc/vmstat
echo "### /proc/vmstat ###" | tee -a ${SUMMARY_FILE}
cat /proc/vmstat | tee -a ${SUMMARY_FILE}
echo -e "\n" | tee -a ${SUMMARY_FILE}
}

function detail {
## lsof
echo "### lsof ###" | tee -a ${SUMMARY_FILE}
lsof | tee -a ${SUMMARY_FILE}
echo -e "\n" | tee -a ${SUMMARY_FILE}

if [ -e ${JBOSS_HOME}/bin/twiddle.sh ]
then
	## twiddle.sh
	echo "### twiddle.sh ###" | tee -a ${SUMMARY_FILE}
	cd /tmp
	export JAVA_OPTS="-Xms16m -Xmx32m"
	${JBOSS_HOME}/bin/twiddle.sh -s jnp://127.0.0.1:1099 invoke "user:service=PollerService,name=PollerManager" printPollerInfo  | tee -a ${SUMMARY_FILE}
	${JBOSS_HOME}/bin/twiddle.sh -s jnp://127.0.0.1:1099 invoke "user:service=SharedTableService,name=SharedTable" printTableInfo | tee -a ${SUMMARY_FILE}
	${JBOSS_HOME}/bin/twiddle.sh -s jnp://127.0.0.1:1099 invoke "user:service=HinemosService,name=HinemosService" printSchedulerStatus | tee -a ${SUMMARY_FILE}
	${JBOSS_HOME}/bin/twiddle.sh -s jnp://127.0.0.1:1099 invoke "user:service=HinemosService,name=HinemosService" printSchedulerDetail | tee -a ${SUMMARY_FILE}
	echo -e "\n" | tee -a ${SUMMARY_FILE}
else
	# manager_cli
	echo "### manager_cli (Hinemos Agent Info) ###" | tee -a ${SUMMARY_FILE}
	${HINEMOS_HOME}/sbin/manager_cli -c com.clustercontrol.commons.util.ManagerCliUtil -m getValidAgentStr | tee -a ${SUMMARY_FILE}
	
	echo "### manager_cli (Scheduler Info) ###" | tee -a ${SUMMARY_FILE}
	${HINEMOS_HOME}/sbin/manager_cli -c com.clustercontrol.commons.util.ManagerCliUtil -m getSchedulerInfoStr | tee -a ${SUMMARY_FILE}
	
	echo "### manager_cli (Poller Info) ###" | tee -a ${SUMMARY_FILE}
	${HINEMOS_HOME}/sbin/manager_cli -c com.clustercontrol.commons.util.ManagerCliUtil -m getPollerInfoStr | tee -a ${SUMMARY_FILE}
	
	echo "### manager_cli (SharedTable Info) ###" | tee -a ${SUMMARY_FILE}
	${HINEMOS_HOME}/sbin/manager_cli -c com.clustercontrol.commons.util.ManagerCliUtil -m getSharedTableInfoStr | tee -a ${SUMMARY_FILE}
	
	echo "### manager_cli (SelfCheck LastFireTime) ###" | tee -a ${SUMMARY_FILE}
	${HINEMOS_HOME}/sbin/manager_cli -c com.clustercontrol.commons.util.ManagerCliUtil -m getSelfCheckLastFireTimeStr | tee -a ${SUMMARY_FILE}
	
	echo "### manager_cli (AsynWorker Statistics) ###" | tee -a ${SUMMARY_FILE}
	${HINEMOS_HOME}/sbin/manager_cli -c com.clustercontrol.commons.util.ManagerCliUtil -m getAsyncWorkerStatistics | tee -a ${SUMMARY_FILE}
	
	echo "### manager_cli (Syslog Statistics) ###" | tee -a ${SUMMARY_FILE}
	${HINEMOS_HOME}/sbin/manager_cli -c com.clustercontrol.commons.util.ManagerCliUtil -m getSyslogStatistics | tee -a ${SUMMARY_FILE}
	
	echo "### manager_cli (SnmpTrap Statistics) ###" | tee -a ${SUMMARY_FILE}
	${HINEMOS_HOME}/sbin/manager_cli -c com.clustercontrol.commons.util.ManagerCliUtil -m getSnmpTrapStatistics | tee -a ${SUMMARY_FILE}
	
	echo "### manager_cli (Job Queue Info) ###" | tee -a ${SUMMARY_FILE}
	${HINEMOS_HOME}/sbin/manager_cli -c com.clustercontrol.commons.util.ManagerCliUtil -m getJobQueueStr | tee -a ${SUMMARY_FILE}
	
	echo -e "\n" | tee -a ${SUMMARY_FILE}
fi

## md5sum
echo "### md5sum ###" | tee -a ${SUMMARY_FILE}
cd ${HINEMOS_HOME}
find `ls --ignore='var'` -type f -print0 | xargs -0 md5sum | tee -a ${SUMMARY_FILE}
cd -
echo -e "\n" | tee -a ${SUMMARY_FILE}
}

function threaddump {

DUMP_FILENAME="threaddump.${FILENAME_POSTFIX}"
DUMP_INFO_FILENAME="threaddump_info.${FILENAME_POSTFIX}"

echo "### Get thread dump  ###"

while [ 1 ]
do
	sleep 0.5 && read -p "input interval (sec) : " INPUT
	case ${INPUT} in
		[0-9]*)
			INTERVAL=${INPUT}
			break;
			;;
		*)
			echo "Your input is invalid. Please input again."
			;;
	esac
done


while [ 1 ]
do
	sleep 0.5 && read -p "input count : " INPUT
	case ${INPUT} in
		[0-9]*)
			COUNT=${INPUT}
			break;
			;;
		*)
			echo "Your input is invalid. Please input again."
			;;
	esac
done


NUM=0
while [ 1 ] ;
do

	NUM=`expr ${NUM} + 1`
	echo -n "getting thread dump ${NUM}/${COUNT} ... "  | tee -a ${HINEMOS_HOME}/var/log/${DUMP_INFO_FILENAME}
	echo -n `date`  | tee -a ${HINEMOS_HOME}/var/log/${DUMP_INFO_FILENAME}
	kill -3 `cat ${HINEMOS_HOME}/var/run/jboss.pid`
	echo " [OK] "  | tee -a ${HINEMOS_HOME}/var/log/${DUMP_INFO_FILENAME}
	echo -e "\n" >> ${HINEMOS_HOME}/var/log/${DUMP_INFO_FILENAME}

	echo "### netstat -s ###" >> ${HINEMOS_HOME}/var/log/${DUMP_INFO_FILENAME}
	netstat -s >> ${HINEMOS_HOME}/var/log/${DUMP_INFO_FILENAME}
	echo -e "\n" >> ${HINEMOS_HOME}/var/log//${DUMP_INFO_FILENAME}

	echo "### netstat -anpo ###" >> ${HINEMOS_HOME}/var/log/${DUMP_INFO_FILENAME}
	netstat -anpo >> ${HINEMOS_HOME}/var/log/${DUMP_INFO_FILENAME}
	echo -e "\n" >> ${HINEMOS_HOME}/var/log/${DUMP_INFO_FILENAME}

	echo "### ps -ef ###" >> ${HINEMOS_HOME}/var/log/${DUMP_INFO_FILENAME}
	ps -ef >> ${HINEMOS_HOME}/var/log/${DUMP_INFO_FILENAME}
	echo -e "\n" >> ${HINEMOS_HOME}/var/log/${DUMP_INFO_FILENAME}

	echo -e "\n" >> ${HINEMOS_HOME}/var/log/${DUMP_INFO_FILENAME}
	echo -e "\n" >> ${HINEMOS_HOME}/var/log/${DUMP_INFO_FILENAME}

	if [ ${NUM} -ge ${COUNT} ] ;
	then
		break;
	fi

	sleep ${INTERVAL}

done

cp -p ${HINEMOS_HOME}/var/log/jboss_stdout.log ${HINEMOS_HOME}/var/log/${DUMP_FILENAME}

echo -e "\n"
echo "### Please get this file ###"
echo "${HINEMOS_HOME}/var/log/${DUMP_FILENAME}"
echo "${HINEMOS_HOME}/var/log/${DUMP_INFO_FILENAME}"
echo -e "\n"

}


function log {

DIR_NAME="hinemos_manager_log.${FILENAME_POSTFIX}"
mkdir ${HINEMOS_HOME}/${DIR_NAME}

echo "### /var/log/syslog ###"
cd /var
ls log/ | grep syslog*
tar cpzf ${HINEMOS_HOME}/${DIR_NAME}/var_log_syslog ./log/syslog*
echo -e "\n"

echo "### ${HINEMOS_HOME}/var/log ###"
cd ${HINEMOS_HOME}/var/
ls log/
tar cpzf ${HINEMOS_HOME}/${DIR_NAME}/hinemos_manager_log.tar.gz ./log/*
echo -e "\n"

echo "### create tarball ###"
cd ${HINEMOS_HOME}
tar cpzf ${HINEMOS_HOME}/${DIR_NAME}.tar.gz ./${DIR_NAME}/*

rm -rf ${HINEMOS_HOME}/${DIR_NAME}

echo ${HINEMOS_HOME}/${DIR_NAME}.tar.gz

echo -e "\n"

}

function diskspacecheck {

echo "### file system disk space usage ###"
df -h
echo -e "\n"

echo "### log directory disk space usage ###"
USAGE=`du -hs /opt/hinemos/var/log/`
echo ${USAGE}
echo -e "\n"


if expr "${USAGE}" : ".*G.*\/opt\/hinemos\/var\/log\/$" > /dev/null;  then
	echo "Disk space used by Hinemos Manager's Log(/opt/hinemos/var/log/) more than 1GByte."
	echo "Please delete log, before getting log about Hinemos Manager."
	exit 1
fi


echo "### Hinemos log archive output directory ###"
echo ${HINEMOS_HOME}/
echo -e "\n"

while [ 1 ]
do
	sleep 0.5 && read -p "Is it OK to collect Hinemos Manager's log ? (Y/N default:Y)" INPUT
	INPUT=${INPUT:-Y}
	case ${INPUT} in
		y|Y)
			break
			;;
		n|N)
			echo "Canceled."
			exit 1
			;;
		*)
			echo "Your input is invalid. Please input again."
			;;
	esac
done

echo -e "\n"

}


if [ $# -eq 0 ]; then
        usage
        exit 1
fi

for OPTION in "$@"
do
        case $OPTION in
                -v)
                        init
                        hinemos
                        ;;
                -vv)
                        init
                        hinemos
                        os
                        ;;
                -vvv)
                        init
                        hinemos
                        os
                        detail
                        ;;
                -o)
                        init
                        diskspacecheck
                        hinemos
                        os
                        detail
                        log
                        exit 0
                        ;;
                -t)
                        init
                        threaddump
                        exit 0
                        ;;
                *)
                        usage
                        exit 1
        esac
done

echo "### Hinemos Manager Summary ###"
echo ${SUMMARY_FILE}

echo ""
echo "#################################################"
echo "###     Script end at `date '+%Y-%m-%d %T'`     ###"
echo "#################################################"

exit 0
