/*
Copyright (C) 2010 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */
package com.clustercontrol.ws.collectiverun;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.Locale;

import javax.annotation.Resource;
import javax.xml.bind.annotation.XmlSeeAlso;
import javax.xml.ws.WebServiceContext;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.fault.CollectiveRunNotFound;
import com.clustercontrol.fault.FacilityNotFound;
import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.fault.InvalidRole;
import com.clustercontrol.fault.InvalidSetting;
import com.clustercontrol.fault.InvalidUserPass;
import com.clustercontrol.accesscontrol.bean.FunctionConstant;
import com.clustercontrol.accesscontrol.bean.SystemPrivilegeInfo;
import com.clustercontrol.accesscontrol.bean.PrivilegeConstant.SystemPrivilegeMode;
import com.clustercontrol.bean.HinemosModuleConstant;
import com.clustercontrol.collectiverun.bean.CommandInfo;
import com.clustercontrol.collectiverun.bean.CommandParameterInfo;
import com.clustercontrol.collectiverun.bean.CommandTypeInfo;
import com.clustercontrol.collectiverun.bean.DetailInfo;
import com.clustercontrol.collectiverun.bean.HistoryFilterInfo;
import com.clustercontrol.collectiverun.bean.HistoryInfo;
import com.clustercontrol.collectiverun.bean.ItemConstant;
import com.clustercontrol.collectiverun.bean.ItemInfo;
import com.clustercontrol.collectiverun.bean.ParameterInfo;
import com.clustercontrol.collectiverun.bean.ParameterSelectInfo;
import com.clustercontrol.collectiverun.bean.CollectiveRunInfo;
import com.clustercontrol.collectiverun.bean.SessionInfo;
import com.clustercontrol.collectiverun.bean.TreeItem;
import com.clustercontrol.collectiverun.bean.TypeInfo;
import com.clustercontrol.collectiverun.session.CollectiveRunControllerBean;
import com.clustercontrol.commons.bean.ViewListInfo;
import com.clustercontrol.ws.util.HashMapInfo;
import com.clustercontrol.ws.util.HttpAuthenticator;

/**
 * 一括制御用のWebAPIエンドポイント
 */
@javax.jws.WebService(targetNamespace = "http://collectiverun.ws.clustercontrol.com")
@XmlSeeAlso({
	HistoryInfo.class
})
public class CollectiveRunEndpoint {
	@Resource
	WebServiceContext wsctx;

	private static Log m_log = LogFactory.getLog( CollectiveRunEndpoint.class );
	private static Log m_opelog = LogFactory.getLog("HinemosOperation");

	/**
	 * echo(WebサービスAPI疎通用)
	 * 
	 * 権限必要なし（ユーザ名チェックのみ実施）
	 * 
	 * @param str
	 * @return
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 * @throws HinemosUnknown
	 */
	public String echo(String str) throws InvalidUserPass, InvalidRole, HinemosUnknown {
		ArrayList<SystemPrivilegeInfo> systemPrivilegeList = new ArrayList<SystemPrivilegeInfo>();
		HttpAuthenticator.authCheck(wsctx, systemPrivilegeList);

		return str + ", " + str;
	}

	/**
	 * 
	 * 一括制御[履歴]ビューで選択されている履歴に対応する種別IDを取得します。
	 * 
	 * CollectiveRunRead権限が必要
	 * 
	 * @param sessionId 選択されている履歴のセッションID
	 * @return 種別ID
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * @throws FacilityNotFound
	 */
	public String getParameterTypeId(String sessionId) throws InvalidUserPass, InvalidRole, HinemosUnknown, FacilityNotFound {
		m_log.debug("getParameterTypeId");
		ArrayList<SystemPrivilegeInfo> systemPrivilegeList = new ArrayList<SystemPrivilegeInfo>();
		systemPrivilegeList.add(new SystemPrivilegeInfo(FunctionConstant.COLLECTIVERUN, SystemPrivilegeMode.READ));
		HttpAuthenticator.authCheck(wsctx, systemPrivilegeList, true);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", SessionID=");
		msg.append(sessionId);
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_COLLECTIVERUN + " Get Info, Method=getParameterTypeId, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		return new CollectiveRunControllerBean().getParameterTypeId(sessionId);
	}

	/**
	 * 一括制御[履歴]ビューに表示する履歴のリストを取得します。
	 * 
	 * CollectiveRunRead権限が必要
	 * 
	 * @param histories 表示履歴数
	 * 
	 * @return 一括制御[履歴]情報
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * 
	 * @see com.clustercontrol.bean.ViewListInfo
	 * @see com.clustercontrol.collectiverun.factory.SelectSession#getHistoryList(Locale, int)
	 * 
	 */
	public ViewListInfo getHistoryListAll(int histories) throws InvalidUserPass, InvalidRole, HinemosUnknown {
		m_log.debug("getHistoryListAll");
		ArrayList<SystemPrivilegeInfo> systemPrivilegeList = new ArrayList<SystemPrivilegeInfo>();
		systemPrivilegeList.add(new SystemPrivilegeInfo(FunctionConstant.COLLECTIVERUN, SystemPrivilegeMode.READ));
		HttpAuthenticator.authCheck(wsctx, systemPrivilegeList, true);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", Count=");
		msg.append(histories);
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_COLLECTIVERUN + " Get Info, Method=getHistoryListAll, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		return new CollectiveRunControllerBean().getHistoryList(histories);
	}

	/**
	 * 一括制御[履歴]ビューに表示する履歴のリストを取得します。
	 * 
	 * CollectiveRunRead権限が必要
	 * 
	 * @param filter フィルタ
	 * @param histories 表示履歴数
	 * 
	 * @return 一括制御[履歴]情報
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * 
	 * @see com.clustercontrol.bean.ViewListInfo
	 * @see com.clustercontrol.collectiverun.factory.SelectSession#getHistoryList(Property, Locale, int)
	 * 
	 */
	public ViewListInfo getHistoryList(HistoryFilterInfo filter, int histories) throws InvalidUserPass, InvalidRole, HinemosUnknown {
		m_log.debug("getHistoryList");
		ArrayList<SystemPrivilegeInfo> systemPrivilegeList = new ArrayList<SystemPrivilegeInfo>();
		systemPrivilegeList.add(new SystemPrivilegeInfo(FunctionConstant.COLLECTIVERUN, SystemPrivilegeMode.READ));
		HttpAuthenticator.authCheck(wsctx, systemPrivilegeList, true);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		if (filter != null) {
			SimpleDateFormat sdf = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss");
			msg.append(", StartFromDate=");
			msg.append(filter.getStartFromDate()==null?null:sdf.format(new Date(filter.getStartFromDate())));
			msg.append(", StartToDate=");
			msg.append(filter.getStartToDate()==null?null:sdf.format(new Date(filter.getStartToDate())));
			msg.append(", EndFromDate=");
			msg.append(filter.getEndFromDate()==null?null:sdf.format(new Date(filter.getEndFromDate())));
			msg.append(", EndToDate=");
			msg.append(filter.getEndToDate()==null?null:sdf.format(new Date(filter.getEndToDate())));
			msg.append(", Type=");
			msg.append(filter.getType());
			msg.append(", Status=");
			msg.append(filter.getStatus());
			msg.append(", StartDate=");
			msg.append(filter.getStartDate()==null?null:sdf.format(new Date(filter.getStartDate())));
			msg.append(", EndDate=");
			msg.append(filter.getEndDate()==null?null:sdf.format(new Date(filter.getEndDate())));
		}
		msg.append(", Count=");
		msg.append(histories);
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_COLLECTIVERUN + " Get Info, Method=getHistoryList, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		return new CollectiveRunControllerBean().getHistoryList(filter, histories);
	}

	/**
	 * 一括制御[詳細]ビューに表示する情報を取得します。<BR>
	 * 
	 * CollectiveRunRead権限が必要
	 * 
	 * @param sessionId セッションID
	 * @return 一括制御[詳細]情報
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * 
	 * @see com.clustercontrol.collectiverun.factory.SelectSession#getDetailList(String)
	 * 
	 */
	public ArrayList<DetailInfo> getDetailList(String sessionId) throws InvalidUserPass, InvalidRole, HinemosUnknown {
		m_log.debug("getDetailList");
		ArrayList<SystemPrivilegeInfo> systemPrivilegeList = new ArrayList<SystemPrivilegeInfo>();
		systemPrivilegeList.add(new SystemPrivilegeInfo(FunctionConstant.COLLECTIVERUN, SystemPrivilegeMode.READ));
		HttpAuthenticator.authCheck(wsctx, systemPrivilegeList, true);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", SessionID=");
		msg.append(sessionId);
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_COLLECTIVERUN + " Get Info, Method=getDetailList, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		return new CollectiveRunControllerBean().getDetailList(sessionId);
	}

	/**
	 * 一括制御を実行する際に呼び出される、インタフェースメソッドです。
	 * セッションIDを生成し、一括制御を実行します。
	 * 
	 * CollectiveRunExecute権限が必要
	 * 
	 * @param typeId 種別ID
	 * @param info 実行情報
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * @throws FacilityNotFound
	 * @throws CollectiveRunNotFound
	 */
	public String run(String typeId, CollectiveRunInfo info) throws InvalidUserPass, InvalidRole, HinemosUnknown, CollectiveRunNotFound,
			FacilityNotFound, InvalidSetting
	{
		m_log.debug("run");
		ArrayList<SystemPrivilegeInfo> systemPrivilegeList = new ArrayList<SystemPrivilegeInfo>();
		systemPrivilegeList.add(new SystemPrivilegeInfo(FunctionConstant.COLLECTIVERUN, SystemPrivilegeMode.EXEC));
		HttpAuthenticator.authCheck(wsctx, systemPrivilegeList, true);

		// 認証済み操作ログ
		String commandId = "";
		try {
			commandId = getCommandId(typeId, info.getName());
		} catch (Exception e) {
			m_log.warn("run() getCommandId, " + e.getMessage(), e);
		}
		StringBuffer msg = new StringBuffer();
		msg.append(", TypeID=");
		msg.append(typeId);
		if (info != null) {
			msg.append(", FacilityID=");
			msg.append(info.getFacilityId());
		}
		msg.append(", CommandID=");
		msg.append(commandId);

		String ret = null;
		try {
			ret = new CollectiveRunControllerBean().run(typeId, info);
		} catch (Exception e){
			m_opelog.warn(HinemosModuleConstant.LOG_PREFIX_COLLECTIVERUN + " Run Failed, Method=run, User="
					+ HttpAuthenticator.getUserAccountString(wsctx)
					+ msg.toString());
			throw e;
		}
		m_opelog.info(HinemosModuleConstant.LOG_PREFIX_COLLECTIVERUN + " Run, Method=run, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		return ret;
	}

	/**
	 * 一括制御コマンドの情報を取得します。<BR>
	 * コマンド情報はツリー構造で表現されます。
	 * 
	 * CollectiveRunRead権限が必要
	 * 
	 * @return ツリー構造で表現された一括制御コマンドの情報
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * 
	 */
	public TreeItem getTree() throws InvalidUserPass, InvalidRole, HinemosUnknown {
		m_log.debug("getTree");
		ArrayList<SystemPrivilegeInfo> systemPrivilegeList = new ArrayList<SystemPrivilegeInfo>();
		systemPrivilegeList.add(new SystemPrivilegeInfo(FunctionConstant.COLLECTIVERUN, SystemPrivilegeMode.READ));
		HttpAuthenticator.authCheck(wsctx, systemPrivilegeList, true);

		// 認証済み操作ログ
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_COLLECTIVERUN + " Get Info, Method=getTree, User="
				+ HttpAuthenticator.getUserAccountString(wsctx));

		return new CollectiveRunControllerBean().getTree();
	}

	/**
	 * 一括制御のマスタ情報を更新します。
	 * 
	 * CollectiveRunAdd権限, CollectiveRunWrite権限が必要
	 * 
	 * @param info
	 * @param parentInfo
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * 
	 */
	public void updateMaster(ItemInfo info, ItemInfo parentInfo) throws InvalidUserPass, InvalidRole, HinemosUnknown {
		m_log.debug("updateMaster");
		ArrayList<SystemPrivilegeInfo> systemPrivilegeList = new ArrayList<SystemPrivilegeInfo>();
		systemPrivilegeList.add(new SystemPrivilegeInfo(FunctionConstant.COLLECTIVERUN, SystemPrivilegeMode.ADD));
		systemPrivilegeList.add(new SystemPrivilegeInfo(FunctionConstant.COLLECTIVERUN, SystemPrivilegeMode.WRITE));
		HttpAuthenticator.authCheck(wsctx, systemPrivilegeList, true);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		if(info != null){
			msg.append(getItemName("", "", info));
		}
		if(parentInfo != null){
			msg.append(getItemName("Parent", "", parentInfo));
		}

		try {
			new CollectiveRunControllerBean().updateMaster(info, parentInfo);
		} catch (Exception e) {
			m_opelog.warn(HinemosModuleConstant.LOG_PREFIX_COLLECTIVERUN + " Change Item Failed, Method=updateMaster, User="
					+ HttpAuthenticator.getUserAccountString(wsctx)
					+ msg.toString());
			throw e;
		}
		m_opelog.info(HinemosModuleConstant.LOG_PREFIX_COLLECTIVERUN + " Change Item, Method=updateMaster, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

	}

	/**
	 * 一括制御マスタ情報を削除します。
	 * 
	 * CollectiveRunWrite権限が必要
	 * 
	 * @param info
	 * @param parentInfo
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * @throws CollectiveRunNotFound
	 */
	public void deleteMaster(ItemInfo info, ItemInfo parentInfo) throws InvalidUserPass, InvalidRole, HinemosUnknown, CollectiveRunNotFound {
		m_log.debug("deleteMaster");
		ArrayList<SystemPrivilegeInfo> systemPrivilegeList = new ArrayList<SystemPrivilegeInfo>();
		systemPrivilegeList.add(new SystemPrivilegeInfo(FunctionConstant.COLLECTIVERUN, SystemPrivilegeMode.WRITE));
		HttpAuthenticator.authCheck(wsctx, systemPrivilegeList, true);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		if(info != null){
			msg.append(getItemName("", "", info));
		}
		if(parentInfo != null){
			msg.append(getItemName("Parent", "", parentInfo));
		}

		try {
			new CollectiveRunControllerBean().deleteMaster(info, parentInfo);
		} catch (Exception e) {
			m_opelog.warn(HinemosModuleConstant.LOG_PREFIX_COLLECTIVERUN + " Delete Item Failed, Method=deleteMaster, User="
					+ HttpAuthenticator.getUserAccountString(wsctx)
					+ msg.toString());
			throw e;
		}
		m_opelog.info(HinemosModuleConstant.LOG_PREFIX_COLLECTIVERUN + " Delete Item, Method=deleteMaster, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

	}

	/**
	 * 一括制御マスタ情報を更新します。<BR>
	 * 表示順を上下させる際に使用します。
	 * 
	 * CollectiveRunWrite権限が必要
	 * 
	 * @param info1
	 * @param info2
	 * @param parentInfo
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * 
	 */
	public void changeOrder(ItemInfo info1, ItemInfo info2, ItemInfo parentInfo) throws InvalidUserPass, InvalidRole, HinemosUnknown {
		m_log.debug("changeOrder");
		ArrayList<SystemPrivilegeInfo> systemPrivilegeList = new ArrayList<SystemPrivilegeInfo>();
		systemPrivilegeList.add(new SystemPrivilegeInfo(FunctionConstant.COLLECTIVERUN, SystemPrivilegeMode.WRITE));
		HttpAuthenticator.authCheck(wsctx, systemPrivilegeList, true);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		if(info1 != null){
			msg.append(getItemName("", "1", info1));
		}
		if(info2 != null){
			msg.append(getItemName("", "2", info2));
		}
		if(parentInfo != null){
			msg.append(getItemName("Parent", "", parentInfo));
		}

		try {
			new CollectiveRunControllerBean().changeOrder(info1, info2, parentInfo);
		} catch (Exception e) {
			m_opelog.warn(HinemosModuleConstant.LOG_PREFIX_COLLECTIVERUN + " Change Order Failed, Method=changeOrder, User="
					+ HttpAuthenticator.getUserAccountString(wsctx)
					+ msg.toString());
			throw e;
		}
		m_opelog.info(HinemosModuleConstant.LOG_PREFIX_COLLECTIVERUN + " Change Order, Method=changeOrder, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

	}

	/**
	 * 一括制御[パラメータ]ビューの種別に表示する文字列のリスト取得します。
	 * 
	 * CollectiveRunRead権限が必要
	 * 
	 * @return 種別文字列
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	public ArrayList<String> getTypeStringList() throws InvalidUserPass, InvalidRole, HinemosUnknown {
		m_log.debug("getTypeStringList");
		ArrayList<SystemPrivilegeInfo> systemPrivilegeList = new ArrayList<SystemPrivilegeInfo>();
		systemPrivilegeList.add(new SystemPrivilegeInfo(FunctionConstant.COLLECTIVERUN, SystemPrivilegeMode.READ));
		HttpAuthenticator.authCheck(wsctx, systemPrivilegeList, true);

		// 認証済み操作ログ
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_COLLECTIVERUN + " Get Info, Method=getTypeStringList, User="
				+ HttpAuthenticator.getUserAccountString(wsctx));

		return new CollectiveRunControllerBean().getTypeStringList();
	}

	/**
	 * 種別のハッシュマップ（key : 種別ID, value : 種別文字列）を取得します。
	 * 
	 * CollectiveRunRead権限が必要
	 * 
	 * @return 種別IDと種別文字列のハッシュマップ
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * 
	 */
	public HashMapInfo getTypeHashMap() throws InvalidUserPass, InvalidRole, HinemosUnknown {
		m_log.debug("getTypeHashMap");
		ArrayList<SystemPrivilegeInfo> systemPrivilegeList = new ArrayList<SystemPrivilegeInfo>();
		systemPrivilegeList.add(new SystemPrivilegeInfo(FunctionConstant.COLLECTIVERUN, SystemPrivilegeMode.READ));
		HttpAuthenticator.authCheck(wsctx, systemPrivilegeList, true);

		// 認証済み操作ログ
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_COLLECTIVERUN + " Get Info, Method=getTypeHashMap, User="
				+ HttpAuthenticator.getUserAccountString(wsctx));

		HashMapInfo info = new HashMapInfo();
		info.getMap1().putAll(new CollectiveRunControllerBean().getTypeHashMap());
		return info;
	}

	/**
	 * コマンドIDを取得します。
	 * 
	 * CollectiveRunRead権限が必要
	 * 
	 * @param name 名前
	 * @return コマンドID
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	public String getCommandId(String typeId, String name) throws InvalidUserPass, InvalidRole, HinemosUnknown {
		m_log.debug("getCommandId");
		ArrayList<SystemPrivilegeInfo> systemPrivilegeList = new ArrayList<SystemPrivilegeInfo>();
		systemPrivilegeList.add(new SystemPrivilegeInfo(FunctionConstant.COLLECTIVERUN, SystemPrivilegeMode.READ));
		HttpAuthenticator.authCheck(wsctx, systemPrivilegeList, true);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", TypeID=");
		msg.append(typeId);
		msg.append(", Name=");
		msg.append(name);
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_COLLECTIVERUN + " Get Info, Method=getCommandId, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		return new CollectiveRunControllerBean().getCommandId(typeId, name);
	}

	/**
	 * コマンドマスタを取得します。
	 * 
	 * CollectiveRunRead権限が必要
	 * 
	 * @param id コマンドID
	 * @return コマンド種別ID
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * @throws CollectiveRunNotFound
	 */
	public CommandInfo getCommandMst(String id) throws InvalidUserPass, InvalidRole, HinemosUnknown, CollectiveRunNotFound {
		m_log.debug("getCommandMst");
		ArrayList<SystemPrivilegeInfo> systemPrivilegeList = new ArrayList<SystemPrivilegeInfo>();
		systemPrivilegeList.add(new SystemPrivilegeInfo(FunctionConstant.COLLECTIVERUN, SystemPrivilegeMode.READ));
		HttpAuthenticator.authCheck(wsctx, systemPrivilegeList, true);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", CommandID=");
		msg.append(id);
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_COLLECTIVERUN + " Get Master, Method=getCommandMst, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		return new CollectiveRunControllerBean().getCommandMst(id);
	}

	/**
	 * セッション情報を取得します。
	 * 
	 * CollectiveRunRead権限が必要
	 * 
	 * @param id セッションID
	 * @return セッション情報
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * @throws CollectiveRunNotFound
	 */
	public SessionInfo getSessionInfo(String id) throws InvalidUserPass, InvalidRole, HinemosUnknown, CollectiveRunNotFound {
		m_log.debug("getSessionInfo");
		ArrayList<SystemPrivilegeInfo> systemPrivilegeList = new ArrayList<SystemPrivilegeInfo>();
		systemPrivilegeList.add(new SystemPrivilegeInfo(FunctionConstant.COLLECTIVERUN, SystemPrivilegeMode.READ));
		HttpAuthenticator.authCheck(wsctx, systemPrivilegeList, true);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", SessionID=");
		msg.append(id);
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_COLLECTIVERUN + " Get Info, Method=getSessionInfo, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		return new CollectiveRunControllerBean().getSessionInfo(id);
	}

	/**
	 * パラメータIDからパラメータ種別を取得します。
	 * 
	 * CollectiveRunRead権限が必要
	 * 
	 * @param id セッションID
	 * @return セッション情報
	 * @throws HinemosUnknown
	 * @throws CollectiveRunNotFound
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	public Integer getParameterType(String id) throws CollectiveRunNotFound, HinemosUnknown, InvalidUserPass, InvalidRole {
		m_log.debug("getParameterType");
		ArrayList<SystemPrivilegeInfo> systemPrivilegeList = new ArrayList<SystemPrivilegeInfo>();
		systemPrivilegeList.add(new SystemPrivilegeInfo(FunctionConstant.COLLECTIVERUN, SystemPrivilegeMode.READ));
		HttpAuthenticator.authCheck(wsctx, systemPrivilegeList, true);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", ParamID=");
		msg.append(id);
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_COLLECTIVERUN + " Get Info, Method=getParameterType, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		return new CollectiveRunControllerBean().getParameterType(id);
	}

	/**
	 * コマンド種別IDからコマンド種別マスタを取得します。
	 * 
	 * CollectiveRunRead権限が必要
	 * 
	 * @param id コマンド種別ID
	 * @return コマンド種別情報
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * @throws CollectiveRunNotFound
	 */
	public CommandTypeInfo getCommandTypeMst(String id) throws InvalidUserPass, InvalidRole, HinemosUnknown, CollectiveRunNotFound {
		m_log.debug("getCommandTypeMst");
		ArrayList<SystemPrivilegeInfo> systemPrivilegeList = new ArrayList<SystemPrivilegeInfo>();
		systemPrivilegeList.add(new SystemPrivilegeInfo(FunctionConstant.COLLECTIVERUN, SystemPrivilegeMode.READ));
		HttpAuthenticator.authCheck(wsctx, systemPrivilegeList, true);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", TypeID=");
		msg.append(id);
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_COLLECTIVERUN + " Get Master, Method=getCommandTypeMst, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		return new CollectiveRunControllerBean().getCommandTypeMst(id);
	}

	/**
	 * コマンドIDからコマンドパラメータマスタを取得します。
	 * 
	 * CollectiveRunRead権限が必要
	 * 
	 * @param commandId コマンドID
	 * @return コマンドパラメータマスタ
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	public ArrayList<CommandParameterInfo> getCommandParameterMstList(String commandId) throws InvalidUserPass, InvalidRole, HinemosUnknown {
		m_log.debug("getCommandParameterMstList");
		ArrayList<SystemPrivilegeInfo> systemPrivilegeList = new ArrayList<SystemPrivilegeInfo>();
		systemPrivilegeList.add(new SystemPrivilegeInfo(FunctionConstant.COLLECTIVERUN, SystemPrivilegeMode.READ));
		HttpAuthenticator.authCheck(wsctx, systemPrivilegeList, true);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", CommandID=");
		msg.append(commandId);
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_COLLECTIVERUN + " Get Master, Method=getCommandParameterMstList, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		return new CollectiveRunControllerBean().getCommandParameterMstList(commandId);
	}

	/**
	 * パラメータIDからパラメータ選択マスタを取得します。
	 * 
	 * CollectiveRunRead権限が必要
	 * 
	 * @param commandId コマンドID
	 * @return パラメータ選択マスタ
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	public ArrayList<ParameterSelectInfo> getParameterSelectMstList(String paramId) throws InvalidUserPass, InvalidRole, HinemosUnknown {
		m_log.debug("getParameterSelectMstList");
		ArrayList<SystemPrivilegeInfo> systemPrivilegeList = new ArrayList<SystemPrivilegeInfo>();
		systemPrivilegeList.add(new SystemPrivilegeInfo(FunctionConstant.COLLECTIVERUN, SystemPrivilegeMode.READ));
		HttpAuthenticator.authCheck(wsctx, systemPrivilegeList, true);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", ParamID=");
		msg.append(paramId);
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_COLLECTIVERUN + " Get Master, Method=getParameterSelectMstList, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		return new CollectiveRunControllerBean().getParameterSelectMstList(paramId);
	}

	/**
	 * アイテム名取得
	 * 
	 * @param header ヘッダ文字列
	 * @param footer フッタ文字列
	 * @param info アイテム情報
	 * @return アイテム名
	 */
	private String getItemName(String header, String footer, ItemInfo info){
		StringBuffer msg = new StringBuffer();
		msg.append(", ");
		msg.append(header);
		if(info.getType() == ItemConstant.TYPE_TYPE){
			msg.append("TypeID");
			msg.append(footer);
			msg.append("=");
			msg.append(((TypeInfo)info).getId());
		}
		else if(info.getType() == ItemConstant.TYPE_COMMAND){
			msg.append("CommandID");
			msg.append(footer);
			msg.append("=");
			msg.append(((CommandInfo)info).getId());
		}
		else if(info.getType() == ItemConstant.TYPE_COMMAND_PARAMETER){
			msg.append("CommandParamID");
			msg.append(footer);
			msg.append("=");
			msg.append(((CommandParameterInfo)info).getId());
		}
		else if(info.getType() == ItemConstant.TYPE_PARAMETER){
			msg.append("ParamID");
			msg.append(footer);
			msg.append("=");
			msg.append(((ParameterInfo)info).getId());
		}
		else if(info.getType() == ItemConstant.TYPE_SELECT_ITEM){
			msg.append("SelectItemID");
			msg.append(footer);
			msg.append("=");
			msg.append(((ParameterSelectInfo)info).getName());
		}
		else{
			return "";
		}
		return msg.toString();
	}
}