/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.repository.util;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.accesscontrol.bean.PrivilegeConstant;
import com.clustercontrol.accesscontrol.bean.PrivilegeConstant.ObjectPrivilegeMode;
import com.clustercontrol.accesscontrol.bean.RoleIdConstant;
import com.clustercontrol.accesscontrol.model.ObjectPrivilegeEntity;
import com.clustercontrol.accesscontrol.util.UserRoleCache;
import com.clustercontrol.bean.HinemosModuleConstant;
import com.clustercontrol.commons.util.HinemosSessionContext;
import com.clustercontrol.commons.util.JpaTransactionManager;
import com.clustercontrol.fault.FacilityNotFound;
import com.clustercontrol.fault.InvalidRole;
import com.clustercontrol.fault.InvalidSetting;
import com.clustercontrol.repository.bean.FacilityConstant;
import com.clustercontrol.repository.bean.FacilityInfo;
import com.clustercontrol.repository.bean.FacilityTreeItem;
import com.clustercontrol.repository.factory.FacilitySelector;
import com.clustercontrol.repository.model.FacilityEntity;
import com.clustercontrol.repository.session.RepositoryControllerBean;
import com.clustercontrol.util.Messages;

/**
 * ロールとそのロールが操作可能なファシリティツリーを管理するクラス。
 */
public class FacilityTreeCache {
	private static Log m_log = LogFactory.getLog( FacilityTreeCache.class );

	// ロールとそのロールが操作可能なファシリティツリーを管理（権限情報のみ保持）
	private static ConcurrentHashMap<String, FacilityTreePrivilege> m_roleFacilityTreePrivilegeMap = new ConcurrentHashMap<String, FacilityTreePrivilege>();
	
	// ファシリティツリー
	private static FacilityTreeItem m_facilityTreeItem = new FacilityTreeItem();
	
	// リポジトリ情報
	private static ConcurrentHashMap<String, FacilityInfo> m_facilityInfoMap = new ConcurrentHashMap<String, FacilityInfo>();
	
	static {
		refresh();
	}

	/**
	 * ロールがファシリティの参照権限があるかどうかを返す。
	 * 
	 * @param facilityId チェック対象のファシリティID
	 * @param roleId チェック対象のロールID
	 * @param isNode ノードチェックが必要な場合はtrue
	 * @throws FacilityNotFound ファシリティ未存在エラー
	 * @throws InvalidRole オブジェクト権限エラー
	 * @throws InvalidSetting ノードでない場合のエラー
	 */
	public static void validateFacilityId(String facilityId, String roleId, boolean isNode) throws FacilityNotFound, InvalidRole, InvalidSetting {
		m_log.debug("validateFacilityId() : facilityId = " + facilityId
					+ ", roleId = " + roleId);
		// 存在確認
		List<FacilityInfo> facilityInfoList = FacilityTreeCache.getAllFacilityList();
		boolean checkFlg = false;
		for (FacilityInfo facilityInfo : facilityInfoList) {
			m_log.debug("validateFacilityId() : checkFacilityId = " + facilityInfo.getFacilityId());
			if (facilityInfo.getFacilityId() != null
					&& facilityInfo.getFacilityId().contains(facilityId)) {
				if (isNode) {
					if(facilityInfo.getFacilityType() != FacilityConstant.TYPE_NODE){
						InvalidSetting e = new InvalidSetting("Src FacilityId is not node. : facilityId = " + facilityId);
						throw e;
					}
				}
				checkFlg = true;
				break;
			}
		}
		if (!checkFlg) {
			FacilityNotFound e = new FacilityNotFound("FacilityId is not exist in repository. : facilityId = " + facilityId);
			throw e;
		}
		// アクセス権限確認
		facilityInfoList = FacilityTreeCache.getFacilityListByRoleId(roleId);
		checkFlg = false;
		for (FacilityInfo facilityInfo : facilityInfoList) {
			if (facilityInfo.getFacilityId().contains(facilityId)) {
				checkFlg = true;
				break;
			}
		}
		if (!checkFlg) {
			InvalidRole e = new InvalidRole(Messages.getString("message.accesscontrol.16") + ", facilityId = " + facilityId);
			throw e;
		}
	}

	/**
	 * ユーザが参照可能なノードを返す。
	 * 
	 * @param userId ユーザID
	 * @return ユーザが参照可能なノード一覧を返す。
	 */
	public static List<FacilityInfo> getNodeListByUserId(String userId){
		m_log.debug("getNodeListByUserId() : userId " + userId);
		FacilityTreePrivilege facilityTreePrivilege = getFacilityTreePrivilegeByUser(userId);
		FacilityTreeItem facilityTreeItem = getFacilityTree(facilityTreePrivilege);
		return getNodeList(facilityTreeItem);
	}

	/**
	 * ロールが参照可能なノードを返す。
	 * 
	 * @param roleId ロールID
	 * @return ロールが参照可能なノード一覧を返す。
	 */
	public static List<FacilityInfo> getNodeListByRoleId(String roleId){
		m_log.debug("getNodeListByRoleId() : roleId " + roleId);
		FacilityTreeItem facilityTreeItem = getFacilityTreeByRoleId(roleId);
		return getNodeList(facilityTreeItem);
	}


	/**
	 * 参照可能なノード一覧を返す。
	 * 
	 * @param facilityTreePrivilege ファシリティのオブジェクト権限
	 * @return 操作可能なノード一覧
	 */
	private static List<FacilityInfo> getNodeList(FacilityTreeItem facilityTreeItem){
		m_log.debug("getNodeList() ");
		List<FacilityInfo> facilityInfoList = new ArrayList<FacilityInfo>();
		List<String> facilityIdList = new ArrayList<String>();
		FacilityTreeItem rootItem = null;
		rootItem = facilityTreeItem.clone();

		if (rootItem.getChildrenArray() != null) {
			for (FacilityTreeItem childItem : rootItem.getChildrenArray()) {
				getNodeListRecursive(childItem, facilityIdList);
			}
		}
		
		// リストにファシリティ情報を格納する
		for (String facilityId : facilityIdList) {
			FacilityInfo facilityInfo = new FacilityInfo(m_facilityInfoMap.get(facilityId));
			facilityInfo.setNotReferFlg(false);
			facilityInfoList.add(facilityInfo);
		}
		return facilityInfoList;
	}

	/**
	 * 参照可能なノード一覧を返す。
	 * 
	 */
	private static void getNodeListRecursive(FacilityTreeItem facilityTreeItem, List<String> facilityIdList){
		// ノードの場合、格納して処理終了
		if (facilityTreeItem.getData().getFacilityType() == FacilityConstant.TYPE_NODE) {
			String facilityId = facilityTreeItem.getData().getFacilityId();
			if (!facilityIdList.contains(facilityId)) {
				facilityIdList.add(facilityId);
			}
			return;
		}
		// 再帰的にノードを格納する
		if (facilityTreeItem.getChildrenArray() != null) {
			for (FacilityTreeItem childItem : facilityTreeItem.getChildrenArray()) {
				getNodeListRecursive(childItem, facilityIdList);
			}
		}
	}

	/**
	 * ファシリティ全件を返す
	 * 
	 * @return ファシリティ全件
	 */
	public static List<FacilityInfo> getAllFacilityList(){
		m_log.debug("getAllFacilityList()");
		FacilityTreeItem facilityTreeItem = getFacilityTreeByRoleId(RoleIdConstant.ADMINISTRATORS);
		return getFacilityList(facilityTreeItem);
	}

	/**
	 * ユーザが参照可能なファシリティを返す。
	 * 
	 * @param userId ユーザID
	 * @return ユーザが参照可能なファシリティ一覧を返す。
	 */
	public static List<FacilityInfo> getFacilityListByUserId(String userId){
		m_log.debug("getFacilityListByUserId() : userId " + userId);
		FacilityTreeItem facilityTreeItem = getFacilityTreeByUserId(userId);
		return getFacilityList(facilityTreeItem);
	}

	/**
	 * ロールが参照可能なファシリティを返す。
	 * 
	 * @param roleId ロールID
	 * @return ロールが参照可能なファシリティ一覧を返す。
	 */
	public static List<FacilityInfo> getFacilityListByRoleId(String roleId){
		m_log.debug("getFacilityListByRoleId() : roleId " + roleId);
		FacilityTreeItem facilityTreeItem = getFacilityTreeByRoleId(roleId);
		return getFacilityList(facilityTreeItem);
	}


	/**
	 * 参照可能なファシリティ一覧を返す。
	 * 
	 * @param facilityTreePrivilege ファシリティのオブジェクト権限
	 * @return 操作可能なノード一覧
	 */
	private static List<FacilityInfo> getFacilityList(FacilityTreeItem facilityTreeItem){
		m_log.debug("getFacilityList() ");
		List<FacilityInfo> facilityInfoList = new ArrayList<FacilityInfo>();
		List<String> facilityIdList = new ArrayList<String>();
		FacilityTreeItem rootItem = null;
		rootItem = facilityTreeItem.clone();
		if (rootItem.getChildrenArray() != null) {
			for (FacilityTreeItem childItem : rootItem.getChildrenArray()) {
				getFacilityListRecursive(childItem, facilityIdList);
			}
		}
		
		// リストにファシリティ情報を格納する
		for (String facilityId : facilityIdList) {
			FacilityInfo facilityInfo = m_facilityInfoMap.get(facilityId);
			facilityInfoList.add(new FacilityInfo(facilityInfo));
		}
		return facilityInfoList;
	}

	/**
	 * 参照可能なファシリティツリーを返す。
	 * 
	 */
	private static void getFacilityListRecursive(FacilityTreeItem facilityTreeItem, List<String> facilityIdList){
		m_log.debug("getFacilityListRecursive() : begin facilityId = " + facilityTreeItem.getData().getFacilityId());
		// ノードの場合、格納して処理終了
		if (facilityTreeItem.getData().getFacilityType() == FacilityConstant.TYPE_NODE) {
			m_log.debug("getFacilityListRecursive() : facilityId = " + facilityTreeItem.getData().getFacilityId());
			String facilityId = facilityTreeItem.getData().getFacilityId();
			if (!facilityIdList.contains(facilityId)) {
				facilityIdList.add(facilityId);
			}
			return;
		}
		// スコープも参照可能な場合は格納する。
		m_log.debug("getFacilityListRecursive() : facilityId = " + facilityTreeItem.getData().getFacilityId());
		m_log.debug("getFacilityListRecursive() : isNotReferFlg = " + facilityTreeItem.getData().isNotReferFlg());
		if (!facilityTreeItem.getData().isNotReferFlg()) {
			String facilityId = facilityTreeItem.getData().getFacilityId();
			if (!facilityIdList.contains(facilityId)) {
				facilityIdList.add(facilityId);
			}
		}

		// 再帰的にノードを格納する
		if (facilityTreeItem.getChildrenArray() != null) {
			for (FacilityTreeItem childItem : facilityTreeItem.getChildrenArray()) {
				getFacilityListRecursive(childItem, facilityIdList);
			}
		}
	}


	/**
	 * ファシリティツリーを返す。
	 * 
	 * @return ファシリティツリーを返す。
	 */
	public static FacilityTreeItem getAllFacilityTree(){
		m_log.debug("getAllFacilityTree()");
		return m_facilityTreeItem;
	}

	/**
	 * ユーザが操作可能なファシリティツリーを返す。
	 * 
	 * @param userId ユーザID
	 * @return ユーザが操作可能なファシリティツリーを返す。
	 */
	public static FacilityTreeItem getFacilityTreeByUserId(String userId){
		m_log.debug("getFacilityTreeByUserId() : userId " + userId);
		FacilityTreePrivilege facilityTreePrivilege = getFacilityTreePrivilegeByUser(userId);
		return getFacilityTree(facilityTreePrivilege);
	}

	/**
	 * ロールが操作可能なファシリティツリーを返す。
	 * 
	 * @param roleId ロールID
	 * @return ロールが操作可能なファシリティツリーを返す。
	 */
	public static FacilityTreeItem getFacilityTreeByRoleId(String roleId){
		m_log.debug("getFacilityTreeByRoleId() : roleId " + roleId);
		FacilityTreePrivilege facilityTreePrivilege = m_roleFacilityTreePrivilegeMap.get(roleId);
		return getFacilityTree(facilityTreePrivilege);
	}

	/**
	 * 操作可能なファシリティツリーを返す。
	 * 
	 * @param facilityTreePrivilege ファシリティのオブジェクト権限
	 * @return 操作可能なファシリティツリーを返す。
	 */
	private static FacilityTreeItem getFacilityTree(FacilityTreePrivilege facilityTreePrivilege){
		m_log.debug("getFacilityTree() ");
		FacilityTreeItem rootItem = null;
		rootItem = m_facilityTreeItem.clone();
		// スコープ
		FacilityTreeItem scopeItem = rootItem.getChildrenArray()[0];
		FacilityTreePrivilege scopePrivilege = facilityTreePrivilege.getChildrenArray()[0];
		scopeItem.getData().setNotReferFlg(!scopePrivilege.getPrivilege(ObjectPrivilegeMode.READ));
		m_log.debug("getFacilityTree() : facilityId=" + scopeItem.getData().getFacilityId() + ", notReferFlg=" +
				scopeItem.getData().isNotReferFlg());
		FacilityTreeItem[] itemChildren = scopeItem.getChildrenArray();
		if (itemChildren != null) {
			FacilityTreePrivilege[] privilegeChildren = scopePrivilege.getChildrenArray();
			for (int itemNum = 0 ; itemNum < itemChildren.length ; itemNum++) {
				for (int privNum = 0 ; privNum < privilegeChildren.length ; privNum++) {
					//FacilityTreeItemの要素のファシリティIDとFacilityTreePrivilegeの要素のファイシティIDが一致した場合
					if (itemChildren[itemNum].getData().getFacilityId()
							.equals(privilegeChildren[privNum].getFacilityId())) {
						getFacilityTreeRecursive(scopeItem, itemChildren[itemNum], privilegeChildren[privNum]);
						break;
					}
				}
				// TOP配下のスコープ確認
				if (itemChildren[itemNum] != null
						&& itemChildren[itemNum].getData().isNotReferFlg()
						&& itemChildren[itemNum].getChildrenArray().length == 0) {
					m_log.debug("getFacilityTree() delete child facilityId = " + itemChildren[itemNum].getData().getFacilityId());
					scopeItem.removeChild(itemChildren[itemNum].getData().getFacilityId());
				}
			}
		}
		return rootItem;
	}

	/**
	 * 操作可能なファシリティツリーを返す。
	 * 
	 */
	private static void getFacilityTreeRecursive(FacilityTreeItem parentFacilityTreeItem, FacilityTreeItem facilityTreeItem, FacilityTreePrivilege facilityTreePrivilege){
		// 親が参照不可の場合、ノードは格納しない。
		if (facilityTreeItem.getData().getFacilityType() == FacilityConstant.TYPE_NODE
			&& parentFacilityTreeItem != null
			&& parentFacilityTreeItem.getData().isNotReferFlg()) {
			return;
		}
		
		if (facilityTreeItem.getData().getFacilityType() == FacilityConstant.TYPE_SCOPE) {
			if (parentFacilityTreeItem != null
					&& !parentFacilityTreeItem.getData().isNotReferFlg()) {
				facilityTreeItem.getData().setNotReferFlg(new Boolean(false));
			} else {
				facilityTreeItem.getData().setNotReferFlg(!facilityTreePrivilege.getPrivilege(ObjectPrivilegeMode.READ));
			}
		} else {
			facilityTreeItem.getData().setNotReferFlg(new Boolean(false));
		}
		
		// 再帰的にファシリティを格納する
		if (facilityTreeItem.getData().getFacilityType() == FacilityConstant.TYPE_SCOPE) {
			FacilityTreeItem[] itemChildren = facilityTreeItem.getChildrenArray();
			if (itemChildren != null) {
				FacilityTreePrivilege[] privilegeChildren = facilityTreePrivilege.getChildrenArray();
				for (int itemNum = 0 ; itemNum < itemChildren.length ; itemNum++) {
					for (int privNum = 0 ; privNum < privilegeChildren.length ; privNum++) {
						//FacilityTreeItemの要素のファシリティIDとFacilityTreePrivilegeの要素のファイシティIDが一致した場合
						if (itemChildren[itemNum].getData().getFacilityId()
								.equals(privilegeChildren[privNum].getFacilityId())) {
							getFacilityTreeRecursive(facilityTreeItem, itemChildren[itemNum], privilegeChildren[privNum]);
							break;
						}
					}
					if (itemChildren[itemNum] != null
							&& itemChildren[itemNum].getData().isNotReferFlg()
							&& itemChildren[itemNum].getChildrenArray().length == 0) {
						m_log.debug("getFacilityTree() delete child facilityId = " + itemChildren[itemNum].getData().getFacilityId());
						facilityTreeItem.removeChild(itemChildren[itemNum].getData().getFacilityId());
					}
				}
			}
		}
	}

	/**
	 * ユーザが操作可能なファシリティのオブジェクト権限を返す
	 * 
	 * @param userId ユーザID
	 * @return オブジェクト権限
	 */
	private static FacilityTreePrivilege getFacilityTreePrivilegeByUser(String userId){
		m_log.debug("getFacilityTreePrivilege() : userId " + userId);
		FacilityTreePrivilege dest = null;
		try {
			// ユーザが所属するロールを取得する
			List<String> roleIds = UserRoleCache.getRoleIdList(userId);
			String destRoleId = null;
			String srcRoleId = null;
			FacilityTreePrivilege src = null;
			for (String roleId : roleIds) {
				if (destRoleId == null || destRoleId.isEmpty()) {
					destRoleId = roleId;
					dest = m_roleFacilityTreePrivilegeMap.get(destRoleId);
					continue;
				}
				srcRoleId = roleId;
				src = m_roleFacilityTreePrivilegeMap.get(srcRoleId);
				dest = merge(dest, src);
			}
			return dest;
		} catch (Exception e) {
			m_log.warn("getFacilityTreePrivilege() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			return null; // エラーの場合はnullを返す
		}
	}

	/**
	 * ファシリティIDを引数にFacilityInfoを返す
	 */
	public static FacilityInfo getFacilityInfo(String facilityId) {
		m_log.debug("getFacilityInfo() : facilityId " + facilityId);
		
		return m_facilityInfoMap.get(facilityId);
	}
	
	/**
	 * 自身が子である親のFacilityInfoのリストを返す
	 * 
	 * @param childFacilityId
	 * @return list
	 */
	public static List<FacilityInfo> getParentFacilityInfo(String childFacilityId) {
		List<FacilityInfo> list = new ArrayList<FacilityInfo>();
		getParentFacilityInfoSub(childFacilityId, m_facilityTreeItem, list);
		return list;
	}
	
	private static void getParentFacilityInfoSub(String childFacilityId, FacilityTreeItem item, List<FacilityInfo> list) {
		
		// ノードの場合は処理終了
		if (item.getData().getFacilityType() == FacilityConstant.TYPE_NODE) {
			return;
		}
		
		for(FacilityTreeItem childItem : item.getChildren()) {
			if(childItem != null && childItem.getData().getFacilityId().equals(childFacilityId)) {
				// 子のデータのファシリティIDがマッチした場合、
				if(item.getData().getFacilityType() == FacilityConstant.TYPE_SCOPE) {
					list.add(item.getData());
				}
			}
			getParentFacilityInfoSub(childFacilityId, childItem, list);
		}
	}
	
	/**
	 * キャッシュ中のファシリティツリー情報から自身が親である直下の子のリストを返す
	 * @param parentFacilityId
	 * @return list
	 */
	public static List<FacilityInfo> getChildrenFacilityInfo(String parentFacilityId) {
		return getChildrenFacilityInfoSub(parentFacilityId, m_facilityTreeItem);
	}
	
	/**
	 * 引数で与えたファシリティツリー情報から自身が親である直下の子のリストを返す
	 * @param parentFacilityId
	 * @param facilityTreeItem
	 * @return list
	 */
	private static List<FacilityInfo> getChildrenFacilityInfo(String parentFacilityId, FacilityTreeItem facilityTreeItem) {
		return getChildrenFacilityInfoSub(parentFacilityId, facilityTreeItem);
	}
	
	private static List<FacilityInfo> getChildrenFacilityInfoSub(String parentFacilityId, FacilityTreeItem item) {
		if (item.getData().getFacilityId().equals(parentFacilityId)) {
			ArrayList<FacilityInfo> ret = new ArrayList<FacilityInfo> ();
			for (FacilityTreeItem childItem : item.getChildren()) {
				ret.add(childItem.getData());
			}
			return ret;
		}
		for (FacilityTreeItem childItem : item.getChildren()) {
			List<FacilityInfo> ret = getChildrenFacilityInfoSub(parentFacilityId, childItem);
			if (ret != null) {
				return ret;
			}
		}
		return null;
	}

	/** ファシリティ関連情報をリフレッシュする */
	public static synchronized void refresh() {
		
		JpaTransactionManager jtm = new JpaTransactionManager();
		if (!jtm.isNestedEm()) {
			m_log.warn("refresh() : transactioin has not been begined.");
			jtm.close();
			return;
		}
		
		long startTime = -1;
		long infoMapRefreshTime = -1;
		long treeItemRefreshTime = -1;
		long treePrivilegeMapRefreshTime = -1;
		
		/*
		 * FacilityInfoMap のリフレッシュ
		 */
		startTime = System.currentTimeMillis();
		ConcurrentHashMap<String, FacilityInfo> facilityInfoMap = setFacilityInfoList();
		if(facilityInfoMap == null) {
			return;
		}
		infoMapRefreshTime = System.currentTimeMillis() - startTime;
		// FacilityInfoはFacilityTreeItem再構築時に参照されるため、先に反映させる
		m_facilityInfoMap = facilityInfoMap;
		m_log.info("refresh() : FacilityInfoMap(Cache) " + infoMapRefreshTime + "ms");
		
		/*
		 * FacilityTreeItem のリフレッシュ
		 */
		startTime = System.currentTimeMillis();
		FacilityTreeItem facilityTreeItem = setFacilityTreeItem();
		treeItemRefreshTime = System.currentTimeMillis() - startTime;
		
		/*
		 * FacilityTreePrivilegeMap のリフレッシュ
		 */
		startTime = System.currentTimeMillis();
		// キャッシュ予定のfacilityTreeItemを引数に渡す
		ConcurrentHashMap<String, FacilityTreePrivilege> roleFacilityTreePrivilegeMap = setFacilityTreePrivilege(facilityTreeItem);
		if(facilityTreeItem == null || roleFacilityTreePrivilegeMap == null) {
			return;
		}
		treePrivilegeMapRefreshTime = System.currentTimeMillis() - startTime;
		
		m_facilityTreeItem = facilityTreeItem;
		m_roleFacilityTreePrivilegeMap = roleFacilityTreePrivilegeMap;
		m_log.info("refresh() : FacilityTreeItem(Cache) " + treeItemRefreshTime + "ms");
		m_log.info("refresh() : FacilityTreePrivilegeMap(Cache) " + treePrivilegeMapRefreshTime + "ms");
			
	}
	
	/** キャッシュの情報を出力する **/
	public static void printCache() {
		
		/*
		 * m_facilityInfoMap を出力
		 */
		m_log.info("printCache() : FacilityInfo start");
		for(String key : m_facilityInfoMap.keySet()) {
			
			FacilityInfo info = m_facilityInfoMap.get(key);
			
			m_log.info("facility id = " + info.getFacilityId() +
					", facility name = " + info.getFacilityName());
			
		}
		m_log.info("printCache() : FacilityInfo end");
		
		/*
		 * m_facilityTreeItem を出力
		 */
		m_log.info("printCache() : FacilityTreeItem start");
		String brank = "  ";
		FacilityTreeItem treeItem = m_facilityTreeItem.clone();
		if (treeItem != null
				&& treeItem.getChildrenArray()[0] != null
				&& treeItem.getChildrenArray()[0].getChildrenArray() != null) {
			m_log.info("facility id = [TOP]");
			for (FacilityTreeItem tree : treeItem.getChildrenArray()[0].getChildrenArray()) {
				m_log.info(brank + "facility id = " + tree.getData().getFacilityId());
				printFacilityTreeItemRecursive(tree, RepositoryControllerBean.ALL, true, brank);
			}
		}
		m_log.info("printCache() : FacilityTreeItem end");
		
		/*
		 * m_roleFacilityTreePrivilegeMap を出力
		 */
		m_log.info("printCache() : FacilityTreePrivilege start");
		brank = "  ";
		for(String roleId : m_roleFacilityTreePrivilegeMap.keySet()) {
			
			m_log.info("role id = " + roleId);
			String print = "facility id = [TOP]";
			FacilityTreePrivilege priv = m_roleFacilityTreePrivilegeMap.get(roleId);
			Map<ObjectPrivilegeMode, Boolean> privileges = priv.getPrivileges();
			for (ObjectPrivilegeMode mode : PrivilegeConstant.objectPrivilegeModes) {
				print = print + " : " + mode.toString() + " = " + privileges.get(mode).toString();
			}
			m_log.info(print);
			
			for (FacilityTreePrivilege parentPrivilege : priv.getChildrenArray()[0].getChildrenArray()) {
				print = "facility id = " + parentPrivilege.getFacilityId();
				privileges = parentPrivilege.getPrivileges();
				for (ObjectPrivilegeMode mode : PrivilegeConstant.objectPrivilegeModes) {
					print = print + " : " + mode.toString() + " = " + privileges.get(mode).toString();
				}
				m_log.info(brank + print);
				printFacilityTreePrivilegeRecursive(parentPrivilege, brank);
			}
			
		}
		m_log.info("printCache() : FacilityTreePrivilege end");
	}
	
	/**
	 * スコープ配下にあるファシリティの一覧を出力する。<BR>
	 * 
	 * @param parentFacilityTreeItem スコープのファシリティインスタンス
	 * @param level 取得する階層数
	 * @param facilityList 格納先となるファシリティの配列
	 * @param scopeFlag スコープ自身を含めるか（含める:true 含めない:false)
	 * @param brank 表示用の空欄
	 */
	private static void printFacilityTreeItemRecursive(FacilityTreeItem parentFacilityTreeItem, 
			int level, boolean scopeFlag, String brank) {
		/** ローカル変数 */
		boolean recursive = false;
		int nextLevel = 0;
		
		// 表示用の空欄
		brank = brank + "  ";

		/** メイン処理 */
		// 階層数による再帰的処理の必要性の確認
		if (level == RepositoryControllerBean.ALL) {
			recursive = true;
			nextLevel = RepositoryControllerBean.ALL;
		} else if (level > 1) {
			recursive = true;
			nextLevel = level - 1;
		}

		// 再帰的にファシリティを配列に追加する
		FacilityTreeItem[] childFacilityTreeItems = parentFacilityTreeItem.getChildrenArray();
		if (childFacilityTreeItems != null) {
			for (FacilityTreeItem childFacilityTreeItem : childFacilityTreeItems) {
				if (childFacilityTreeItem.getData().getFacilityType() == FacilityConstant.TYPE_SCOPE) {
					if (scopeFlag) {
						m_log.info(brank + "facility id = " + childFacilityTreeItem.getData().getFacilityId());
					}
				} else {
					m_log.info(brank + "facility id = " + childFacilityTreeItem.getData().getFacilityId());
				}
				if (recursive) {
					printFacilityTreeItemRecursive(childFacilityTreeItem, nextLevel, scopeFlag, brank);
				}
			}
		}
	}
	
	/**
	 * スコープ配下にあるファシリティの一覧をオブジェクト権限と共に出力する
	 * 
	 * @param parentPrivilege 
	 * @param brank 表示用の空欄
	 */
	private static void printFacilityTreePrivilegeRecursive(FacilityTreePrivilege parentPrivilege, String brank) {
		
		// 表示用の空欄
		brank = brank + "  ";
		
		// 再帰的に表示する
		FacilityTreePrivilege[] childFacilityTreePrivileges = parentPrivilege.getChildrenArray();
		if (childFacilityTreePrivileges != null && childFacilityTreePrivileges.length > 0) {
			for (FacilityTreePrivilege childFacilityTreePrivilege : childFacilityTreePrivileges) {
				Map<ObjectPrivilegeMode, Boolean> privileges = childFacilityTreePrivilege.getPrivileges();
				String print = "facility id = " + childFacilityTreePrivilege.getFacilityId();
				for (ObjectPrivilegeMode mode : PrivilegeConstant.objectPrivilegeModes) {
					print = print + " : " + mode.toString() + " = " + privileges.get(mode).toString();
				}
				m_log.info(brank + print);
				
				// 再起呼び出しの判定を行い、必要応じて呼び出す
				if (childFacilityTreePrivilege.getChildrenArray() != null &&
						childFacilityTreePrivilege.getChildrenArray().length > 0) {
					printFacilityTreePrivilegeRecursive(childFacilityTreePrivilege, brank);
				}
			}
		}
	}
	
	/**
	 * 2つのオブジェクト権限をマージする。
	 * 
	 * @param dest マージするオブジェクト権限
	 * @param src マージするオブジェクト権限
	 * @return マージしたオブジェクト権限
	 */
	private static FacilityTreePrivilege merge(FacilityTreePrivilege dest, FacilityTreePrivilege src) {
		FacilityTreePrivilege privilege = null;
		privilege = dest.clone();

		for (ObjectPrivilegeMode mode : PrivilegeConstant.objectPrivilegeModes) {
			privilege.setPrivilege(mode, (privilege.getPrivilege(mode) || src.getPrivilege(mode)));
		}
		FacilityTreePrivilege[] privilegeChildren = privilege.getChildrenArray();
		if (privilegeChildren != null) {
			FacilityTreePrivilege[] srcChildren = src.getChildrenArray();
			for (int i = 0 ; i < privilegeChildren.length ; i++) {
				mergeRecursive(privilegeChildren[i], srcChildren[i]);
			}
				
		}
		return privilege;
	}
	
	/**
	 * 2つのオブジェクト権限をマージする。
	 * 
	 * @param dest マージするオブジェクト権限
	 * @param src マージするオブジェクト権限
	 */
	private static void mergeRecursive(FacilityTreePrivilege dest, FacilityTreePrivilege src) {
		for (ObjectPrivilegeMode mode : PrivilegeConstant.objectPrivilegeModes) {
			dest.setPrivilege(mode, (dest.getPrivilege(mode) || src.getPrivilege(mode)));
		}
		FacilityTreePrivilege[] destChildren = dest.getChildrenArray();
		if (destChildren != null) {
			FacilityTreePrivilege[] srcChildren = src.getChildrenArray();
			for (int i = 0 ; i < destChildren.length ; i++) {
				mergeRecursive(destChildren[i], srcChildren[i]);
			}
				
		}
	}
	
	/**
	 * ファシリティのオブジェクト権限を取得し、キャッシュに設定する<BR>
	 * 
	 * @param facilityTreeItem ファシリティツリー情報
	 * @return ファシリティのオブジェクト権限
	 */
	private static ConcurrentHashMap<String, FacilityTreePrivilege> setFacilityTreePrivilege(FacilityTreeItem facilityTreeItem) {
		
		// トランザクションが開始されていない場合には処理を終了する
		JpaTransactionManager jtm = new JpaTransactionManager();
		if (!jtm.isNestedEm()) {
			m_log.warn("refresh() : transactioin has not been begined.");
			return null;
		}
		
		ConcurrentHashMap<String, FacilityTreePrivilege> roleFacilityTreePrivilegeMap = 
				new ConcurrentHashMap<String, FacilityTreePrivilege>();
		
		/** メイン処理 */
		m_log.debug("getting tree data of facilities...");

		// 木構造最上位インスタンスの生成
		Map<ObjectPrivilegeMode, Boolean> privileges = new HashMap<ObjectPrivilegeMode, Boolean>(){
			{
				for (ObjectPrivilegeMode mode : PrivilegeConstant.objectPrivilegeModes) {
					super.put(mode, true);
				}
			}
		};
		
		/** ローカル変数 */
		FacilityTreePrivilege rootTree = null;
		FacilityTreePrivilege scopeTree = null;
		List<String> roleIds = UserRoleCache.getAllRoleIdList();
		
		ArrayList<FacilityEntity> facilityEntityList = null;
		try {
			facilityEntityList = FacilitySelector.getRootScopeList();
		} catch (FacilityNotFound e) {
			return null;
		} catch (Exception e) {
			m_log.warn("setFacilityTreePrivilege() failure to get a tree data of facilities. : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
		}
		
		// FacilityEntity を FacilityInfo に変換
		ArrayList<FacilityInfo> facilityInfoList = new ArrayList<FacilityInfo>();
		for (FacilityEntity entity : facilityEntityList) {
			facilityInfoList.add(m_facilityInfoMap.get(entity.getFacilityId()));
		}
		
		for(String roleId : roleIds) {
		
			//オブジェクト権限チェックに使用するファシリティ関連の権限リストを作成（ロールIDごと）
			List<ObjectPrivilegeEntity> objectPrivilegeFacilityEntitiesByRoleId 
			= com.clustercontrol.accesscontrol.util.QueryUtil.getAllObjectPrivilegeByFilter(HinemosModuleConstant.PLATFORM_REPOSITORY, null, roleId, null);
			
			rootTree = new FacilityTreePrivilege(null, null, FacilityConstant.STRING_COMPOSITE, FacilityConstant.TYPE_COMPOSITE, privileges);
			
			// コンポジットアイテムの生成
			scopeTree = new FacilityTreePrivilege(rootTree, null, Messages.getString("scope", Locale.getDefault()), FacilityConstant.TYPE_COMPOSITE, null);
	
			for (FacilityInfo facility : facilityInfoList) {
				setFacilityTreePrivilegeRecursive(scopeTree, facility, roleId, facilityTreeItem, objectPrivilegeFacilityEntitiesByRoleId);
			}
	
			m_log.debug("successful in getting tree data of facilities.");
			roleFacilityTreePrivilegeMap.put(roleId, rootTree);
		}
		
		return roleFacilityTreePrivilegeMap;
	}

	/**
	 * ファシリティのオブジェクト権限を再帰的に取得する。<BR>
	 * 
	 * @param parentPrivilege 親となるファシリティのオブジェクト権限
	 * @param facility 格納するファシリティインスタンス
	 * @param facilityTreeItem ファシリティツリー情報
	 * @param objectPrivilegeFacilityEntitiesByRoleId ファシリティ関連のオブジェクト権限リスト（ロールIDで絞込み済み）
	 * @param roleId ロールID
	 */
	private static void setFacilityTreePrivilegeRecursive(FacilityTreePrivilege parentPrivilege, FacilityInfo facility, 
			String roleId, FacilityTreeItem facilityTreeItem, List<ObjectPrivilegeEntity> objectPrivilegeFacilityEntitiesByRoleId) {
		m_log.debug("setFacilityTreePrivilegeRecursive()"
				+ " parent = " + (parentPrivilege==null?"":parentPrivilege.getFacilityId())
				+ " child = " + facility.getFacilityId()
				+ " loginuser = " + (String)HinemosSessionContext.instance().getProperty(HinemosSessionContext.LOGIN_USER_ID));

		/** ローカル変数 */
		FacilityTreePrivilege childPrivilege = null;

		// ファシリティへのアクセス権限を設定する
		Map<ObjectPrivilegeMode, Boolean> privileges = new HashMap<ObjectPrivilegeMode, Boolean>(){
			{
				for (ObjectPrivilegeMode mode : PrivilegeConstant.objectPrivilegeModes) {
					super.put(mode, false);
				}
			}};
		// 特権ロールの場合は許可する
		if (RoleIdConstant.isAdministratorRole(roleId)) {
			for (ObjectPrivilegeMode mode : PrivilegeConstant.objectPrivilegeModes) {
				privileges.put(mode, true);
			}
		} 
		// オーナーロールの場合は許可する
		else if (facility.getOwnerRoleId().equals(roleId)) {
			for (ObjectPrivilegeMode mode : PrivilegeConstant.objectPrivilegeModes) {
				privileges.put(mode, true);
			}
		} 
		// オブジェクト権限が設定されている場合は許可する
		else {
			for(ObjectPrivilegeEntity entity : objectPrivilegeFacilityEntitiesByRoleId) {
				if(facility.getFacilityId().equals(entity.getId().getObjectId())){
					privileges.put(ObjectPrivilegeMode.valueOf(entity.getId().getObjectPrivilege()), true);
				}
			}
		}
		// 親スコープに参照権限がある場合は参照権限を設定する
		if (!privileges.get(ObjectPrivilegeMode.READ)
			&& parentPrivilege != null
			&& parentPrivilege.getFacilityType() == FacilityConstant.TYPE_SCOPE
			&& parentPrivilege.getPrivilege(ObjectPrivilegeMode.READ)) {
			privileges.put(ObjectPrivilegeMode.READ, true);
		}

		childPrivilege = new FacilityTreePrivilege(parentPrivilege, facility.getFacilityId(), facility.getFacilityId(), facility.getFacilityType(), privileges);

		// 再帰的にファシリティを格納する
		List<FacilityInfo> childFacilityInfos = getChildrenFacilityInfo(facility.getFacilityId(), facilityTreeItem);
		if (childFacilityInfos != null) {
			for (FacilityInfo child : childFacilityInfos) {
				setFacilityTreePrivilegeRecursive(childPrivilege, child, roleId, facilityTreeItem, objectPrivilegeFacilityEntitiesByRoleId);
			}
		}
	}
	
	/**
	 * ファシリティ情報一覧をキャッシュに設定する。<BR>
	 * 
	 * @return ConcurrentHashMap<String, FacilityInfo>
	 */
	private static ConcurrentHashMap<String, FacilityInfo> setFacilityInfoList() {

		// トランザクションが開始されていない場合には処理を終了する
		JpaTransactionManager jtm = new JpaTransactionManager();
		if (!jtm.isNestedEm()) {
			m_log.warn("refresh() : transactioin has not been begined.");
			return null;
		}
		
		 ConcurrentHashMap<String, FacilityInfo> facilityInfoMap = new ConcurrentHashMap<String, FacilityInfo>();;
		
		// ファシリティ情報を全件取得する
		List<FacilityEntity> facilityEntities = QueryUtil.getAllFacility_NONE();
		for (FacilityEntity facilityEntity : facilityEntities) {
			// ファシリティの格納
			FacilityInfo facilityInfo = new FacilityInfo();
			facilityInfo.setFacilityId(facilityEntity.getFacilityId());
			facilityInfo.setFacilityName(facilityEntity.getFacilityName());
			facilityInfo.setFacilityType(facilityEntity.getFacilityType());
			facilityInfo.setDisplaySortOrder(facilityEntity.getDisplaySortOrder());
			facilityInfo.setIconImage(facilityEntity.getIconImage());
			facilityInfo.setBuiltInFlg(FacilitySelector.isBuildinScope(facilityEntity));
			facilityInfo.setValid(FacilityUtil.isValid(facilityEntity));
			facilityInfo.setOwnerRoleId(facilityEntity.getOwnerRoleId());
			facilityInfo.setDescription(facilityEntity.getDescription());
			facilityInfoMap.put(facilityEntity.getFacilityId(), facilityInfo);
		}
		
		return facilityInfoMap;
	}
	
	/**
	 * ファシリティの木構造を取得し、キャッシュに設定する。<BR>
	 * 
	 * @return FacilityTreeItem
	 */
	private static FacilityTreeItem setFacilityTreeItem() {

		// トランザクションが開始されていない場合には処理を終了する
		JpaTransactionManager jtm = new JpaTransactionManager();
		if (!jtm.isNestedEm()) {
			m_log.warn("refresh() : transactioin has not been begined.");
			return null;
		}

		/** ローカル変数 */
		FacilityTreeItem rootTree = null;
		FacilityInfo rootInfo = null;
		FacilityInfo scopeInfo = null;
		FacilityTreeItem scopeTree = null;

		/** メイン処理 */
		m_log.debug("getting tree data of facilities...");

		// 木構造最上位インスタンスの生成
		rootInfo = new FacilityInfo();
		rootInfo.setFacilityName(FacilityConstant.STRING_COMPOSITE);
		rootInfo.setFacilityType(FacilityConstant.TYPE_COMPOSITE);
		rootTree = new FacilityTreeItem(null, rootInfo);

		// コンポジットアイテムの生成
		scopeInfo = new FacilityInfo();
		scopeInfo.setFacilityName(Messages.getString("scope", Locale.getDefault()));
		scopeInfo.setFacilityType(FacilityConstant.TYPE_COMPOSITE);
		scopeTree = new FacilityTreeItem(rootTree, scopeInfo);

		try {
			for (FacilityEntity facility : FacilitySelector.getRootScopeList()) {
				setFacilityTreeItemRecursive(scopeTree, facility);
			}
		} catch (FacilityNotFound e) {
		} catch (Exception e) {
			m_log.warn("setFacilityTreeItem() failure to get a tree data of facilities. : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
		}

		m_log.debug("successful in getting tree data of facilities.");
		return rootTree;
	}

	/**
	 * ファシリティの木構造を再帰的に取得する。<BR>
	 * 
	 * @param parentTree 親となるファシリティの木構造
	 * @param facility 格納するファシリティインスタンス
	 * @param roleId ロールID
	 */
	private static void setFacilityTreeItemRecursive(FacilityTreeItem parentTree, FacilityEntity facility) {
		m_log.debug("setFacilityTreeItemRecursive()"
				+ " parent = " + (parentTree==null?"":parentTree.getData().getFacilityId())
				+ " child = " + facility.getFacilityId()
				+ " loginuser = " + (String)HinemosSessionContext.instance().getProperty(HinemosSessionContext.LOGIN_USER_ID));

		/** ローカル変数 */
		FacilityInfo facilityInfo = null;
		FacilityTreeItem childTree = null;

		// ファシリティの格納
		facilityInfo = new FacilityInfo(m_facilityInfoMap.get(facility.getFacilityId()));
		childTree = new FacilityTreeItem(parentTree, facilityInfo);

		// 再帰的にファシリティを格納する
		if (FacilityUtil.isScope(facility)) {
			List<FacilityEntity> childFacilityEntities = QueryUtil.getChildFacilityEntity(facility.getFacilityId());
			if (childFacilityEntities != null) {
				for (FacilityEntity child : childFacilityEntities) {
					setFacilityTreeItemRecursive(childTree, child);
				}
			}
		}
	}
}
