/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.notify.factory;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.fault.InvalidRole;
import com.clustercontrol.fault.NotifyNotFound;
import com.clustercontrol.notify.bean.NotifyCommandInfo;
import com.clustercontrol.notify.bean.NotifyEventInfo;
import com.clustercontrol.notify.bean.NotifyInfo;
import com.clustercontrol.notify.bean.NotifyJobInfo;
import com.clustercontrol.notify.bean.NotifyLogEscalateInfo;
import com.clustercontrol.notify.bean.NotifyMailInfo;
import com.clustercontrol.notify.bean.NotifyStatusInfo;
import com.clustercontrol.notify.model.NotifyCommandInfoEntity;
import com.clustercontrol.notify.model.NotifyEventInfoEntity;
import com.clustercontrol.notify.model.NotifyInfoEntity;
import com.clustercontrol.notify.model.NotifyJobInfoEntity;
import com.clustercontrol.notify.model.NotifyLogEscalateInfoEntity;
import com.clustercontrol.notify.model.NotifyMailInfoEntity;
import com.clustercontrol.notify.model.NotifyStatusInfoEntity;
import com.clustercontrol.notify.util.QueryUtil;
import com.clustercontrol.repository.session.RepositoryControllerBean;
import com.clustercontrol.util.apllog.AplLogger;

/**
 * 通知情報を検索するクラスです。
 *
 * @version 3.0.0
 * @since 1.0.0
 */
public class SelectNotify {

	/** ログ出力のインスタンス。 */
	private static Log m_log = LogFactory.getLog( SelectNotify.class );

	/**
	 * 通知情報を返します。
	 * 
	 * @param notifyId 取得対象の通知ID
	 * @return 通知情報
	 * @throws NotifyNotFound
	 * @throws InvalidRole
	 * @throws HinemosUnknown
	 * 
	 * @see com.clustercontrol.notify.ejb.entity.NotifyInfoBean
	 * @see com.clustercontrol.notify.ejb.entity.NotifyEventInfoBean
	 */
	public NotifyInfo getNotify(String notifyId) throws NotifyNotFound, InvalidRole, HinemosUnknown {
		// 通知情報を取得
		NotifyInfo info = null;

		try {
			NotifyInfoEntity notify = QueryUtil.getNotifyInfoPK(notifyId);
			info = getNotifyInfo(notify);
		} catch (NotifyNotFound e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
			String[] args = { notifyId };
			apllog.put("SYS", "004", args);
			throw e;
		} catch (InvalidRole e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
			String[] args = { notifyId };
			apllog.put("SYS", "004", args);
			throw e;
		} catch (HinemosUnknown e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
			String[] args = { notifyId };
			apllog.put("SYS", "004", args);
			throw e;
		}

		return info;
	}

	private NotifyInfo getNotifyInfo(NotifyInfoEntity notify) throws HinemosUnknown {

		NotifyInfo bean = null;
		String notifyId = notify.getNotifyId();

		ArrayList<NotifyCommandInfo> notifyCommands = getNotifyCommand(notifyId,notify);
		ArrayList<NotifyEventInfo> notifyEvents = getNotifyEvent(notifyId,notify);
		ArrayList<NotifyJobInfo> notifyJobs = getNotifyJob(notifyId,notify);
		ArrayList<NotifyLogEscalateInfo> notifyLogEscalates = getNotifyLogEscalate(notifyId,notify);
		ArrayList<NotifyMailInfo> notifyMails = getNotifyMail(notifyId,notify);
		ArrayList<NotifyStatusInfo> notifyStatuses = getNotifyStatus(notifyId,notify);
		Long regDate = notify.getRegDate() == null ? null:notify.getRegDate().getTime();
		Long updateDate = notify.getUpdateDate() == null ? null:notify.getUpdateDate().getTime();

		bean = new NotifyInfo(
				notify.getNotifyId(),
				notify.getDescription(),
				notify.getNotifyType(),
				notify.getInitialCount(),
				notify.getRenotifyType(),
				notify.getRenotifyPeriod(),
				regDate,
				updateDate,
				notify.getRegUser(),
				notify.getUpdateUser(),
				notify.getValidFlg(),
				notify.getOwnerRoleId(),
				notifyCommands,
				notifyEvents,
				notifyJobs,
				notifyLogEscalates,
				notifyMails,
				notifyStatuses
				);
		return bean;
	}

	/**
	 * コマンド通知詳細情報を取得します。
	 * 
	 * @param notifyId
	 * @param notify
	 * @return
	 */
	private ArrayList<NotifyCommandInfo> getNotifyCommand(String notifyId, NotifyInfoEntity notify){
		ArrayList<NotifyCommandInfo> notifyEvents = new ArrayList<NotifyCommandInfo>();
		Collection<NotifyCommandInfoEntity> ct = notify.getNotifyCommandInfoEntities();

		for (NotifyCommandInfoEntity entity : ct){
			NotifyCommandInfo info = new NotifyCommandInfo();
			info.setNotifyId(entity.getId().getNotifyId() );
			info.setPriority(entity.getId().getPriority() );
			info.setValidFlg(entity.getValidFlg() );
			info.setCommand(entity.getCommand() );
			info.setEffectiveUser(entity.getEffectiveUser() );
			info.setSetEnvironment(entity.getSetEnvironment() );
			info.setTimeout(entity.getCommandTimeout());
			notifyEvents.add(info);
		}
		return notifyEvents;
	}

	/**
	 * イベント通知詳細情報を取得します。
	 * 
	 * @param notifyId
	 * @param notify
	 * @return
	 */
	private ArrayList<NotifyEventInfo> getNotifyEvent(String notifyId, NotifyInfoEntity notify){
		ArrayList<NotifyEventInfo> notifyEvents = new ArrayList<NotifyEventInfo>();
		Collection<NotifyEventInfoEntity> ct = notify.getNotifyEventInfoEntities();

		Iterator<NotifyEventInfoEntity> itr = ct.iterator();
		NotifyEventInfoEntity entity = null;
		while(itr.hasNext()){
			entity = itr.next();
			NotifyEventInfo event = new NotifyEventInfo(
					notifyId,
					entity.getId().getPriority(),
					entity.getEventNormalFlg(),
					entity.getEventNormalState());

			notifyEvents.add(event);
		}
		return notifyEvents;
	}

	/**
	 * ジョブ通知詳細情報を取得します。
	 * 
	 * @param notifyId
	 * @param notify
	 * @return
	 * @throws HinemosUnknown
	 */
	private ArrayList<NotifyJobInfo> getNotifyJob(String notifyId, NotifyInfoEntity notify) throws HinemosUnknown {
		ArrayList<NotifyJobInfo> notifyEvents = new ArrayList<NotifyJobInfo>();
		Collection<NotifyJobInfoEntity> ct = notify.getNotifyJobInfoEntities();

		Iterator<NotifyJobInfoEntity> itr = ct.iterator();

		NotifyJobInfoEntity entity = null;
		while(itr.hasNext()){
			entity = itr.next();
			String facilityPath = null;
			facilityPath = new RepositoryControllerBean().getFacilityPath(entity.getJobExecFacilityId(), null);

			NotifyJobInfo status = new NotifyJobInfo(
					notifyId,
					entity.getId().getPriority(),
					entity.getJobRun(),
					entity.getJobunitId(),
					entity.getJobId(),
					entity.getJobFailurePriority(),
					entity.getJobExecFacilityFlg(),
					entity.getJobExecFacilityId(),
					facilityPath);

			notifyEvents.add(status);
		}

		return notifyEvents;
	}

	/**
	 * ログエスカレーション通知詳細情報を取得します。
	 * 
	 * @param notifyId
	 * @param notify
	 * @return
	 * @throws HinemosUnknown
	 */
	private ArrayList<NotifyLogEscalateInfo> getNotifyLogEscalate(String notifyId, NotifyInfoEntity notify) throws HinemosUnknown {
		ArrayList<NotifyLogEscalateInfo> notifyEvents = new ArrayList<NotifyLogEscalateInfo>();
		Collection<NotifyLogEscalateInfoEntity> ct = notify.getNotifyLogEscalateInfoEntities();

		Iterator<NotifyLogEscalateInfoEntity> itr = ct.iterator();
		NotifyLogEscalateInfoEntity entity = null;
		while(itr.hasNext()){
			entity = itr.next();
			String facilityPath = null;
			facilityPath = new RepositoryControllerBean().getFacilityPath(entity.getEscalateFacilityId(), null);

			NotifyLogEscalateInfo status = new NotifyLogEscalateInfo(
					notifyId,
					entity.getId().getPriority(),
					entity.getEscalateFlg(),
					entity.getEscalateMessage(),
					entity.getSyslogPriority(),
					entity.getSyslogFacility(),
					entity.getEscalateFacilityFlg(),
					entity.getEscalateFacilityId(),
					facilityPath,
					entity.getEscalatePort());

			notifyEvents.add(status);
		}

		return notifyEvents;
	}

	/**
	 * メール通知詳細情報を取得します。
	 * 
	 * @param notifyId
	 * @param notify
	 * @return
	 */
	private ArrayList<NotifyMailInfo> getNotifyMail(String notifyId, NotifyInfoEntity notify){
		ArrayList<NotifyMailInfo> notifyEvents = new ArrayList<NotifyMailInfo>();
		Collection<NotifyMailInfoEntity> ct = notify.getNotifyMailInfoEntities();

		Iterator<NotifyMailInfoEntity> itr = ct.iterator();
		NotifyMailInfoEntity entity = null;
		while(itr.hasNext()){
			entity = itr.next();
			NotifyMailInfo status = new NotifyMailInfo(
					notifyId,
					entity.getId().getPriority(),
					entity.getMailTemplateInfoEntity() == null ? null : entity.getMailTemplateInfoEntity().getMailTemplateId(),
							entity.getMailFlg(),
							entity.getMailAddress());

			notifyEvents.add(status);
		}
		return notifyEvents;
	}

	/**
	 * ステータス通知詳細情報を取得します。
	 * 
	 * @param notifyId
	 * @param notify
	 * @return
	 */
	private ArrayList<NotifyStatusInfo> getNotifyStatus(String notifyId, NotifyInfoEntity notify){
		ArrayList<NotifyStatusInfo> notifyEvents = new ArrayList<NotifyStatusInfo>();
		Collection<NotifyStatusInfoEntity> ct = notify.getNotifyStatusInfoEntities();

		Iterator<NotifyStatusInfoEntity> itr = ct.iterator();
		NotifyStatusInfoEntity entity = null;
		while(itr.hasNext()){
			entity = itr.next();
			NotifyStatusInfo status = new NotifyStatusInfo(
					notifyId,
					entity.getId().getPriority(),
					entity.getStatusFlg(),
					entity.getStatusInvalidFlg(),
					entity.getStatusUpdatePriority(),
					entity.getStatusValidPeriod());

			notifyEvents.add(status);
		}
		return notifyEvents;
	}

	/**
	 * 通知情報一覧を返します(障害検知用通知を除く)。
	 * <p>
	 * <ol>
	 * <li>通知IDの昇順に並んだ全ての通知情報を取得します。</li>
	 * <li>１通知情報をテーブルのカラム順（{@link com.clustercontrol.notify.bean.NotifyTableDefine}）に、リスト（{@link ArrayList}）にセットします。</li>
	 * <li>この１通知情報を保持するリストを、通知情報一覧を保持するリスト（{@link ArrayList}）に格納し返します。<BR>
	 *  <dl>
	 *  <dt>通知情報一覧（Objectの2次元配列）</dt>
	 *  <dd>{ 通知情報1 {カラム1の値, カラム2の値, … }, 通知情報2{カラム1の値, カラム2の値, …}, … }</dd>
	 *  </dl>
	 * </li>
	 * </ol>
	 * 
	 * @return 通知情報一覧（Objectの2次元配列）
	 * @throws NotifyNotFound
	 * @throws InvalidRole
	 * @throws HinemosUnknown
	 * 
	 * @see com.clustercontrol.notify.bean.NotifyTableDefine
	 * @see #collectionToArray(Collection)
	 */
	public ArrayList<NotifyInfo> getNotifyList() throws NotifyNotFound, InvalidRole, HinemosUnknown {
		m_log.debug("getNotifyList() : start");
		ArrayList<NotifyInfo> list = new ArrayList<NotifyInfo>();

		try {
			// 通知情報一覧を取得
			List<NotifyInfoEntity> ct = QueryUtil.getAllNotifyInfoOrderByNotifyId();
			for(NotifyInfoEntity notify : ct){
				list.add(getNotify(notify.getNotifyId()));
			}
		} catch (NotifyNotFound e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
			apllog.put("SYS", "006");
			throw e;
		} catch (InvalidRole e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
			apllog.put("SYS", "006");
			throw e;
		} catch (HinemosUnknown e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
			apllog.put("SYS", "006");
			throw e;
		}

		return list;
	}

	/**
	 * オーナーロールIDを条件として通知情報一覧を返します(障害検知用通知を除く)。
	 * 
	 * @param ownerRoleId
	 * @return 通知情報一覧（Objectの2次元配列）
	 * @throws HinemosUnknown
	 * 
	 * @see com.clustercontrol.notify.bean.NotifyTableDefine
	 * @see #collectionToArray(Collection)
	 */
	public ArrayList<NotifyInfo> getNotifyListByOwnerRole(String ownerRoleId) throws HinemosUnknown {
		m_log.debug("getNotifyListByOwnerRole() : start");
		ArrayList<NotifyInfo> list = new ArrayList<NotifyInfo>();

		try {
			// 通知情報一覧を取得
			List<NotifyInfoEntity> ct = QueryUtil.getAllNotifyInfoOrderByNotifyId_OR(ownerRoleId);
			for(NotifyInfoEntity notify : ct){
				list.add(getNotifyInfo(notify));
			}
		} catch (HinemosUnknown e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
			apllog.put("SYS", "006");
			throw e;
		}

		return list;
	}

	/**
	 * 通知情報マップを返します。
	 * <p>
	 * <ol>
	 * <li>全ての通知情報を取得します。</li>
	 * <li>通知情報マップに、通知情報をセットします。</li>
	 *  <dl>
	 *   <dt>通知情報マップ</dt>
	 *   <dd>キー：通知ID</dd>
	 *   <dd>値　：通知情報</dd>
	 *  </dl>
	 * </li>
	 * </ol>
	 * 
	 * @return 通知情報マップ
	 * @throws HinemosUnknown
	 */
	public HashMap<String, NotifyInfo> getNotifyMap() throws HinemosUnknown {

		HashMap<String, NotifyInfo> map = new HashMap<String, NotifyInfo>();
		try
		{
			// 通知情報一覧を取得
			List<NotifyInfoEntity> ct = QueryUtil.getAllNotifyInfo();

			Iterator<NotifyInfoEntity> itr = ct.iterator();
			NotifyInfo bean = null;
			NotifyInfoEntity notify = null;

			while(itr.hasNext()){

				notify = itr.next();
				bean = getNotifyInfo(notify);

				map.put(notify.getNotifyId(), bean);
			}

		} catch (HinemosUnknown e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
			apllog.put("SYS", "006");
			throw e;
		}
		return map;
	}
}
