/*

 Copyright (C) 2006 NTT DATA Corporation

 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation, version 2.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 */

package com.clustercontrol.performance.dialog;

import java.lang.reflect.InvocationTargetException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.dialogs.ProgressMonitorDialog;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.layout.FormLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.DirectoryDialog;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;

import com.clustercontrol.ClusterControlPlugin;
import com.clustercontrol.bean.DataRangeConstant;
import com.clustercontrol.composite.FacilityTreeComposite;
import com.clustercontrol.composite.action.StringVerifyListener;
import com.clustercontrol.monitor.util.MonitorSettingEndpointWrapper;
import com.clustercontrol.performance.action.RecordDataWriter;
import com.clustercontrol.performance.util.Messages;
import com.clustercontrol.repository.FacilityPath;
import com.clustercontrol.repository.util.RepositoryEndpointWrapper;
import com.clustercontrol.ws.monitor.InvalidRole_Exception;
import com.clustercontrol.ws.monitor.MonitorInfo;
import com.clustercontrol.ws.repository.FacilityTreeItem;

/**
 * 収集した実績データをエクスポートするためのダイアログクラス
 * 
 * @version 4.0.0
 * @since 1.0.0
 */
public class ExportDialog extends Dialog {

	// ログ
	private static Log m_log = LogFactory.getLog( ExportDialog.class );

	// Dialog Composite
	private Composite compositeAll;
	private Composite compositeScope;
	private Composite compositeCheck;	// チェックBox用コンポジット
	private Composite compositeOutpuFile;
	private FacilityTreeComposite treeComposite;
	private Label pathLabel = null;
	private Text folderNameText = null;
	private Button checkBoxHeader = null; // ヘッダ出力有無用チェックボックス

	// Dialog Setting
	private String monitorId = null; //
	private String monitorTypeId = null; //
	private String facilityId = null; // monitor target
	private FacilityTreeItem treeItem = null;

	// Export Setting
	private RecordDataWriter writer;

	private String targetFacilityId = null; // export target
	private boolean headerFlag = false;
	private boolean archiveFlag = false;
	private String folderName = null;

	/**
	 * コンストラクタ
	 */
	public ExportDialog(Shell parent, String monitorId, String monitorTypeId) {
		super(parent);
		this.monitorId = monitorId;
		this.monitorTypeId = monitorTypeId;
	}

	/**
	 * 初期サイズの設定
	 */
	@Override
	protected Point getInitialSize() {
		return new Point(640, 480);
	}

	/**
	 * タイトルの設定
	 */
	@Override
	protected void configureShell(Shell newShell) {
		super.configureShell(newShell);
		newShell.setText(Messages.getString("EXPORT_DIALOG_TITLE")); // "性能[エクスポート]"
	}

	/**
	 * ダイアログの設定
	 */
	@Override
	protected Control createDialogArea(Composite parent) {

		// コンポジット全体
		compositeAll = (Composite) super.createDialogArea(parent);
		compositeAll.setLayout(new FillLayout());

		// スコープ表示
		compositeScope = new Composite(compositeAll, SWT.NONE);
		pathLabel = new Label(compositeScope, SWT.NONE);
		compositeScope.setLayout(new GridLayout());
		pathLabel.setText(Messages.getString("SCOPE") + " : "); // "スコープ"
		pathLabel.pack();

		// スコープツリー表示
		createTree();

		// チェックボックス
		createCompositeCheck();

		// 出力ファイル先指定BOX
		createCompositeOutpuFile();

		// セパレータ
		createSeparator();

		return compositeAll;
	}

	/**
	 * FacilityTreeのコンポジット。
	 * ノード指定時のみOKボタンを
	 */
	private void createTree() {

		try{
			MonitorInfo monitorInfo = MonitorSettingEndpointWrapper.getMonitor(this.monitorId, this.monitorTypeId);
			this.facilityId = monitorInfo.getFacilityId();
			this.treeItem = RepositoryEndpointWrapper.getExecTargetFacilityTreeByFacilityId(this.facilityId, null);
		} catch (InvalidRole_Exception e) {
			m_log.warn("createTree() : getMonitor, " + e.getMessage(), e);
			MessageDialog.openInformation(null, com.clustercontrol.util.Messages.getString("message"), com.clustercontrol.util.Messages.getString("message.accesscontrol.16"));
		} catch (Exception e) {
			m_log.warn("createTree() getMonitor, " + e.getMessage(), e);
		}

		treeComposite = new FacilityTreeComposite(compositeScope, SWT.NONE, null, false,
				true, true, false);
		GridData gridDataTree = new GridData(GridData.FILL_BOTH);
		treeComposite.setLayoutData(gridDataTree);
		treeComposite.setScopeTree(this.treeItem);
		treeComposite.getTreeViewer().addSelectionChangedListener(
				new ISelectionChangedListener() {

					@Override
					public void selectionChanged(SelectionChangedEvent event) {
						StructuredSelection selection = (StructuredSelection) event
								.getSelection();
						FacilityTreeItem selectItem = (FacilityTreeItem) selection
								.getFirstElement();
						if (selectItem instanceof FacilityTreeItem) {
							// パスラベルの更新
							FacilityPath path = new FacilityPath(
									ClusterControlPlugin.getDefault()
									.getSeparator());
							pathLabel.setText(Messages.getString("SCOPE")
									+ " : " + path.getPath(selectItem));
							pathLabel.pack();
						}
					}
				});
	}

	/**
	 * 「ヘッダを出力」のチェックボックス。
	 */
	private void createCompositeCheck() {
		compositeCheck = new Composite(compositeScope, SWT.NULL);
		GridLayout buttonLayout = new GridLayout(1, true);
		buttonLayout.marginHeight = 0;
		buttonLayout.marginWidth = 0;
		compositeCheck.setLayout(buttonLayout);


		GridData layoutData =  null;

		// ヘッダ出力有無チェックボックス
		checkBoxHeader = new Button(compositeCheck, SWT.CHECK);
		checkBoxHeader.setText(Messages.getString("OUTPUT_HEADER")); // "ヘッダを出力"
		layoutData =  new GridData(GridData.FILL_HORIZONTAL);
		checkBoxHeader.setLayoutData(layoutData);
	}

	/**
	 * 出力するフォルダ名を決めるためのコンポジット。
	 */
	private void createCompositeOutpuFile() {
		compositeOutpuFile = new Composite(compositeScope, SWT.NULL);
		GridData gridDataComposite3 = new GridData(GridData.FILL_HORIZONTAL);
		compositeOutpuFile.setLayoutData(gridDataComposite3);
		compositeOutpuFile.setLayout(new FormLayout());

		// ラベルフィールド
		Label label = new Label(compositeOutpuFile, SWT.NULL);
		label.setText(Messages.getString("FOLDER_NAME") + " : "); // "ファイル名"
		FormData labelData = new FormData();
		labelData.top = new FormAttachment(0, 0); // ウィンドウの上側にはりつく
		labelData.left = new FormAttachment(0, 0); // ウィンドウの左側にはりつく
		label.setLayoutData(labelData);

		// ボタン
		Button button = new Button(compositeOutpuFile, SWT.NULL);
		button.setText(Messages.getString("REFER")); // "参照"
		FormData buttonData = new FormData();
		buttonData.top = new FormAttachment(0, 0); // ウィンドウの上側にはりつく
		buttonData.right = new FormAttachment(100, 0); // ウィンドウの右側にはりつく
		button.setLayoutData(buttonData);

		button.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {

				// 出力先ファイルを選択するダイアログを開く
				DirectoryDialog saveDialog = new DirectoryDialog(getShell());
				String folderName = saveDialog.open();
				if (folderName != null){
					folderNameText.setText(folderName);
				}
			}
		});

		// テキストフィールド
		folderNameText = new Text(compositeOutpuFile, SWT.BORDER);

		// 初期値をカレントディレクトリとする。

		FormData textData = new FormData();
		textData.top = new FormAttachment(0, 0); // ウィンドウの上側にはりつく
		textData.left = new FormAttachment(label, 0); // ラベルの右側にはりつく
		textData.right = new FormAttachment(button, 0); // ボタンの左側にはりつく
		folderNameText.setLayoutData(textData);
		folderNameText.addVerifyListener(new StringVerifyListener(
				DataRangeConstant.VARCHAR_4096));
	}

	/**
	 * セパレータの作成
	 */
	private void createSeparator() {
		// セパレータ(水平線)を作成
		Label h_separator = new Label(compositeScope, SWT.SEPARATOR
				| SWT.HORIZONTAL);
		GridData gridDataLabel = new GridData(GridData.FILL_HORIZONTAL);
		h_separator.setLayoutData(gridDataLabel);
	}

	/**
	 * OKボタン押下時に呼ばれます。
	 */
	@Override
	protected void okPressed() {
		// DataWriterへの入力
		this.folderName = this.folderNameText.getText();
		this.targetFacilityId = treeComposite.getSelectItem().getData().getFacilityId();
		this.headerFlag = checkBoxHeader.getSelection();
		this.archiveFlag = true;

		// 入力チェック
		if (this.targetFacilityId  == null) {
			MessageDialog.openWarning(getShell(),
					Messages.getString("MESSAGE_0020"), // "確認"
					Messages.getString("MSG_SPECIFY_SCOPE")); // "スコープを指定してください"
			return;
		}
		if (this.folderName == null || folderName.equals("")) {
			MessageDialog.openWarning(getShell(),
					Messages.getString("INPUT_ERROR"), // "入力エラー"
					Messages.getString("MSG_SPECIFY_FOLDER_NAME")); // "ファイル名を指定してください"
			return;
		}

		// 書き込み準備
		writer = new RecordDataWriter(
				this.monitorId,
				this.targetFacilityId,
				this.headerFlag,
				this.archiveFlag,
				this.folderName);

		// Download & 書き込み
		try {
			IRunnableWithProgress op = new IRunnableWithProgress() {
				@Override
				public void run(IProgressMonitor monitor)
						throws InvocationTargetException, InterruptedException {
					// エクスポートを開始
					Thread exportThread = new Thread(writer);
					exportThread.start();

					monitor.beginTask(Messages.getString("EXPORT"), 100); // "エクスポート"

					int progress = 0;
					int buff = 0;
					while (progress < 100) {
						progress = writer.getProgress();

						if (monitor.isCanceled()) {
							throw new InterruptedException("");
						}
						if (writer.isCanceled()) {
							throw new InterruptedException(writer.getCancelMessage());
						}
						Thread.sleep(50);
						monitor.worked(progress - buff);
						buff = progress;
					}
					monitor.done();
				}
			};

			// ダイアログの表示
			new ProgressMonitorDialog(getShell()).run(true, true, op);

			MessageDialog.openInformation(getShell(),
					Messages.getString("MESSAGE_0020"), // "確認"
					Messages.getString("MSG_EXPORT_SUCCESS")); // "エクスポート処理が成功しました。

		} catch (InvocationTargetException e) {
		} catch (InterruptedException e) {
			// キャンセルされた場合の処理
			MessageDialog.openInformation(getShell(),
					Messages.getString("MESSAGE_0020"), // "確認"
					Messages.getString("MSG_EXPORT_CANCEL") + " : " + e.getMessage()); // "エクスポート処理はキャンセルされました"
			writer.setCanceled(true);
		}

		super.okPressed();
	}
}