/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.jobmanagement.dialog;

import java.util.Iterator;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.RowData;
import org.eclipse.swt.layout.RowLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.TabFolder;
import org.eclipse.swt.widgets.TabItem;
import org.eclipse.swt.widgets.Text;

import com.clustercontrol.accesscontrol.bean.RoleIdConstant;
import com.clustercontrol.bean.DataRangeConstant;
import com.clustercontrol.bean.EndStatusConstant;
import com.clustercontrol.bean.RequiredFieldColorConstant;
import com.clustercontrol.bean.SizeConstant;
import com.clustercontrol.bean.YesNoConstant;
import com.clustercontrol.composite.RoleIdListComposite;
import com.clustercontrol.composite.RoleIdListComposite.Mode;
import com.clustercontrol.composite.action.StringVerifyListener;
import com.clustercontrol.dialog.CommonDialog;
import com.clustercontrol.dialog.ValidateResult;
import com.clustercontrol.fault.JobInvalid;
import com.clustercontrol.jobmanagement.bean.JobConstant;
import com.clustercontrol.jobmanagement.bean.SystemParameterConstant;
import com.clustercontrol.jobmanagement.composite.CommandComposite;
import com.clustercontrol.jobmanagement.composite.ControlComposite;
import com.clustercontrol.jobmanagement.composite.EndDelayComposite;
import com.clustercontrol.jobmanagement.composite.EndStatusComposite;
import com.clustercontrol.jobmanagement.composite.FileComposite;
import com.clustercontrol.jobmanagement.composite.MultiplicityComposite;
import com.clustercontrol.jobmanagement.composite.NotificationsComposite;
import com.clustercontrol.jobmanagement.composite.ParameterComposite;
import com.clustercontrol.jobmanagement.composite.ReferComposite;
import com.clustercontrol.jobmanagement.composite.StartDelayComposite;
import com.clustercontrol.jobmanagement.composite.WaitRuleComposite;
import com.clustercontrol.jobmanagement.util.JobEditStateUtil;
import com.clustercontrol.jobmanagement.util.JobEndpointWrapper;
import com.clustercontrol.jobmanagement.util.JobPropertyUtil;
import com.clustercontrol.jobmanagement.util.JobTreeItemUtil;
import com.clustercontrol.jobmanagement.util.JobUtil;
import com.clustercontrol.notify.util.NotifyGroupIdGenerator;
import com.clustercontrol.util.Messages;
import com.clustercontrol.ws.jobmanagement.JobCommandInfo;
import com.clustercontrol.ws.jobmanagement.JobEndStatusInfo;
import com.clustercontrol.ws.jobmanagement.JobFileInfo;
import com.clustercontrol.ws.jobmanagement.JobInfo;
import com.clustercontrol.ws.jobmanagement.JobNotificationsInfo;
import com.clustercontrol.ws.jobmanagement.JobParameterInfo;
import com.clustercontrol.ws.jobmanagement.JobTreeItem;
import com.clustercontrol.ws.jobmanagement.JobWaitRuleInfo;
import com.clustercontrol.ws.jobmanagement.OtherUserGetLock_Exception;
import com.clustercontrol.ws.notify.NotifyRelationInfo;

/**
 * ジョブ[ジョブの作成・変更]ダイアログクラスです。
 * 
 * @version 4.0.0
 * @since 1.0.0
 */
public class JobDialog extends CommonDialog {
	// ログ
	private static Log m_log = LogFactory.getLog( JobDialog.class );

	/** ジョブID用テキスト */
	private Text m_jobIdText = null;
	/** ジョブ名用テキスト */
	private Text m_jobNameText = null;
	/** 説明テキスト */
	private Text m_jobAnnotationText = null;
	/** 待ち条件タブ用コンポジット */
	private WaitRuleComposite m_startComposite = null;
	/** 制御タブ用コンポジット */
	private ControlComposite m_controlComposite = null;
	/** 終了状態タブ用コンポジット */
	private EndStatusComposite m_endComposite = null;
	/** コマンドタブ用コンポジット */
	private CommandComposite m_executeComposite = null;
	/** ファイル転送タブ用コンポジット */
	private FileComposite m_fileComposite = null;
	/** 通知先の指定タブ用コンポジット */
	private NotificationsComposite m_messageComposite = null;
	/** 開始遅延タブ用コンポジット */
	private StartDelayComposite m_startDelayComposite = null;
	/** 終了遅延タブ用コンポジット */
	private EndDelayComposite m_endDelayComposite = null;
	/** 多重度制御用コンポジット */
	private MultiplicityComposite m_multiplicityComposite = null;
	/** ジョブ変数タブ用コンポジット */
	private ParameterComposite m_parameterComposite = null;
	/** 参照タブ用コンポジット */
	private ReferComposite m_referComposite = null;

	/** ジョブツリーアイテム */
	private JobTreeItem m_jobTreeItem = null;
	/** タブフォルダー */
	private TabFolder m_tabFolder = null;
	/** シェル */
	private Shell m_shell = null;
	/** 読み取り専用フラグ */
	private boolean m_readOnly = false;

	/** オーナーロールID用テキスト */
	private RoleIdListComposite m_ownerRoleId = null;

	/**
	 * コンストラクタ
	 * 
	 * @param parent 親シェル
	 * @param readOnly 読み取り専用フラグ true：変更不可、false：変更可
	 */
	public JobDialog(Shell parent, boolean readOnly) {
		super(parent);
		m_readOnly = readOnly;
	}

	/**
	 * ダイアログエリアを生成します。
	 * <P>
	 * ジョブ種別により、表示するタブを切り替えます。
	 * 
	 * @param parent 親コンポジット
	 * 
	 * @see com.clustercontrol.jobmanagement.bean.JobInfo
	 * @see com.clustercontrol.bean.JobConstant
	 */
	@Override
	protected void customizeDialog(Composite parent) {
		m_shell = this.getShell();

		RowLayout layout = new RowLayout();
		layout.type = SWT.VERTICAL;
		layout.spacing = 0;
		layout.marginWidth = 10;
		layout.marginHeight = 10;
		layout.marginBottom = 0;
		layout.fill = true;
		parent.setLayout(layout);

		Composite composite = null;
		RowLayout rowLayout = null;

		JobInfo info = m_jobTreeItem.getData();
		// 本メソッドで詳細情報を取得するため、setJobFull実行
		JobPropertyUtil.setJobFull(info);

		if (info instanceof JobInfo) {
			if (info.getType() == JobConstant.TYPE_JOBUNIT) {
				parent.getShell().setText(
						Messages.getString("dialog.job.create.modify.jobunit"));
			} else if (info.getType() == JobConstant.TYPE_JOBNET) {
				parent.getShell().setText(
						Messages.getString("dialog.job.create.modify.jobnet"));
			} else if (info.getType() == JobConstant.TYPE_JOB) {
				parent.getShell().setText(
						Messages.getString("dialog.job.create.modify.job"));
			} else if (info.getType() == JobConstant.TYPE_FILEJOB) {
				parent.getShell().setText(
						Messages.getString("dialog.job.create.modify.forward.file.job"));
			} else if (info.getType() == JobConstant.TYPE_REFERJOB){
				parent.getShell().setText(
						Messages.getString("dialog.job.create.modify.refer.job"));
			}
		}
		boolean initFlag = true;
		if (info.getId() != null && info.getId().length() > 0) {
			initFlag = false;
		}

		// ジョブID
		composite = new Composite(parent, SWT.NONE);
		rowLayout = new RowLayout();
		rowLayout.type = SWT.HORIZONTAL;
		rowLayout.spacing = 10;
		composite.setLayout(rowLayout);
		Label jobIdTitle = new Label(composite, SWT.NONE);
		jobIdTitle.setText(Messages.getString("job.id") + " : ");
		jobIdTitle
		.setLayoutData(new RowData(120, SizeConstant.SIZE_LABEL_HEIGHT));
		this.m_jobIdText = new Text(composite, SWT.BORDER);
		this.m_jobIdText.setLayoutData(new RowData(200,
				SizeConstant.SIZE_TEXT_HEIGHT));
		this.m_jobIdText.addVerifyListener(
				new StringVerifyListener(DataRangeConstant.VARCHAR_64));
		this.m_jobIdText.addModifyListener(new ModifyListener(){
			@Override
			public void modifyText(ModifyEvent arg0) {
				update();
			}
		});

		// ジョブ名
		composite = new Composite(parent, SWT.NONE);
		rowLayout = new RowLayout();
		rowLayout.type = SWT.HORIZONTAL;
		rowLayout.spacing = 10;
		composite.setLayout(rowLayout);
		Label jobNameTitle = new Label(composite, SWT.NONE);
		jobNameTitle.setText(Messages.getString("job.name") + " : ");
		jobNameTitle.setLayoutData(new RowData(120,
				SizeConstant.SIZE_LABEL_HEIGHT));
		this.m_jobNameText = new Text(composite, SWT.BORDER);
		this.m_jobNameText.setLayoutData(new RowData(200,
				SizeConstant.SIZE_TEXT_HEIGHT));
		this.m_jobNameText.addVerifyListener(
				new StringVerifyListener(DataRangeConstant.VARCHAR_64));
		this.m_jobNameText.addModifyListener(new ModifyListener(){
			@Override
			public void modifyText(ModifyEvent arg0) {
				update();
			}
		});

		// 説明
		composite = new Composite(parent, SWT.NONE);
		rowLayout = new RowLayout();
		rowLayout.type = SWT.HORIZONTAL;
		rowLayout.spacing = 10;
		composite.setLayout(rowLayout);
		Label jobAnnotationTitle = new Label(composite, SWT.NONE);
		jobAnnotationTitle.setText(Messages.getString("description") + " : ");
		jobAnnotationTitle.setLayoutData(new RowData(120,
				SizeConstant.SIZE_LABEL_HEIGHT));
		m_jobAnnotationText = new Text(composite, SWT.BORDER);
		m_jobAnnotationText.setLayoutData(new RowData(200,
				SizeConstant.SIZE_TEXT_HEIGHT));
		m_jobAnnotationText.addVerifyListener(
				new StringVerifyListener(DataRangeConstant.VARCHAR_256));

		// オーナーロールID
		composite = new Composite(parent, SWT.NONE);
		rowLayout = new RowLayout();
		rowLayout.type = SWT.HORIZONTAL;
		rowLayout.spacing = 10;
		composite.setLayout(rowLayout);
		Label ownerRoleIdTitle = new Label(composite, SWT.NONE);
		ownerRoleIdTitle.setText(Messages.getString("owner.role.id") + " : ");
		ownerRoleIdTitle.setLayoutData(new RowData(120,
				SizeConstant.SIZE_LABEL_HEIGHT));
		// 新規登録、コピー時のみ変更可能
		// 新規登録、コピーの判定はJobInfo.createTimeで行う。
		if (info.getType() == JobConstant.TYPE_JOBUNIT && info.getCreateTime() == null) {
			this.m_ownerRoleId = new RoleIdListComposite(composite, SWT.NONE, true, Mode.OWNER_ROLE);
			this.m_ownerRoleId.getComboRoleId().addSelectionListener(new SelectionAdapter() {
				@Override
				public void widgetSelected(SelectionEvent e) {
					m_messageComposite.getNotifyId().setOwnerRoleId(m_ownerRoleId.getText());
				}
			});
		} else {
			this.m_ownerRoleId = new RoleIdListComposite(composite, SWT.NONE, false, Mode.OWNER_ROLE);
		}
		this.m_ownerRoleId.setLayoutData(new RowData(200,
				SizeConstant.SIZE_COMBO_HEIGHT + 5));

		m_tabFolder = new TabFolder(parent, SWT.NONE);

		if (info instanceof JobInfo) {
			if (info.getType() == JobConstant.TYPE_JOBNET) {
				//待ち条件
				m_startComposite = new WaitRuleComposite(m_tabFolder, SWT.NONE);
				TabItem tabItem1 = new TabItem(m_tabFolder, SWT.NONE);
				tabItem1.setText(Messages.getString("wait.rule"));
				tabItem1.setControl(m_startComposite);

				//制御
				m_controlComposite = new ControlComposite(m_tabFolder, SWT.NONE);
				TabItem tabItem2 = new TabItem(m_tabFolder, SWT.NONE);
				tabItem2.setText(Messages.getString("control"));
				tabItem2.setControl(m_controlComposite);

				//開始遅延
				m_startDelayComposite = new StartDelayComposite(m_tabFolder, SWT.NONE);
				TabItem tabItem3 = new TabItem(m_tabFolder, SWT.NONE);
				tabItem3.setText(Messages.getString("start.delay"));
				tabItem3.setControl(m_startDelayComposite);

				//終了遅延
				m_endDelayComposite = new EndDelayComposite(m_tabFolder, SWT.NONE, false);
				TabItem tabItem4 = new TabItem(m_tabFolder, SWT.NONE);
				tabItem4.setText(Messages.getString("end.delay"));
				tabItem4.setControl(m_endDelayComposite);
			}
			else if (info.getType() == JobConstant.TYPE_JOB) {
				//待ち条件
				m_startComposite = new WaitRuleComposite(m_tabFolder, SWT.NONE);
				TabItem tabItem1 = new TabItem(m_tabFolder, SWT.NONE);
				tabItem1.setText(Messages.getString("wait.rule"));
				tabItem1.setControl(m_startComposite);

				//制御
				m_controlComposite = new ControlComposite(m_tabFolder, SWT.NONE);
				TabItem tabItem2 = new TabItem(m_tabFolder, SWT.NONE);
				tabItem2.setText(Messages.getString("control"));
				tabItem2.setControl(m_controlComposite);

				//コマンド
				m_executeComposite = new CommandComposite(m_tabFolder, SWT.NONE);
				TabItem tabItem3 = new TabItem(m_tabFolder, SWT.NONE);
				tabItem3.setText(Messages.getString("command"));
				tabItem3.setControl(m_executeComposite);

				//開始遅延
				m_startDelayComposite = new StartDelayComposite(m_tabFolder, SWT.NONE);
				TabItem tabItem4 = new TabItem(m_tabFolder, SWT.NONE);
				tabItem4.setText(Messages.getString("start.delay"));
				tabItem4.setControl(m_startDelayComposite);

				//終了遅延
				m_endDelayComposite = new EndDelayComposite(m_tabFolder, SWT.NONE, false);
				TabItem tabItem5 = new TabItem(m_tabFolder, SWT.NONE);
				tabItem5.setText(Messages.getString("end.delay"));
				tabItem5.setControl(m_endDelayComposite);

				//多重度
				m_multiplicityComposite = new MultiplicityComposite(m_tabFolder, SWT.NONE, false);
				TabItem tabItem6 = new TabItem(m_tabFolder, SWT.NONE);
				tabItem6.setText(Messages.getString("multiplicity"));
				tabItem6.setControl(m_multiplicityComposite);
			}
			else if (info.getType() == JobConstant.TYPE_FILEJOB) {
				//待ち条件
				m_startComposite = new WaitRuleComposite(m_tabFolder, SWT.NONE);
				TabItem tabItem1 = new TabItem(m_tabFolder, SWT.NONE);
				tabItem1.setText(Messages.getString("wait.rule"));
				tabItem1.setControl(m_startComposite);

				//制御
				m_controlComposite = new ControlComposite(m_tabFolder, SWT.NONE);
				TabItem tabItem2 = new TabItem(m_tabFolder, SWT.NONE);
				tabItem2.setText(Messages.getString("control"));
				tabItem2.setControl(m_controlComposite);

				//ファイル転送
				m_fileComposite = new FileComposite(m_tabFolder, SWT.NONE);
				TabItem tabItem3 = new TabItem(m_tabFolder, SWT.NONE);
				tabItem3.setText(Messages.getString("forward.file"));
				tabItem3.setControl(m_fileComposite);

				//開始遅延
				m_startDelayComposite = new StartDelayComposite(m_tabFolder, SWT.NONE);
				TabItem tabItem4 = new TabItem(m_tabFolder, SWT.NONE);
				tabItem4.setText(Messages.getString("start.delay"));
				tabItem4.setControl(m_startDelayComposite);

				//終了遅延
				m_endDelayComposite = new EndDelayComposite(m_tabFolder, SWT.NONE, true);
				TabItem tabItem5 = new TabItem(m_tabFolder, SWT.NONE);
				tabItem5.setText(Messages.getString("end.delay"));
				tabItem5.setControl(m_endDelayComposite);

				//多重度
				m_multiplicityComposite = new MultiplicityComposite(m_tabFolder, SWT.NONE, true);
				TabItem tabItem6 = new TabItem(m_tabFolder, SWT.NONE);
				tabItem6.setText(Messages.getString("multiplicity"));
				tabItem6.setControl(m_multiplicityComposite);
			}
			//参照ジョブ
			else if(info.getType() == JobConstant.TYPE_REFERJOB){
				//待ち条件
				m_startComposite = new WaitRuleComposite(m_tabFolder, SWT.NONE);
				TabItem tabItem1 = new TabItem(m_tabFolder, SWT.NONE);
				tabItem1.setText(Messages.getString("wait.rule"));
				tabItem1.setControl(m_startComposite);

				//参照
				m_referComposite = new ReferComposite(m_tabFolder, SWT.NONE);
				TabItem tabItem2 = new TabItem(m_tabFolder, SWT.NONE);
				tabItem2.setText(Messages.getString("refer"));
				tabItem2.setControl(m_referComposite);
			}
			//参照ジョブ以外では使用する
			if (info.getType() != JobConstant.TYPE_REFERJOB) {
				//終了状態
				m_endComposite = new EndStatusComposite(m_tabFolder, SWT.NONE);
				TabItem tabItem7 = new TabItem(m_tabFolder, SWT.NONE);
				tabItem7.setText(Messages.getString("end.status"));
				tabItem7.setControl(m_endComposite);

				//通知先の指定
				m_messageComposite = new NotificationsComposite(m_tabFolder, SWT.NONE);
				TabItem tabItem8 = new TabItem(m_tabFolder, SWT.NONE);
				tabItem8.setText(Messages.getString("notifications"));
				tabItem8.setControl(m_messageComposite);
			}
			if (info.getType() == JobConstant.TYPE_JOBUNIT ) {

				//ジョブパラメータ
				m_parameterComposite = new ParameterComposite(m_tabFolder, SWT.NONE, initFlag);
				TabItem tabItem9 = new TabItem(m_tabFolder, SWT.NONE);
				tabItem9.setText(Messages.getString("job.parameter"));
				tabItem9.setControl(m_parameterComposite);
			}
		}

		m_tabFolder.setSelection(0);

		// 画面中央に
		Display display = m_shell.getDisplay();
		m_shell.setLocation(
				(display.getBounds().width - m_shell.getSize().x) / 2, (display
						.getBounds().height - m_shell.getSize().y) / 2);

		//ジョブ情報反映
		reflectJobInfo(info);

		if(m_readOnly){
			m_jobIdText.setEnabled(false);
			m_jobNameText.setEnabled(false);
			m_jobAnnotationText.setEnabled(false);
			m_ownerRoleId.setEnabled(false);
			if(m_startComposite != null)
				m_startComposite.setEnabled(false);
			if(m_controlComposite != null)
				m_controlComposite.setEnabled(false);
			if(m_executeComposite != null)
				m_executeComposite.setEnabled(false);
			if(m_fileComposite != null)
				m_fileComposite.setEnabled(false);
			if(m_startDelayComposite != null)
				m_startDelayComposite.setEnabled(false);
			if(m_endDelayComposite != null)
				m_endDelayComposite.setEnabled(false);
			if(m_multiplicityComposite != null)
				m_multiplicityComposite.setEnabled(false);

			if(info.getType() != JobConstant.TYPE_REFERJOB){
				m_endComposite.setEnabled(false);
				m_messageComposite.setEnabled(false);
			} else {
				if(m_referComposite != null){
					m_referComposite.setEnabled(false);
				}
			}

			if(m_parameterComposite != null)
				m_parameterComposite.setEnabled(false);
		}
	}

	/**
	 * 更新処理
	 * 
	 */
	public void update(){
		// 必須項目を明示
		if("".equals(this.m_jobIdText.getText())){
			this.m_jobIdText.setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
		}else{
			this.m_jobIdText.setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
		}
		if("".equals(this.m_jobNameText.getText())){
			this.m_jobNameText.setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
		}else{
			this.m_jobNameText.setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
		}
	}

	/**
	 * ＯＫボタンテキスト取得
	 * 
	 * @return ＯＫボタンのテキスト
	 * @since 1.0.0
	 */
	@Override
	protected String getOkButtonText() {
		return Messages.getString("ok");
	}

	/**
	 * キャンセルボタンテキスト取得
	 * 
	 * @return キャンセルボタンのテキスト
	 * @since 1.0.0
	 */
	@Override
	protected String getCancelButtonText() {
		return Messages.getString("cancel");
	}

	/**
	 * ジョブ情報をダイアログ及び各タブのコンポジットに反映します。
	 * 
	 * @see com.clustercontrol.jobmanagement.bean.JobInfo
	 * @see com.clustercontrol.bean.JobConstant
	 */
	private void reflectJobInfo(JobInfo info) {
		if (info instanceof JobInfo) {
			//ジョブID設定
			String jobId = info.getId();
			if (jobId != null) {
				m_jobIdText.setText(jobId);
			} else {
				m_jobIdText.setText("");
			}
			if (info.getType() == JobConstant.TYPE_JOBUNIT &&
					JobEditStateUtil.getLockedJobunitBackup(info) != null) {
				// すでにマネージャに登録してあるジョブユニットはIDを変更できない
				m_jobIdText.setEditable(false);
			}

			//ジョブ名設定
			if (info.getName() != null) {
				m_jobNameText.setText(info.getName());
			} else {
				m_jobNameText.setText("");
			}
			//注釈設定
			if (info.getDescription() != null) {
				m_jobAnnotationText.setText(info.getDescription());
			} else {
				m_jobAnnotationText.setText("");
			}

			// オーナーロール取得
			if (info.getOwnerRoleId() != null) {
				this.m_ownerRoleId.setText(info.getOwnerRoleId());
			} else {
				if (info.getType() == JobConstant.TYPE_JOBUNIT) {
					this.m_ownerRoleId.setText(RoleIdConstant.ALL_USERS);
				} else {
					JobTreeItem parentItem = m_jobTreeItem.getParent();
					if (parentItem != null) {
						JobInfo parentInfo = parentItem.getData();
						this.m_ownerRoleId.setText(parentInfo.getOwnerRoleId());
					}
				}
			}
			// すでにマネージャに登録してあるオーナロールIDを変更できない
			if (JobEditStateUtil.getLockedJobunitBackup(info) != null) {
				this.m_ownerRoleId.setEnabled(false);
			}
			//参照タブ以外
			if (info.getType() != JobConstant.TYPE_REFERJOB) {
				this.m_messageComposite.getNotifyId().setOwnerRoleId(m_ownerRoleId.getText());
			}

			JobWaitRuleInfo jobWaitRuleInfo = info.getWaitRule();
			if (jobWaitRuleInfo == null) {
				jobWaitRuleInfo = JobTreeItemUtil.getNewJobWaitRuleInfo();
			}

			//タブ内のコンポジットにジョブ情報を反映
			if (info.getType() == JobConstant.TYPE_JOBNET) {
				//開始待ち条件
				m_startComposite.setWaitRuleInfo(jobWaitRuleInfo);
				m_startComposite.setJobTreeItem(m_jobTreeItem);
				m_startComposite.reflectWaitRuleInfo();

				//制御
				m_controlComposite.setWaitRuleInfo(jobWaitRuleInfo);
				m_controlComposite.getCalendarId().setOwnerRoleId(this.m_ownerRoleId.getText());
				m_controlComposite.reflectWaitRuleInfo();

				//開始遅延
				m_startDelayComposite.setWaitRuleInfo(jobWaitRuleInfo);
				m_startDelayComposite.reflectWaitRuleInfo();

				//終了遅延
				m_endDelayComposite.setWaitRuleInfo(jobWaitRuleInfo);
				m_endDelayComposite.reflectWaitRuleInfo();
			}
			else if (info.getType() == JobConstant.TYPE_JOB) {
				//開始待ち条件
				m_startComposite.setWaitRuleInfo(jobWaitRuleInfo);
				m_startComposite.setJobTreeItem(m_jobTreeItem);
				m_startComposite.reflectWaitRuleInfo();

				//制御
				m_controlComposite.setWaitRuleInfo(jobWaitRuleInfo);
				m_controlComposite.getCalendarId().setOwnerRoleId(this.m_ownerRoleId.getText());
				m_controlComposite.reflectWaitRuleInfo();

				//実行内容
				m_executeComposite.setCommandInfo(info.getCommand());
				m_executeComposite.setOwnerRoleId(this.m_ownerRoleId.getText());
				m_executeComposite.reflectCommandInfo();

				//開始遅延
				m_startDelayComposite.setWaitRuleInfo(jobWaitRuleInfo);
				m_startDelayComposite.reflectWaitRuleInfo();

				//終了遅延
				m_endDelayComposite.setWaitRuleInfo(jobWaitRuleInfo);
				m_endDelayComposite.reflectWaitRuleInfo();

				//多重度
				m_multiplicityComposite.setWaitRuleInfo(jobWaitRuleInfo);
				m_multiplicityComposite.reflectWaitRuleInfo();
			}
			else if (info.getType() == JobConstant.TYPE_FILEJOB) {
				//開始待ち条件
				m_startComposite.setWaitRuleInfo(jobWaitRuleInfo);
				m_startComposite.setJobTreeItem(m_jobTreeItem);
				m_startComposite.reflectWaitRuleInfo();

				//制御
				m_controlComposite.setWaitRuleInfo(jobWaitRuleInfo);
				m_controlComposite.getCalendarId().setOwnerRoleId(this.m_ownerRoleId.getText());
				m_controlComposite.reflectWaitRuleInfo();

				//ファイル転送
				m_fileComposite.setFileInfo(info.getFile());
				m_fileComposite.setOwnerRoleId(this.m_ownerRoleId.getText());
				m_fileComposite.reflectFileInfo();

				//開始遅延
				m_startDelayComposite.setWaitRuleInfo(jobWaitRuleInfo);
				m_startDelayComposite.reflectWaitRuleInfo();

				//終了遅延
				m_endDelayComposite.setWaitRuleInfo(jobWaitRuleInfo);
				m_endDelayComposite.reflectWaitRuleInfo();

				//多重度
				m_multiplicityComposite.setWaitRuleInfo(jobWaitRuleInfo);
				m_multiplicityComposite.reflectWaitRuleInfo();
			}
			//参照ジョブ
			else if(info.getType() == JobConstant.TYPE_REFERJOB){
				//開始待ち条件
				m_startComposite.setWaitRuleInfo(jobWaitRuleInfo);
				m_startComposite.setJobTreeItem(m_jobTreeItem);
				m_startComposite.reflectWaitRuleInfo();
				//参照ジョブ
				m_referComposite.setReferJobUnitId(info.getReferJobUnitId());
				m_referComposite.setReferJobId(info.getReferJobId());
				m_referComposite.setJobTreeItem(m_jobTreeItem);
				m_referComposite.reflectReferInfo();
			}
			//参照タブ以外で使用する
			if (info.getType() != JobConstant.TYPE_REFERJOB) {
				//終了状態の定義
				m_endComposite.setEndInfo(info.getEndStatus());
				m_endComposite.reflectEndInfo();

				//メッセージの指定
				m_messageComposite.setNotificationsInfo(info.getNotifications());
				m_messageComposite.getNotifyId().setOwnerRoleId(this.m_ownerRoleId.getText());
				m_messageComposite.reflectNotificationsInfo();
			}

			if (info.getType() == JobConstant.TYPE_JOBUNIT) {

				//ジョブパラメータ
				m_parameterComposite.setParamInfo(info.getParam());
				m_parameterComposite.reflectParamInfo();
			}
		}
	}

	/**
	 * 入力値チェックをします。
	 * 
	 * @return 検証結果
	 * 
	 * @see com.clustercontrol.dialog.CommonDialog#validate()
	 */
	@Override
	protected ValidateResult validate() {
		ValidateResult result = null;

		result = createJobInfo();
		if (result != null) {
			return result;
		}

		JobInfo info = m_jobTreeItem.getData();
		if (info instanceof JobInfo) {
			if (info.getType() == JobConstant.TYPE_JOBNET) {
				//開始待ち条件
				result = m_startComposite.createWaitRuleInfo();
				if (result != null) {
					return result;
				}

				//制御
				result = m_controlComposite.createWaitRuleInfo();
				if (result != null) {
					return result;
				}

				//開始遅延
				result = m_startDelayComposite.createWaitRuleInfo();
				if (result != null) {
					return result;
				}

				//終了遅延
				result = m_endDelayComposite.createWaitRuleInfo();
				if (result != null) {
					return result;
				}
			} else if (info.getType() == JobConstant.TYPE_JOB) {
				//開始待ち条件
				result = m_startComposite.createWaitRuleInfo();
				if (result != null) {
					return result;
				}

				//制御
				result = m_controlComposite.createWaitRuleInfo();
				if (result != null) {
					return result;
				}

				//実行内容
				result = m_executeComposite.createCommandInfo();
				if (result != null) {
					return result;
				}

				//開始遅延
				result = m_startDelayComposite.createWaitRuleInfo();
				if (result != null) {
					return result;
				}

				//終了遅延
				result = m_endDelayComposite.createWaitRuleInfo();
				if (result != null) {
					return result;
				}

				//多重度
				result = m_multiplicityComposite.createWaitRuleInfo();
				if (result != null) {
					return result;
				}
			} else if (info.getType() == JobConstant.TYPE_FILEJOB) {
				//開始待ち条件
				result = m_startComposite.createWaitRuleInfo();
				if (result != null) {
					return result;
				}

				//制御
				result = m_controlComposite.createWaitRuleInfo();
				if (result != null) {
					return result;
				}

				//ファイル転送
				result = m_fileComposite.createFileInfo();
				if (result != null) {
					return result;
				}

				//開始遅延
				result = m_startDelayComposite.createWaitRuleInfo();
				if (result != null) {
					return result;
				}

				//終了遅延
				result = m_endDelayComposite.createWaitRuleInfo();
				if (result != null) {
					return result;
				}

				//多重度
				result = m_multiplicityComposite.createWaitRuleInfo();
				if (result != null) {
					return result;
				}
			}
			//参照ジョブ
			else if(info.getType() == JobConstant.TYPE_REFERJOB){
				//開始待ち条件
				result = m_startComposite.createWaitRuleInfo();
				if (result != null) {
					return result;
				}
				//参照ジョブ
				result = m_referComposite.createReferInfo();
				if(result != null){
					return result;
				}
			}
			//参照ジョブ以外で使用する
			if(info.getType() != JobConstant.TYPE_REFERJOB){
				//終了状態の定義
				result = m_endComposite.createEndInfo();
				if (result != null) {
					return result;
				}

				//メッセージの指定
				result = m_messageComposite.createNotificationsInfo();
				if (result != null) {
					return result;
				}
			}

			if (info.getType() == JobConstant.TYPE_JOBUNIT) {
				//ジョブパラメータ
				result = m_parameterComposite.createParamInfo();
				if (result != null) {
					return result;
				}
			}

			if (m_startComposite != null)
				info.setWaitRule(
						m_startComposite.getWaitRuleInfo());
			if (m_controlComposite != null)
				info.setWaitRule(
						m_controlComposite.getWaitRuleInfo());
			if (m_executeComposite != null)
				info.setCommand(
						m_executeComposite.getCommandInfo());
			if (m_fileComposite != null)
				info.setFile(
						m_fileComposite.getFileInfo());
			if (m_endComposite != null) {
				List<JobEndStatusInfo> jobEndStatusInfoList = info.getEndStatus();
				jobEndStatusInfoList.clear();
				if (m_endComposite.getEndInfo() != null) {
					jobEndStatusInfoList.addAll(m_endComposite.getEndInfo());
				}
			} if (m_startDelayComposite != null)
				info.setWaitRule(
						m_startDelayComposite.getWaitRuleInfo());
			if (m_endDelayComposite != null)
				info.setWaitRule(
						m_endDelayComposite.getWaitRuleInfo());
			if (m_multiplicityComposite != null)
				info.setWaitRule(
						m_multiplicityComposite.getWaitRuleInfo());
			if (m_messageComposite != null){
				List<JobNotificationsInfo> al = m_messageComposite.getNotificationsInfo();


				//JobNotificationsInfoにある、NotifyRelationInfoに
				//notifyGroupIdをセットします。
				Iterator<NotifyRelationInfo> it;
				NotifyRelationInfo nrInfo;
				JobNotificationsInfo jnInfo;
				String notifyGroupId = NotifyGroupIdGenerator.createNotifyGroupIdJob(info.getJobunitId(), info.getId(),0);
				for(int i = 0 ; i< al.size(); i++){
					jnInfo = al.get(i);
					jnInfo.setNotifyGroupId(notifyGroupId);

					//通知設定は正常の時のみに入っているので、
					//正常の時の通知設定に通知グループIDを設定する。
					if(jnInfo.getNotifyId() != null){
						it = (jnInfo.getNotifyId()).iterator();

						while(it.hasNext()){
							nrInfo = it.next();
							nrInfo.setNotifyGroupId(notifyGroupId);

						}
					}
				}
				List<JobNotificationsInfo> jobNotificationsInfoList = info.getNotifications();
				jobNotificationsInfoList.clear();
				jobNotificationsInfoList.addAll(al);
			}

			if (m_parameterComposite != null){
				List<JobParameterInfo> jobParameterInfoinfoList = info.getParam();
				jobParameterInfoinfoList.clear();
				if (m_parameterComposite.getParamInfo() != null) {
					jobParameterInfoinfoList.addAll(m_parameterComposite.getParamInfo());
				}
			}

			//参照ジョブ
			if(m_referComposite != null){
				if(m_referComposite.getReferJobUnitId() != null){
					info.setReferJobUnitId(m_referComposite.getReferJobUnitId());
				}
				if(m_referComposite.getReferJobId() != null){
					info.setReferJobId(m_referComposite.getReferJobId());
				}
			}

			info.setPropertyFull(true);
		}

		return null;
	}

	/**
	 * ダイアログの情報から、ジョブ情報を作成します。
	 * 
	 * @return 入力値の検証結果
	 */
	private ValidateResult createJobInfo() {
		ValidateResult result = null;

		JobInfo info = m_jobTreeItem.getData();
		String oldJobunitId;

		// ジョブユニットIDの重複チェック(ジョブを編集しているときだけチェックする）
		if (!m_readOnly && info.getType() == JobConstant.TYPE_JOBUNIT) {
			// ジョブユニットIDの重複チェック
			oldJobunitId = info.getJobunitId();
			info.setJobunitId(m_jobIdText.getText());
			try {
				JobUtil.findDuplicateJobunitId(m_jobTreeItem.getParent().getParent());
			} catch (JobInvalid e) {
				result = new ValidateResult();
				result.setValid(false);
				result.setID(Messages.getString("message.hinemos.1"));
				String[] args1 = { m_jobIdText.getText() };
				result.setMessage(Messages.getString("message.job.64", args1));
				return result;
			} finally {
				info.setJobunitId(oldJobunitId);
			}
			// ジョブユニットIDの文字制約のチェック
			if(!m_jobIdText.getText().matches("^[A-Za-z0-9-_]+$")){
				result = new ValidateResult();
				result.setValid(false);
				result.setID(Messages.getString("message.hinemos.1"));
				String[] args1 = { m_jobIdText.getText(), Messages.getString("job.id")};
				result.setMessage(Messages.getString("message.common.6", args1));

				info.setJobunitId(oldJobunitId);
				return result;
			}

			if (JobEditStateUtil.getEditSession(m_jobTreeItem.getData()) == null) {
				// 新規ジョブユニット作成の場合
				Integer editSession = null;
				try {
					editSession =JobUtil.getEditLock(m_jobIdText.getText(), null, false);
				} catch (OtherUserGetLock_Exception e) {
					// 他のユーザがロックを取得している
					String message = e.getMessage();
					if (MessageDialog.openQuestion(
							null,
							Messages.getString("confirmed"),
							message)) {
						try {
							editSession = JobUtil.getEditLock(m_jobIdText.getText(), null, true);
						} catch (Exception e1) {
							// ここには絶対にこないはず
							m_log.error("run() : logical error");
						}
					}
				}
				if (editSession == null) {
					result = new ValidateResult();
					result.setValid(false);
					result.setID(Messages.getString("message.hinemos.1"));
					String[] args1 = { m_jobIdText.getText() };
					result.setMessage(Messages.getString("message.job.105", args1));
					return result;
				}
				JobEditStateUtil.addLockedJobunit(info, null, editSession);
			} else if (!m_jobIdText.getText().equals(oldJobunitId)) {
				// ジョブユニットID変更の場合
				Integer oldEditSession = JobEditStateUtil.getEditSession(info);
				Integer editSession = null;
				try {
					editSession =JobUtil.getEditLock(m_jobIdText.getText(), null, false);
				} catch (OtherUserGetLock_Exception e) {
					// 他のユーザがロックを取得している
					String message = e.getMessage();
					if (MessageDialog.openQuestion(
							null,
							Messages.getString("confirmed"),
							message)) {
						try {
							editSession = JobUtil.getEditLock(m_jobIdText.getText(), null, true);
						} catch (Exception e1) {
							// ここには絶対にこないはず
							m_log.error("run() : logical error");
						}
					}
				}
				if (editSession == null) {
					result = new ValidateResult();
					result.setValid(false);
					result.setID(Messages.getString("message.hinemos.1"));
					String[] args1 = { m_jobIdText.getText() };
					result.setMessage(Messages.getString("message.job.105", args1));
					return result;
				}
				JobEditStateUtil.addLockedJobunit(info, null, editSession);
				try {
					JobEndpointWrapper.releaseEditLock(oldEditSession);
				} catch (Exception e) {
					result = new ValidateResult();
					result.setValid(false);
					result.setID(Messages.getString("message.hinemos.1"));
					String[] args1 = { m_jobIdText.getText() };
					result.setMessage(Messages.getString("message.job.105", args1));
					return result;
				}
			}
		}

		//ジョブID取得
		if (m_jobIdText.getText().length() > 0) {
			String oldId = info.getId();
			info.setId("");
			//ジョブIDの重複チェック（所属するジョブユニット配下のみ）
			JobTreeItem unit = JobUtil.getTopJobUnitTreeItem(m_jobTreeItem);
			if(unit != null && JobUtil.findJobId(m_jobIdText.getText(), unit)){
				result = new ValidateResult();
				result.setValid(false);
				result.setID(Messages.getString("message.hinemos.1"));
				String[] args1 = { m_jobIdText.getText() };
				result.setMessage(Messages.getString("message.job.42", args1));

				info.setId(oldId);
				return result;
			}
			// ジョブIDの文字制約のチェック
			if(!m_jobIdText.getText().matches("^[A-Za-z0-9-_]+$")){
				result = new ValidateResult();
				result.setValid(false);
				result.setID(Messages.getString("message.hinemos.1"));
				String[] args1 = { m_jobIdText.getText(), Messages.getString("job.id")};
				result.setMessage(Messages.getString("message.common.6", args1));

				info.setId(oldId);
				return result;
			}
			info.setId(m_jobIdText.getText());

			// ジョブユニットの場合はジョブユニットIDをセットする。
			if (info.getType() == JobConstant.TYPE_JOBUNIT) {
				info.setJobunitId(m_jobIdText.getText());
			}


		} else {
			result = new ValidateResult();
			result.setValid(false);
			result.setID(Messages.getString("message.hinemos.1"));
			result.setMessage(Messages.getString("message.job.22"));
			return result;
		}

		//ジョブ名取得
		if (m_jobNameText.getText().length() > 0) {
			info.setName(m_jobNameText.getText());
		} else {
			result = new ValidateResult();
			result.setValid(false);
			result.setID(Messages.getString("message.hinemos.1"));
			result.setMessage(Messages.getString("message.job.23"));
			return result;
		}

		//注釈取得
		if (m_jobAnnotationText.getText().length() > 0) {
			info.setDescription(m_jobAnnotationText.getText());
		} else {
			info.setDescription("");
		}

		//オーナーロールID取得
		String newOwnerRoleId = m_ownerRoleId.getText();
		if (newOwnerRoleId.length() > 0) {
			if (!newOwnerRoleId.equals(info.getOwnerRoleId())) {
				changeOwnerRoleId(m_jobTreeItem, newOwnerRoleId);
			}
		} else {
			result = new ValidateResult();
			result.setValid(false);
			result.setID(Messages.getString("message.hinemos.1"));
			result.setMessage(Messages.getString("owner.role.id"));
			return result;
		}

		return null;
	}

	/**
	 * オーナーロールIDを変更する。<BR>
	 * 
	 * @param item 反映するJobTreeItem
	 * @param ownerRoleId オーナーロールID
	 */
	private static void changeOwnerRoleId(JobTreeItem item, String ownerRoleId) {

		// 直下のJobTreeItemのOwnerRoleIdを変更する
		List<JobTreeItem> children = item.getChildren();
		if (children != null && children.size() > 0) {
			Iterator<JobTreeItem> iter = children.iterator();
			while(iter.hasNext()) {
				JobTreeItem child = iter.next();
				changeOwnerRoleId(child, ownerRoleId);
			}
		}
		JobInfo info = item.getData();
		info.setOwnerRoleId(ownerRoleId);

		// ジョブ種別ごとにオーナーロールIDに依存する設定を削除
		if (info.getType() == JobConstant.TYPE_JOBNET) {
			// ジョブネット
			//制御（カレンダID）
			JobWaitRuleInfo jobWaitRuleInfo = info.getWaitRule();
			if (jobWaitRuleInfo != null) {
				jobWaitRuleInfo.setCalendar(YesNoConstant.TYPE_NO);
				jobWaitRuleInfo.setCalendarId(null);
			}

		} else if (info.getType() == JobConstant.TYPE_JOB) {
			// ジョブ
			//制御（カレンダID）
			JobWaitRuleInfo jobWaitRuleInfo = info.getWaitRule();
			if (jobWaitRuleInfo != null) {
				jobWaitRuleInfo.setCalendar(YesNoConstant.TYPE_NO);
				jobWaitRuleInfo.setCalendarId(null);
			}

			//実行内容（スコープ）
			JobCommandInfo jobCommandInfo = info.getCommand();
			if (jobCommandInfo != null) {
				//ジョブ変数を選択させる
				jobCommandInfo.setFacilityID(
						SystemParameterConstant.getParamText(SystemParameterConstant.FACILITY_ID));
				jobCommandInfo.setScope(null);
			}

		} else if (info.getType() == JobConstant.TYPE_FILEJOB) {
			// ファイル転送ジョブ
			//制御（カレンダID）
			JobWaitRuleInfo jobWaitRuleInfo = info.getWaitRule();
			if (jobWaitRuleInfo != null) {
				jobWaitRuleInfo.setCalendar(YesNoConstant.TYPE_NO);
				jobWaitRuleInfo.setCalendarId(null);
			}

			//ファイル転送（転送元スコープ、転送先スコープ）
			JobFileInfo jobFileInfo = info.getFile();
			if (jobFileInfo != null) {
				jobFileInfo.setSrcScope(null);
				jobFileInfo.setSrcFacilityID(null);
				jobFileInfo.setDestScope(null);
				jobFileInfo.setDestFacilityID(null);
			}

		} else if(info.getType() == JobConstant.TYPE_REFERJOB){
			// 参照ジョブ
			//参照ジョブ(参照ジョブユニットID、ジョブID)
			info.setReferJobUnitId(null);
			info.setReferJobId(null);
		}
		//参照タブ以外で使用する
		if (info.getType() != JobConstant.TYPE_REFERJOB) {
			List<JobNotificationsInfo> notifications = info.getNotifications();
			if (notifications != null && notifications.size() > 0) {
				Iterator<JobNotificationsInfo> iter = notifications.iterator();
				while(iter.hasNext()) {
					JobNotificationsInfo child = iter.next();
					if (child.getType() == EndStatusConstant.TYPE_NORMAL) {
						child.getNotifyId().clear();
					}
				}
			}
		}

		m_log.debug("changeOwnerRoleId() "
				+ " jobunitId = " + info.getJobunitId()
				+ " jobId = " + info.getId()
				+ " ownerRoleId = " + info.getOwnerRoleId());
	}

	/**
	 * ジョブツリーアイテムを返します。
	 * 
	 * @return ジョブツリーアイテム
	 */
	public JobTreeItem getJobTreeItem() {
		return m_jobTreeItem;
	}

	/**
	 * ジョブツリーアイテムを設定します。
	 * 
	 * @param jobTreeItem ジョブツリーアイテム
	 */
	public void setJobTreeItem(JobTreeItem jobTreeItem) {
		this.m_jobTreeItem = jobTreeItem;
	}
}