/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.collectiverun.composite;

import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;

import com.clustercontrol.bean.DataRangeConstant;
import com.clustercontrol.collectiverun.action.ModifyMaster;
import com.clustercontrol.collectiverun.bean.ParameterTypeConstant;
import com.clustercontrol.collectiverun.composite.action.MasterInfoModifyListener;
import com.clustercontrol.collectiverun.util.TreeItemUtil;
import com.clustercontrol.composite.action.StringVerifyListener;
import com.clustercontrol.dialog.ValidateResult;
import com.clustercontrol.util.Messages;
import com.clustercontrol.ws.collectiverun.ItemInfo;
import com.clustercontrol.ws.collectiverun.ParameterInfo;
import com.clustercontrol.ws.collectiverun.TreeItem;

/**
 * 一括制御階層ツリーコンポジット
 * 
 * @version 2.1.0
 * @since 2.1.0
 */
public class ParameterComposite extends Composite {
	private Text id = null;

	private Text name = null;

	private Button typeText1 = null;

	private Button typePassword = null;

	private Button typeSelect = null;

	private Button typeText2 = null;

	private TreeItem selectItem = null;

	private TreeComposite composite = null;

	public ParameterComposite(Composite parent, int style, TreeComposite composite) {
		super(parent, style);
		this.composite = composite;
		this.selectItem = composite.getSelectItem();
		initialize();
	}

	private void initialize() {
		GridLayout layout = new GridLayout(10, true);
		this.setLayout(layout);
		layout.marginHeight = 5;
		layout.marginWidth = 5;

		GridData gridData = null;
		Label label = null;

		Group group = new Group(this, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.verticalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.grabExcessVerticalSpace = true;
		gridData.horizontalSpan = 10;
		group.setLayoutData(gridData);
		group.setText(Messages.getString("parameter"));

		layout = new GridLayout(10, true);
		group.setLayout(layout);
		layout.marginHeight = 10;
		layout.marginWidth = 10;
		layout.verticalSpacing = 10;

		// パラメータID
		label = new Label(group, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.horizontalSpan = 3;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("parameter.id") + " : ");

		this.id = new Text(group, SWT.BORDER);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.horizontalSpan = 7;
		this.id.setLayoutData(gridData);
		this.id.addModifyListener(new MasterInfoModifyListener(this.composite));
		this.id.addVerifyListener(
				new StringVerifyListener(DataRangeConstant.VARCHAR_64));

		// 名前
		label = new Label(group, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.horizontalSpan = 3;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("name") + " : ");

		this.name = new Text(group, SWT.BORDER);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.horizontalSpan = 7;
		this.name.setLayoutData(gridData);
		this.name.addModifyListener(new MasterInfoModifyListener(this.composite));
		this.name.addVerifyListener(
				new StringVerifyListener(DataRangeConstant.VARCHAR_256));

		//入力形式
		Group typeGroup = new Group(group, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.verticalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.horizontalSpan = 10;
		typeGroup.setLayoutData(gridData);
		typeGroup.setText(Messages.getString("input.type"));

		layout = new GridLayout(10, true);
		typeGroup.setLayout(layout);
		layout.marginHeight = 10;
		layout.marginWidth = 10;
		layout.verticalSpacing = 0;

		//編集
		this.typeText1 = new Button(typeGroup, SWT.RADIO);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.horizontalSpan = 10;
		this.typeText1.setLayoutData(gridData);
		this.typeText1.setText(Messages.getString("edit"));
		this.typeText1.addSelectionListener(new SelectionListener() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				Button check = (Button) e.getSource();
				if (check.getSelection()) {
					typeText2.setEnabled(true);
					typePassword.setEnabled(true);
				}
			}

			@Override
			public void widgetDefaultSelected(SelectionEvent e) {

			}
		});

		//編集形式
		Group textGroup = new Group(typeGroup, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.verticalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.horizontalSpan = 10;
		textGroup.setLayoutData(gridData);

		layout = new GridLayout(10, true);
		textGroup.setLayout(layout);
		layout.marginHeight = 5;
		layout.marginWidth = 10;
		layout.verticalSpacing = 0;

		//テキスト表示
		this.typeText2 = new Button(textGroup, SWT.RADIO);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.horizontalSpan = 5;
		this.typeText2.setLayoutData(gridData);
		this.typeText2.setText(Messages.getString("show.text"));

		//アステリスク表示
		this.typePassword = new Button(textGroup, SWT.RADIO);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.horizontalSpan = 5;
		this.typePassword.setLayoutData(gridData);
		this.typePassword.setText(Messages.getString("show.asterisk"));

		label = new Label(typeGroup, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.horizontalSpan = 10;
		label.setLayoutData(gridData);

		label = new Label(typeGroup, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.horizontalSpan = 10;
		label.setLayoutData(gridData);

		//選択
		this.typeSelect = new Button(typeGroup, SWT.RADIO);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.horizontalSpan = 10;
		this.typeSelect.setLayoutData(gridData);
		this.typeSelect.setText(Messages.getString("select"));
		this.typeSelect.addSelectionListener(new SelectionListener() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				Button check = (Button) e.getSource();
				if (check.getSelection()) {
					typeText2.setEnabled(false);
					typePassword.setEnabled(false);
				}
			}

			@Override
			public void widgetDefaultSelected(SelectionEvent e) {

			}
		});

		// 適用ボタン
		label = new Label(this, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.horizontalSpan = 8;
		label.setLayoutData(gridData);

		Button buttonCreate = new Button(this, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.horizontalSpan = 2;
		buttonCreate.setLayoutData(gridData);
		buttonCreate.setText(Messages.getString("apply"));
		buttonCreate.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				ValidateResult result = validate();

				if (result != null) {
					displayError(result);
				}
			}
		});

		setEnabled();

		update();
	}

	/**
	 * コントロールの使用可/不可の設定処理
	 * 
	 */
	public void setEnabled() {
		if(this.composite.getCreateItem() == null){
			this.id.setEnabled(false);
		}
		else{
			this.id.setEnabled(true);
		}
	}

	/**
	 * 更新処理
	 * 
	 */
	@Override
	public void update() {
		if(selectItem instanceof TreeItem){
			ParameterInfo info = selectItem.getParameterInfo();

			//パラメータIDを設定
			if(info.getId() != null){
				this.id.setText(info.getId());
			}

			//名前を設定
			if(info.getName() != null){
				this.name.setText(info.getName());
			}

			//パラメータ種別を設定
			if(info.getParamType() == ParameterTypeConstant.TYPE_STRING){
				this.typeText1.setSelection(true);
				this.typeText2.setSelection(true);
			}
			else if(info.getParamType() == ParameterTypeConstant.TYPE_PASSWORD){
				this.typeText1.setSelection(true);
				this.typePassword.setSelection(true);
			}
			else if(info.getParamType() == ParameterTypeConstant.TYPE_SELECT){
				this.typeSelect.setSelection(true);
			}

			if(info.getParamType() == ParameterTypeConstant.TYPE_SELECT){
				typeText2.setEnabled(false);
				typePassword.setEnabled(false);
			}
			else{
				typeText2.setEnabled(true);
				typePassword.setEnabled(true);
			}
		}
		this.composite.setUpdateItem(null);
	}

	/**
	 * パラメータ情報作成
	 * 
	 * @return
	 */
	public ValidateResult createParameterInfo() {
		ValidateResult result = null;

		if(selectItem instanceof TreeItem){
			ParameterInfo info = selectItem.getParameterInfo();
			ParameterInfo cloneInfo = TreeItemUtil.clone(info);

			ItemInfo parentInfo = selectItem.getParent().getParameterInfo();

			//パラメータID取得
			if (this.id.getText().length() > 0) {
				cloneInfo.setId(this.id.getText());
			} else {
				result = new ValidateResult();
				result.setValid(false);
				result.setID(Messages.getString("message.hinemos.1"));
				result.setMessage(Messages.getString("message.collective.run.6"));
				return result;
			}

			//パラメータID重複チェック
			if (this.id.isEnabled() &&
					TreeItemUtil.findId(cloneInfo.getId(), selectItem.getParent())) {
				result = new ValidateResult();
				result.setValid(false);
				result.setID(Messages.getString("message.hinemos.1"));
				String args[] = { Messages.getString("parameter.id"), cloneInfo.getId() };
				result.setMessage(Messages.getString("message.collective.run.16", args));
				return result;
			}

			//名前取得
			if (this.name.getText().length() > 0) {
				cloneInfo.setName(this.name.getText());
			} else {
				result = new ValidateResult();
				result.setValid(false);
				result.setID(Messages.getString("message.hinemos.1"));
				result.setMessage(Messages.getString("message.collective.run.3"));
				return result;
			}

			//パラメータ種別取得
			if (this.typeText1.getSelection()) {
				if (this.typeText2.getSelection()) {
					cloneInfo.setParamType(ParameterTypeConstant.TYPE_STRING);
				}else{
					cloneInfo.setParamType(ParameterTypeConstant.TYPE_PASSWORD);
				}
			}else{
				cloneInfo.setParamType(ParameterTypeConstant.TYPE_SELECT);
			}

			ModifyMaster modify = new ModifyMaster();
			if(modify.update(cloneInfo, parentInfo)){
				selectItem.setParameterInfo(cloneInfo);
				this.composite.setCreateItem(null);
				this.composite.setUpdateItem(null);
				this.composite.getTreeViewer().refresh(selectItem);
			}
		}

		return null;
	}

	/**
	 * 入力値チェックを行います。
	 * 
	 * 必要に応じて、入力値チェックを実装して下さい。
	 * 
	 * @return ValidateResultオブジェクト
	 */
	protected ValidateResult validate() {
		ValidateResult result = null;

		result = createParameterInfo();
		if (result != null) {
			return result;
		}

		return null;
	}

	/**
	 * エラー内容を通知します。
	 * <p>
	 * 
	 * 警告メッセージボックスにて、クライアントに通知します。
	 * 
	 * @param result
	 *            ValidateResultオブジェクト
	 */
	private void displayError(ValidateResult result) {
		MessageDialog.openWarning(
				null,
				result.getID(),
				result.getMessage());
	}

	/**
	 * @return
	 */
	public TreeItem getSelectItem() {
		return selectItem;
	}

	/**
	 * @param selectItem
	 */
	public void setSelectItem(TreeItem selectItem) {
		this.selectItem = selectItem;
	}
}