/*
Copyright (C) 2014 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */
package com.clustercontrol.cloud.cloudn.base.presenter;

import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.ui.IViewPart;

import com.clustercontrol.cloud.cloudn.base.Activator;
import com.clustercontrol.cloud.cloudn.base.commons.util.CloudnConstants;
import com.clustercontrol.cloud.cloudn.base.commons.util.MessageManagerExt;
import com.clustercontrol.cloud.cloudn.base.ui.dialogs.ShowPasswordDialog;
import com.clustercontrol.cloud.commons.util.AccountControlUtil;
import com.clustercontrol.cloud.presenter.ICloudInstance;
import com.clustercontrol.cloud.presenter.ICloudStorage;
import com.clustercontrol.util.Messages;
import com.clustercontrol.ws.cloudn.AttachStorageResult;
import com.clustercontrol.ws.cloudn.CloudManagerFault_Exception;
import com.clustercontrol.ws.cloudn.CloudnOptionEndpoint;
import com.clustercontrol.ws.cloudn.DeleteInstanceResult;
import com.clustercontrol.ws.cloudn.InvalidRole_Exception;
import com.clustercontrol.ws.cloudn.InvalidUserPass_Exception;
import com.clustercontrol.ws.cloudn.JobResult;
import com.clustercontrol.ws.cloudn.JobStatus;
import com.clustercontrol.ws.cloudn.PasswordResult;

public class JobTrackingService {
	private static ExecutorService service = Executors.newFixedThreadPool(5);
	private static MessageManagerExt messages = MessageManagerExt.getInstance("messages");
	
	public interface GetResultAction<T extends JobResult>{
		T getResult();
	}
	
	public interface AfterAction<T extends JobResult>{
		boolean invoke(T result);
	}
	
	public static void trackPassword(final IViewPart view, final ICloudInstance instance){
		trackJob(
			new GetResultAction<PasswordResult>() {
				@Override
				public PasswordResult getResult() {
					try {
						return Activator.getEndpointManager().getEndpoint(CloudnOptionEndpoint.class).getPassword(AccountControlUtil.getRoleId(instance.getCloudInstanceManager().getRegion().getAccountResource().getAccountResourceId()), instance.getRegion(), instance.getInstanceId());
					} catch (CloudManagerFault_Exception | InvalidRole_Exception | InvalidUserPass_Exception e) {
						PasswordResult result = new PasswordResult();
						result.setStatus(JobStatus.ERROR);
						if(e instanceof CloudManagerFault_Exception && ((CloudManagerFault_Exception) e).getFaultInfo().getErrorCode().equals("JOB_ID_UNREGISTERED")){
							result.setErrorText(messages.getString("message.error.password_is_fixed_and_unreferable"));
						} else {
							result.setErrorText(e.getMessage());
						}
						return result;
					}
				}
			}
			, 
			new AfterAction<PasswordResult>(){
				@Override
				public boolean invoke(final PasswordResult result) {
					String tmpText = null;
					switch (result.getStatus()){
					case RUNNING:
						return false;
					case SUCCESS:
						tmpText = result.getPassword();
						break;
					case FAILURE:
					case ERROR:
					default:
						tmpText = messages.getString("message.error.track_password_failed");
						if(result.getErrorCode() != null){
							tmpText = tmpText + "\nerror code : " + result.getErrorCode();
						}
						if(result.getErrorText() != null && !result.getErrorText().isEmpty()){
							tmpText = tmpText + "\nerror message : " + result.getErrorText();
						}
					}
					
					final String displayText = tmpText;
					view.getSite().getShell().getDisplay().asyncExec(new Runnable() {
						@Override
						public void run() {
							ShowPasswordDialog dialog = new ShowPasswordDialog(view.getSite().getShell());
							dialog.setText(displayText);
							dialog.setInstance(instance);
							dialog.open();
						}
					});
					
					return true;
				}
			}
		);
		
	}
	
	public static void trackDeleteInstance(IViewPart view, final ICloudInstance instance){
		String errorMessage = messages.getString("message.error.delete_instance_failed") + "\nInstanceId : " + instance.getInstanceId();
		trackJobResult(view,
				new GetResultAction<DeleteInstanceResult>() {
					@Override
					public DeleteInstanceResult getResult() {
						try {
							return Activator.getEndpointManager().getEndpoint(CloudnOptionEndpoint.class).getDeleteInstanceResult(AccountControlUtil.getRoleId(instance.getCloudInstanceManager().getRegion().getAccountResource().getAccountResourceId()), instance.getRegion(), instance.getInstanceId());
						} catch (CloudManagerFault_Exception | InvalidRole_Exception | InvalidUserPass_Exception e) {
							DeleteInstanceResult result = new DeleteInstanceResult();
							result.setStatus(JobStatus.ERROR);
							result.setErrorText(e.getMessage());
							return result;
						}
					}
				}
				, 
				errorMessage);
	}


	public static void trackAttachStorage(IViewPart view, ICloudStorage storage, String instanceId){
		trackAttachStorage(view, AccountControlUtil.getRoleId(storage.getCloudStorageManager().getRegion().getAccountResource().getAccountResourceId()), storage.getCloudStorageManager().getRegion().getRegion(), storage.getStorageId(), instanceId);
	}
	
	public static void trackAttachStorage(IViewPart view, ICloudInstance instance, String storageId){
		trackAttachStorage(view, AccountControlUtil.getRoleId(instance.getCloudInstanceManager().getRegion().getAccountResource().getAccountResourceId()), instance.getCloudInstanceManager().getRegion().getRegion(), storageId, instance.getInstanceId());
	}
	
	public static void trackAttachStorage(IViewPart view, final String roleId, final String region, final String storageId, final String instanceId){
		String errorMessage = messages.getString("message.error.attach_storage_failed") + "\nStorageId : " + storageId;
		trackJobResult(view,
				new GetResultAction<AttachStorageResult>() {
					@Override
					public AttachStorageResult getResult() {
						try {
							return Activator.getEndpointManager().getEndpoint(CloudnOptionEndpoint.class).getAttachStorageResult(roleId, region, storageId, instanceId);
						} catch (CloudManagerFault_Exception | InvalidRole_Exception | InvalidUserPass_Exception e) {
							AttachStorageResult result = new AttachStorageResult();
							result.setStatus(JobStatus.ERROR);
							result.setErrorText(e.getMessage());
							return result;
						}
					}
				}
				, 
				errorMessage);
	}

	public static <T extends JobResult> void trackJobResult(final IViewPart view, final GetResultAction<T> action, final String errorMessage){
		trackJob(
			action
			,
			new AfterAction<T>(){
				@Override
				public boolean invoke(final T result) {
					String tmpText = null;
					switch (result.getStatus()){
					case RUNNING:
						return false;
					case SUCCESS:
						return true;
					case FAILURE:
					case ERROR:
					default:
						tmpText = errorMessage;
						if(result.getErrorCode() != null){
							tmpText = tmpText + "\nerror code : " + result.getErrorCode();
						}
						if(result.getErrorText() != null && !result.getErrorText().isEmpty()){
							tmpText = tmpText + "\nerror message : " + result.getErrorText();
						}
					}
					
					final String displayText = tmpText;
					view.getSite().getShell().getDisplay().asyncExec(new Runnable() {
						@Override
						public void run() {
							MessageDialog.openError(view.getSite().getShell(), Messages.getString("failed"), displayText);
						}
					});
					
					return true;
				}
			}
		);
		
	}
	
	public static <T extends JobResult> void trackJob(final GetResultAction<T> getAction, final AfterAction<T> afterAction){
		service.submit(new Runnable(){
			@Override
			public void run() {
				int count = 0;
				boolean finish = false;
				for(;;){
					try{
						T result = getAction.getResult();
						finish = afterAction.invoke(result);
						if(!finish && count >= CloudnConstants.JOB_TRACKING_COUNT){
							result.setStatus(JobStatus.ERROR);
							result.setErrorCode("JOB_TRACKING_IS_TIMED_OUT");
							result.setErrorText(messages.getString("message.error.job_tracking_is_timed_out"));
							afterAction.invoke(result);
						}
					} catch (Exception e) {
					}
					if(finish || count >= CloudnConstants.JOB_TRACKING_COUNT){
						break;
					}
					try {
						Thread.sleep(CloudnConstants.JOB_TRACKING_INTERVAL);
					} catch (InterruptedException e) {
					}
					++count;
				}
			}
		});
	}
}
