/*
Copyright (C) 2013 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */
package com.clustercontrol.cloud.util;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.TreeSet;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.atomic.AtomicInteger;

import javax.xml.ws.WebServiceContext;

import org.apache.log4j.Logger;

import com.clustercontrol.accesscontrol.bean.ObjectPrivilegeFilterInfo;
import com.clustercontrol.accesscontrol.bean.ObjectPrivilegeInfo;
import com.clustercontrol.accesscontrol.bean.SystemPrivilegeInfo;
import com.clustercontrol.accesscontrol.bean.PrivilegeConstant.ObjectPrivilegeMode;
import com.clustercontrol.accesscontrol.session.AccessControllerBean;
import com.clustercontrol.bean.EndStatusConstant;
import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.bean.StatusConstant;
import com.clustercontrol.bean.YesNoConstant;
import com.clustercontrol.cloud.SessionService;
import com.clustercontrol.commons.util.HinemosSessionContext;
import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.fault.InvalidRole;
import com.clustercontrol.fault.InvalidSetting;
import com.clustercontrol.fault.InvalidUserPass;
import com.clustercontrol.fault.JobMasterNotFound;
import com.clustercontrol.fault.PrivilegeDuplicate;
import com.clustercontrol.fault.UsedObjectPrivilege;
import com.clustercontrol.jobmanagement.bean.CommandStopTypeConstant;
import com.clustercontrol.jobmanagement.bean.ConditionTypeConstant;
import com.clustercontrol.jobmanagement.bean.JobCommandInfo;
import com.clustercontrol.jobmanagement.bean.JobConstant;
import com.clustercontrol.jobmanagement.bean.JobEndStatusInfo;
import com.clustercontrol.jobmanagement.bean.JobInfo;
import com.clustercontrol.jobmanagement.bean.JobNotificationsInfo;
import com.clustercontrol.jobmanagement.bean.JobTreeItem;
import com.clustercontrol.jobmanagement.bean.JobWaitRuleInfo;
import com.clustercontrol.jobmanagement.bean.OperationConstant;
import com.clustercontrol.jobmanagement.bean.ProcessingMethodConstant;
import com.clustercontrol.notify.bean.OutputBasicInfo;
import com.clustercontrol.notify.factory.NotifyEventTaskFactory;
import com.clustercontrol.notify.util.NotifyGroupIdGenerator;
import com.clustercontrol.plugin.impl.AsyncWorkerPlugin;
import com.clustercontrol.repository.bean.FacilityConstant;
import com.clustercontrol.repository.bean.NodeConstant;
import com.clustercontrol.repository.bean.NodeDiskInfo;
import com.clustercontrol.repository.bean.NodeInfo;
import com.clustercontrol.repository.bean.NodeVariableInfo;
import com.clustercontrol.repository.bean.ScopeInfo;
import com.clustercontrol.repository.session.RepositoryControllerBean;
import com.clustercontrol.ws.util.HttpAuthenticator;

public class HinemosUtil {
	// TODO スレッドにひとつしか  newSingleThreadExecutor を確保しないので汎用性が非常に低い。
	// いづれ改善しないと。
	private static ThreadLocal<ExecutorService> executorService  = new ThreadLocal<ExecutorService>() {
		protected ExecutorService initialValue()
		{
			return null;
		}
	};

	protected HinemosUtil() {
	}

	private static final AtomicInteger poolNumber = new AtomicInteger(1);

	private static ConcurrentHashMap <String, String> userDefinedNodePropertyList = new ConcurrentHashMap<String, String>();
	public static void putUserDefinedNodePropertyList(String key, String value){
		if(key != null && value != null){
			userDefinedNodePropertyList.put(key, value);
		}
	}
	
	public static ExecutorService executorService() {
		ExecutorService service = executorService.get();
		if (service == null) {
			service = Executors.newSingleThreadExecutor(new ThreadFactory() {
				private final AtomicInteger threadNumber = new AtomicInteger(1);
				@Override
				public Thread newThread(final Runnable r) {
					final Object accountName = HinemosSessionContext.instance().getProperty(HinemosSessionContext.LOGIN_USER_ID);
					final Object isAdministrator = HinemosSessionContext.instance().getProperty(HinemosSessionContext.IS_ADMINISTRATOR);
					return new Thread(new Runnable() {
						@Override
						public void run() {
							HinemosSessionContext.instance().setProperty(HinemosSessionContext.LOGIN_USER_ID, accountName);
							HinemosSessionContext.instance().setProperty(HinemosSessionContext.IS_ADMINISTRATOR, isAdministrator);
							r.run();
						}
					}, "HinemosUtil-" + poolNumber.getAndIncrement() + "-thread-" + threadNumber.getAndIncrement());
				}
			});
			executorService.set(service);
		}
		return service;
	}

	public static void execute(Runnable command) {
		executorService().execute(command);
	}

	public static <T> Future<T> submit(Callable<T> task) {
		return executorService().submit(task);
	}

	public static <T> Future<T> submit(Runnable task, T result) {
		return executorService().submit(task, result);
	}

	public static Future<?> submit(Runnable task) {
		return executorService().submit(task);
	}

	public static void shutdown() {
		ExecutorService service = executorService.get();
		if (service != null) {
			service.shutdown();
			executorService.set(null);
		}
	}

	public static void authCheck(WebServiceContext wsctx, SystemPrivilegeInfo... demandingRight) throws InvalidUserPass, InvalidRole, HinemosUnknown {
		HttpAuthenticator.authCheck(wsctx, new ArrayList<SystemPrivilegeInfo>(Arrays.asList(demandingRight)));
	}

	public static ScopeInfo createScope(String facilityId, String facilityName, String roleId) {
		ScopeInfo scope = new ScopeInfo();

		scope.setFacilityName(facilityName);
		scope.setFacilityId(facilityId);
		//		scope.setDescription("AWS用");

		scope.setFacilityType(FacilityConstant.TYPE_SCOPE);
		scope.setDisplaySortOrder(100);
		scope.setValid(true);
		scope.setCreateDatetime(System.currentTimeMillis());
		scope.setModifyDatetime(System.currentTimeMillis());
		scope.setBuiltInFlg(false);
		scope.setOwnerRoleId(roleId);

		return scope;
	}

	public static com.clustercontrol.repository.bean.NodeInfo createNodeInfo(
			String facilityId,
			String facilityName,
			String platform,
			String subPlatform,
			String nodeName,
			String description,
			String nodeType,
			String serviceId,
			String accountResourceId,
			String resourceType,
			String instanceId,
			String region,
			String availabilityZone,
			String roleId,
			NodeVariableInfo...variables
			) {
		NodeInfo nodeInfo = new NodeInfo();
		setDefaultValue(nodeInfo);

		nodeInfo.setFacilityId(facilityId);
		nodeInfo.setFacilityName(facilityName);
		nodeInfo.setPlatformFamily(platform);
		nodeInfo.setSubPlatformFamily(subPlatform);
		nodeInfo.setIpAddressVersion(4);
		nodeInfo.setIpAddressV4("123.123.123.123");
		nodeInfo.setNodeName(nodeName);
		nodeInfo.setDescription(description);

		nodeInfo.setCloudNodeType(nodeType);
		nodeInfo.setCloudService(serviceId);
		nodeInfo.setCloudAccountResouce(accountResourceId);
		nodeInfo.setCloudResourceType(resourceType);
		nodeInfo.setCloudResourceId(instanceId);
		nodeInfo.setCloudRegion(region);
		nodeInfo.setCloudZone(availabilityZone);

		nodeInfo.setOwnerRoleId(roleId);

		if(nodeInfo.getNodeVariableInfo() != null){
			nodeInfo.getNodeVariableInfo().addAll(Arrays.asList(variables));
		}else{
			nodeInfo.setNodeVariableInfo(new ArrayList<NodeVariableInfo>(Arrays.asList(variables)));
		}

		return nodeInfo;
	}

	public static com.clustercontrol.repository.bean.NodeInfo resetNodeInfo(
			NodeInfo nodeInfo, 
			String facilityName,
			String platform,
			String subPlatform,
			String nodeName,
			String description,
			String nodeType,
			String serviceId,
			String accountResourceId,
			String resourceType,
			String instanceId,
			String region,
			String availabilityZone
			) {
		nodeInfo.setFacilityName(facilityName);
		nodeInfo.setPlatformFamily(platform);
		nodeInfo.setSubPlatformFamily(subPlatform);
		nodeInfo.setIpAddressVersion(4);
		nodeInfo.setIpAddressV4("123.123.123.123");
		nodeInfo.setNodeName(nodeName);
		nodeInfo.setDescription(description);

		nodeInfo.setCloudNodeType(nodeType);
		nodeInfo.setCloudService(serviceId);
		nodeInfo.setCloudAccountResouce(accountResourceId);
		nodeInfo.setCloudResourceType(resourceType);
		nodeInfo.setCloudResourceId(instanceId);
		nodeInfo.setCloudRegion(region);
		nodeInfo.setCloudZone(availabilityZone);

		ArrayList<NodeDiskInfo> disklist = new ArrayList<NodeDiskInfo>();
		nodeInfo.setNodeDiskInfo(disklist);

		return nodeInfo;
	}

	public static void clearNodeInfo(NodeInfo nodeInfo) {
//		nodeInfo.setNodeVariableInfo(new ArrayList<NodeVariableInfo>(nodeInfo.getNodeVariableInfo()));
//		String[] names = {"cloudId", "instanceId", "instanceType", "region", "availabilityZone"};
//		for (String name: names) {
//			for (int i = 0; i < nodeInfo.getNodeVariableInfo().size(); ++i) {
//				NodeVariableInfo info = nodeInfo.getNodeVariableInfo().get(i);
//				if (info.getNodeVariableName().equals(name)) {
//					nodeInfo.getNodeVariableInfo().remove(i);
//					break;
//				}
//			}
//		}

		nodeInfo.setNodeDiskInfo(new ArrayList<NodeDiskInfo>(nodeInfo.getNodeDiskInfo()));
		Iterator<NodeDiskInfo> iter = nodeInfo.getNodeDiskInfo().iterator();
		while (iter.hasNext()) {
			NodeDiskInfo info = iter.next();
			if (info.getDeviceDescription().startsWith("storageId=")) {
				iter.remove();
			}
		}
		
		nodeInfo.setCloudNodeType("");
		nodeInfo.setCloudService("");
		nodeInfo.setCloudAccountResouce("");
		nodeInfo.setCloudResourceType("");
		nodeInfo.setCloudResourceId("");
		nodeInfo.setCloudRegion("");
		nodeInfo.setCloudZone("");
	}

	private static boolean checkEmpty(String s) {
		return s == null || s.isEmpty();
	}

	public static void setDefaultValue(NodeInfo nodeInfo) {
		nodeInfo.setFacilityType(FacilityConstant.TYPE_NODE);
		nodeInfo.setDisplaySortOrder(100);
		if (nodeInfo.getFacilityId() == null) {
			nodeInfo.setFacilityId("");
		}
		if (nodeInfo.getFacilityName() == null) {
			nodeInfo.setFacilityName("");
		}
		if (nodeInfo.getDescription() == null) {
			nodeInfo.setDescription("");
		}
		if (nodeInfo.isValid() == null) {
			nodeInfo.setValid(Boolean.TRUE);
		}
		if (nodeInfo.getCreateUserId() == null) {
			nodeInfo.setCreateUserId("");
		}
		if (nodeInfo.getCreateDatetime() == null) {
			nodeInfo.setCreateDatetime(null);
		}
		if (nodeInfo.getModifyUserId() == null) {
			nodeInfo.setModifyUserId("");
		}
		if (nodeInfo.getModifyDatetime() == null) {
			nodeInfo.setModifyDatetime(null);
		}

		// HW
		if (nodeInfo.getPlatformFamily() == null) {
			nodeInfo.setPlatformFamily("");
		}
		if (nodeInfo.getHardwareType() == null) {
			nodeInfo.setHardwareType("");
		}
		if (nodeInfo.getIconImage() == null) {
			nodeInfo.setIconImage("");
		}

		// IPアドレス
		if (nodeInfo.getIpAddressVersion() == null) {
			nodeInfo.setIpAddressVersion(-1);
		}
		if (nodeInfo.getIpAddressV4() == null) {
			nodeInfo.setIpAddressV4("");
		}
		if (nodeInfo.getIpAddressV6() == null) {
			nodeInfo.setIpAddressV6("");
		}

		// OS
		if (nodeInfo.getNodeName() == null) {
			nodeInfo.setNodeName("");
		}
		if (nodeInfo.getOsName() == null) {
			nodeInfo.setOsName("");
		}
		if (nodeInfo.getOsRelease() == null) {
			nodeInfo.setOsRelease("");
		}
		if (nodeInfo.getOsVersion() == null) {
			nodeInfo.setOsVersion("");
		}
		if (nodeInfo.getCharacterSet() == null) {
			nodeInfo.setCharacterSet("");
		}

		// Hinemosエージェント
		if (nodeInfo.getAgentAwakePort() == null || nodeInfo.getAgentAwakePort() == -1) {
			nodeInfo.setAgentAwakePort(24005);

			String value = userDefinedNodePropertyList.get(NodeConstant.AGENT_AWAKE_PORT);
			if(value != null){
				try{
					int intValue = Integer.parseInt(value);
					nodeInfo.setAgentAwakePort(intValue);
				}catch(NumberFormatException e){
				}
			}
		}

		// JOB
		if (nodeInfo.getJobPriority() == null) {
			nodeInfo.setJobPriority(16);

			String value = userDefinedNodePropertyList.get(NodeConstant.JOB_PRIORITY);
			if(value != null){
				try{
					int intValue = Integer.parseInt(value);
					nodeInfo.setJobPriority(intValue);
				}catch(NumberFormatException e){
					nodeInfo.setJobPriority(16);
				}
			}
		}
		if (nodeInfo.getJobMultiplicity() == null) {
			nodeInfo.setJobMultiplicity(0);

			String value = userDefinedNodePropertyList.get(NodeConstant.JOB_MULTIPLICITY);
			if(value != null){
				try{
					int intValue = Integer.parseInt(value);
					nodeInfo.setJobMultiplicity(intValue);
				}catch(NumberFormatException e){
				}
			}
		}
		
		// SNMP
		if (nodeInfo.getSnmpPort() == null) {
			nodeInfo.setSnmpPort(161);

			String value = userDefinedNodePropertyList.get(NodeConstant.SNMP_PORT);
			if(value != null){
				try{
					int intValue = Integer.parseInt(value);
					nodeInfo.setSnmpPort(intValue);
				}catch(NumberFormatException e){
				}
			}
		}
		if (checkEmpty(nodeInfo.getSnmpCommunity())) {
			nodeInfo.setSnmpCommunity("public");

			String value = userDefinedNodePropertyList.get(NodeConstant.SNMP_COMMUNITY);
			if(value != null){
				nodeInfo.setSnmpCommunity(value);
			}
		}
		if (checkEmpty(nodeInfo.getSnmpVersion())) {
			nodeInfo.setSnmpVersion("2c");

			String value = userDefinedNodePropertyList.get(NodeConstant.SNMP_VERSION);
			if(value != null){
				if(value.equals("1") || value.equals("2c")){
					nodeInfo.setSnmpVersion(value);
				}
			}
		}
		if (nodeInfo.getSnmpTimeout() == null || nodeInfo.getSnmpTimeout() == -1) {
			nodeInfo.setSnmpTimeout(5000);

			String value = userDefinedNodePropertyList.get(NodeConstant.SNMPTIMEOUT);
			if(value != null){
				try{
					int intValue = Integer.parseInt(value);
					nodeInfo.setSnmpTimeout(intValue);
				}catch(NumberFormatException e){
				}
			}
		}
		if (nodeInfo.getSnmpRetryCount() == null || nodeInfo.getSnmpRetryCount() == -1) {
			nodeInfo.setSnmpRetryCount(3);

			String value = userDefinedNodePropertyList.get(NodeConstant.SNMPRETRIES);
			if(value != null){
				try{
					int intValue = Integer.parseInt(value);
					nodeInfo.setSnmpRetryCount(intValue);
				}catch(NumberFormatException e){
				}
			}
		}

		// WBEM
		if (checkEmpty(nodeInfo.getWbemUser())) {
			nodeInfo.setWbemUser("root");

			String value = userDefinedNodePropertyList.get(NodeConstant.WBEM_USER);
			if(value != null){
				nodeInfo.setWbemUser(value);
			}
		}
		if (nodeInfo.getWbemUserPassword() == null || nodeInfo.getWbemUserPassword() == "") {
			nodeInfo.setWbemUserPassword("");

			String value = userDefinedNodePropertyList.get(NodeConstant.WBEM_USER_PASSWORD);
			if(value != null){
				nodeInfo.setWbemUserPassword(value);
			}
		}
		if (nodeInfo.getWbemPort() == null || nodeInfo.getWbemPort() == -1) {
			nodeInfo.setWbemPort(5988);

			String value = userDefinedNodePropertyList.get(NodeConstant.WBEM_PORT);
			if(value != null){
				try{
					int intValue = Integer.parseInt(value);
					nodeInfo.setWbemPort(intValue);
				}catch(NumberFormatException e){
				}
			}
		}
		if (checkEmpty(nodeInfo.getWbemProtocol())) {
			nodeInfo.setWbemProtocol("http");

			String value = userDefinedNodePropertyList.get(NodeConstant.WBEM_PROTOCOL);
			if(value != null){
				if(value.equals("http") || value.equals("https")){
					nodeInfo.setWbemProtocol(value);
				}
			}
		}
		if (nodeInfo.getWbemTimeout() == null || nodeInfo.getWbemTimeout() == -1) {
			nodeInfo.setWbemTimeout(5000);

			String value = userDefinedNodePropertyList.get(NodeConstant.WBEM_TIMEOUT);
			if(value != null){
				try{
					int intValue = Integer.parseInt(value);
					nodeInfo.setWbemTimeout(intValue);
				}catch(NumberFormatException e){
				}
			}
		}
		if (nodeInfo.getWbemRetryCount() == null || nodeInfo.getWbemRetryCount() == -1) {
			nodeInfo.setWbemRetryCount(3);

			String value = userDefinedNodePropertyList.get(NodeConstant.WBEM_RETRIES);
			if(value != null){
				try{
					int intValue = Integer.parseInt(value);
					nodeInfo.setWbemRetryCount(intValue);
				}catch(NumberFormatException e){
				}
			}
		}

		// IPMI
		if (nodeInfo.getIpmiIpAddress() == null || nodeInfo.getIpmiIpAddress() == "") {
			nodeInfo.setIpmiIpAddress("");

			String value = userDefinedNodePropertyList.get(NodeConstant.IPMI_IP_ADDRESS);
			if(value != null){
				nodeInfo.setIpmiIpAddress(value);
			}
		}
		if (nodeInfo.getIpmiPort() == null || nodeInfo.getIpmiPort() == -1) {
			nodeInfo.setIpmiPort(0);

			String value = userDefinedNodePropertyList.get(NodeConstant.IPMI_PORT);
			if(value != null){
				try{
					int intValue = Integer.parseInt(value);
					nodeInfo.setIpmiPort(intValue);
				}catch(NumberFormatException e){
				}
			}
		}
		if (nodeInfo.getIpmiUser() == null || nodeInfo.getIpmiUser() == "") {
			nodeInfo.setIpmiUser("");

			String value = userDefinedNodePropertyList.get(NodeConstant.IPMI_USER);
			if(value != null){
				nodeInfo.setIpmiUser(value);
			}
		}
		if (nodeInfo.getIpmiUserPassword() == null || nodeInfo.getIpmiUserPassword() == "") {
			nodeInfo.setIpmiUserPassword("");

			String value = userDefinedNodePropertyList.get(NodeConstant.IPMI_USER_PASSWORD);
			if(value != null){
				nodeInfo.setIpmiUserPassword(value);
			}
		}
		if (nodeInfo.getIpmiTimeout() == null || nodeInfo.getIpmiTimeout() == -1) {
			nodeInfo.setIpmiTimeout(0);

			String value = userDefinedNodePropertyList.get(NodeConstant.IPMI_TIMEOUT);
			if(value != null){
				try{
					int intValue = Integer.parseInt(value);
					nodeInfo.setIpmiTimeout(intValue);
				}catch(NumberFormatException e){
				}
			}
		}
		if (nodeInfo.getIpmiRetries() == null || nodeInfo.getIpmiRetries() == -1) {
			nodeInfo.setIpmiRetries(3);

			String value = userDefinedNodePropertyList.get(NodeConstant.IPMI_RETRIES);
			if(value != null){
				try{
					int intValue = Integer.parseInt(value);
					nodeInfo.setIpmiRetries(intValue);
				}catch(NumberFormatException e){
				}
			}
		}
		if (checkEmpty(nodeInfo.getIpmiProtocol())) {
			nodeInfo.setIpmiProtocol("RMCP+");

			String value = userDefinedNodePropertyList.get(NodeConstant.IPMI_PROTOCOL);
			if(value != null){
				nodeInfo.setIpmiProtocol(value);
			}
		}
		if (nodeInfo.getIpmiLevel() == null || nodeInfo.getIpmiLevel() == "") {
			nodeInfo.setIpmiLevel("");

			String value = userDefinedNodePropertyList.get(NodeConstant.IPMI_LEVEL);
			if(value != null){
				nodeInfo.setIpmiLevel(value);
			}
		}

		// WinRM
		if (nodeInfo.getWinrmUser() == null || nodeInfo.getWinrmUser() == "") {
			nodeInfo.setWinrmUser("");

			String value = userDefinedNodePropertyList.get(NodeConstant.WINRM_USER);
			if(value != null){
				nodeInfo.setWinrmUser(value);
			}
		}
		if (nodeInfo.getWinrmUserPassword() == null || nodeInfo.getWinrmUserPassword() == "") {
			nodeInfo.setWinrmUserPassword("");

			String value = userDefinedNodePropertyList.get(NodeConstant.WINRM_USER_PASSWORD);
			if(value != null){
				nodeInfo.setWinrmUserPassword(value);
			}
		}
		if (checkEmpty(nodeInfo.getWinrmVersion())) {
			nodeInfo.setWinrmVersion("2.0");

			String value = userDefinedNodePropertyList.get(NodeConstant.WINRM_VERSION);
			if(value != null){
				nodeInfo.setWinrmVersion(value);
			}
		}
		if (nodeInfo.getWinrmPort() == null || nodeInfo.getWinrmPort() == -1) {
			nodeInfo.setWinrmPort(5985);

			String value = userDefinedNodePropertyList.get(NodeConstant.WINRM_PORT);
			if(value != null){
				try{
					int intValue = Integer.parseInt(value);
					nodeInfo.setWinrmPort(intValue);
				}catch(NumberFormatException e){
				}
			}
		}
		if (checkEmpty(nodeInfo.getWinrmProtocol())) {
			nodeInfo.setWinrmProtocol("http");

			String value = userDefinedNodePropertyList.get(NodeConstant.WINRM_PROTOCOL);
			if(value != null){
				if(value.equals("http") || value.equals("https")){
					nodeInfo.setWinrmProtocol(value);
				}
			}
		}
		if (nodeInfo.getWinrmTimeout() == null || nodeInfo.getWinrmTimeout() == -1) {
			nodeInfo.setWinrmTimeout(5000);

			String value = userDefinedNodePropertyList.get(NodeConstant.WINRM_TIMEOUT);
			if(value != null){
				try{
					int intValue = Integer.parseInt(value);
					nodeInfo.setWinrmTimeout(intValue);
				}catch(NumberFormatException e){
				}
			}
		}
		if (nodeInfo.getWinrmRetries() == null || nodeInfo.getWinrmRetries() == -1) {
			nodeInfo.setWinrmRetries(3);

			String value = userDefinedNodePropertyList.get(NodeConstant.WINRM_RETRIES);
			if(value != null){
				try{
					int intValue = Integer.parseInt(value);
					nodeInfo.setWinrmRetries(intValue);
				}catch(NumberFormatException e){
				}
			}
		}

		// サーバ仮想化
		//		if (nodeInfo.getVirtualizationNodeType() == null) {
		//			nodeInfo.setVirtualizationNodeType("");
		//		}
		if (nodeInfo.getVmManagementNode() == null) {
			nodeInfo.setVmManagementNode("");
		}
		//		if (nodeInfo.getVmIndex() == null) {
		//			nodeInfo.setVmIndex(-1);
		//		}
		if (nodeInfo.getVmName() == null) {
			nodeInfo.setVmName("");
		}
		//		if (nodeInfo.getVirtualizationSolution() == null) {
		//			nodeInfo.setVirtualizationSolution("");
		//		}
		if (nodeInfo.getVmId() == null) {
			nodeInfo.setVmId("");
		}
		if (nodeInfo.getVmUser() == null) {
			nodeInfo.setVmUser("");
		}
		if (nodeInfo.getVmUserPassword() == null) {
			nodeInfo.setVmUserPassword("");
		}
		if (nodeInfo.getVmProtocol() == null) {
			nodeInfo.setVmProtocol("");
		}

		// ネットワーク仮想化
		//		if (nodeInfo.getVNetSwitchType() == null) {
		//			nodeInfo.setVNetSwitchType("");
		//		}
		//		if (nodeInfo.getVNetHostNode() == null) {
		//			nodeInfo.setVNetHostNode("");
		//		}
		if (nodeInfo.getOpenFlowDataPathId() == null) {
			nodeInfo.setOpenFlowDataPathId("");
		}
		if (nodeInfo.getOpenFlowCtrlIpAddress() == null) {
			nodeInfo.setOpenFlowCtrlIpAddress("");
		}

		// ノード変数
		if(nodeInfo.getNodeVariableInfo() == null){
			ArrayList<NodeVariableInfo> list = new ArrayList<>();
			
			String nameString = userDefinedNodePropertyList.get(NodeConstant.NODE_VARIABLE_NAME);
			String valueString = userDefinedNodePropertyList.get(NodeConstant.NODE_VARIABLE_VALUE);
			if(nameString != null && valueString != null){
				String[] names = nameString.split(",");
				String[] values = valueString.split(",");
				if(names.length == values.length){
					for (int i = 0; i < values.length; i++) {
						list.add(new NodeVariableInfo(names[i], values[i]));
					}
				}
			}
			
			nodeInfo.setNodeVariableInfo(list);
		}
		
		// 保守
		if (nodeInfo.getAdministrator() == null || nodeInfo.getAdministrator() == "") {
			nodeInfo.setAdministrator("");

			String value = userDefinedNodePropertyList.get(NodeConstant.ADMINISTRATOR);
			if(value != null){
				nodeInfo.setAdministrator(value);
			}
		}
		if (nodeInfo.getContact() == null || nodeInfo.getContact() == "") {
			nodeInfo.setContact("");

			String value = userDefinedNodePropertyList.get(NodeConstant.CONTACT);
			if(value != null){
				nodeInfo.setContact(value);
			}
		}
	}

	public static String getFacilityName(RepositoryControllerBean repositoryController, String facilityId) {
		try {
			NodeInfo nodeInfo = repositoryController.getNode(facilityId);
			return nodeInfo.getFacilityName();
		}
		catch(Exception e) {
			Logger logger = Logger.getLogger(HinemosUtil.class);
			logger.error("Not got facilityId (" + facilityId + ")");
		}

		return null;
	}

	public static String getFacilityName(String facilityId) {
		try {
			return getFacilityName(new RepositoryControllerBean(), facilityId);
		}
		catch(Exception e) {
			Logger logger = Logger.getLogger(HinemosUtil.class);
			logger.error("Not got facilityId (" + facilityId + ")");
		}

		return null;
	}

	public static JobTreeItem searchJobTreeItem(JobTreeItem treeItem, String jobunitId, String id) {
		if (treeItem.getData().getJobunitId().equals(jobunitId) && treeItem.getData().getId().equals(id)) {
			return treeItem;
		}
		for (JobTreeItem child: treeItem.getChildren()) {
			// Hinemos から取得した情報には、親が設定されていないので、むりやりここで代入。
			child.setParent(treeItem);

			JobTreeItem result = searchJobTreeItem(child, jobunitId, id);
			if (result != null) {
				return result;
			}
		}
		return null;
	}

	public static void relateJobTreeItems(JobTreeItem parent, JobTreeItem child) {
		child.setParent(parent);
		child.getData().setJobunitId(parent.getData().getJobunitId());
		if (parent.getChildren() == null) {
			parent.setChildren(new ArrayList<JobTreeItem>());
		}
		parent.getChildren().add(child);
	}

	public static final String MESSAGE_ID_INFO = "001";
	public static final String MESSAGE_ID_WARNING = "002";
	public static final String MESSAGE_ID_CRITICAL = "003";
	public static final String MESSAGE_ID_UNKNOWN = "100";
	public static final String MESSAGE_ID_FAILURE = "200";

	public static enum Priority {
		INFO(PriorityConstant.TYPE_INFO, MESSAGE_ID_INFO),
		WARNING(PriorityConstant.TYPE_WARNING, MESSAGE_ID_WARNING),
		CRITICAL(PriorityConstant.TYPE_CRITICAL, MESSAGE_ID_CRITICAL),
		UNKNOWN(PriorityConstant.TYPE_UNKNOWN, MESSAGE_ID_UNKNOWN),
		FAILURE(PriorityConstant.TYPE_FAILURE, MESSAGE_ID_FAILURE),
		;

		private Priority(int type, String messageId) {
			this.type = type;
			this.messageId = messageId;
		}

		public final int type;
		public final String messageId;

		public static Priority priority(int type) {
			for (Priority p: values()) {
				if (p.type == type) {
					return p;
				}
			}
			return null;
		}
	}

	public static OutputBasicInfo createOutputBasicInfo(
			Priority priority,
			String pluginId,
			String monitorId,
			String subKey,
			String application,
			String facilityId,
			String facilityPath,
			String message,
			String messageOrg,
			Long generationDate) {
		OutputBasicInfo output = new OutputBasicInfo();

		// 通知情報を設定
		output.setPluginId(pluginId);
		output.setMonitorId(monitorId);
		output.setApplication(application);

		// 通知抑制用のサブキーを設定。
		output.setSubKey(subKey);

		output.setFacilityId(facilityId);

		if (facilityPath == null) {
			try {
				facilityPath = RepositoryControllerBeanWrapper.bean().getFacilityPath(facilityId, null);
			}
			catch (Exception e) {
				Logger.getLogger(HinemosUtil.class).error(e.getMessage(), e);
			}
		}
		output.setScopeText(facilityPath);

		output.setPriority(priority.type);
		output.setMessageId(priority.messageId);
		output.setMessage(message);
		output.setMessageOrg(messageOrg);
		output.setGenerationDate(generationDate);

		return output;
	}

	public static OutputBasicInfo createOutputBasicInfoEx(
			Priority priority,
			String pluginId,
			String monitorId,
			String subKey,
			String application,
			String facilityId,
			String message,
			String messageOrg) {
		return createOutputBasicInfo(
				priority,
				pluginId,
				monitorId,
				subKey,
				application,
				facilityId,
				null,
				message,
				messageOrg,
				new Date().getTime());
	}

	public static OutputBasicInfo createInternalOutputBasicInfo(
			Priority priority,
			String pluginId,
			String monitorId,
			String subKey,
			String scopeText,
			String application,
			String message,
			String messageOrg) {
		return createOutputBasicInfo(
				priority,
				pluginId,
				monitorId,
				subKey,
				application,
				"INTERNAL",
				scopeText,
				message,
				messageOrg,
				new Date().getTime());
	}

	public static void notifyInternalMessage (
			Priority priority,
			String pluginId,
			String monitorId,
			String subKey,
			String scopeText,
			String application,
			String message,
			String messageOrg) {

		final OutputBasicInfo output = createInternalOutputBasicInfo(
				priority,
				pluginId,
				monitorId,
				subKey,
				scopeText,
				application,
				message,
				messageOrg);

		SessionService.execute(new Runnable() {
			@Override
			public void run() {
				try {
					AsyncWorkerPlugin.addTask(NotifyEventTaskFactory.class.getSimpleName(), output, false);
				}
				catch (HinemosUnknown e) {
					Logger.getLogger(this.getClass()).warn(e.getMessage(), e);
				}
			}
		});
	}
	
	public static class ObjectPriviledgeOperator {
		private List<ObjectPrivilegeInfo> privileges;
		private String objectType;
		private String objectId;
		private AccessControllerBean accessBean;
		
		public ObjectPriviledgeOperator(AccessControllerBean accessBean, String objectType, String objectId) throws HinemosUnknown {
			this.accessBean = accessBean;
			this.objectType = objectType;
			this.objectId = objectId;
			
			ObjectPrivilegeFilterInfo filter = new ObjectPrivilegeFilterInfo();
			filter.setObjectId(this.objectId);
			filter.setObjectType(this.objectType);
			privileges = accessBean.getObjectPrivilegeInfoList(filter);
		}
		
		public ObjectPriviledgeOperator addFullRightToObject(String roleId) {
			return addRightToObject(roleId, ObjectPrivilegeMode.READ, ObjectPrivilegeMode.WRITE, ObjectPrivilegeMode.EXEC);
		}

		public ObjectPriviledgeOperator removeFullRightFromObject(String roleId) {
			return removeRightFromObject(roleId, ObjectPrivilegeMode.READ, ObjectPrivilegeMode.WRITE, ObjectPrivilegeMode.EXEC);
		}

		public ObjectPriviledgeOperator addRightToObject(String roleId, ObjectPrivilegeMode...modes) {
			Set<ObjectPrivilegeInfo> addingPrivileges = new TreeSet<>(new Comparator<ObjectPrivilegeInfo>() {
				@Override
				public int compare(ObjectPrivilegeInfo o1, ObjectPrivilegeInfo o2) {
					return o1.getObjectPrivilege().compareTo(o2.getObjectPrivilege());
				}
			});
			for (ObjectPrivilegeMode mode: modes) {
				ObjectPrivilegeInfo objectPrivilegeInfo = new ObjectPrivilegeInfo();
				objectPrivilegeInfo.setObjectId(objectId);
				objectPrivilegeInfo.setObjectType(objectType);
				objectPrivilegeInfo.setRoleId(roleId);
				objectPrivilegeInfo.setObjectPrivilege(mode.name());
				addingPrivileges.add(objectPrivilegeInfo);
			}
			for (ObjectPrivilegeInfo priviledge: privileges) {
				for (Iterator<ObjectPrivilegeInfo> iter = addingPrivileges.iterator(); iter.hasNext();) {
					ObjectPrivilegeInfo adding = iter.next();
					if (priviledge.getObjectPrivilege().equals(adding.getObjectPrivilege()) && priviledge.getRoleId().equals(adding.getRoleId())) {
						iter.remove();
						break;
					}
				}
			}
			privileges.addAll(addingPrivileges);
			return this;
		}

		public ObjectPriviledgeOperator removeRightFromObject(String roleId, ObjectPrivilegeMode...modes) {
			for (Iterator<ObjectPrivilegeInfo> iter = privileges.iterator(); iter.hasNext();) {
				ObjectPrivilegeInfo priviledge = iter.next();
				if (priviledge.getRoleId().equals(roleId) && ObjectPrivilegeMode.valueOf(priviledge.getObjectPrivilege()) != null) {
					iter.remove();
				}
			}
			return this;
		}
		
		public void commit() throws PrivilegeDuplicate, UsedObjectPrivilege, HinemosUnknown, InvalidSetting, InvalidRole, JobMasterNotFound {
			accessBean.replaceObjectPrivilegeInfo(objectType, objectId, privileges);
		}
	}
	
	public static void addFullRightToObject(AccessControllerBean accessBean, String roleId, String objectType, String objectId) throws PrivilegeDuplicate, UsedObjectPrivilege, HinemosUnknown, InvalidSetting, InvalidRole, JobMasterNotFound {
		new ObjectPriviledgeOperator(accessBean, objectType, objectId).addFullRightToObject(roleId).commit();
	}

	public static void removeFullRightFromObject(AccessControllerBean accessBean, String roleId, String objectType, String objectId) throws PrivilegeDuplicate, UsedObjectPrivilege, HinemosUnknown, InvalidSetting, InvalidRole, JobMasterNotFound {
		new ObjectPriviledgeOperator(accessBean, objectType, objectId).removeFullRightFromObject(roleId).commit();
	}

	public static void addRightToObject(AccessControllerBean accessBean, String roleId, String objectType, String objectId, ObjectPrivilegeMode...modes) throws PrivilegeDuplicate, UsedObjectPrivilege, HinemosUnknown, InvalidSetting, InvalidRole, JobMasterNotFound {
		new ObjectPriviledgeOperator(accessBean, objectType, objectId).addRightToObject(roleId, modes).commit();
	}

	public static void removeRightFromObject(AccessControllerBean accessBean, String roleId, String objectType, String objectId, ObjectPrivilegeMode...modes) throws PrivilegeDuplicate, UsedObjectPrivilege, HinemosUnknown, InvalidSetting, InvalidRole, JobMasterNotFound {
		new ObjectPriviledgeOperator(accessBean, objectType, objectId).removeRightFromObject(roleId, modes).commit();
	}
	
	public static JobInfo createJobInfo(String jobId, int type, String name, String description, String roleId) {
		JobInfo info = new JobInfo();
		info.setId(jobId);
		info.setType(type);

		info.setName(name);
		info.setDescription(description);

		ArrayList<JobEndStatusInfo> statuses = new ArrayList<>();
		JobEndStatusInfo status0 = new JobEndStatusInfo();
		status0.setType(EndStatusConstant.TYPE_NORMAL);
		status0.setValue(EndStatusConstant.INITIAL_VALUE_NORMAL);
		status0.setStartRangeValue(0);
		status0.setEndRangeValue(0);
		statuses.add(status0);
		JobEndStatusInfo status1 = new JobEndStatusInfo();
		status1.setType(EndStatusConstant.TYPE_WARNING);
		status1.setValue(EndStatusConstant.INITIAL_VALUE_WARNING);
		status1.setStartRangeValue(1);
		status1.setEndRangeValue(1);
		statuses.add(status1);
		JobEndStatusInfo status2 = new JobEndStatusInfo();
		status2.setType(EndStatusConstant.TYPE_ABNORMAL);
		status2.setValue(EndStatusConstant.INITIAL_VALUE_ABNORMAL);
		status2.setStartRangeValue(0);
		status2.setEndRangeValue(0);
		statuses.add(status2);
		info.setEndStatus(statuses);

		String notifyGroupId = NotifyGroupIdGenerator.createNotifyGroupIdJob(info.getJobunitId(), info.getId(),0);
		ArrayList<JobNotificationsInfo> notifications = new ArrayList<JobNotificationsInfo>();
		JobNotificationsInfo infoStart = new JobNotificationsInfo();
		infoStart.setType(EndStatusConstant.TYPE_BEGINNING);
		infoStart.setPriority(PriorityConstant.TYPE_CRITICAL);
		infoStart.setNotifyGroupId(notifyGroupId);
		notifications.add(infoStart);
		JobNotificationsInfo infoNormal = new JobNotificationsInfo();
		infoNormal.setType(EndStatusConstant.TYPE_NORMAL);
		infoNormal.setPriority(PriorityConstant.TYPE_CRITICAL);
		infoNormal.setNotifyGroupId(notifyGroupId);
		notifications.add(infoNormal);
		JobNotificationsInfo infoWarning = new JobNotificationsInfo();
		infoWarning.setType(EndStatusConstant.TYPE_WARNING);
		infoWarning.setPriority(PriorityConstant.TYPE_CRITICAL);
		infoWarning.setNotifyGroupId(notifyGroupId);
		notifications.add(infoWarning);
		JobNotificationsInfo infoAbnormal = new JobNotificationsInfo();
		infoAbnormal.setType(EndStatusConstant.TYPE_ABNORMAL);
		infoAbnormal.setPriority(PriorityConstant.TYPE_CRITICAL);
		infoAbnormal.setNotifyGroupId(notifyGroupId);
		notifications.add(infoAbnormal);

		info.setNotifications(notifications);
		info.setOwnerRoleId(roleId);

		info.setPropertyFull(true);

		return info;
	}

	public static JobTreeItem createJobnet(String jobnetId, String name, String description, String roleId) {
		JobInfo info = createJobInfo(jobnetId, JobConstant.TYPE_JOBNET, name, description, roleId);
		setJobWaitRuleInfo(info);

		JobTreeItem jobnet = new JobTreeItem();
		jobnet.setData(info);

		return jobnet;
	}

	public static JobTreeItem createJob(RepositoryControllerBean repositoryBean, String jobId, String name, String description, String facilityId, String command, String roleId) throws HinemosUnknown {
		JobInfo info = createJobInfo(jobId, JobConstant.TYPE_JOB, name, description, roleId);
		setJobWaitRuleInfo(info);

		JobCommandInfo jci = new JobCommandInfo();
		info.setCommand(jci);

		jci.setFacilityID(facilityId);
		jci.setScope(repositoryBean.getFacilityPath(facilityId, null));
		jci.setStartCommand(command);
		jci.setStopType(CommandStopTypeConstant.DESTROY_PROCESS);
		jci.setSpecifyUser(YesNoConstant.TYPE_NO);
		jci.setProcessingMethod(ProcessingMethodConstant.TYPE_ALL_NODE);
		jci.setMessageRetry(10);
		jci.setErrorEndFlg(YesNoConstant.booleanToType(true));
		jci.setErrorEndValue(EndStatusConstant.INITIAL_VALUE_ABNORMAL);

		JobTreeItem job = new JobTreeItem();
		job.setData(info);

		return job;
	}

	public static void setJobWaitRuleInfo(JobInfo jobInfo) {
		JobWaitRuleInfo jobWaitRuleInfo = new JobWaitRuleInfo();

		// 待ち条件タブ
		jobWaitRuleInfo.setCondition(ConditionTypeConstant.TYPE_AND);
		jobWaitRuleInfo.setEndCondition(YesNoConstant.TYPE_YES);
		jobWaitRuleInfo.setEndStatus(EndStatusConstant.TYPE_ABNORMAL);
		jobWaitRuleInfo.setEndValue(EndStatusConstant.INITIAL_VALUE_ABNORMAL);

		// 制御タブ
		jobWaitRuleInfo.setCalendar(YesNoConstant.TYPE_NO);
		jobWaitRuleInfo.setCalendarEndStatus(EndStatusConstant.TYPE_ABNORMAL);
		jobWaitRuleInfo.setCalendarEndValue(EndStatusConstant.INITIAL_VALUE_NORMAL);
		jobWaitRuleInfo.setSuspend(YesNoConstant.TYPE_NO);
		jobWaitRuleInfo.setSkip(YesNoConstant.TYPE_NO);
		jobWaitRuleInfo.setSkipEndStatus(EndStatusConstant.TYPE_ABNORMAL);
		jobWaitRuleInfo.setSkipEndValue(EndStatusConstant.INITIAL_VALUE_NORMAL);

		// 開始遅延タブ
		jobWaitRuleInfo.setStart_delay(YesNoConstant.TYPE_NO);
		jobWaitRuleInfo.setStart_delay_condition_type(ConditionTypeConstant.TYPE_AND);
		jobWaitRuleInfo.setStart_delay_notify(YesNoConstant.TYPE_NO);
		jobWaitRuleInfo.setStart_delay_notify_priority(PriorityConstant.TYPE_CRITICAL);
		jobWaitRuleInfo.setStart_delay_operation_type(OperationConstant.TYPE_STOP_AT_ONCE);
		jobWaitRuleInfo.setStart_delay_operation(YesNoConstant.TYPE_NO);
		jobWaitRuleInfo.setStart_delay_operation_end_status(EndStatusConstant.TYPE_ABNORMAL);
		jobWaitRuleInfo.setStart_delay_operation_end_value(EndStatusConstant.INITIAL_VALUE_ABNORMAL);
		jobWaitRuleInfo.setStart_delay_session(YesNoConstant.TYPE_NO);
		jobWaitRuleInfo.setStart_delay_session_value(1);
		jobWaitRuleInfo.setStart_delay_time(YesNoConstant.TYPE_NO);

		// 終了遅延タブ
		jobWaitRuleInfo.setEnd_delay(YesNoConstant.TYPE_NO);
		jobWaitRuleInfo.setEnd_delay_condition_type(ConditionTypeConstant.TYPE_AND);
		jobWaitRuleInfo.setEnd_delay_job(YesNoConstant.TYPE_NO);
		jobWaitRuleInfo.setEnd_delay_job_value(1);
		jobWaitRuleInfo.setEnd_delay_notify(YesNoConstant.TYPE_NO);
		jobWaitRuleInfo.setEnd_delay_notify_priority(PriorityConstant.TYPE_CRITICAL);
		jobWaitRuleInfo.setEnd_delay_operation(YesNoConstant.TYPE_NO);
		jobWaitRuleInfo.setEnd_delay_operation_end_status(EndStatusConstant.TYPE_ABNORMAL);
		jobWaitRuleInfo.setEnd_delay_operation_end_value(EndStatusConstant.INITIAL_VALUE_ABNORMAL);
		jobWaitRuleInfo.setEnd_delay_operation_type(OperationConstant.TYPE_STOP_AT_ONCE);
		jobWaitRuleInfo.setEnd_delay_session(YesNoConstant.TYPE_NO);
		jobWaitRuleInfo.setEnd_delay_session_value(1);
		jobWaitRuleInfo.setEnd_delay_time(YesNoConstant.TYPE_NO);

		// 多重度タブ
		jobWaitRuleInfo.setMultiplicityEndValue(EndStatusConstant.INITIAL_VALUE_ABNORMAL);
		jobWaitRuleInfo.setMultiplicityNotify(YesNoConstant.TYPE_YES);
		jobWaitRuleInfo.setMultiplicityNotifyPriority(PriorityConstant.TYPE_WARNING);
		jobWaitRuleInfo.setMultiplicityOperation(StatusConstant.TYPE_WAIT);

		jobInfo.setWaitRule(jobWaitRuleInfo);
	}
}