/*
Copyright (C) 2013 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
*/
package com.clustercontrol.ws.aws;

import java.util.ArrayList;
import java.util.List;

import javax.annotation.Resource;
import javax.jws.WebService;
import javax.xml.ws.WebServiceContext;

import com.amazonaws.AmazonServiceException;
import com.amazonaws.services.ec2.AmazonEC2;
import com.amazonaws.services.ec2.model.DescribeKeyPairsResult;
import com.amazonaws.services.ec2.model.DescribeSecurityGroupsRequest;
import com.amazonaws.services.ec2.model.DescribeSecurityGroupsResult;
import com.amazonaws.services.ec2.model.DescribeSubnetsResult;
import com.amazonaws.services.ec2.model.GetPasswordDataRequest;
import com.clustercontrol.accesscontrol.bean.PrivilegeConstant.SystemPrivilegeMode;
import com.clustercontrol.cloud.CloudManagerFault;
import com.clustercontrol.cloud.Filter;
import com.clustercontrol.cloud.ICloudContext;
import com.clustercontrol.cloud.IResourceManagement;
import com.clustercontrol.cloud.SessionService;
import com.clustercontrol.cloud.aws.util.AWSConstants;
import com.clustercontrol.cloud.aws.util.AWSUtil;
import com.clustercontrol.cloud.aws.util.SecurityGroupUtil;
import com.clustercontrol.cloud.factory.CloudRoleConstants;
import com.clustercontrol.cloud.validation.AuthorizingValidator_role_region;
import com.clustercontrol.cloud.validation.annotation.CustomMethodValidation;
import com.clustercontrol.cloud.validation.annotation.ElementId;
import com.clustercontrol.cloud.validation.annotation.Identity;
import com.clustercontrol.cloud.validation.annotation.Into;
import com.clustercontrol.cloud.validation.annotation.NotNull;
import com.clustercontrol.cloud.validation.annotation.NotNullContainer;
import com.clustercontrol.cloud.validation.annotation.Size;
import com.clustercontrol.fault.InvalidRole;
import com.clustercontrol.fault.InvalidUserPass;
import com.clustercontrol.ws.cloud.IWebServiceBase;
import com.clustercontrol.ws.cloud.security.HinemosAccessRight;

/**
 * クラウド管理オプション用のエンドポイント
 *
 */
@WebService(serviceName = "AWSOptionEndpointService", portName = "AWSOptionEndpointPort", targetNamespace = "http://aws.ws.clustercontrol.com", endpointInterface="com.clustercontrol.ws.aws.AWSOptionEndpoint")
public class AWSOptionEndpointImpl implements AWSOptionEndpoint, IWebServiceBase, AWSConstants, CloudRoleConstants {
	@Resource
	private WebServiceContext wsctx;

	private static AmazonEC2 getAmazonEC2() throws CloudManagerFault {
		final ICloudContext context = SessionService.current().get(ICloudContext.class);
		return AWSUtil.createAmazonEC2(
				new IResourceManagement.ICredential() {
					@Override
					public String getSecretKey() throws CloudManagerFault {
						return context.getAccessDestionation().getCloudUser().getSecretKey();
					}
					@Override
					public String getAccessKey() throws CloudManagerFault {
						return context.getAccessDestionation().getCloudUser().getAccessKey();
					}
				},
				context.getCurrentRegion().getEndpoint(EP_TYPE_EC2).getLocation());
	}
	
	@Override
	@HinemosAccessRight(roleName=CLOUDMANAGEMENT, right=SystemPrivilegeMode.READ)
	@CustomMethodValidation(AuthorizingValidator_role_region.class)
	public List<KeyPair> getKeyNames(
			@ElementId("roleId") @Identity String roleId,
			@ElementId("region") @NotNull @Size(max=64) String region
			) throws CloudManagerFault, InvalidUserPass, InvalidRole {
		try {
			DescribeKeyPairsResult awsResult = getAmazonEC2().describeKeyPairs();
			List<KeyPair> keyPairs = new ArrayList<>();
			for (com.amazonaws.services.ec2.model.KeyPairInfo awsKeyPair: awsResult.getKeyPairs()) {
				KeyPair kp = new KeyPair();
				kp.setKeyName(awsKeyPair.getKeyName());
				keyPairs.add(kp);
			}
			return keyPairs;
		}
		catch (AmazonServiceException e) {
			throw new CloudManagerFault(e.getMessage(), e.getErrorCode(), e);
		}
	}

	@Override
	@HinemosAccessRight(roleName=CLOUDMANAGEMENT, right=SystemPrivilegeMode.READ)
	@CustomMethodValidation(AuthorizingValidator_role_region.class)
	public List<Subnet> getSubnets(
			@ElementId("roleId") @Identity String roleId,
			@ElementId("region") @NotNull @Size(max=64) String region
			) throws CloudManagerFault, InvalidUserPass, InvalidRole {
		try {
			List<Subnet> subnets = new ArrayList<>(); 
			DescribeSubnetsResult awsResult = getAmazonEC2().describeSubnets();
			for (com.amazonaws.services.ec2.model.Subnet awsSubnet: awsResult.getSubnets()) {
				Subnet subnet = new Subnet();
				subnet.setSubnetId(awsSubnet.getSubnetId());
				subnet.setVpcId(awsSubnet.getVpcId());
				subnet.setAvailabilityZone(awsSubnet.getAvailabilityZone());
				subnet.setCidrBlock(awsSubnet.getCidrBlock());
				subnets.add(subnet);
			}
			return subnets;
		}
		catch (AmazonServiceException e) {
			throw new CloudManagerFault(e.getMessage(), e.getErrorCode(), e);
		}
	}

	@Override
	@HinemosAccessRight(roleName=CLOUDMANAGEMENT, right=SystemPrivilegeMode.READ)
	@CustomMethodValidation(AuthorizingValidator_role_region.class)
	public String getWindowsPassword(
			@ElementId("roleId") @Identity String roleId,
			@ElementId("region") @NotNull @Size(max=64) String region,
			@ElementId("instanceId") @NotNull @Size(max=64) String instanceId
			) throws CloudManagerFault, InvalidUserPass, InvalidRole {
		try {
			return getAmazonEC2().getPasswordData(new GetPasswordDataRequest().withInstanceId(instanceId)).getPasswordData();
		}
		catch (AmazonServiceException e) {
			throw new CloudManagerFault(e.getMessage(), e.getErrorCode(), e);
		}
	}

	@Override
	@HinemosAccessRight(roleName=CLOUDMANAGEMENT, right=SystemPrivilegeMode.READ)
	@CustomMethodValidation(AuthorizingValidator_role_region.class)
	public List<SecurityGroup> getSecurityGroups(
			@ElementId("roleId") @Identity String roleId,
			@ElementId("region") @NotNull @Size(max=64) String region
			) throws CloudManagerFault, InvalidUserPass, InvalidRole {
		try{
			DescribeSecurityGroupsResult dsgResult = getAmazonEC2().describeSecurityGroups();
			List<SecurityGroup> groups = new ArrayList<SecurityGroup>();
			for(com.amazonaws.services.ec2.model.SecurityGroup awsSg: dsgResult.getSecurityGroups()){
				SecurityGroup sg = SecurityGroupUtil.convertToSecrurityGroup(awsSg);
				groups.add(sg);
			}
			
			return groups;
		}
		catch(com.amazonaws.AmazonServiceException e){
			throw new CloudManagerFault(e);
		}
	}

	@Override
	@HinemosAccessRight(roleName=CLOUDMANAGEMENT, right=SystemPrivilegeMode.READ)
	@CustomMethodValidation(AuthorizingValidator_role_region.class)
	public List<SecurityGroup> getSecurityGroupsWithFilters(
			@ElementId("roleId") @Identity String roleId,
			@ElementId("region") @NotNull @Size(max=64) String region,
			@ElementId("filters") @NotNull @NotNullContainer @Into List<Filter> filters
			) throws CloudManagerFault, InvalidUserPass, InvalidRole {
		try {
			DescribeSecurityGroupsRequest awsRequest = new DescribeSecurityGroupsRequest();
			for (Filter f: filters) {
				com.amazonaws.services.ec2.model.Filter awsFilter = new com.amazonaws.services.ec2.model.Filter(f.getName(), f.getValues());
				awsRequest.getFilters().add(awsFilter);
			}
			
			DescribeSecurityGroupsResult result = getAmazonEC2().describeSecurityGroups(awsRequest);
			List<SecurityGroup> groups = new ArrayList<SecurityGroup>();
			for(com.amazonaws.services.ec2.model.SecurityGroup awsSg: result.getSecurityGroups()){
				SecurityGroup sg = SecurityGroupUtil.convertToSecrurityGroup(awsSg);
				groups.add(sg);
			}
			
			return groups;
		}
		catch(AmazonServiceException e){
			throw new CloudManagerFault(e.getMessage(), e.getErrorCode(), e);
		}
	}
	
	@Override
	public void start() {
	}

	@Override
	public void stop() {
	}
}