/*
Copyright (C) 2013 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */

package com.clustercontrol.cloud.aws.base.ui.views.actions;

import java.util.ArrayList;

import org.eclipse.jface.action.IAction;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;

import com.clustercontrol.cloud.aws.base.commons.util.MessageManagerExt;
import com.clustercontrol.cloud.aws.base.resources.AvailabilityZone;
import com.clustercontrol.cloud.aws.base.resources.GroupIdentifier;
//
import com.clustercontrol.cloud.aws.base.resources.CloudResourceProviderImpl;
import com.clustercontrol.cloud.aws.base.ui.dialogs.CreateInstanceDialog;
import com.clustercontrol.cloud.presenter.CloudModelException;
import com.clustercontrol.cloud.presenter.CreateCloudInstanceByTemplateRequest;
import com.clustercontrol.cloud.presenter.CreateCloudInstanceRequest;
import com.clustercontrol.cloud.presenter.ICloudComputeService;
import com.clustercontrol.cloud.presenter.ICloudInstance;
import com.clustercontrol.cloud.presenter.ICloudTemplate;
import com.clustercontrol.cloud.presenter.ICloudTemplateManager;
import com.clustercontrol.cloud.presenter.IAccountResource.IRegion;
import com.clustercontrol.cloud.presenter.ICloudInstance.StateType;
import com.clustercontrol.cloud.ui.views.InstanceMonitorService;
import com.clustercontrol.cloud.ui.views.InstancesView;
import com.clustercontrol.cloud.ui.views.actions.AbstractViewAction;
import com.clustercontrol.cloud.ui.views.actions.InvocationException;
import com.clustercontrol.util.Messages;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectWriter;


public class CreateInstanceAction extends AbstractViewAction {
	private MessageManagerExt messages = MessageManagerExt.getInstance("messages");
	
	private String msgCreateInstanceConfirm = messages.getString("message.confirm_some_action_to_something", new Object[]{"word.instance", "word.create"});
	private String msgCreateInstanceResult = messages.getString("message.result_some_action_to_something", new Object[]{"word.instance", "word.create"});

	
	@Override
	protected void invoke(IAction action) throws InvocationException {
		CreateInstanceDialog dialog = new CreateInstanceDialog(this.viewPart.getSite().getShell());
		
		final InstancesView view = (InstancesView)viewPart;
		IRegion region = view.getActiveRegion();
		
		if (region == null) {
			assert view.getCurrentAccountResource().getRegions().length > 0;
			region = view.getCurrentAccountResource().getRegions()[0];
		}
		
		ICloudComputeService service = region.getCloudService(ICloudComputeService.class);
		final ICloudTemplateManager templateManager = region.getAccountResource().getCloudTemplateManager();
		dialog.setCloudTemplateManager(templateManager);
		
		if (region != null) {
			CreateInstanceDialog.Model m = new CreateInstanceDialog.Model();
			m.setRegionName(region.getRegion());
			if (view.getActiveZone() != null) {
				m.setAvailabilityZone(new AvailabilityZone());
				m.getAvailabilityZone().setZoneName(view.getActiveZone());
			}
			dialog.setInput(m);
		}

		dialog.setCloudInfoProvider(new CloudResourceProviderImpl(region.getAccountResource().getAccountResourceId()));
		dialog.setBehavior(new CreateInstanceDialog.DialogBehavior(){
		private Combo region;

			@Override
			public void setupRegion(Combo region) {
				for (String regionName: provider.getAllRegion()) {
					String regionLocaleName = messages.getString(regionName);
					region.add(regionLocaleName);
					region.setData(regionLocaleName, regionName);
				}
				if (input != null && input.getRegionName() != null) {
					String regionLocaleName = messages.getString(input.getRegionName());
					region.select(region.indexOf(regionLocaleName));
				} else {
					region.select(0);
				}
				this.region = region;
			}

			@Override
			public void setupAvailabilityZone(Combo availabilityZone) {
			}

			@Override
			public void setupKeyPair(Combo keyPair) {
			}
			
			@Override
			public void setupSelectImageFlg(Button rdoAmi, Button rdoTemplate){
				rdoAmi.setSelection(true);
				rdoTemplate.setSelection(false);
			}

			@Override
			public void setupTemplate(Button useTemplate, Combo template, Button editArgument){
				for(ICloudTemplate tmpTemplate: templateManager.getCloudTemplates()){
					if(tmpTemplate.getRegion().equals(region.getData(region.getText()))){
						String tmpLabel = tmpTemplate.getTemplateName() + " ( " + tmpTemplate.getTemplateId() + " )";
						template.add(tmpLabel);
						template.setData(tmpLabel, tmpTemplate);
					}
				}
				template.setEnabled(false);
				editArgument.setEnabled(false);
			}
		});
		
		if(dialog.open() == Window.OK){
			if (MessageDialog.openConfirm(
				null,
				Messages.getString("confirmed"),
				msgCreateInstanceConfirm)){

				CreateInstanceDialog.Model m = dialog.getOutput();

				IRegion r = service.getRegion();
				final ICloudInstance instance;
				
				CreateInstanceDialog.InstanceDetail detail = new CreateInstanceDialog.InstanceDetail();

				if(m.getSubnet() != null){
					detail.subnetId = m.getSubnet().getSubnetId();
				} else {
					detail.subnetId = null;
				}
				detail.keyName = m.getKeyName();
				detail.monitoring = m.getMonitoring();
				detail.disableApiTermination = m.getDisableApiTermination();
				detail.instanceInitiatedShutdownBehavior = m.getInstanceInitiatedShutdownBehavior();
				detail.ebsOptimized = m.getEbsOptimized();
				if(m.getRootBlockDevice() != null){
					detail.rootBlockDevice = new CreateInstanceDialog.EbsBlockDevice();
					detail.rootBlockDevice.iops = m.getRootBlockDevice().iops;
					detail.rootBlockDevice.volumeSize = m.getRootBlockDevice().volumeSize;
					detail.rootBlockDevice.volumeType = m.getRootBlockDevice().volumeType;
					detail.rootBlockDevice.deleteOnTermination = m.getRootBlockDevice().deleteOnTermination;
				} else {
//					detail.rootBlockDevice = new CreateInstanceDialog.EbsBlockDevice();
//					detail.rootBlockDevice.volumeSize = 10;
//					detail.rootBlockDevice.volumeType = "standard";
//					detail.rootBlockDevice.deleteOnTermination = true;
				}
				
				if(m.getSecurityGroups() != null){
					detail.securityGroupIds = new ArrayList<>();
					for(GroupIdentifier securityGroup: m.getSecurityGroups()){
						detail.securityGroupIds.add(securityGroup.getGroupId());
					}
				} else {
//					detail.securityGroupIds = new ArrayList<>();
//					detail.securityGroupIds.add("hinemos-agent-nonVPC");
				}
				
				ObjectMapper om = new ObjectMapper();
				ObjectWriter dw = om.writerWithType(CreateInstanceDialog.InstanceDetail.class);
				String detailJson;
				try {
					detailJson = dw.writeValueAsString(detail);
				} catch (Exception e) {
					throw new CloudModelException(e);
				}

				if(m.isTemplateUsed()){
					CreateCloudInstanceByTemplateRequest request = new CreateCloudInstanceByTemplateRequest();

					request.setZone(m.getZone());
					
					request.setTemplateId(m.getTemplateId());
					
					request.setFlavor(m.getFlavor());
					
					request.setFacilityId(m.getFacilityId());
					request.setFacilityName(m.getFacilityName());
					request.setDescription(m.getDescription());
					request.setNodeName(m.getNodeName());
					request.setReplaceNode(m.getReplaceNode());
					
					request.setInstanceDetail(detailJson);
					
					request.setTags(m.getTags());
					request.setArguments(m.getArguments());

					instance = r.getAccountResource().getRegion(m.getRegionName()).getCloudService(ICloudComputeService.class).getInstanceManager().createCloudInstanceByTemplate(request);
				} else {
					CreateCloudInstanceRequest request = new CreateCloudInstanceRequest();

					request.setZone(m.getZone());
					if(m.getImage() != null){
						request.setImageId(m.getImage().getImageId());
					}
					request.setFlavor(m.getFlavor());
					
					request.setFacilityId(m.getFacilityId());
					request.setFacilityName(m.getFacilityName());
					request.setDescription(m.getDescription());
					request.setNodeName(m.getNodeName());
					request.setReplaceNode(m.getReplaceNode());
					
					request.setInstanceDetail(detailJson);
					
					request.setTags(m.getTags());
					
					instance = r.getAccountResource().getRegion(m.getRegionName()).getCloudService(ICloudComputeService.class).getInstanceManager().createCloudInstance(request);
				}

				// 成功報告ダイアログを生成
				MessageDialog.openInformation(
					null,
					Messages.getString("successful"),
					msgCreateInstanceResult);
				
				view.getSite().getShell().getDisplay().asyncExec(new Runnable() {
					@Override
					public void run() {
						InstanceMonitorService.getInstanceMonitorService().startMonitor(instance, StateType.running, StateType.terminated, StateType.stopped);
						com.clustercontrol.action.FacilityTree.refresh();
					}
				});
			}
			else {
				return;
			}
		}
	}
	
	@Override
	public void selectionChanged(IAction action, ISelection selection) {
		action.setEnabled(true);
	}
}