/*

Copyright (C) 2012 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.snmptrap.service;

import java.net.InetAddress;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.calendar.session.CalendarControllerBean;
import com.clustercontrol.commons.util.HinemosProperties;
import com.clustercontrol.commons.util.JpaTransactionManager;
import com.clustercontrol.commons.util.MonitoredThreadPoolExecutor;
import com.clustercontrol.fault.CalendarNotFound;
import com.clustercontrol.monitor.run.bean.MonitorInfo;
import com.clustercontrol.monitor.run.bean.MonitorTrapValueInfo;
import com.clustercontrol.monitor.session.MonitorSettingControllerBean;
import com.clustercontrol.repository.bean.FacilityTreeAttributeConstant;
import com.clustercontrol.repository.session.RepositoryControllerBean;
import com.clustercontrol.snmptrap.bean.MonitorTrapConstant;
import com.clustercontrol.snmptrap.bean.SnmpTrapMasterInfo;
import com.clustercontrol.snmptrap.bean.SnmpTrapV1;
import com.clustercontrol.snmptrap.bean.SnmpVarbind;
import com.clustercontrol.snmptrap.bean.TrapCheckInfo;
import com.clustercontrol.snmptrap.factory.SelectMibMaster;
import com.clustercontrol.snmptrap.model.SnmpTrapMstEntityPK;
import com.clustercontrol.snmptrap.util.SnmpTrapHandler;
import com.clustercontrol.snmptrap.util.SnmpTrapNotifier;
import com.clustercontrol.snmptrap.util.SnmpTrapReceiver;
import com.clustercontrol.util.Messages;
import com.clustercontrol.util.apllog.AplLogger;

public class SnmpTrapMonitor implements SnmpTrapHandler {

	private static final Log log = LogFactory.getLog(SnmpTrapMonitor.class);

	private static final long _shutdownTimeoutMsec;
	private static final int _statsInterval;
	private static final boolean _orgMessageVarbind;
	private static final boolean _orgMessageCommunity;

	private ExecutorService _executor;
	private SnmpTrapNotifier _notifier = new SnmpTrapNotifier();

	private final Charset _defaultCharset;

	private final int _threadSize;
	private final int _queueSize;

	private long receivedCount = 0;
	private long discardedCount = 0;
	private long notifiedCount = 0;

	private SnmpTrapSuppressor _suppressor = new SnmpTrapSuppressor();

	static {
		_shutdownTimeoutMsec = Long.parseLong(HinemosProperties.getProperty("monitor.snmptrap.shutdown.timeoue", "60000"));
		_statsInterval = Integer.parseInt(HinemosProperties.getProperty("monitor.snmptrap.stats.interval", "100"));
		_orgMessageVarbind = Boolean.parseBoolean(HinemosProperties.getProperty("monitor.snmptrap.org.message.varbind", "true"));
		_orgMessageCommunity = Boolean.parseBoolean(HinemosProperties.getProperty("monitor.snmptrap.org.message.community", "true"));

		log.info("shutdownTimeout=" + _shutdownTimeoutMsec +
				", statsInterval=" + _statsInterval +
				", orgMessageVarbind=" + _orgMessageVarbind +
				", orgMessageCommunity=" + _orgMessageCommunity);
	}

	public SnmpTrapMonitor(Charset defaultCharset, int threadSize, int queueSize) {
		_defaultCharset = defaultCharset;
		_threadSize = threadSize;
		_queueSize = queueSize;
	}

	@Override
	public void snmptrapReceived(SnmpTrapV1 snmptrap) {
		countupReceived();
		_executor.execute(new SnmpTrapFilterTask(snmptrap));
	}

	public synchronized void snmptrapReceived(String receiverId, SnmpTrapV1 snmptrap) {
		if (! _suppressor.isSuppress(receiverId, snmptrap)) {
			countupReceived();
			try {
				new SnmpTrapFilterTask(snmptrap).run();
			} catch (Exception e) {
				log.warn("snmptrap monitor failure. (snmptrap = " + snmptrap + ")", e);
			}
		}
	}

	private synchronized void countupReceived() {
		receivedCount = receivedCount >= Long.MAX_VALUE ? 0 : receivedCount + 1;
		if (receivedCount % _statsInterval == 0) {
			log.info("The number of snmptrap (received) : " + receivedCount);
		}
	}

	private synchronized void countupDiscarded() {
		discardedCount = discardedCount >= Long.MAX_VALUE ? 0 : discardedCount + 1;
		if (discardedCount % _statsInterval == 0) {
			log.info("The number of snmptrap (discarded) : " + discardedCount);
		}
	}

	private synchronized void countupNotified() {
		notifiedCount = notifiedCount >= Long.MAX_VALUE ? 0 : notifiedCount + 1;
		if (notifiedCount % _statsInterval == 0) {
			log.info("The number of snmptrap (notified) : " + notifiedCount);
		}
	}

	public long getReceivedCount() {
		return receivedCount;
	}

	public long getDiscardedCount() {
		return discardedCount;
	}

	public long getNotifiedCount() {
		return notifiedCount;
	}

	public int getQueuedCount() {
		return ((ThreadPoolExecutor)_executor).getQueue().size();
	}

	@Override
	public synchronized void start() {
		_executor = new MonitoredThreadPoolExecutor(_threadSize, _threadSize,
				0L, TimeUnit.MICROSECONDS, new LinkedBlockingQueue<Runnable>(_queueSize),
				new ThreadFactory() {
			private volatile int _count = 0;
			@Override
			public Thread newThread(Runnable r) {
				return new Thread(r, "SnmpTrapFilter-" + _count++);
			}
		}, new SnmpTrapRejectionHandler());
	}

	private class SnmpTrapRejectionHandler extends ThreadPoolExecutor.DiscardPolicy {

		@Override
		public void rejectedExecution(Runnable r, ThreadPoolExecutor e) {
			if (r instanceof SnmpTrapRejectionHandler) {
				countupDiscarded();
				log.warn("too many snmptrap. snmptrap discarded : " + r);
			}
		}
	}

	@Override
	public void shutdown() {
		_executor.shutdown();
		try {
			if (! _executor.awaitTermination(_shutdownTimeoutMsec, TimeUnit.MILLISECONDS)) {
				List<Runnable> remained = _executor.shutdownNow();
				if (remained != null) {
					log.info("shutdown timeout. runnable remained. (size = " + remained.size() + ")");
				}
			}
		} catch (InterruptedException e) {
			_executor.shutdownNow();
		}
	}

	/**
	 * フィルタリング処理を実装したタスククラス
	 */
	private class SnmpTrapFilterTask implements Runnable {

		public final SnmpTrapV1 snmptrap;

		public SnmpTrapFilterTask(SnmpTrapV1 snmptrap) {
			this.snmptrap = snmptrap;
		}

		@Override
		public void run() {

			JpaTransactionManager tm = null;
			boolean warn = true;

			try {
				tm = new JpaTransactionManager();
				tm.begin();

				RepositoryControllerBean repositoryCtrl = new RepositoryControllerBean();

				List<MonitorInfo> monitorList = new MonitorSettingControllerBean().getTrapList();

				if (monitorList == null) {
					if (log.isDebugEnabled()) {
						log.debug("snmptrap monitor not found. skip filtering. [" + snmptrap + "]");
					}
					return;
				}

				for (MonitorInfo monitor : monitorList) {
					TrapCheckInfo check = monitor.getTrapCheckInfo();

					// 無効となっている設定はスキップする
					if (monitor.getMonitorFlg() == ValidConstant.TYPE_INVALID) {
						if (log.isDebugEnabled()) {
							log.debug("snmptrap monitor " + monitor.getMonitorId()
									+ " is not enabled, skip filtering. [" + snmptrap + "]");
						}
						continue;
					}

					// カレンダの有効期間外の場合、スキップする
					String calendarId = monitor.getCalendarId();
					if (calendarId != null && ! "".equals(calendarId)) {
						try {
							if (! new CalendarControllerBean().isRun(calendarId, snmptrap.receivedTime)) {
								if (log.isDebugEnabled()) {
									log.debug("calendar " + calendarId + " is not enabled term. [" + snmptrap + "]");
								}
								continue;
							}
						} catch (CalendarNotFound e) {
							// カレンダが未定義の場合は、スキップせずに継続する（予期せぬロストの回避）
							log.info("calendar " + calendarId
									+ " is not found, skip calendar check. [" + snmptrap + "]");
						}
					}

					// コミュニティのチェック
					String community = check.getCommunityName();
					if (check.getCommunityCheck() != MonitorTrapConstant.COMMUNITY_CHECK_OFF) {
						if (! community.equals(snmptrap.community)) {
							if (log.isDebugEnabled()) {
								log.debug("community " + community + " is not matched. [" + snmptrap + "]");
							}
							continue;
						}
					}

					// varbindの文字列変換
					Charset charset = _defaultCharset;
					if (check.getCharsetConvert() == MonitorTrapConstant.CHARSET_CONVERT_ON) {
						if (check.getCharsetName() != null) {
							if (Charset.isSupported(check.getCharsetName())) {
								charset = Charset.forName(check.getCharsetName());
							} else {
								log.warn("not supported charset : " + check.getCharsetName());
							}
						}
					}

					List<SnmpVarbind> varbinds = snmptrap.getVarbinds();
					String[] varbind = null;
					String varbindStr = "";
					if(varbinds != null) {
						varbind = new String[varbinds.size()];
						for (int i = 0; i < varbinds.size(); i++) {
							switch (varbinds.get(i).type) {
							case OctetString :
								varbind[i] = new String(varbinds.get(i).object, charset);
								break;
							case Opaque :
								varbind[i] = new String(varbinds.get(i).object, charset);
								break;
							default :
								varbind[i] = new String(varbinds.get(i).object);
							}
						}
						if (_orgMessageVarbind) {
							for (String value : varbind) {
								if (varbindStr.length() == 0) {
									varbindStr = "VarBind=" + value;
								} else {
									varbindStr += ", " + value;
								}
							}
							varbindStr += " \n";
						}
					}
					String communityStr = "";
					if (_orgMessageCommunity) {
						communityStr = Messages.getString("CommunityName") + "=" + snmptrap.community + " \n";
					}

					// snmptrapのagent address(joesnmpではpacketのsrc address)から該当するファシリティ一覧を取得
					List<String> matchedFacilities = null;
					matchedFacilities = repositoryCtrl.getFacilityIdByIpAddress(InetAddress.getByName(snmptrap.agentAddr));
					if (matchedFacilities == null || matchedFacilities.size() == 0) {
						matchedFacilities = new ArrayList<String>();
						matchedFacilities.add(FacilityTreeAttributeConstant.UNREGISTEREFD_SCOPE);
					}
					if (log.isDebugEnabled()) {
						log.debug("matched facilities : " + matchedFacilities + " [" + snmptrap + "]");
					}

					// 監視対象のファシリティID一覧を取得する
					List<String> targetFacilities = null;
					if (FacilityTreeAttributeConstant.UNREGISTEREFD_SCOPE.equals(monitor.getFacilityId())) {
						targetFacilities = new ArrayList<String>();
						targetFacilities.add(FacilityTreeAttributeConstant.UNREGISTEREFD_SCOPE);
					} else {
						targetFacilities = repositoryCtrl.getExecTargetFacilityIdList(monitor.getFacilityId(), monitor.getOwnerRoleId());
					}
					if (log.isDebugEnabled()) {
						log.debug("target facilities : " + targetFacilities + " [" + snmptrap + "]");
					}

					// 通知対象のファシリティID一覧を絞り込む
					List<String> notifyFacilities = new ArrayList<String>(matchedFacilities);
					notifyFacilities.retainAll(targetFacilities);
					if (notifyFacilities.size() == 0) {
						if (log.isDebugEnabled()) {
							log.debug("notification facilities not found [" + snmptrap + "]");
						}
						continue;
					}

					// OIDのチェック
					String facilityPath = "";
					SelectMibMaster mibSelector = new SelectMibMaster();
					if (check.getCheckMode() == MonitorTrapConstant.ALL_OID) {
						// マスタ登録済みの全てのOIDが対象の場合
						SnmpTrapMasterInfo mibMst = mibSelector.findMasterInfo(snmptrap.enterpriseId, snmptrap.genericId, snmptrap.specificId);

						if (mibMst != null && mibMst.getMib() != null) {

							String msg = mibMst.getLogmsg();
							String msgOrig = communityStr + varbindStr + "Message=" + mibMst.getDescr();

							msg = getBindedString(msg, varbind);
							msgOrig = getBindedString(msgOrig, varbind);

							msgOrig = "OID=" + snmptrap.enterpriseId + "\nTrapName=" + mibMst.getUei() + "\n" + msgOrig;

							int priority = mibMst.getPriority();

							for (String facilityId : notifyFacilities) {
								if (FacilityTreeAttributeConstant.UNREGISTEREFD_SCOPE.equals(facilityId)) {
									facilityPath = snmptrap.agentAddr;
								} else {
									facilityPath = repositoryCtrl.getFacilityPath(facilityId, null);
								}

								_notifier.put(snmptrap, monitor.getMonitorId(), monitor.getNotifyGroupId(),
										priority, facilityId, facilityPath, monitor.getApplication(), msg, msgOrig);
								countupNotified();
							}

						} else {
							//一般トラップが1件もマッチしない場合は、再検索
							if (snmptrap.genericId != 6) {
								//SNMPv1からSNMPv2形式の変換において、genericId=6以外(標準トラップ)の場合はOIDはTrapSnmp.GENERIC_TRAPSに従う
								mibMst = mibSelector.findMasterInfo(SnmpTrapReceiver.genericTraps.get(snmptrap.genericId).toString(),
										snmptrap.genericId, snmptrap.specificId);

								if (mibMst != null && mibMst.getMib() != null) {
									String msg = mibMst.getLogmsg();
									String msgOrig = communityStr + varbindStr + "Message=" +  mibMst.getDescr();

									msg = getBindedString(msg, varbind);
									msgOrig = getBindedString(msgOrig, varbind);

									msgOrig = "OID=" + snmptrap.enterpriseId + "\nTrapName=" + mibMst.getUei() + "\n" + msgOrig;

									int priority = mibMst.getPriority();

									for (String facilityId : notifyFacilities) {
										if (FacilityTreeAttributeConstant.UNREGISTEREFD_SCOPE.equals(facilityId)) {
											facilityPath = snmptrap.agentAddr;
										} else {
											facilityPath = repositoryCtrl.getFacilityPath(facilityId, null);
										}

										_notifier.put(snmptrap, monitor.getMonitorId(), monitor.getNotifyGroupId(),
												priority, facilityId, facilityPath, monitor.getApplication(), msg, msgOrig);
										countupNotified();
									}
								}
							}
						}


					} else if (check.getCheckMode() == MonitorTrapConstant.UNREGISTERED_OID) {
						// マスタ未登録てのOIDが対象の場合

						if (snmptrap.genericId == 6) {
							SnmpTrapMasterInfo mibMst = mibSelector.findMasterInfo(snmptrap.enterpriseId, snmptrap.genericId, snmptrap.specificId);

							if (mibMst != null && mibMst.getMib() == null) {
								String msg = communityStr + varbindStr
										+ ", oid : " + snmptrap.enterpriseId
										+ ", specificId : " + snmptrap.specificId
										+ ", generic_id : " + snmptrap.genericId;
								String msgOrig = msg;

								int priority = PriorityConstant.TYPE_UNKNOWN;

								for (String facilityId : notifyFacilities) {
									if (FacilityTreeAttributeConstant.UNREGISTEREFD_SCOPE.equals(facilityId)) {
										facilityPath = snmptrap.agentAddr;
									} else {
										facilityPath = repositoryCtrl.getFacilityPath(facilityId, null);
									}

									_notifier.put(snmptrap, monitor.getMonitorId(), monitor.getNotifyGroupId(),
											priority, facilityId, facilityPath, monitor.getApplication(), msg, msgOrig);
									countupNotified();
								}
							}

						}


					} else {
						// ユーザ指定のOIDが対象の場合

						List<MonitorTrapValueInfo> oidList = monitor.getTrapValueInfo();
						int matched = 0;

						for (MonitorTrapValueInfo oid : oidList) {
							if (oid.isValidFlg()
									&& snmptrap.enterpriseId.equals(oid.getTrapOid())
									&& snmptrap.genericId == oid.getGenericId()
									&& snmptrap.specificId == oid.getSpecificId()) {
								String mib = oid.getMib();

								SnmpTrapMstEntityPK pk
								= new SnmpTrapMstEntityPK(mib,
										snmptrap.enterpriseId,
										snmptrap.genericId,
										snmptrap.specificId);
								SnmpTrapMasterInfo mibMst = mibSelector.getMasterInfo(pk);

								String msg = oid.getLogmsg();
								String msgOrig = communityStr + varbindStr + oid.getDescr();

								msg = getBindedString(msg, varbind);
								msgOrig = getBindedString(msgOrig, varbind);

								msgOrig = "OID=" + snmptrap.enterpriseId + "\nTrapName=" + mibMst.getUei() + "\n" + msgOrig;

								int priority = oid.getPriority();

								for (String facilityId : notifyFacilities) {
									if (FacilityTreeAttributeConstant.UNREGISTEREFD_SCOPE.equals(facilityId)) {
										facilityPath = snmptrap.agentAddr;
									} else {
										facilityPath = repositoryCtrl.getFacilityPath(facilityId, null);
									}

									_notifier.put(snmptrap, monitor.getMonitorId(), monitor.getNotifyGroupId(),
											priority, facilityId, facilityPath, monitor.getApplication(), msg, msgOrig);
									countupNotified();
								}

								matched++;
								break;
							}
						}

						if (matched == 0 && snmptrap.genericId != 6) {
							for (MonitorTrapValueInfo oid : oidList) {
								if (oid.isValidFlg() && oid.getTrapOid().equals(SnmpTrapReceiver.genericTraps.get(snmptrap.genericId).toString())) {

									String mib = oid.getMib();

									SnmpTrapMstEntityPK pk = new SnmpTrapMstEntityPK(
											mib,
											SnmpTrapReceiver.genericTraps.get(snmptrap.genericId).toString(),
											snmptrap.genericId,
											snmptrap.specificId);
									SnmpTrapMasterInfo mibMst = mibSelector.getMasterInfo(pk);

									String msg = oid.getLogmsg();
									String msgOrig = communityStr + varbindStr + oid.getDescr();

									msg = getBindedString(msg, varbind);
									msgOrig = getBindedString(msgOrig, varbind);

									msgOrig = "OID=" + snmptrap.enterpriseId + "\nTrapName=" + mibMst.getUei() + "\n" + msgOrig;

									int priority = oid.getPriority();

									for (String facilityId : notifyFacilities) {
										if (FacilityTreeAttributeConstant.UNREGISTEREFD_SCOPE.equals(facilityId)) {
											facilityPath = snmptrap.agentAddr;
										} else {
											facilityPath = repositoryCtrl.getFacilityPath(facilityId, null);
										}

										_notifier.put(snmptrap, monitor.getMonitorId(), monitor.getNotifyGroupId(),
												priority, facilityId, facilityPath, monitor.getApplication(), msg, msgOrig);
										countupNotified();
									}
								}
							}
						}
					}

				}

				tm.commit();
				warn = false;
			} catch (Exception e) {
				log.warn("unexpected internal error. [" + snmptrap + "] : "
						+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			} finally {
				if (tm != null) {
					tm.close();
				}
			}
			
			if(warn) {
				// Internal Event
				AplLogger apllog = new AplLogger("TRAP", "trap");
				String[] args = {snmptrap.enterpriseId, String.valueOf(snmptrap.genericId), String.valueOf(snmptrap.specificId)};
				apllog.put("SYS", "009", args);
			}
		}
	}

	private String getBindedString(String str, String[] varbinds) {
		if (str == null) {
			return "";
		}
		if (varbinds == null) {
			return str;
		}

		for (int i = 0; i < varbinds.length; i++) {
			if (log.isDebugEnabled()) {
				log.debug("binding : " + str + "  " + "%parm[#" + (i + 1) + "]% to " + varbinds[i] + "]");
			}
			str = str.replace("%parm[#" + (i + 1) + "]%", varbinds[i]);
			if (log.isDebugEnabled()) {
				log.debug("binded : " + str);
			}
		}

		return str;
	}
}
