/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.repository.factory;

import java.util.ArrayList;
import java.util.concurrent.ConcurrentHashMap;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.fault.FacilityNotFound;
import com.clustercontrol.repository.bean.NodeCpuInfo;
import com.clustercontrol.repository.bean.NodeDeviceInfo;
import com.clustercontrol.repository.bean.NodeDiskInfo;
import com.clustercontrol.repository.bean.NodeFilesystemInfo;
import com.clustercontrol.repository.bean.NodeHostnameInfo;
import com.clustercontrol.repository.bean.NodeInfo;
import com.clustercontrol.repository.bean.NodeMemoryInfo;
import com.clustercontrol.repository.bean.NodeNetworkInterfaceInfo;
import com.clustercontrol.repository.bean.NodeNoteInfo;
import com.clustercontrol.repository.bean.NodeVariableInfo;
import com.clustercontrol.repository.model.FacilityEntity;
import com.clustercontrol.repository.model.NodeCpuEntity;
import com.clustercontrol.repository.model.NodeDeviceEntity;
import com.clustercontrol.repository.model.NodeDiskEntity;
import com.clustercontrol.repository.model.NodeEntity;
import com.clustercontrol.repository.model.NodeFilesystemEntity;
import com.clustercontrol.repository.model.NodeHostnameEntity;
import com.clustercontrol.repository.model.NodeMemoryEntity;
import com.clustercontrol.repository.model.NodeNetworkInterfaceEntity;
import com.clustercontrol.repository.model.NodeNoteEntity;
import com.clustercontrol.repository.model.NodeVariableEntity;
import com.clustercontrol.repository.util.FacilityUtil;
import com.clustercontrol.repository.util.QueryUtil;

/**
 * ノード用プロパティを作成するクラス<BR>
 * 
 * @version 1.0.0
 * @since 1.0.0
 */
public class NodeProperty {

	private static Log m_log = LogFactory.getLog(NodeProperty.class);

	/** ----- 初期値キャッシュ ----- */

	private static Object nodeLock = new Object();
	private static ConcurrentHashMap<String, NodeInfo> nodeCache =
			new ConcurrentHashMap<String, NodeInfo>();

	public static void removeNode (String facilityId) {
		m_log.info("remove NodeCache : " + facilityId);
		synchronized(nodeLock) {
			nodeCache.remove(facilityId);
		}
	}

	public static void clearCache () {
		m_log.info("clear NodeCache");
		synchronized(nodeLock) {
			nodeCache.clear();
		}
	}

	/**
	 * 与えられたファシリティIDに基づき、該当するノード情報を返す。<BR>
	 * 
	 * @param facilityId ファシリティID
	 * @param mode ノード情報扱い種別（参照、追加、変更）
	 * @return ノード情報
	 * @throws FacilityNotFound
	 */
	public static NodeInfo getProperty(String facilityId) throws FacilityNotFound {

		m_log.debug("getProperty() : facilityId = " + facilityId);

		/** ローカル変数 */
		NodeInfo property = null;
		NodeEntity node = null;
		FacilityEntity facility = null;

		/** メイン処理 */
		if (facilityId == null || facilityId.compareTo("") == 0) {
			return new NodeInfo();
		}

		synchronized(nodeLock) {
			property = nodeCache.get(facilityId);
			if (property != null) {
				return property;
			} else {
				property = new NodeInfo();
			}

			facility = QueryUtil.getFacilityPK_NONE(facilityId);
			// ファシリティがノードかどうかを確認する
			if (!FacilityUtil.isNode(facility)) {
				FacilityNotFound e = new FacilityNotFound("this facility is not a node. (facilityId = " + facilityId + ")");
				m_log.info("getProperty() : "
						+ e.getClass().getSimpleName() + ", " + e.getMessage());
				throw e;
			}
			node = facility.getNodeEntity();

			// ----- ファシリティ関連 -----
			// ファシリティID
			property.setFacilityId(facility.getFacilityId());
			// ファシリティ名
			property.setFacilityName(facility.getFacilityName());
			// 説明
			property.setDescription(facility.getDescription());
			// 有効/無効
			property.setValid(FacilityUtil.isValid(facility));
			// アイコン名
			property.setIconImage(facility.getIconImage());
			// オーナーロールID
			property.setOwnerRoleId(facility.getOwnerRoleId());
			// 登録ユーザID
			property.setCreateUserId(facility.getCreateUserId());
			// 登録日時
			if (facility.getCreateDatetime() != null
					&& facility.getCreateDatetime().getTime() != 0) {
				property.setCreateDatetime(facility.getCreateDatetime().getTime());
			}
			// 最終更新ユーザID
			property.setModifyUserId(facility.getModifyUserId());
			// 最終更新日時
			if (facility.getModifyDatetime() != null
					&& facility.getModifyDatetime().getTime() != 0) {
				property.setModifyDatetime(facility.getModifyDatetime().getTime());
			}

			// ----- ノード基本情報 -----

			// ----- HW -----
			// プラットフォーム
			property.setPlatformFamily(node.getPlatformFamily());
			// サブプラットフォーム
			property.setSubPlatformFamily(node.getSubPlatformFamily());
			// H/Wタイプ
			property.setHardwareType(node.getHardwareType());
			// 画面アイコンイメージ
			property.setIconImage(facility.getIconImage());

			// ----- IPアドレス関連 -----
			// IPバージョン
			if (node.getIpAddressVersion() != -1) {
				property.setIpAddressVersion(node.getIpAddressVersion());
			}
			// IPアドレスV4
			property.setIpAddressV4(node.getIpAddressV4());
			// IPアドレスV6
			property.setIpAddressV6(node.getIpAddressV6());
			// ホスト名
			ArrayList<NodeHostnameInfo> hostnameList = new ArrayList<NodeHostnameInfo>();
			if (node.getNodeHostnameEntities() != null) {
				for (NodeHostnameEntity nodeHostname : node.getNodeHostnameEntities()) {
					// ホスト名
					hostnameList.add(new NodeHostnameInfo(nodeHostname.getId().getHostname()));
				}
			}
			property.setNodeHostnameInfo(hostnameList);

			// ----- OS関連 -----
			// ノード名
			property.setNodeName(node.getNodeName());
			// OS名
			property.setOsName(node.getOsName());
			// OSリリース
			property.setOsRelease(node.getOsRelease());
			// OSバージョン
			property.setOsVersion(node.getOsVersion());
			// 文字セット
			property.setCharacterSet(node.getCharacterSet());

			// ----- Hinemosエージェント関連 -----
			// 即時反映用ポート番号
			if (node.getAgentAwakePort() != -1) {
				property.setAgentAwakePort(node.getAgentAwakePort());
			}

			// ----- ジョブ -----
			// ジョブ優先度
			if (node.getJobPriority() != -1) {
				property.setJobPriority(node.getJobPriority());
			}
			// ジョブ多重度
			if (node.getJobMultiplicity() != -1) {
				property.setJobMultiplicity(node.getJobMultiplicity());
			}


			// ----- サービス -----

			// ----- SNMP関連 -----
			// SNMPポート番号
			if (node.getSnmpPort() != -1) {
				property.setSnmpPort(node.getSnmpPort());
			}
			// SNMPコミュニティ名
			property.setSnmpCommunity(node.getSnmpCommunity());
			// SNMPバージョン
			property.setSnmpVersion(node.getSnmpVersion());
			// SNMPタイムアウト
			if (node.getSnmpTimeout() != -1) {
				property.setSnmpTimeout(node.getSnmpTimeout());
			}
			// SNMPリトライ回数
			if (node.getSnmpRetryCount() != -1) {
				property.setSnmpRetryCount(node.getSnmpRetryCount());
			}

			// ----- WBEM関連 -----
			// WBEM接続ポート番号
			if (node.getWbemPort() != -1) {
				property.setWbemPort(node.getWbemPort());
			}
			// WBEM接続ユーザ
			property.setWbemUser(node.getWbemUser());
			// WBEM接続ユーザパスワード
			property.setWbemUserPassword(node.getWbemUserPassword());
			// WBEM接続プロトコル
			property.setWbemProtocol(node.getWbemProtocol());
			// WBEM接続タイムアウト
			if (node.getWbemTimeout() != -1) {
				property.setWbemTimeout(node.getWbemTimeout());
			}
			// WBEM接続リトライ回数
			if (node.getWbemRetryCount() != -1) {
				property.setWbemRetryCount(node.getWbemRetryCount());
			}

			// ----- IPMI関連 -----
			// IPMI接続アドレス
			property.setIpmiIpAddress(node.getIpmiIpAddress());
			// IPMIポート番号
			if (node.getIpmiPort() != -1) {
				property.setIpmiPort(node.getIpmiPort());
			}
			// IPMI接続ユーザ
			property.setIpmiUser(node.getIpmiUser());
			// IPMI接続ユーザパスワード
			property.setIpmiUserPassword(node.getIpmiUserPassword());
			// IPMI接続タイムアウト
			if (node.getIpmiTimeout() != -1) {
				property.setIpmiTimeout(node.getIpmiTimeout());
			}
			// IPMI接続リトライ回数
			if (node.getIpmiRetryCount() != -1) {
				property.setIpmiRetries(node.getIpmiRetryCount());
			}
			// IPMI接続プロトコル
			property.setIpmiProtocol(node.getIpmiProtocol());
			// IPMI特権レベル
			property.setIpmiLevel(node.getIpmiLevel());

			// ----- WinRM関連 -----
			// WinRM接続ユーザ
			property.setWinrmUser(node.getWinrmUser());
			// WinRM接続ユーザパスワード
			property.setWinrmUserPassword(node.getWinrmUserPassword());
			// WinRMバージョン
			property.setWinrmVersion(node.getWinrmVersion());
			// WinRM接続ポート番号
			if (node.getWinrmPort() != -1) {
				property.setWinrmPort(node.getWinrmPort());
			}
			// WinRM接続プロトコル
			property.setWinrmProtocol(node.getWinrmProtocol());
			// WinRM接続タイムアウト
			if (node.getWinrmTimeout() != -1) {
				property.setWinrmTimeout(node.getWinrmTimeout());
			}
			// WinRM接続リトライ回数
			if (node.getWinrmRetryCount() != -1) {
				property.setWinrmRetries(node.getWinrmRetryCount());
			}

			// ----- デバイス関連 -----

			// ----- 汎用デバイス情報 -----
			ArrayList<NodeDeviceInfo> deviceList = new ArrayList<NodeDeviceInfo>();
			if (node.getNodeDeviceEntities() != null) {
				for (NodeDeviceEntity nodeDevice : node.getNodeDeviceEntities()) {
					NodeDeviceInfo device = new NodeDeviceInfo();

					// デバイス種別
					device.setDeviceType(nodeDevice.getId().getDeviceType());
					// デバイス表示名
					device.setDeviceDisplayName(nodeDevice.getDeviceDisplayName());
					// デバイスINDEX
					device.setDeviceIndex(nodeDevice.getId().getDeviceIndex());
					// デバイス名
					device.setDeviceName(nodeDevice.getId().getDeviceName());
					// デバイスサイズ
					device.setDeviceSize(nodeDevice.getDeviceSize());
					// デバイスサイズ単位
					device.setDeviceSizeUnit(nodeDevice.getDeviceSizeUnit());
					// 説明
					device.setDeviceDescription(nodeDevice.getDeviceDescription());
					deviceList.add(device);
				}
			}
			property.setNodeDeviceInfo(deviceList);

			// ----- CPUデバイス情報 -----
			ArrayList<NodeCpuInfo> cpuList = new ArrayList<NodeCpuInfo>();
			if (node.getNodeCpuEntities() != null) {
				for(NodeCpuEntity nodeCpu : node.getNodeCpuEntities()){
					NodeCpuInfo cpu = new NodeCpuInfo();

					// デバイス種別
					cpu.setDeviceType(nodeCpu.getId().getDeviceType());
					// デバイス表示名
					cpu.setDeviceDisplayName(nodeCpu.getDeviceDisplayName());
					// デバイスINDEX
					cpu.setDeviceIndex(nodeCpu.getId().getDeviceIndex());
					// デバイス名
					cpu.setDeviceName(nodeCpu.getId().getDeviceName());
					// デバイスサイズ
					cpu.setDeviceSize(nodeCpu.getDeviceSize());
					// デバイスサイズ単位
					cpu.setDeviceSizeUnit(nodeCpu.getDeviceSizeUnit());
					// 説明
					cpu.setDeviceDescription(nodeCpu.getDeviceDescription());
					cpuList.add(cpu);
				}
			}
			property.setNodeCpuInfo(cpuList);

			// ----- MEMデバイス情報 -----
			ArrayList<NodeMemoryInfo> memList = new ArrayList<NodeMemoryInfo>();
			if (node.getNodeMemoryEntities() != null) {
				for(NodeMemoryEntity nodeMem : node.getNodeMemoryEntities()){
					NodeMemoryInfo mem = new NodeMemoryInfo();

					// デバイス種別
					mem.setDeviceType(nodeMem.getId().getDeviceType());
					// デバイス表示名
					mem.setDeviceDisplayName(nodeMem.getDeviceDisplayName());
					// デバイスINDEX
					mem.setDeviceIndex(nodeMem.getId().getDeviceIndex());
					// デバイス名
					mem.setDeviceName(nodeMem.getId().getDeviceName());
					// デバイスサイズ
					mem.setDeviceSize(nodeMem.getDeviceSize());
					// デバイスサイズ単位
					mem.setDeviceSizeUnit(nodeMem.getDeviceSizeUnit());
					// 説明
					mem.setDeviceDescription(nodeMem.getDeviceDescription());
					memList.add(mem);
				}
			}
			property.setNodeMemoryInfo(memList);

			// ----- NICデバイス情報 -----
			ArrayList<NodeNetworkInterfaceInfo> nicList = new ArrayList<NodeNetworkInterfaceInfo>();
			if (node.getNodeNetworkInterfaceEntities() != null) {
				for(NodeNetworkInterfaceEntity nodeNic : node.getNodeNetworkInterfaceEntities()){
					NodeNetworkInterfaceInfo nic = new NodeNetworkInterfaceInfo();

					// デバイス種別
					nic.setDeviceType(nodeNic.getId().getDeviceType());
					// デバイス表示名
					nic.setDeviceDisplayName(nodeNic.getDeviceDisplayName());
					// デバイスINDEX
					nic.setDeviceIndex(nodeNic.getId().getDeviceIndex());
					// デバイス名
					nic.setDeviceName(nodeNic.getId().getDeviceName());
					// デバイスサイズ
					nic.setDeviceSize(nodeNic.getDeviceSize());
					// デバイスサイズ単位
					nic.setDeviceSizeUnit(nodeNic.getDeviceSizeUnit());
					// 説明
					nic.setDeviceDescription(nodeNic.getDeviceDescription());
					// NIC ip
					nic.setNicIpAddress(nodeNic.getDeviceNicIpAddress());
					// NIC MAC
					nic.setNicMacAddress(nodeNic.getDeviceNicMacAddress());
					nicList.add(nic);
				}
			}
			property.setNodeNetworkInterfaceInfo(nicList);

			// ----- DISKデバイス情報 -----
			ArrayList<NodeDiskInfo> diskList = new ArrayList<NodeDiskInfo>();
			if (node.getNodeDiskEntities() != null) {
				for(NodeDiskEntity nodeDisk : node.getNodeDiskEntities()){
					NodeDiskInfo disk = new NodeDiskInfo();

					// デバイス種別
					disk.setDeviceType(nodeDisk.getId().getDeviceType());
					// デバイス表示名
					disk.setDeviceDisplayName(nodeDisk.getDeviceDisplayName());
					// デバイスINDEX
					disk.setDeviceIndex(nodeDisk.getId().getDeviceIndex());
					// デバイス名
					disk.setDeviceName(nodeDisk.getId().getDeviceName());
					// デバイスサイズ
					disk.setDeviceSize(nodeDisk.getDeviceSize());
					// デバイスサイズ単位
					disk.setDeviceSizeUnit(nodeDisk.getDeviceSizeUnit());
					// 説明
					disk.setDeviceDescription(nodeDisk.getDeviceDescription());
					// 回転数
					disk.setDiskRpm(nodeDisk.getDeviceDiskRpm());
					diskList.add(disk);
				}
			}
			property.setNodeDiskInfo(diskList);

			// ---- ファイルシステム情報 -----
			ArrayList<NodeFilesystemInfo> filesystemList = new ArrayList<NodeFilesystemInfo>();
			if (node.getNodeFilesystemEntities() != null) {
				for (NodeFilesystemEntity nodeFilesystem : node.getNodeFilesystemEntities()) {
					NodeFilesystemInfo filesystem = new NodeFilesystemInfo();

					// デバイス種別
					filesystem.setDeviceType(nodeFilesystem.getId().getDeviceType());
					// デバイス表示名
					filesystem.setDeviceDisplayName(nodeFilesystem.getDeviceDisplayName());
					// デバイスINDEX
					filesystem.setDeviceIndex(nodeFilesystem.getId().getDeviceIndex());
					// デバイス名
					filesystem.setDeviceName(nodeFilesystem.getId().getDeviceName());
					// デバイスサイズ
					filesystem.setDeviceSize(nodeFilesystem.getDeviceSize());
					// デバイスサイズ単位
					filesystem.setDeviceSizeUnit(nodeFilesystem.getDeviceSizeUnit());
					// 説明
					filesystem.setDeviceDescription(nodeFilesystem.getDeviceDescription());
					// ファイルシステム種別
					filesystem.setFilesystemType(nodeFilesystem.getDeviceFilesystemType());
					filesystemList.add(filesystem);
				}
			}
			property.setNodeFilesystemInfo(filesystemList);

			// ----- サーバ仮想化関連 -----
			// 仮想化ノード種別
			property.setVmNodeType(node.getVmNodeType());
			// 仮想マシン管理ノード
			property.setVmManagementNode(node.getVmManagementNode());
			// 仮想マシン名
			property.setVmName(node.getVmName());
			// 仮想マシンID
			property.setVmId(node.getVmId());
			// 仮想化ソフト接続ユーザ
			property.setVmUser(node.getVmUser());
			// 仮想化ソフト接続ユーザパスワード
			property.setVmUserPassword(node.getVmUserPassword());
			// 仮想化ソフト接続プロトコル
			property.setVmProtocol(node.getVmProtocol());

			// ----- ネットワーク仮想化関連 -----
			// 仮想スイッチ種別
			property.setvNetSwitchType(node.getVnetSwitchType());
			// 仮想スイッチホストノード
			property.setvNetHostNode(node.getVnetHostNode());
			// OpenFlow データパスID
			property.setOpenFlowDataPathId(node.getOpenflowDataPathId());
			// OpenFlow コントローラIPアドレス
			property.setOpenFlowCtrlIpAddress(node.getOpenflowCtrlIpAddress());

			// ----- クラウド関連 -----
			// クラウドノード種別
			property.setCloudNodeType(node.getCloudNodeType());
			// クラウドサービス
			property.setCloudService(node.getCloudService());
			// クラウドアカウントリソース
			property.setCloudAccountResouce(node.getCloudAccountResouce());
			// クラウドリソースタイプ
			property.setCloudResourceType(node.getCloudResourceType());
			// クラウドリソースID
			property.setCloudResourceId(node.getCloudResourceId());
			// クラウドリージョン
			property.setCloudRegion(node.getCloudRegion());
			// クラウドゾーン
			property.setCloudZone(node.getCloudZone());

			// ----- 	ノード変数 -----
			ArrayList<NodeVariableInfo> nodeVariableList = new ArrayList<NodeVariableInfo>();
			if (node.getNodeVariableEntities() != null) {
				for (NodeVariableEntity nodeVariable : node.getNodeVariableEntities()) {
					NodeVariableInfo variable = new NodeVariableInfo();

					// ノード変数名
					variable.setNodeVariableName(nodeVariable.getId().getNodeVariableName());
					// ノード変数値
					variable.setNodeVariableValue(nodeVariable.getNodeVariableValue());
					nodeVariableList.add(variable);
				}
			}
			property.setNodeVariableInfo(nodeVariableList);

			// ----- 管理情報関連 -----
			// 連絡先
			property.setContact(node.getContact());
			// 管理者
			property.setAdministrator(node.getAdministrator());


			// ----- 備考 -----
			ArrayList<NodeNoteInfo> noteList = new ArrayList<NodeNoteInfo>();
			if (node.getNodeNoteEntities() != null) {
				int cnt = 0;
				for (NodeNoteEntity nodeNote : node.getNodeNoteEntities()) {
					// 備考
					noteList.add(new NodeNoteInfo(cnt, nodeNote.getNote()));
					cnt++;
				}
			}
			property.setNodeNoteInfo(noteList);

			if (nodeCache.get(facilityId) == null) {
				m_log.debug("add NodeCache : " + facilityId);
				nodeCache.put(facilityId, property);
			}
		}
		return property;
	}
}
