/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.snmptrap.composite;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerFilter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.Text;

import com.clustercontrol.bean.DataRangeConstant;
import com.clustercontrol.composite.action.StringVerifyListener;
import com.clustercontrol.snmptrap.action.GetMaster;
import com.clustercontrol.snmptrap.action.GetMasterTableDefine;
import com.clustercontrol.snmptrap.util.MonitorSnmpTrapEndpointWrapper;
import com.clustercontrol.util.Messages;
import com.clustercontrol.viewer.CommonTableViewer;
import com.clustercontrol.ws.monitor.InvalidRole_Exception;
import com.clustercontrol.ws.monitor.MonitorTrapValueInfo;
import com.clustercontrol.ws.monitor.SnmpTrapMasterInfo;

/**
 * SNMPTRAP監視マスタ一覧コンポジットクラス<BR>
 * 
 * @version 4.0.0
 * @since 2.1.0
 */
public class MasterListComposite extends Composite {

	// ログ
	private static Log m_log = LogFactory.getLog( MasterListComposite.class );

	public static final int WIDTH_TITLE = 5;
	public static final int WIDTH_VALUE = 2;

	// ----- instance フィールド ----- //

	/** テーブルビューア */
	private CommonTableViewer tableViewer = null;

	/** マスタ情報 */
	private List<SnmpTrapMasterInfo> masterList = null;

	/** OID情報 */
	private List<MonitorTrapValueInfo> oidList = null;

	/** MIB */
	private Combo comboMib = null;

	/** フィルタ */
	private Text textFilter = null;

	/** フィルタクリアボタン */
	private Button buttonClearFilter = null;

	/** 表示ボタン */
	private Button buttonShowTable = null;

	// ----- コンストラクタ ----- //

	/**
	 * インスタンスを返します。
	 * 
	 * @param parent
	 *            親のコンポジット
	 * @param style
	 *            スタイル
	 */
	public MasterListComposite(Composite parent, int style) {
		super(parent, style);

		this.initialize();
	}

	// ----- instance メソッド ----- //

	/**
	 * コンポジットを生成・構築します。
	 */
	private void initialize() {
		GridLayout layout = new GridLayout(1, true);
		this.setLayout(layout);
		layout.marginHeight = 0;
		layout.marginWidth = 0;
		layout.numColumns = 15;

		//変数として利用されるラベル
		Label label = null;

		// グループ
		Group groupFilter = new Group(this, SWT.NONE);
		layout = new GridLayout(1, true);
		layout.marginWidth = 5;
		layout.marginHeight = 5;
		layout.numColumns = 15;
		groupFilter.setLayout(layout);
		GridData gridData = new GridData();
		gridData.horizontalSpan = 15;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		groupFilter.setLayoutData(gridData);
		groupFilter.setText(Messages.getString("filter") + " : ");

		/*
		 * MIB
		 */
		// ラベル
		label = new Label(groupFilter, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_TITLE;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("mib") + " : ");
		// コンボボックス
		this.comboMib = new Combo(groupFilter, SWT.DROP_DOWN | SWT.READ_ONLY);
		gridData = new GridData();
		gridData.horizontalSpan = 8;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.comboMib.setLayoutData(gridData);
		this.comboMib.setVisibleItemCount(10);
		// MIBリスト取得
		List<String> mibList = new GetMaster().getMibList();
		for(int i = 0; i < mibList.size(); i++){
			this.comboMib.add(mibList.get(i));
		}
		this.comboMib.select(0);
		this.comboMib.addSelectionListener(new SelectionListener(){
			/* (non-Javadoc)
			 * @see org.eclipse.swt.events.SelectionListener#widgetSelected(org.eclipse.swt.events.SelectionEvent)
			 */
			@Override
			public void widgetSelected(SelectionEvent e) {
				//表示処理
				update();
			}

			/* (non-Javadoc)
			 * @see org.eclipse.swt.events.SelectionListener#widgetDefaultSelected(org.eclipse.swt.events.SelectionEvent)
			 */
			@Override
			public void widgetDefaultSelected(SelectionEvent e) {

			}
		});
		// 空白
		label = new Label(groupFilter, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 2;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);

		/*
		 * フィルタ
		 */
		// ラベル
		label = new Label(groupFilter, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_TITLE;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("trap.name") + " : ");
		// テキスト
		this.textFilter = new Text(groupFilter, SWT.BORDER | SWT.LEFT);
		gridData = new GridData();
		gridData.horizontalSpan = 6;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.textFilter.setLayoutData(gridData);
		this.textFilter.addVerifyListener(
				new StringVerifyListener(DataRangeConstant.VARCHAR_256));
		// クリアボタン
		this.buttonClearFilter = new Button(groupFilter, SWT.NONE);
		this.buttonClearFilter.setText(Messages.getString("clear"));
		gridData = new GridData();
		gridData.horizontalSpan = 2;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.buttonClearFilter.setLayoutData(gridData);
		this.buttonClearFilter.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				textFilter.setText("");
			}
		});
		// 表示ボタン
		this.buttonShowTable = new Button(groupFilter, SWT.NONE);
		this.buttonShowTable.setText(Messages.getString("show"));
		gridData = new GridData();
		gridData.horizontalSpan = 2;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.buttonShowTable.setLayoutData(gridData);
		this.buttonShowTable.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				//表示処理
				update();
			}
		});

		Table table = new Table(this, SWT.MULTI | SWT.H_SCROLL | SWT.V_SCROLL
				| SWT.FULL_SELECTION | SWT.BORDER);
		table.setHeaderVisible(true);
		table.setLinesVisible(true);

		gridData = new GridData();
		gridData.horizontalSpan = 15;
		gridData.minimumHeight = 200;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.verticalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.grabExcessVerticalSpace = true;
		table.setLayoutData(gridData);

		// テーブルビューアの作成
		this.tableViewer = new CommonTableViewer(table);
		this.tableViewer.createTableColumn(GetMasterTableDefine.get(),
				GetMasterTableDefine.SORT_COLUMN_INDEX,
				GetMasterTableDefine.SORT_ORDER);

		this.update();
	}

	/**
	 * tableViewerを返します。
	 * 
	 * @return tableViewer
	 */
	public CommonTableViewer getTableViewer() {
		return this.tableViewer;
	}

	/**
	 * コンポジットを更新します。
	 * <p>
	 * 
	 */
	@Override
	public void update() {

		//マスタ一覧情報を取得
		try {
			this.masterList = MonitorSnmpTrapEndpointWrapper.getMasterList(comboMib.getText());

		} catch (InvalidRole_Exception e) {
			// アクセス権なしの場合、エラーダイアログを表示する
			MessageDialog.openInformation(
					null,
					Messages.getString("message"),
					Messages.getString("message.accesscontrol.16"));
		} catch (Exception e) {
			// 上記以外の例外
			m_log.warn("update(), " + e.getMessage(), e);
			MessageDialog.openInformation(
					null,
					Messages.getString("message"),
					Messages.getString("message.hinemos.failure.unexpected") + ", " + e.getMessage());
		}

		ArrayList list = collectionToArray(this.masterList);

		//フィルタを設定
		this.tableViewer.addFilter(new ViewerFilter(){

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ViewerFilter#select(org.eclipse.jface.viewers.Viewer, java.lang.Object, java.lang.Object)
			 */
			@Override
			public boolean select(Viewer viewer, Object parentElement, Object element) {
				boolean select = true;

				ArrayList info = (ArrayList)element;

				String mib = (String)info.get(GetMasterTableDefine.MIB);
				String trapName = (String)info.get(GetMasterTableDefine.TRAP_NAME);

				if(oidList != null){
					String trapOid = (String)info.get(GetMasterTableDefine.TRAP_OID);
					Integer genericId = (Integer)info.get(GetMasterTableDefine.GENERIC_ID);
					Integer specificId = (Integer)info.get(GetMasterTableDefine.SPECIFIC_ID);

					for(int i = 0 ; i < oidList.size(); i++){
						MonitorTrapValueInfo oidInfo = oidList.get(i);
						if(mib.equals(oidInfo.getMib()) &&
								trapOid.equals(oidInfo.getTrapOid()) &&
								genericId.equals(oidInfo.getGenericId()) &&
								specificId.equals(oidInfo.getSpecificId())){
							select = false;
						}
					}
				}

				if(!comboMib.getText().equals("")){
					if(!mib.equals(comboMib.getText())){
						select = false;
					}
				}

				if(!textFilter.getText().equals("")){
					if(!trapName.matches(textFilter.getText())){
						select = false;
					}
				}

				return select;
			}
		});

		// テーブル更新
		this.tableViewer.setInput(list);
	}

	/**
	 * 選択したマスタ情報を取得
	 * 
	 * @return
	 */
	public ArrayList<SnmpTrapMasterInfo> getSelectMaster() {

		ArrayList<SnmpTrapMasterInfo> data = new ArrayList<SnmpTrapMasterInfo>();

		//選択されたアイテムを取得
		StructuredSelection selection =
				(StructuredSelection)tableViewer.getSelection();
		List list = selection.toList();

		if (list != null) {
			for(int index = 0; index < list.size(); index++){

				ArrayList info = (ArrayList)list.get(index);
				if (info != null && info.size() > 0) {
					String mib = (String)info.get(GetMasterTableDefine.MIB);
					String oid = (String)info.get(GetMasterTableDefine.TRAP_OID);
					Integer genericId = (Integer)info.get(GetMasterTableDefine.GENERIC_ID);
					Integer specificId = (Integer)info.get(GetMasterTableDefine.SPECIFIC_ID);

					if (this.masterList != null) {
						for(int i = 0; i < this.masterList.size(); i++){
							SnmpTrapMasterInfo master = this.masterList.get(i);

							if(mib.equals(master.getMib()) &&
									oid.equals(master.getTrapOid()) &&
									genericId.equals(master.getGenericId()) &&
									specificId.equals(master.getSpecificId())){

								data.add(master);
								break;
							}
						}
					}
				}
			}
		}

		return data;
	}

	/**
	 * マスタ情報をObjectの2次元配列に格納
	 * 
	 * @param masterList
	 * @return
	 */
	@SuppressWarnings("unchecked")
	public ArrayList collectionToArray(List<SnmpTrapMasterInfo> masterList) {

		ArrayList list = new ArrayList();
		if(masterList != null){
			Iterator itr = masterList.iterator();
			while(itr.hasNext())
			{
				SnmpTrapMasterInfo master = (SnmpTrapMasterInfo)itr.next();

				ArrayList info = new ArrayList();
				info.add(master.getMib());
				info.add(master.getUei());
				info.add(master.getTrapOid());
				info.add(new Integer(master.getGenericId()));
				info.add(new Integer(master.getSpecificId()));
				info.add(new Integer(master.getPriority()));
				info.add(master.getLogmsg());
				list.add(info);
			}
		}
		return list;
	}

	/**
	 * @return
	 */
	public List<MonitorTrapValueInfo> getOidList() {
		return oidList;
	}

	/**
	 * @param oidList
	 */
	public void setOidList(List<MonitorTrapValueInfo> oidList) {
		this.oidList = oidList;
	}
}