/*

Copyright (C) 2007 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.maintenance.composite;

import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;

import com.clustercontrol.maintenance.util.MaintenanceEndpointWrapper;
import com.clustercontrol.util.Messages;
import com.clustercontrol.ws.maintenance.InvalidRole_Exception;
import com.clustercontrol.ws.maintenance.MaintenanceTypeMst;


/**
 * メンテナンス種別コンポジットクラスです。
 * 
 * @version 2.2.0
 * @since 2.2.0
 */
public class MaintenanceTypeListComposite extends Composite {

	// ログ
	private static Log m_log = LogFactory.getLog( MaintenanceTypeListComposite.class );

	public static final int WIDTH_TITLE = 4;
	public static final int WIDTH_TEXT = 8;

	// ----- instance フィールド ----- //

	/** メンテナンス種別ラベル */
	private Label labelMaintenanceType = null;

	/** メンテナンス種別コンボボックス */
	private Combo comboMaintenanceType = null;

	/** メンテナンス種別一覧リスト */
	List<MaintenanceTypeMst> maintenanceTypeList = null;

	// ----- コンストラクタ ----- //

	/**
	 * インスタンスを返します。
	 * 
	 * @param parent 親のコンポジット
	 * @param style スタイル
	 * @param labelFlg メンテナンス種別ラベル表示フラグ
	 */
	public MaintenanceTypeListComposite(Composite parent, int style, boolean labelFlg) {
		super(parent, style);

		this.initialize(parent, labelFlg);
	}


	// ----- instance メソッド ----- //

	/**
	 * コンポジットを生成・構築します。
	 */
	private void initialize(Composite parent, boolean labelFlg) {

		// 変数として利用されるグリッドデータ
		GridData gridData = null;

		GridLayout layout = new GridLayout(1, true);
		layout.marginHeight = 0;
		layout.marginWidth = 0;
		if(labelFlg){
			layout.numColumns = 15;
		}
		else{
			layout.numColumns = 10;
		}
		this.setLayout(layout);

		/*
		 * メンテナンス種別
		 */
		if(labelFlg){
			// ラベル
			this.labelMaintenanceType = new Label(this, SWT.NONE);
			gridData = new GridData();
			gridData.horizontalSpan = WIDTH_TITLE;
			gridData.horizontalAlignment = GridData.FILL;
			gridData.grabExcessHorizontalSpace = true;
			this.labelMaintenanceType.setLayoutData(gridData);
			this.labelMaintenanceType.setText(Messages.getString("maintenance.type") + " : ");
		}

		// コンボボックス
		this.comboMaintenanceType = new Combo(this, SWT.DROP_DOWN | SWT.READ_ONLY);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_TEXT;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.comboMaintenanceType.setLayoutData(gridData);

		this.update();
	}

	/**
	 * コンポジットを更新します。
	 * <p>
	 * 
	 */
	@Override
	public void update() {

		// データ取得
		if(maintenanceTypeList == null){
			getMaintenanceTypeList();
		}

		// メンテナンス種別プルダウンメニューの作成
		for(MaintenanceTypeMst type : maintenanceTypeList){
			this.comboMaintenanceType.add(Messages.getString(type.getNameId()));
		}
	}

	/**
	 * メンテナンス種別のプルダウンメニューより選択された種別の選択番号を取得
	 * @return
	 */
	public int getSelectionIndex(){
		return comboMaintenanceType.getSelectionIndex();
	}

	/**
	 * メンテナンス種別のプルダウンメニューより選択された種別の取得
	 * @param index
	 * @return
	 */
	public String getSelectionTypeId(){
		String typeId = null;
		// データ取得
		if(maintenanceTypeList == null){
			getMaintenanceTypeList();
		}
		int index = getSelectionIndex();
		if(index >= 0){
			MaintenanceTypeMst maintenanceType = maintenanceTypeList.get(index);
			typeId = maintenanceType.getTypeId();
		}else{
			typeId = "";
		}

		return typeId;
	}

	/**
	 * メンテナンス種別IDよりメンテナンス種別名の取得
	 * @param type_id
	 * @return
	 */
	public String getMaintenanceTypeName(String type_id){
		String name = null;
		// データ取得
		if(maintenanceTypeList == null){
			getMaintenanceTypeList();
		}

		// メンテナンス種別IDよりメンテナンス種別名を取得
		for(MaintenanceTypeMst type : maintenanceTypeList){
			if((type.getTypeId()).equals(type_id)){
				name = Messages.getString(type.getNameId());
				break;
			}
		}
		return name;
	}

	/**
	 * メンテナンス種別一覧の取得
	 */
	private void getMaintenanceTypeList(){
		try {
			maintenanceTypeList = MaintenanceEndpointWrapper.getMaintenanceTypeList();
		} catch (InvalidRole_Exception e) {
			MessageDialog.openInformation(null, Messages.getString("message"),
					Messages.getString("message.accesscontrol.16"));
		} catch (Exception e) {
			m_log.warn("getMaintenanceTypeList(), " + e.getMessage(), e);
			MessageDialog.openError(
					null,
					Messages.getString("failed"),
					Messages.getString("message.hinemos.failure.unexpected") + ", " + e.getMessage());
		}
	}

	/* (非 Javadoc)
	 * @see org.eclipse.swt.widgets.Control#setEnabled(boolean)
	 */
	@Override
	public void setEnabled(boolean enabled) {
		this.comboMaintenanceType.setEnabled(enabled);
	}

	/* (非 Javadoc)
	 * @see org.eclipse.swt.widgets.Combo#getText()
	 */
	public String getText() {
		return this.comboMaintenanceType.getText();
	}

	/* (非 Javadoc)
	 * @see org.eclipse.swt.widgets.Combo#setText(java.lang.String)
	 */
	public void setText(String string) {
		this.comboMaintenanceType.setText(string);
	}

	/* (非 Javadoc)
	 * @see org.eclipse.swt.widgets.Combo#setBackground(org.eclipse.swt.graphics.Color)
	 */
	@Override
	public void setBackground(Color color) {
		this.comboMaintenanceType.setBackground(color);
	}

	/* (非 Javadoc)
	 * @see org.eclipse.swt.widgets.Combo#addModifyListener(org.eclipse.swt.events.ModifyListener)
	 */
	public void addModifyListener(ModifyListener listener) {
		this.comboMaintenanceType.addModifyListener(listener);
	}

}