/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.jobmanagement.view.action;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.ui.IViewActionDelegate;
import org.eclipse.ui.IViewPart;

import com.clustercontrol.jobmanagement.composite.JobTreeComposite;
import com.clustercontrol.jobmanagement.util.JobEditStateUtil;
import com.clustercontrol.jobmanagement.util.JobEndpointWrapper;
import com.clustercontrol.jobmanagement.view.JobListView;
import com.clustercontrol.util.Messages;
import com.clustercontrol.ws.jobmanagement.InvalidObjectPrivilege_Exception;
import com.clustercontrol.ws.jobmanagement.InvalidRole_Exception;
import com.clustercontrol.ws.jobmanagement.JobInfo;
import com.clustercontrol.ws.jobmanagement.JobTreeItem;

/**
 * ジョブ[一覧]ビューの「登録」のクライアント側アクションクラス<BR>
 * 
 * @version 1.0.0
 * @since 1.0.0
 */
public class RegisterJobAction implements IViewActionDelegate {

	// ログ
	private static Log m_log = LogFactory.getLog( RegisterJobAction.class );

	/** ビュー */
	private IViewPart m_view;

	/**
	 * ビューを保持します。ワークベンチにロードされた際に呼ばれます。
	 * 
	 * @param view ビューのインスタンス
	 * 
	 * @see org.eclipse.ui.IViewActionDelegate#init(org.eclipse.ui.IViewPart)
	 */
	@Override
	public void init(IViewPart view) {
		m_view = view;
	}

	/**
	 * ジョブ[一覧]ビューの「登録」が押された場合に、ジョブを登録します。
	 * <p>
	 * <ol>
	 * <li>登録の確認ダイアログを表示します。</li>
	 * <li>ジョブ[一覧]ビューからジョブツリーアイテムを取得します。</li>
	 * <li>ジョブツリーアイテムを登録します。</li>
	 * </ol>
	 * 
	 * @see org.eclipse.ui.IActionDelegate#run(org.eclipse.jface.action.IAction)
	 * @see com.clustercontrol.jobmanagement.view.JobListView
	 * @see com.clustercontrol.jobmanagement.composite.JobTreeComposite
	 * @see com.clustercontrol.jobmanagement.action.RegisterJob#registerJob(JobTreeItem)
	 */
	@Override
	public void run(IAction action) {
		JobTreeItem selectedItem = null;
		JobListView view = (JobListView) m_view.getAdapter(JobListView.class);
		if (view instanceof JobListView) {
			JobTreeComposite tree = view.getJobTreeComposite();
			selectedItem = tree.getSelectItem();

			if (selectedItem instanceof JobTreeItem) {

				// 確認ダイアログを生成
				if (MessageDialog.openQuestion(
						null,
						Messages.getString("confirmed"),
						Messages.getString("message.job.31"))) {

					// Exceptionが発生したかどうかのフラグ
					boolean error = false;
					// ジョブユニットごとのメッセージのリスト(成功、失敗の結果の詳細が入る）
					HashMap<String, String> resultList = new HashMap<String, String>();
					// 成功して開放予定のジョブユニットのリスト
					ArrayList<JobTreeItem> releaseList = new ArrayList<JobTreeItem>();


					// ジョブ登録の開始
					m_log.debug("registerJob start " + new Date());
					Long start = System.currentTimeMillis();
					Long t_start = System.currentTimeMillis();
					// ジョブユニットの削除
					m_log.debug("deleteJobunit start");
					for (JobTreeItem jobunit : JobEditStateUtil.getDeletedJobunitList()) {
						try {
							String jobunitId = jobunit.getData().getJobunitId();
							m_log.debug("delete " + jobunitId);
							JobEndpointWrapper.checkEditLock(jobunitId, JobEditStateUtil.getEditSession(jobunit.getData()));
							JobEndpointWrapper.deleteJobunit(jobunitId);
							if (!JobEditStateUtil.getEditedJobunitList().contains(jobunit)) {
								// 削除のみの場合は、編集ロック開放リストに追加する
								releaseList.add(jobunit);
								JobEditStateUtil.removeJobunitUpdateTime(jobunitId);
							}
							resultList.put(jobunitId, Messages.getString("message.job.75"));
						} catch (InvalidRole_Exception e) {
							// システム権限エラーの場合は次のジョブユニットを処理する
							m_log.error("run() delete: " + e.getMessage());
							error = true;
							String jobunitId = jobunit.getData().getJobunitId();
							resultList.put(jobunitId, Messages.getString("message.accesscontrol.16"));
						} catch (InvalidObjectPrivilege_Exception e) {
							// オブジェクト権限エラーは次のジョブユニットを処理する
							m_log.error("run() delete: " + e.getMessage());
							error = true;
							String jobunitId = jobunit.getData().getJobunitId();
							resultList.put(jobunitId, Messages.getString("message.accesscontrol.16"));
						} catch (Exception e) {
							m_log.error("run() delete: " + e.getMessage(), e);
							error = true;
							String jobunitId = jobunit.getData().getJobunitId();
							resultList.put(jobunitId, Messages.getString("message.job.76") + "[" + e.getMessage() + "]");
						}
					}
					long t_end = System.currentTimeMillis();
					m_log.debug("delete: " + (t_end-t_start) +" ms");
					t_start = System.currentTimeMillis();

					// ジョブユニットの登録
					m_log.debug("registerJobunit start");
					for (JobInfo info : JobEditStateUtil.getLockedJobunitList()) {
						try {
							JobTreeItem jobunit = null; // ループで一致すると判定された場合にはjobunitに値が設定される
							
							//削除したジョブユニットかチェックする
							for (JobTreeItem item : releaseList) {
								if (item.getData().equals(info)) {
									jobunit = item;
									break;
								}
							}
							if (jobunit != null) {
								// 削除したジョブユニットの場合は何もしない
								continue;
							}
							
							// 編集したジョブユニットかどうかをチェックする
							for (JobTreeItem item : JobEditStateUtil.getEditedJobunitList()) {
								if (item.getData().equals(info)) {
									jobunit = item;
									break;
								}
							}
							
							if (jobunit == null) {
								// 編集していないジョブユニットは、ロックを開放する
								JobEndpointWrapper.releaseEditLock(JobEditStateUtil.getEditSession(info));
								JobEditStateUtil.removeLockedJobunit(info);
							} else {
								// 編集したジョブユニットはマネージャに登録する
								m_log.debug("register " + jobunit.getData().getJobunitId());
								String jobunitId = jobunit.getData().getJobunitId();
								JobEndpointWrapper.checkEditLock(jobunitId, JobEditStateUtil.getEditSession(jobunit.getData()));
								JobEndpointWrapper.registerJobunit(jobunit);

								// 編集ロック開放リストに追加する
								releaseList.add(jobunit);
								Long updateTime = JobEndpointWrapper.getUpdateTime(jobunitId);
								JobEditStateUtil.putJobunitUpdateTime(jobunitId, updateTime);

								resultList.put(jobunitId, Messages.getString("message.job.79"));
							}

						} catch (InvalidRole_Exception e) {
							// システム権限エラーの場合は次のジョブユニットを処理する
							m_log.error("run() register: " + e.getMessage());
							error = true;
							resultList.put(info.getJobunitId(), Messages.getString("message.accesscontrol.16"));
						} catch (InvalidObjectPrivilege_Exception e) {
							// オブジェクト権限エラーは次のジョブユニットを処理する
							m_log.error("run() register: " + e.getMessage());
							error = true;
							resultList.put(info.getJobunitId(), Messages.getString("message.accesscontrol.16"));
						} catch (Exception e) {
							m_log.error("run() register: " + e.getMessage(), e);
							error = true;
							resultList.put(info.getJobunitId(), Messages.getString("message.job.80") + "[" + e.getMessage() + "]");
						}
					}
					t_end = System.currentTimeMillis();
					m_log.debug("register: " + (t_end-t_start) +" ms");

					// 編集ロックをまとめて解除し、propertyFullをクリアする
					for (JobTreeItem item : releaseList) {
						try {
							// 編集ロックの解除
							m_log.debug("release " + item.getData().getJobunitId());
							JobEndpointWrapper.releaseEditLock(JobEditStateUtil.getEditSession(item.getData()));
							JobEditStateUtil.exitEditMode(item);
							
							// 更新時刻の最新化のため、propertyFullをクリアする
							String jobunitId = item.getData().getJobunitId();
							JobEditStateUtil.clearPropertyFull(jobunitId);
						} catch (Exception e) {
							m_log.warn("run() : " + e.getMessage());
						}
					}

					// 登録結果のメッセージの作成
					String message = "";

					ArrayList<String> entries = new ArrayList<String>(resultList.keySet());
					Collections.sort(entries);

					for(String key : entries){
						message = message + key + ":" + resultList.get(key) + "\n";
					}

					String[] args = {message};
					
					if (error) {
						m_log.info("run() : register job failure " + message);
						// 登録に失敗したジョブユニットがある場合
						MessageDialog.openWarning(
								null,
								Messages.getString("message.hinemos.1"),
								Messages.getString("message.job.67", args));
					} else {
						// 登録に失敗したジョブユニットがない場合
						if (entries.isEmpty()) {
							// ジョブを編集してなかった場合など、登録に成功したジョブユニットが0件の場合
							MessageDialog.openInformation(null, Messages.getString("message"),
									Messages.getString("message.job.114"));
						} else {
							// 1件以上のジョブユニットを登録した場合
							MessageDialog.openInformation(null, Messages.getString("message"),
									Messages.getString("message.job.111", args));
						}
					}
					
					Long end = System.currentTimeMillis();
					m_log.debug("register() : " + (end - start) + "ms");
					m_log.debug("registerJob end   " + new Date());

					view.getJobTreeComposite().refresh();
					view.getJobTreeComposite().getTreeViewer().setSelection(
							new StructuredSelection(selectedItem), true);
				}
			}
		}
	}


	/* (non-Javadoc)
	 * @see org.eclipse.ui.IActionDelegate#selectionChanged(org.eclipse.jface.action.IAction, org.eclipse.jface.viewers.ISelection)
	 */
	@Override
	public void selectionChanged(IAction action, ISelection selection) {

	}
}