package com.clustercontrol.jobmanagement.util;

import java.util.ArrayList;
import java.util.concurrent.ConcurrentHashMap;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.jface.dialogs.MessageDialog;

import com.clustercontrol.jobmanagement.bean.JobConstant;
import com.clustercontrol.util.Messages;
import com.clustercontrol.ws.jobmanagement.InvalidRole_Exception;
import com.clustercontrol.ws.jobmanagement.JobInfo;
import com.clustercontrol.ws.jobmanagement.JobTreeItem;

/**
 * JobEditStateユーティリティクラス
 * 
 * 以下を提供します。<BR>
 * <li>編集ロックに関するユーティリティ
 * <li>ジョブ登録時に更新する必要があるジョブユニットを管理するユーティリティ
 * 
 * @version 4.1.0
 * @since 4.1.0
 */
public class JobEditStateUtil {
	/** ログ出力のインスタンス<BR> */
	private static Log m_log = LogFactory.getLog( JobEditStateUtil.class );

	/** 新規作成または編集したマネージャに登録する必要があるジョブユニットのリスト */
	private static ArrayList<JobTreeItem> editedJobunitList = new ArrayList<JobTreeItem>();

	/** 削除されたマネージャから削除する必要があるジョブユニットのリスト */
	private static ArrayList<JobTreeItem> deletedJobunitList = new ArrayList<JobTreeItem>();

	/** 編集ロックを取得したジョブユニットのバックアップ */
	private static ConcurrentHashMap<JobInfo, JobTreeItem> lockedJobunitMap = new ConcurrentHashMap<JobInfo, JobTreeItem>();

	/** 編集ロックを取得したジョブユニットのリスト */
	private static ConcurrentHashMap<JobInfo, Integer> editSessionMap = new ConcurrentHashMap<JobInfo, Integer>();

	/** ジョブユニットと最終更新日時の組 */
	private static ConcurrentHashMap<String, Long> jobunitUpdateTimeMap = new ConcurrentHashMap<String, Long>();
	
	/** ジョブツリー */
	private static JobTreeItem m_jobTreeItem = null;
	
	/**
	 * マネージャに登録する必要があるジョブユニットを追加します
	 * @param jobunit 編集したジョブユニット
	 */
	public static void addEditedJobunit(JobTreeItem jobunit) {
		if (jobunit.getData().getType() == JobConstant.TYPE_COMPOSITE) {
			// ツリーのトップが呼ばれることはないはず
			m_log.warn("addEditJobunit() : jobunit is TOP");
			return;
		} else if (jobunit.getData().getType() == JobConstant.TYPE_JOBUNIT) {
			// ジョブユニットの場合は編集済みジョブユニットに登録する
			if (!editedJobunitList.contains(jobunit)) {
				m_log.debug("addEditJobunit() : add " + jobunit.getData().getJobunitId());
				editedJobunitList.add(jobunit);
			}
		} else {
			// ジョブユニットでない場合は親を再帰的に呼び出す
			addEditedJobunit(jobunit.getParent());
		}
	}

	/**
	 * マネージャに登録する必要があるジョブユニットのリストを返します。
	 * 
	 * @return ArrayList<JobTreeItem>
	 */
	public static ArrayList<JobTreeItem> getEditedJobunitList() {
		return editedJobunitList;
	}

	/**
	 *マネージャに登録する必要があるジョブユニットのリストからジョブユニットを削除します。
	 * 
	 */
	public static void removeEditedJobunit(JobTreeItem jobunit) {
		editedJobunitList.remove(jobunit);
	}

	/**
	 * マネージャから削除する必要があるジョブユニットを追加します
	 * @param jobunit 削除操作したジョブユニット
	 */
	public static void addDeletedJobunit(JobTreeItem jobunit) {
		deletedJobunitList.add(jobunit);
	}

	/**
	 * マネージャから削除する必要があるジョブユニットのリストを返します。<BR>
	 * 
	 * @return ArrayList<String>
	 */
	public static ArrayList<JobTreeItem> getDeletedJobunitList() {
		return deletedJobunitList;
	}

	/**
	 * ジョブツリーが編集された時にtrueを返します。
	 * @return ジョブツリーが編集されている場合true
	 */
	public static boolean isEdit() {
		m_log.debug("isEdit() : lockedJobunitList.size()=" + lockedJobunitMap.size());
		boolean edit = true;
		if (lockedJobunitMap.size() == 0) {
			// 編集モードのジョブが存在しない場合
			edit = false;
		}
		return edit;
	}

	/**
	 * 編集ロックを取得したjobunitのリストを返します。
	 * @return ArrayList<JobInfo> 編集ロックを取得したジョブユニットのリスト
	 */
	public static ArrayList<JobInfo> getLockedJobunitList() {
		ArrayList<JobInfo> list = new ArrayList<JobInfo>();
		for (JobInfo info : editSessionMap.keySet()) {
			m_log.debug("list add " + info.getJobunitId());
			list.add(info);
		}
		return list;
	}

	/**
	 * ジョブユニットのバックアップ(編集ロック取得前)を返す
	 * @param info
	 * @return ジョブユニットのバックアップ
	 */
	public static JobTreeItem getLockedJobunitBackup(JobInfo info) {
		return lockedJobunitMap.get(info);
	}

	/**
	 * ジョブユニットの編集ロック状態をクリアする
	 * 
	 * @param info
	 */
	public static void removeLockedJobunit(JobInfo info) {
		lockedJobunitMap.remove(info);
		editSessionMap.remove(info);
	}

	/**
	 * 編集ロックを取得したジョブユニットを追加する
	 * @param info ジョブユニット
	 * @param item ジョブツリーのバックアップ
	 * @param editSession セッション
	 */
	public static void addLockedJobunit(JobInfo info, JobTreeItem item, Integer editSession) {
		editSessionMap.put(info, editSession);
		if (item != null) {
			lockedJobunitMap.put(info, item);
		}
	}

	/**
	 * 編集ロックを取得したジョブユニット配下のジョブかどうかをチェックする。
	 * @param jobunitId 選択されたジョブのジョブユニットID
	 * @return 編集ロックを取得したジョブユニット配下のジョブである場合true
	 */
	public static boolean isLockedJobunitId(String jobunitId) {
		boolean ret = false;

		for (JobInfo info : editSessionMap.keySet()) {
			if (jobunitId.equals(info.getJobunitId())) {
				ret = true;
			}
		}
		return ret;
	}

	/**
	 * ジョブユニットのセッションを取得する
	 * 
	 * @param info ジョブユニット
	 * @return セッション
	 */
	public static Integer getEditSession(JobInfo info) {
		return editSessionMap.get(info);
	}

	/**
	 * 編集ロックの状態をすべてクリアする
	 */
	public static void clearEditStateAll() {
		m_log.debug("clearEditStateAll()");
		editedJobunitList.clear();
		deletedJobunitList.clear();
		lockedJobunitMap.clear();
		editSessionMap.clear();
		jobunitUpdateTimeMap.clear();
		m_jobTreeItem = null;
	}

	/**
	 * 編集したジョブユニットの編集状態をクリアする
	 * 編集していない場合は、removeLockedJobunit()の呼び出しだけでよい
	 * 
	 * @param item ジョブツリーアイテム
	 */
	public static void exitEditMode(JobTreeItem item) {
		editedJobunitList.remove(item);
		deletedJobunitList.remove(item);
		removeLockedJobunit(item.getData());
	}
	
	/**
	 * 指定したジョブユニットの最終更新日時を取得します
	 * 
	 * @param jobunitId ジョブユニットID
	 * @return 最終更新日時
	 */
	public static Long getJobunitUpdateTime(String jobunitId) {
		return jobunitUpdateTimeMap.get(jobunitId);
	}
	
	/**
	 * 指定したジョブユニットの最終更新日時を設定します
	 * 
	 * @param jobunitId ジョブユニットID
	 * @param updateTime 最終更新日時
	 */
	public static void putJobunitUpdateTime(String jobunitId, Long updateTime) {
		jobunitUpdateTimeMap.put(jobunitId, updateTime);
	}
	
	/**
	 * 最終更新日時を保存するマップから指定したジョブユニットの情報を削除します
	 * 
	 * @param jobunitId 情報を削除するジョブユニットID
	 */
	public static void removeJobunitUpdateTime(String jobunitId) {
		jobunitUpdateTimeMap.remove(jobunitId);
	}
	
	public static JobTreeItem getJobTreeItem() {
		return m_jobTreeItem;
	}
	
	/**
	 * ジョブツリーを更新する<BR>
	 * 
	 * @param ownerRoleId
	 * @param m_treeOnly
	 * @return
	 */
	public static JobTreeItem updateJobTree(String ownerRoleId, boolean m_treeOnly) {
		try {
			m_jobTreeItem = JobEndpointWrapper.getJobTree(ownerRoleId, m_treeOnly);
			updateJobunitUpdateTime();
		} catch (InvalidRole_Exception e) {
			// アクセス権なしの場合、エラーダイアログを表示する
			MessageDialog.openInformation(
					null,
					Messages.getString("message"),
					Messages.getString("message.accesscontrol.16"));
		} catch (Exception e) {
			m_log.warn("update() getJobTree, " + e.getMessage(), e);
			MessageDialog.openError(
					null,
					Messages.getString("failed"),
					Messages.getString("message.hinemos.failure.unexpected") + ", " + e.getMessage());
		}
		return m_jobTreeItem;
	}
	
	private static void updateJobunitUpdateTime() {
		try {
			for (JobTreeItem jobunit : m_jobTreeItem.getChildren().get(0).getChildren()) {
				String jobunitId = jobunit.getData().getJobunitId();
				Long updateTime = JobEndpointWrapper.getUpdateTime(jobunitId);
				if (updateTime != null) {
					JobEditStateUtil.putJobunitUpdateTime(jobunitId, updateTime);
				}
			}
		} catch (InvalidRole_Exception e) {
			// アクセス権なしの場合、エラーダイアログを表示する
			MessageDialog.openInformation(
					null,
					Messages.getString("message"),
					Messages.getString("message.accesscontrol.16"));
		} catch (Exception e) {
			m_log.warn("updateJobunitUpdateTime() : " + e.getMessage(), e);
			MessageDialog.openError(
					null,
					Messages.getString("failed"),
					Messages.getString("message.hinemos.failure.unexpected") + ", " + e.getMessage());
		}
	}
	
	/**
	 * 特定のジョブユニット配下に関して、JobTreeItemキャッシュ中のpropertyFullをクリアする
	 */
	public static void clearPropertyFull(String jobunitId) {
		
		for (JobTreeItem jobunitItem : m_jobTreeItem.getChildren().get(0).getChildren()) {
			
			if(jobunitItem.getData().getJobunitId().equals(jobunitId)) {
				jobunitItem.getData().setPropertyFull(false);
				for(JobTreeItem item : jobunitItem.getChildren()) {
					item.getData().setPropertyFull(false);
					clearPropertyFullRecursive(item);
				}
			}
		}
	}
	
	private static void clearPropertyFullRecursive(JobTreeItem parentItem) {
		
		for (JobTreeItem item : parentItem.getChildren()) {
			item.getData().setPropertyFull(false);
			clearPropertyFullRecursive(item);
			
		}
	}
}
