/*

Copyright (C) 2013 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.calendar.composite;

import java.util.ArrayList;
import java.util.Iterator;

import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.PlatformUI;

import com.clustercontrol.calendar.dialog.CalendarDetailDialog;
import com.clustercontrol.dialog.ValidateResult;
import com.clustercontrol.util.Messages;
import com.clustercontrol.ws.calendar.CalendarDetailInfo;

/**
 * カレンダ詳細情報グループのコンポジットクラス<BR>
 * <p>
 * <dl>
 *  <dt>コンポジット</dt>
 *  <dd>値取得の成功時</dd>
 *  <dd>　カレンダ詳細情報一覧コンポジット</dd>
 *  <dd>　「追加」ボタン</dd>
 *  <dd>　「変更」ボタン</dd>
 *  <dd>　「削除」ボタン</dd>
 *  <dd>　「上へ」ボタン</dd>
 *  <dd>　「下へ」ボタン</dd>
 *  <dd>値取得の失敗時</dd>
 *  <dd>　「重要度」 コンボボックス</dd>
 * </dl>
 * 
 * @version 4.1.0
 * @since 4.1.0
 */
public class CalendarDetailInfoComposite extends Composite {

	/** カレンダ詳細情報一覧 コンポジット。 */
	private CalendarDetailListComposite m_infoListComposite = null;

	/** 追加 ボタン。 */
	private Button m_buttonAdd = null;

	/** 変更 ボタン。 */
	private Button m_buttonModify = null;

	/** 削除 ボタン。 */
	private Button m_buttonDelete = null;

	/** 上へ ボタン。 */
	private Button m_buttonUp = null;

	/** 下へ ボタン。 */
	private Button m_buttonDown = null;

	/**
	 * インスタンスを返します。
	 * <p>
	 * 初期処理を呼び出し、コンポジットを配置します。
	 * 
	 * @param parent 親のコンポジット
	 * @param style スタイル
	 * @param tableDefine カレンダ詳細情報一覧のテーブル定義情報（{@link com.clustercontrol.bean.TableColumnInfo}のリスト）
	 * 
	 * @see org.eclipse.swt.SWT
	 * @see org.eclipse.swt.widgets.Composite#Composite(Composite parent, int style)
	 * @see com.clustercontrol.bean.TableColumnInfo#TableColumnInfo(java.lang.String, int, int, int)
	 * @see com.clustercontrol.monitor.run.action.GetStringFilterTableDefine
	 * @see #initialize(ArrayList)
	 */
	public CalendarDetailInfoComposite(Composite parent, int style){
		super(parent, style);
		this.initialize();
	}

	/**
	 * コンポジットを配置します。
	 */
	private void initialize(){

		// 変数として利用されるグリッドデータ
		GridData gridData = null;

		GridLayout layout = new GridLayout(1, true);
		layout.marginHeight = 0;
		layout.marginWidth = 0;
		layout.numColumns = 15;
		this.setLayout(layout);

		/*
		 * カレンダ詳細情報一覧
		 */
		this.m_infoListComposite = new CalendarDetailListComposite(this, SWT.BORDER);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.verticalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.grabExcessVerticalSpace = true;
		gridData.horizontalSpan = 13;
		this.m_infoListComposite.setLayoutData(gridData);
		/*
		 * 操作ボタン
		 */
		Composite composite = new Composite(this, SWT.NONE);
		layout = new GridLayout(1, true);
		layout.numColumns = 1;
		composite.setLayout(layout);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.verticalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.grabExcessVerticalSpace = true;
		gridData.horizontalSpan = 2;
		composite.setLayoutData(gridData);

		// 追加ボタン
		this.m_buttonAdd = this.createButton(composite, Messages.getString("add"));
		this.m_buttonAdd.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				// シェルを取得
				Shell shell = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getShell();
				//FIXME
				//CalendarDetailDialog dialog = new CalendarDetailDialog(shell, m_ownerRoleId);
				CalendarDetailDialog dialog = new CalendarDetailDialog(shell, m_infoListComposite.getOwnerRoleId());
				if (dialog.open() == IDialogConstants.OK_ID) {
					m_infoListComposite.getDetailList().add(dialog.getInputData());
					m_infoListComposite.update();
				}
			}
		});
		// 変更ボタン
		this.m_buttonModify = this.createButton(composite, Messages.getString("modify"));
		this.m_buttonModify.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				Integer order = m_infoListComposite.getSelection();
				if (order != null) {
					// シェルを取得
					Shell shell = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getShell();
					//FIXME
					//CalendarDetailDialog dialog = new CalendarDetailDialog(shell,m_infoListComposite.getDetailList().get(order - 1), m_ownerRoleId);
					CalendarDetailDialog dialog = new CalendarDetailDialog(shell,m_infoListComposite.getDetailList().get(order - 1), m_infoListComposite.getOwnerRoleId());
					if (dialog.open() == IDialogConstants.OK_ID) {
						m_infoListComposite.getDetailList().remove(m_infoListComposite.getDetailList().get(order - 1));
						m_infoListComposite.getDetailList().add(order - 1,dialog.getInputData());
						m_infoListComposite.setSelection();
					}
				}else{
					MessageDialog.openWarning(
							null,
							Messages.getString("warning"),
							Messages.getString("message.monitor.30"));
				}
			}
		});

		// 削除ボタン
		this.m_buttonDelete = this.createButton(composite, Messages.getString("delete"));
		this.m_buttonDelete.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				Integer order = m_infoListComposite.getSelection();
				if (order != null) {
					m_infoListComposite.getDetailList().remove(order - 1);
					m_infoListComposite.update();
				}else{
					MessageDialog.openWarning(
							null,
							Messages.getString("warning"),
							Messages.getString("message.monitor.30"));
				}
			}
		});

		// 上へボタン
		this.m_buttonUp = this.createButton(composite, Messages.getString("up"));
		this.m_buttonUp.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				Integer order = m_infoListComposite.getSelection();
				if (order != null) {
					m_infoListComposite.up();
					m_infoListComposite.update();
				}else{
					MessageDialog.openWarning(
							null,
							Messages.getString("warning"),
							Messages.getString("message.monitor.30"));
				}
			}
		});

		// 下へボタン
		this.m_buttonDown = this.createButton(composite, Messages.getString("down"));
		this.m_buttonDown.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				Integer order = m_infoListComposite.getSelection();
				if (order != null) {
					m_infoListComposite.down();
					m_infoListComposite.update();
				}
				else{
					MessageDialog.openWarning(
							null,
							Messages.getString("warning"),
							Messages.getString("message.monitor.30"));
				}
			}
		});
	}

	/**
	 * 
	 * @return
	 */
	public ArrayList<CalendarDetailInfo> getDetailList(){
		return this.m_infoListComposite.getDetailList();
	}
	/**
	 * カレンダ詳細情報をコンポジット内リストに反映させる
	 * @param detailList
	 */
	public void setDetailList(ArrayList<CalendarDetailInfo> detailList){
		if (detailList != null) {
			this.m_infoListComposite.setDetailList(detailList);
		}
		this.update();
	}

	/* (非 Javadoc)
	 * @see org.eclipse.swt.widgets.Control#setEnabled(boolean)
	 */
	@Override
	public void setEnabled(boolean enabled) {
		this.m_infoListComposite.setEnabled(enabled);
		this.m_buttonAdd.setEnabled(enabled);
		this.m_buttonModify.setEnabled(enabled);
		this.m_buttonDelete.setEnabled(enabled);
		this.m_buttonUp.setEnabled(enabled);
		this.m_buttonDown.setEnabled(enabled);
	}

	/**
	 * ボタンを返します。
	 * 
	 * @param parent 親のコンポジット
	 * @param label ボタンに表示するテキスト
	 * @return ボタン
	 */
	private Button createButton(Composite parent, String label) {
		Button button = new Button(parent, SWT.NONE);

		GridData gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		button.setLayoutData(gridData);

		button.setText(label);

		return button;
	}

	/**
	 * 無効な入力値の情報を設定します。
	 * 
	 * @param id ID
	 * @param message メッセージ
	 * @return 認証結果
	 */
	protected ValidateResult setValidateResult(String id, String message) {

		ValidateResult validateResult = new ValidateResult();
		validateResult.setValid(false);
		validateResult.setID(id);
		validateResult.setMessage(message);

		return validateResult;
	}

	public void setOwnerRoleId(String ownerRoleId) {
		this.m_infoListComposite.setOwnerRoleId(ownerRoleId);
		//this.m_ownerRoleId = ownerRoleId;
	}

	public void changeOwnerRoleId(String ownerRoleId) {

		//if (ownerRoleId == null
		//		|| !ownerRoleId.equals(this.m_ownerRoleId)) {
		if (ownerRoleId == null
				|| !ownerRoleId.equals(this.m_infoListComposite.getOwnerRoleId())) {

			//Iterator<CalendarDetailInfo> iter = m_info.getCalendarDetailList().iterator();
			Iterator<CalendarDetailInfo> iter = m_infoListComposite.getDetailList().iterator();
			while (iter.hasNext()) {
				CalendarDetailInfo composite = iter.next();
				composite.setCalPatternId(null);
				composite.setCalPatternInfo(null);
			}
		}
		setOwnerRoleId(ownerRoleId);
	}
}