/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.accesscontrol.view.action;

import java.util.List;

import org.eclipse.jface.action.IAction;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.ui.IViewActionDelegate;
import org.eclipse.ui.IViewPart;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.PlatformUI;

import com.clustercontrol.accesscontrol.action.GetRoleListTableDefine;
import com.clustercontrol.accesscontrol.util.AccessEndpointWrapper;
import com.clustercontrol.accesscontrol.view.RoleListView;
import com.clustercontrol.accesscontrol.view.RoleSettingTreeView;
import com.clustercontrol.bean.PluginConstant;
import com.clustercontrol.util.Messages;
import com.clustercontrol.ws.access.InvalidRole_Exception;
import com.clustercontrol.ws.access.UnEditableRole_Exception;
import com.clustercontrol.ws.access.UsedFacility_Exception;
import com.clustercontrol.ws.access.UsedOwnerRole_Exception;
import com.clustercontrol.ws.access.UsedRole_Exception;

/**
 * アクセス[ロール]ビューの「削除」のアクションクラス<BR>
 * 
 * @version 2.0.0
 * @since 2.0.0
 */
public class RoleDeleteAction implements IViewActionDelegate {
	/** アクションID */
	public static final String ID = "com.clustercontrol.accesscontrol.view.action.RoleDeleteAction";
	/** ビュー */
	private IViewPart viewPart;

	/**
	 * ビューを保持します。ワークベンチにロードされた際に呼ばれます。
	 * 
	 * @param view ビューのインスタンス
	 * 
	 * @see org.eclipse.ui.IViewActionDelegate#init(org.eclipse.ui.IViewPart)
	 */
	@Override
	public void init(IViewPart viewPart) {
		this.viewPart = viewPart;
	}

	/**
	 * アクセス[ロール]ビューの「削除」が押された場合に、<BR>
	 * ロールを削除します。
	 * <p>
	 * <ol>
	 * <li>アクセス[ロール]ビューから選択されているロールのロールIDを取得します。</li>
	 * <li>ロールIDが一致するロールを削除します。</li>
	 * <li>アクセス[ロール]ビューを更新します。</li>
	 * </ol>
	 * 
	 * @see org.eclipse.ui.IActionDelegate#run(org.eclipse.jface.action.IAction)
	 * @see com.clustercontrol.accesscontrol.view.UserListView
	 * @see com.clustercontrol.accesscontrol.action.DeleteUserProperty#delete(String)
	 */
	@Override
	public void run(IAction action) {
		// ロール一覧より、選択されているロールのUIDを取得
		RoleListView roleListView = (RoleListView) this.viewPart
				.getAdapter(RoleListView.class);

		StructuredSelection selection = (StructuredSelection) roleListView
				.getComposite().getTableViewer().getSelection();

		List list = (List) selection.getFirstElement();
		String roleId = null;
		if (list != null) {
			roleId = (String) list.get(GetRoleListTableDefine.ROLE_ID);
		}

		if (roleId != null) {
			// 確認ダイアログにて変更が選択された場合、削除処理を行う。
			String[] args = { roleId };
			if (MessageDialog.openConfirm(
					null,
					Messages.getString("confirmed"),
					Messages.getString("message.accesscontrol.32", args))) {

				try {
					// 削除処理
					AccessEndpointWrapper.deleteRoleInfo(roleId);
					// 完了メッセージ
					MessageDialog.openInformation(
							null,
							Messages.getString("successful"),
							Messages.getString("message.accesscontrol.30"));

				} catch (UsedFacility_Exception e) {
					// ロールIDのスコープが使用されている場合のエラーダイアログを表示する
					Object[] errorArgs ={roleId, PluginConstant.typeToString(
							e.getFaultInfo().getPlugin())};

					MessageDialog.openInformation(
							null,
							Messages.getString("message"),
							Messages.getString("message.repository.27", errorArgs));
				} catch (UsedOwnerRole_Exception e) {
					// ロールがオーナーロールとして使用されている場合はエラー
					Object[] errorArgs = {roleId,
							 PluginConstant.typeToString(((UsedOwnerRole_Exception) e).getFaultInfo().getPlugin()) };
					MessageDialog.openInformation(null, 
							Messages.getString("message"),
							Messages.getString("message.accesscontrol.52", errorArgs));
				} catch (Exception e) {
					String errMessage = "";
					if (e instanceof InvalidRole_Exception) {
						// 権限なし
						MessageDialog.openInformation(null, Messages.getString("message"),
								Messages.getString("message.accesscontrol.16"));
					} else if (e instanceof UsedRole_Exception) {
						// ロールに所属するユーザが存在する場合はエラー
						MessageDialog.openInformation(null, Messages.getString("message"),
								Messages.getString("message.accesscontrol.42"));
					} else if (e instanceof UnEditableRole_Exception) {
						// 削除不可のロールを削除する場合はエラー（システムロール、内部モジュール用ロール）
						MessageDialog.openInformation(null, Messages.getString("message"),
								Messages.getString("message.accesscontrol.41"));
					} else {
						errMessage = ", " + e.getMessage();
					}

					// 上記以外の例外
					MessageDialog.openError(
							null,
							Messages.getString("failed"),
							Messages.getString("message.accesscontrol.31") + errMessage);
				}
				// ビューを更新
				roleListView.update();

				//アクティブページを手に入れる
				IWorkbenchPage page = PlatformUI.getWorkbench()
						.getActiveWorkbenchWindow().getActivePage();

				//ツリービューを更新する
				IViewPart roleTreeViewPart = page.findView(RoleSettingTreeView.ID);
				if (roleTreeViewPart != null) {
					RoleSettingTreeView treeView = (RoleSettingTreeView) roleTreeViewPart
							.getAdapter(RoleSettingTreeView.class);
					treeView.update();
				}
			}
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.IActionDelegate#selectionChanged(org.eclipse.jface.action.IAction, org.eclipse.jface.viewers.ISelection)
	 */
	@Override
	public void selectionChanged(IAction action, ISelection selection) {
	}
}