/*

Copyright (C) 2012 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.snmptrap.service;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.Map.Entry;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.commons.util.HinemosProperties;
import com.clustercontrol.snmptrap.bean.SnmpTrapV1;

public class SnmpTrapSuppressor {

	private static final Log log = LogFactory.getLog(SnmpTrapSuppressor.class);

	private final Map<Long, Map<SnmpTrapV1, ReceiverStat>> _recentSnmpTrap;
	private int gcCounter = 0;
	private int recentSnmpTrapCounter = 0;

	private static final int _maxRecentSnmpTrap;
	private static final long _maxLifetimeSnmpTrapMsec;

	static {
		// 30 [msg/sec] * 60 [sec] * 15 [min] = 27000
		_maxRecentSnmpTrap = Integer.parseInt(HinemosProperties.getProperty("monitor.snmptrap.recent.maxsize", "27000"));
		// 15 [min] * 60 [sec] * 1000 [msec] = 900000 [msec]
		_maxLifetimeSnmpTrapMsec = Long.parseLong(HinemosProperties.getProperty("monitor.snmptrap.recent.lifetime", "900000"));
	}

	public SnmpTrapSuppressor() {
		_recentSnmpTrap = new TreeMap<Long, Map<SnmpTrapV1, ReceiverStat>>(
				new Comparator<Long>() {
					@Override
					public int compare(Long key1, Long key2) {
						// snmptrap受信日時の降順に並ぶように実装
						long diff = key2 - key1;
						if (diff > Integer.MAX_VALUE) {
							return Integer.MAX_VALUE;
						} else if (diff < Integer.MIN_VALUE) {
							return Integer.MIN_VALUE;
						} else {
							return (int)diff;
						}
					}
				}
				);
	}

	public synchronized boolean isSuppress(String receiverId, SnmpTrapV1 snmptrap) {
		if (++gcCounter > _maxRecentSnmpTrap) {
			gcRecentSnmpTrap();
			gcCounter = 0;
		}

		if (log.isDebugEnabled()) {
			log.debug("checking receiverId = " + receiverId + ", snmptrap = " + snmptrap + ")");
		}

		ReceiverStat counter = null;
		if (_recentSnmpTrap.containsKey(snmptrap.receivedTime)) {
			// 該当時間帯に出力したsnmptrapが存在するか
			Map<SnmpTrapV1, ReceiverStat> snmptrapMap = _recentSnmpTrap.get(snmptrap.receivedTime);

			if (snmptrapMap.containsKey(snmptrap)) {
				// 該当時間に出力された同一のsnmptrapを受信している場合、
				// 各receiverの受信統計情報を取得する
				counter = snmptrapMap.get(snmptrap);
			} else {
				if (recentSnmpTrapCounter < _maxRecentSnmpTrap) {
					// 該当時間帯に出力された同一のsnmptrapを受信していない場合、
					// そのsnmptrapの受信統計情報を新規生成する
					counter = new ReceiverStat();
					snmptrapMap.put(snmptrap, counter);
					recentSnmpTrapCounter++;
				} else {
					// 抑制判定用のsnmptrapストアが一定量を超えたら、snmptrapストアを更新しない
					// よって、生存期間で空きが出るまで、重複して監視される
					log.warn("too many snmptrap. skip update list of suppression : " + snmptrap);
					return false;
				}
			}
		} else {
			if (recentSnmpTrapCounter < _maxRecentSnmpTrap) {
				// 該当時間に出力されたsnmptrapが存在しない場合、
				// その時間の統計情報一覧を新規生成する
				Map<SnmpTrapV1, ReceiverStat> snmptrapMap = new HashMap<SnmpTrapV1, ReceiverStat>();
				counter = new ReceiverStat();
				snmptrapMap.put(snmptrap, counter);
				_recentSnmpTrap.put(snmptrap.receivedTime, snmptrapMap);
				recentSnmpTrapCounter++;
			} else {
				// 抑制判定用のsnmptrapストアが一定量を超えたら、snmptrapストアを更新しない
				// よって、生存期間で空きが出るまで、重複して監視される
				log.warn("too many snmptrap. skip update list of suppression : " + snmptrap);
				return false;
			}
		}

		// 該当snmptrapの受信統計情報から抑制するかどうかを返す
		return counter.isSuppress(receiverId);
	}

	private void gcRecentSnmpTrap() {
		long now = System.currentTimeMillis();
		long thresTime = now - _maxLifetimeSnmpTrapMsec;

		log.info("cleaning recent snmptrap map. (garbage snmptrap before " + new Date(thresTime) + ")");

		List<Long> removeList = new ArrayList<Long>();

		recentSnmpTrapCounter = 0;
		for (Iterator<Entry<Long, Map<SnmpTrapV1, ReceiverStat>>> it = _recentSnmpTrap.entrySet().iterator(); it.hasNext();) {
			Entry<Long, Map<SnmpTrapV1, ReceiverStat>> entry = it.next();

			if (log.isDebugEnabled()) {
				log.debug("checking recent snmptrap. (date = " + new Date(entry.getKey()) + ", snmptrap = " + entry.getKey() + ")");
			}

			// 生存期間を超えたsnmptrap情報は破棄する
			// (万が一、これを超えて同一のsnmptrap情報が届いた場合、抑制されずに通知される)
			if (entry.getKey() < thresTime) {
				if (log.isDebugEnabled()) {
					log.debug("removing recent snmptrap. (date = " + new Date(entry.getKey()) + ")");
				}
				removeList.add(entry.getKey());
			} else {
				recentSnmpTrapCounter += entry.getValue().size();
			}
		}

		for (Long key : removeList) {
			_recentSnmpTrap.remove(key);
		}
	}

	private class ReceiverStat {
		private final Map<String, Integer> counter = new HashMap<String, Integer>();

		public boolean isSuppress(String receiverId) {
			int receiverCount = 0;
			int maxCount = 0;

			// receiverが初めてそのsnmptrapを受信した場合
			if (! counter.containsKey(receiverId)) {
				counter.put(receiverId, 0);
			}

			// 各receiverの統計情報を確認し、最大受信数とreceiverの受信数を取得
			for (String key : counter.keySet()) {
				if (counter.get(key) > maxCount) {
					maxCount = counter.get(key);
				}
				if (key.equals(receiverId)) {
					receiverCount = counter.get(key);
				}
			}

			// 該当receiverの受信数をカウントアップする
			counter.put(receiverId, receiverCount + 1);

			if (receiverCount < maxCount) {
				// 既に監視している系が存在するため、抑制対象となる
				return true;
			} else {
				// 監視している系が存在しないため、監視対象となる
				return false;
			}
		}
	}

}
