/*

Copyright (C) 2010 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.selfcheck.monitor;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.selfcheck.util.FileSystemPoller;
import com.clustercontrol.util.apllog.AplLogger;

/**
 * ファイルシステムの使用率を確認する処理の実装クラス
 */
public class FileSystemMonitor extends SelfCheckMonitorBase {

	private static Log m_log = LogFactory.getLog( FileSystemMonitor.class );

	public final String mountPoint;
	public final int thresholdPer;

	public final String monitorId = "SYS_FS";
	public final String subKey;
	public final String application = "SELFCHECK (FileSystem)";

	public final int snmpPort;
	public final String snmpVersion;
	public final String snmpCommunity;
	public final int snmpRetries;
	public final int snmpTimeout;

	/**
	 * コンストラクタ
	 * @param mountPoint 対象となるファイルシステムのマウントポイント
	 * @param fileSystemUsagePer ファイルシステム使用率の上限値
	 * @param snmpPort SNMPポート番号
	 * @param snmpVersion SNMPバージョン
	 * @param snmpCommunity SNMPコミュニティ名
	 * @param snmpRetries SNMPリトライ回数
	 * @param snmpTimeout SNMPタイムアウト[msec]
	 */
	public FileSystemMonitor(String mountPoint, int fileSystemUsagePer, int snmpPort, String snmpVersion, String snmpCommunity, int snmpRetries, int snmpTimeout) {
		this.mountPoint = mountPoint;
		this.thresholdPer = fileSystemUsagePer;

		this.snmpPort = snmpPort;
		this.snmpVersion = snmpVersion;
		this.snmpCommunity = snmpCommunity;
		this.snmpRetries = snmpRetries;
		this.snmpTimeout = snmpTimeout;

		this.subKey = mountPoint;
	}

	/**
	 * セルフチェック処理名
	 */
	@Override
	public String toString() {
		return String.format("monitoring a usage of file system. (monitorId = %s, thresdholdPer = %d, snmpPort = %d, snmpVersion = %s, snmpCommunity = %s, snmpRetries = %d, snmpTimeout = %d)",
				mountPoint, thresholdPer, snmpPort, snmpVersion, snmpCommunity, snmpRetries, snmpTimeout);
	}

	/**
	 * 監視項目ID
	 */
	@Override
	public String getMonitorId() {
		return monitorId;
	}

	/**
	 * 監視項目subKey
	 */
	@Override
	public String getSubKey() {
		return subKey;
	}

	/**
	 * ファイルシステムの使用率が上限値以下であるかを確認する処理
	 * @return 通知情報（アプリケーション名は未格納）
	 */
	@Override
	public void execute() {
		/** ローカル変数 */
		int fileSystemUsage = 0;
		int fileSystemTotal = 0;
		double fileSystemUsagePer = 0;
		boolean warn = true;

		/** メイン処理 */
		m_log.debug("monitoring file system usage. (mountPoint = " + mountPoint + ", thresholdPer = " + thresholdPer + ")");

		// 利用可能なヒープ容量をMByte単位で取得する
		try {
			fileSystemUsage = new FileSystemPoller(mountPoint, snmpPort, snmpVersion, snmpCommunity, snmpRetries, snmpTimeout).getFileSystemUsage();
			fileSystemTotal = new FileSystemPoller(mountPoint, snmpPort, snmpVersion, snmpCommunity, snmpRetries, snmpTimeout).getFileSystemTotal();
		} catch (Exception e) {
			m_log.warn("filesystem usage collection failure. (mountPoint = " + mountPoint + ", threshold = " + thresholdPer + " [%])", e);
		}

		fileSystemUsagePer = (double)fileSystemUsage / (double)fileSystemTotal * 100.0;

		if (fileSystemUsage == -1 || fileSystemTotal == -1) {
			m_log.info("skipped monitoring file system usage. (mountPoint = " + mountPoint + ", threshold = " + thresholdPer + " [%])");
			return;
		} else {
			if (fileSystemUsagePer <= thresholdPer) {
				m_log.debug("usage of file system is low. (mountPoint = " + mountPoint
						+ ", usage = " + String.format("%.2f", fileSystemUsagePer) + " [%], threshold = " + thresholdPer + " [%])");
				warn = false;
			}
		}
		if (warn) {
			m_log.info("usage of file system is high. (mountPoint = " + mountPoint
					+ ", usage = " + String.format("%.2f", fileSystemUsagePer) + " [%], threshold = " + thresholdPer + " [%])");
		}

		if (!isNotify(warn)) {
			return;
		}
		String[] msgAttr1 = { mountPoint, String.format("%.2f", fileSystemUsagePer), new Integer(thresholdPer).toString() };
		AplLogger aplLogger = new AplLogger(PLUGIN_ID, APL_ID);
		aplLogger.put(MESSAGE_ID, "002", msgAttr1,
				"usage of filesystem(" +
						mountPoint +
						") is too high (" +
						String.format("%.2f", fileSystemUsagePer) +
						" [%] > threshold " +
						thresholdPer +
				" [%]).");

		return;
	}

}
