/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.notify.util;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.HinemosModuleConstant;
import com.clustercontrol.commons.util.JpaTransactionManager;
import com.clustercontrol.notify.bean.NotifyRelationInfo;
import com.clustercontrol.notify.model.NotifyRelationInfoEntity;

/**
 * 通知グループIDと通知IDとの関連をマップで管理するクラス。
 * ジョブセッションで利用されている通知グループはキャッシュされない。
 */
public class NotifyRelationCache {
	private static Log m_log = LogFactory.getLog( NotifyRelationCache.class );

	private static ConcurrentHashMap<String, List<NotifyRelationInfo>> m_notifyMap = new ConcurrentHashMap<String, List<NotifyRelationInfo>>();

	static {
		refresh();
	}

	/**
	 * 通知グループIDが関連を持つ通知IDのリストを返す。
	 * 
	 * @param notifyGroupId 通知グループID
	 * @return 通知IDのリスト。エラー時は空のリストを返す。
	 */
	public static List<String> getNotifyIdList(String notifyGroupId){
		List<String> list = new ArrayList<String>();
		for (NotifyRelationInfo info : getNotifyList(notifyGroupId)) {
			list.add(info.getNotifyId());
		}
		Collections.sort(list);
		return list;
	}

	/**
	 * 通知グループIDが関連を持つ通知情報のリストを返す。
	 * 
	 * @param notifyGroupId 通知グループID
	 * @return 通知情報のリスト。エラー時は空のリストを返す。
	 */
	public static List<NotifyRelationInfo> getNotifyList(String notifyGroupId){
		try {
			// キャッシュから取得
			// ジョブセッションで利用されている通知グループの以外はキャッシュされているはず。
			List<NotifyRelationInfo> notifyList = m_notifyMap.get(notifyGroupId);
			if(notifyList != null){
				return notifyList;
			}
			m_log.debug("getNotifyIdList() : Job Session. " + notifyGroupId);
			List<NotifyRelationInfoEntity> nriList
			= QueryUtil.getNotifyRelationInfoByNotifyGroupId(notifyGroupId);
			notifyList = new ArrayList<NotifyRelationInfo>();
			for(NotifyRelationInfoEntity nri : nriList){
				NotifyRelationInfo info
				= new NotifyRelationInfo(
						nri.getId().getNotifyGroupId(),
						nri.getId().getNotifyId(),
						nri.getNotifyType(),
						nri.getNotifyFlg());
				notifyList.add(info);
			}
			return notifyList;
		} catch (Exception e) {
			m_log.warn("getNotifyList() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			return new ArrayList<NotifyRelationInfo>(); // 空のリストを返す
		}
	}

	public static void refresh(){
		JpaTransactionManager jtm = new JpaTransactionManager();
		if (!jtm.isNestedEm()) {
			m_log.warn("refresh() : transactioin has not been begined.");
			jtm.close();
			return;
		}

		long start = System.currentTimeMillis();
		ConcurrentHashMap<String, List<NotifyRelationInfo>> notifyMap = new ConcurrentHashMap<String, List<NotifyRelationInfo>>();
		List<NotifyRelationInfoEntity> nriList = null;
		try {
			nriList = QueryUtil.getAllNotifyRelationInfoWithoutJobSes();
		} catch (Exception e) {
			m_log.warn("refresh() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			return;
		}
		for (NotifyRelationInfoEntity nri : nriList) {
			String notifyGroupId = nri.getId().getNotifyGroupId();
			// ジョブセッションで利用されている通知グループの場合は、キャッシュしない。
			if(notifyGroupId.startsWith(HinemosModuleConstant.JOB_SESSION + "_") == false){
				List<NotifyRelationInfo> notifyList = notifyMap.get(notifyGroupId);
				if (notifyList == null) {
					notifyList = new ArrayList<NotifyRelationInfo>();
					NotifyRelationInfo info
					= new NotifyRelationInfo(
							nri.getId().getNotifyGroupId(),
							nri.getId().getNotifyId(),
							nri.getNotifyType(),
							nri.getNotifyFlg());
					notifyList.add(info);
					notifyMap.put(notifyGroupId, notifyList);
				} else {
					NotifyRelationInfo info
					= new NotifyRelationInfo(
							nri.getId().getNotifyGroupId(),
							nri.getId().getNotifyId(),
							nri.getNotifyType(),
							nri.getNotifyFlg());
					notifyList.add(info);
				}
			}
		}
		for (List<NotifyRelationInfo> notifyList : notifyMap.values()) {
			if (notifyList == null) {
				continue;
			}
			Collections.sort(notifyList);
		}
		m_notifyMap = notifyMap;
		m_log.info("refresh NotifyRelationCache. " + (System.currentTimeMillis() - start)
				+ "ms. size=" + notifyMap.size());
	}
}
