/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.collectiverun.factory;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.collectiverun.bean.CommandInfo;
import com.clustercontrol.collectiverun.bean.CommandParameterInfo;
import com.clustercontrol.collectiverun.bean.EndStatusInfo;
import com.clustercontrol.collectiverun.bean.ItemConstant;
import com.clustercontrol.collectiverun.bean.ParameterInfo;
import com.clustercontrol.collectiverun.bean.ParameterSelectInfo;
import com.clustercontrol.collectiverun.bean.TreeItem;
import com.clustercontrol.collectiverun.bean.TypeInfo;
import com.clustercontrol.collectiverun.model.CrunCmdMstEntity;
import com.clustercontrol.collectiverun.model.CrunCmdParamMstEntity;
import com.clustercontrol.collectiverun.model.CrunEndMstEntity;
import com.clustercontrol.collectiverun.model.CrunParamMstEntity;
import com.clustercontrol.collectiverun.model.CrunParamSelectMstEntity;
import com.clustercontrol.collectiverun.model.CrunTypeMstEntity;
import com.clustercontrol.collectiverun.util.QueryUtil;
import com.clustercontrol.fault.CollectiveRunNotFound;
import com.clustercontrol.util.Messages;

/**
 * マスタ検索クラス
 *
 * @version 2.1.0
 * @since 2.1.0
 */
public class SelectMaster {
	private static Log m_log = LogFactory.getLog( SelectMaster.class );

	/**
	 * ツリー情報取得
	 * 
	 * @return
	 */
	public TreeItem getTree() {

		//TreeItemの最上位インスタンスを作成
		CommandInfo commandInfo = new CommandInfo();
		commandInfo.setName("");
		commandInfo.setType(ItemConstant.TYPE_COMPOSITE);
		commandInfo.setOrderNo(0);
		TreeItem tree = new TreeItem(null, commandInfo);

		//コマンドツリーのルートを生成
		commandInfo = new CommandInfo();
		commandInfo.setName(ItemConstant.STRING_COMMAND_COMPOSITE);
		commandInfo.setType(ItemConstant.TYPE_COMMAND_COMPOSITE);
		commandInfo.setOrderNo(0);
		TreeItem cmdTree = new TreeItem(tree, commandInfo);

		List<CrunTypeMstEntity> collectionTypeMst
		= QueryUtil.getAllCrunTypeMst();

		Iterator<CrunTypeMstEntity> itrCrunTypeMst = collectionTypeMst.iterator();
		while(itrCrunTypeMst.hasNext()){
			//種別を取得
			CrunTypeMstEntity type = itrCrunTypeMst.next();
			//種別ツリーを作成
			createTypeTree(type, cmdTree, Locale.getDefault());
		}

		//パラメータツリーのルートを生成
		ParameterInfo parameterInfo = new ParameterInfo();
		parameterInfo.setName(ItemConstant.STRING_PARAMETER_COMPOSITE);
		parameterInfo.setType(ItemConstant.TYPE_PARAMETER_COMPOSITE);
		parameterInfo.setOrderNo(1);
		TreeItem paramTree = new TreeItem(tree, parameterInfo);

		List<CrunParamMstEntity> collectionParamMst = QueryUtil.getAllCrunParamMst();
		Iterator<CrunParamMstEntity> itrCrunParamMst = collectionParamMst.iterator();
		while(itrCrunParamMst.hasNext()){
			//種別を取得
			CrunParamMstEntity param = itrCrunParamMst.next();
			//種別ツリーを作成
			createParamTree(param, paramTree, Locale.getDefault());
		}

		return tree;
	}

	/**
	 * 種別情報作成
	 * 
	 * @param type
	 * @param parent
	 * @param locale
	 */
	private void createTypeTree(CrunTypeMstEntity type, TreeItem parent, Locale locale) {

		//TypeInfoを作成
		TypeInfo info = new TypeInfo();
		info.setId(type.getTypeId());
		info.setName(Messages.getString(type.getNameId(), locale));
		info.setType(ItemConstant.TYPE_TYPE);
		info.setOrderNo(type.getOrderNo().intValue());

		TreeItem item = new TreeItem(parent, info);

		//種別に紐づく全コマンドマスタ取得
		Collection<CrunCmdMstEntity> collection = type.getCrunCmdMstEntities();
		if(collection != null && collection.size() > 0){
			Iterator<CrunCmdMstEntity> itr = collection.iterator();
			while(itr.hasNext()){
				//コマンドマスタを取得
				CrunCmdMstEntity cmd = itr.next();

				//コマンドツリーを作成
				createCommandTree(cmd, item, locale);
			}
		}
	}

	/**
	 * コマンド情報作成
	 * 
	 * @param cmd
	 * @param parent
	 * @param locale
	 */
	private void createCommandTree(CrunCmdMstEntity cmd, TreeItem parent, Locale locale) {

		//CommandInfoを作成
		CommandInfo info = new CommandInfo();
		info.setId(cmd.getCommandId());
		info.setName(Messages.getString(cmd.getNameId(), locale));
		info.setType(ItemConstant.TYPE_COMMAND);
		info.setOrderNo(cmd.getOrderNo().intValue());
		info.setCommand(cmd.getCommand());
		info.setCommandType(cmd.getCommandType().intValue());

		//コマンドに紐づく終了状態マスタ取得
		ArrayList<EndStatusInfo> list = new ArrayList<EndStatusInfo>();
		Collection<CrunEndMstEntity> collectionEndMst = cmd.getCrunEndMstEntities();
		if(collectionEndMst != null && collectionEndMst.size() > 0){
			Iterator<CrunEndMstEntity> itr = collectionEndMst.iterator();
			while(itr.hasNext()){
				//終了状態マスタを取得
				CrunEndMstEntity endStatus = itr.next();

				//EndStatusInfoを作成
				EndStatusInfo endStatusInfo = new EndStatusInfo();
				endStatusInfo.setEndStatus(endStatus.getId().getEndStatus());
				endStatusInfo.setStartRangeValue(endStatus.getEndValueFrom().intValue());
				endStatusInfo.setEndRangeValue(endStatus.getEndValueTo().intValue());

				list.add(endStatusInfo);
			}
		}
		info.setEndStatus(list);

		TreeItem item = new TreeItem(parent, info);

		//コマンドに紐づくコマンドパラメータマスタ取得
		Collection<CrunCmdParamMstEntity> collectionCmdParamMst = cmd.getCrunCmdParamMstEntities();
		if(collectionCmdParamMst != null && collectionCmdParamMst.size() > 0){
			Iterator<CrunCmdParamMstEntity> itr = collectionCmdParamMst.iterator();
			while(itr.hasNext()){
				//コマンドパラメータマスタを取得
				CrunCmdParamMstEntity param = itr.next();

				//コマンドパラメータツリーを作成
				createCommandParameterTree(param, item, locale);
			}
		}
	}

	/**
	 * コマンドパラメータ情報作成
	 * 
	 * @param cmdParam
	 * @param parent
	 * @param locale
	 */
	private void createCommandParameterTree(CrunCmdParamMstEntity cmdParam, TreeItem parent, Locale locale) {

		//CommandParameterInfoを作成
		CommandParameterInfo info = new CommandParameterInfo();
		info.setId(cmdParam.getId().getParamId());

		CrunParamMstEntity param = cmdParam.getCrunParamMstEntity();
		info.setName(Messages.getString(param.getNameId(), locale));

		info.setPrefix(cmdParam.getParamPrefix());
		info.setOrderNo(cmdParam.getOrderNo().intValue());
		info.setType(ItemConstant.TYPE_COMMAND_PARAMETER);

		new TreeItem(parent, info);
	}

	/**
	 * パラメータ情報作成
	 * 
	 * @param param
	 * @param parent
	 * @param locale
	 */
	private void createParamTree(CrunParamMstEntity param, TreeItem parent, Locale locale) {
		//ParameterInfoを作成
		ParameterInfo info = new ParameterInfo();
		info.setId(param.getParamId());
		info.setName(Messages.getString(param.getNameId(), locale));
		info.setType(ItemConstant.TYPE_PARAMETER);
		info.setParamType(param.getParamType().intValue());

		TreeItem item = new TreeItem(parent, info);

		//パラメータに紐づく選択肢マスタ取得
		Collection<CrunParamSelectMstEntity> collection = param.getCrunParamSelectMstEntities();
		if(collection != null && collection.size() > 0){
			Iterator<CrunParamSelectMstEntity> itr = collection.iterator();
			while(itr.hasNext()){
				//コマンドマスタを取得
				CrunParamSelectMstEntity select = itr.next();

				//選択肢ツリーを作成
				createParamSelectTree(select, item, locale);
			}
		}
	}

	/**
	 * パラメータ選択肢情報作成
	 * 
	 * @param select
	 * @param parent
	 * @param locale
	 */
	private void createParamSelectTree(CrunParamSelectMstEntity select, TreeItem parent, Locale locale) {
		//ParameterSelectInfoを作成
		ParameterSelectInfo info = new ParameterSelectInfo();
		info.setId(select.getId().getParamId());
		info.setName(Messages.getString(select.getNameId(), locale));
		info.setType(ItemConstant.TYPE_SELECT_ITEM);
		info.setOrderNo(select.getId().getOrderNo().intValue());
		info.setValue(select.getParamValue());

		new TreeItem(parent, info);
	}

	/**
	 * 種別IDから文字列を取得
	 * 
	 * @param typeId
	 * @param locale
	 * @return
	 * @throws CollectiveRunNotFound
	 */
	public static String typeIdToString(String typeId, Locale locale) throws CollectiveRunNotFound {
		CrunTypeMstEntity type = QueryUtil.getCrunTypeMstPK(typeId);
		return Messages.getString(type.getNameId(), locale);
	}

	/**
	 * 文字列から種別IDを取得
	 * 
	 * @param string
	 * @param locale
	 * @return
	 * @throws CollectiveRunNotFound
	 * 
	 */
	public static String stringToTypeId(String string, Locale locale) {
		List<CrunTypeMstEntity> collection = QueryUtil.getAllCrunTypeMst();
		Iterator<CrunTypeMstEntity> itr = collection.iterator();
		while(itr.hasNext()){
			//種別を取得
			CrunTypeMstEntity type = itr.next();
			String name = Messages.getString(type.getNameId(), locale);
			if(name.equals(string))
				return type.getTypeId();
		}
		return null;
	}

	/**
	 * 種別文字列リストを取得
	 * 
	 * @return
	 */
	public static ArrayList<String> getTypeStringList() {
		ArrayList<String> list = new ArrayList<String>();
		List<CrunTypeMstEntity> collection
		= QueryUtil.getAllCrunTypeMstSortOrderNo();

		Iterator<CrunTypeMstEntity> itr = collection.iterator();
		while(itr.hasNext()){
			//種別を取得
			CrunTypeMstEntity type = itr.next();
			String name = Messages.getString(type.getNameId(), Locale.getDefault());
			list.add(name);
		}
		return list;
	}

	/**
	 * 種別情報のハッシュマップを取得
	 * 
	 * @return
	 */
	public static HashMap<String, String> getTypeHashMap() {

		HashMap<String, String> map = new HashMap<String, String>();
		List<CrunTypeMstEntity> collection
		= QueryUtil.getAllCrunTypeMst();

		Iterator<CrunTypeMstEntity> itr = collection.iterator();
		while(itr.hasNext()){
			//種別を取得
			CrunTypeMstEntity type = itr.next();
			map.put(type.getTypeId(), Messages.getString(type.getNameId(), Locale.getDefault()));
			map.put(Messages.getString(type.getNameId(), Locale.getDefault()), type.getTypeId());
		}
		return map;
	}
}
