/*
Copyright (C) 2013 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */
package com.clustercontrol.cloud.azure.factory;


import java.net.URI;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.TimeZone;
import java.util.regex.Pattern;

import javax.persistence.Query;
import javax.xml.bind.JAXBContext;
import javax.xml.bind.Unmarshaller;

import org.apache.log4j.Logger;

import com.clustercontrol.cloud.CloudManagerFault;
import com.clustercontrol.cloud.Filter;
import com.clustercontrol.cloud.ICloudContext;
import com.clustercontrol.cloud.IResourceManagement;
import com.clustercontrol.cloud.IResourceManagement.InstanceBackup.BackupedData;
import com.clustercontrol.cloud.IResourceManagement.Storage.StorageAttachment;
import com.clustercontrol.cloud.InternalManagerError;
import com.clustercontrol.cloud.SessionService;
import com.clustercontrol.cloud.azure.AzureOptionPropertyConstants;
import com.clustercontrol.cloud.azure.bean.AzurePendingInstances;
import com.clustercontrol.cloud.azure.bean.AzurePendingStartInstance;
import com.clustercontrol.cloud.azure.bean.AzurePendingStopInstance;
import com.clustercontrol.cloud.azure.factory.ListBlobsResult.Blob;
import com.clustercontrol.cloud.azure.util.AzureConstants;
import com.clustercontrol.cloud.azure.util.AzureUtil;
import com.clustercontrol.cloud.bean.Image;
import com.clustercontrol.cloud.bean.InstanceStateChange;
import com.clustercontrol.cloud.bean.InstanceStateKind;
import com.clustercontrol.cloud.bean.PlatformKind;
import com.clustercontrol.cloud.bean.Snapshot;
import com.clustercontrol.cloud.bean.StorageAttachmentStateKind;
import com.clustercontrol.cloud.bean.StorageStateKind;
import com.clustercontrol.cloud.bean.Tag;
import com.clustercontrol.cloud.bean.Zone;
import com.clustercontrol.cloud.dao.CloudInstanceDao;
import com.clustercontrol.cloud.dao.CloudStorageBackupDao;
import com.clustercontrol.cloud.persistence.EntityManagerEx;
import com.clustercontrol.cloud.persistence.Transactional;

import com.microsoft.windowsazure.Configuration;
import com.microsoft.windowsazure.core.OperationStatus;
import com.microsoft.windowsazure.core.OperationStatusResponse;
import com.microsoft.windowsazure.exception.ServiceException;
import com.microsoft.windowsazure.management.AffinityGroupOperations;
import com.microsoft.windowsazure.management.ManagementClient;
import com.microsoft.windowsazure.management.ManagementService;
import com.microsoft.windowsazure.management.compute.ComputeManagementService;
import com.microsoft.windowsazure.management.compute.HostedServiceOperations;
import com.microsoft.windowsazure.management.compute.DeploymentOperations;
import com.microsoft.windowsazure.management.compute.ComputeManagementClient;
import com.microsoft.windowsazure.management.compute.VirtualMachineOSImageOperations;
import com.microsoft.windowsazure.management.compute.models.DataVirtualHardDisk;
import com.microsoft.windowsazure.management.compute.models.DeploymentGetResponse;
import com.microsoft.windowsazure.management.compute.models.DeploymentSlot;
import com.microsoft.windowsazure.management.compute.models.HostedServiceListResponse;
import com.microsoft.windowsazure.management.compute.models.Role;
import com.microsoft.windowsazure.management.compute.models.RoleInstance;
import com.microsoft.windowsazure.management.compute.models.VirtualMachineDiskGetResponse;
import com.microsoft.windowsazure.management.compute.models.VirtualMachineDiskListResponse;
import com.microsoft.windowsazure.management.compute.models.VirtualMachineOSImageCreateParameters;
import com.microsoft.windowsazure.management.compute.models.VirtualMachineOSImageCreateResponse;
import com.microsoft.windowsazure.management.compute.models.VirtualMachineOSImageListResponse.VirtualMachineOSImage;
import com.microsoft.windowsazure.management.compute.models.HostedServiceListResponse.HostedService;
import com.microsoft.windowsazure.management.compute.models.VirtualMachineDiskListResponse.VirtualMachineDisk;
import com.microsoft.windowsazure.management.models.AffinityGroupListResponse;
import com.microsoft.windowsazure.management.storage.StorageAccountOperations;
import com.microsoft.windowsazure.management.storage.StorageManagementClient;
import com.microsoft.windowsazure.management.storage.StorageManagementService;
import com.microsoft.windowsazure.management.storage.models.StorageAccount;
import com.microsoft.windowsazure.management.storage.models.StorageAccountListResponse;
import com.microsoft.windowsazure.services.blob.BlobContract;
import com.microsoft.windowsazure.services.blob.BlobService;
import com.microsoft.windowsazure.services.blob.models.CreateBlobOptions;
import com.microsoft.windowsazure.services.blob.models.GetBlobResult;

@Transactional(Transactional.TransactionType.Supported)
public class AzureResourceManagement implements IResourceManagement, AzureConstants {
	public static String RT_Instance = "AZURE";
	public static String RT_Image = "AZURE_Template";
	public static String RT_Storage = "AZURE_Volume";
	public static String RT_Snapshot = "AZURE_Snapshot";
	public static String RT_LoadBalancer = "LBA";
	public static String RT_InstanceBackup = "InstanceBackup";
	public static String RT_StorageBackup = "StorageBackup";

	private IRegion region;
	private ICredential credential;
	@SuppressWarnings("unused")
	private IStore store;

	@Override
	public void setAccessDestination(ICredential credential, IRegion region) {
		this.region = region;
		this.credential = credential;
	}

	@Override
	public void setStore(IStore store) {
		this.store = store;
	}

	@Override
	public ICredential getCledential() {
		return credential;
	}

	@Override
	public IRegion getRegion() {
		return region;
	}

	@Override
	public void disconnect() {
	}

	@Override
	public Instance createInstance(String id, String name, String flavor, String imageId, String zone, String instanceDetail, List<Tag> tags) throws CloudManagerFault {
		Logger logger = Logger.getLogger(this.getClass());
		logger.info("Create Instance start");
		logger.info("Param instanceId:" + id + " instanceName:" + name + " ImageId:" + imageId + " Zone:" + zone);
		
		Instance instance = new Instance();
		instance.setName(name);
		instance.setFlavor(flavor);
		instance.setImageId(imageId);
		instance.setZone(zone);
		instance.setTags(tags);
		
		for(Tag tag:tags){
			logger.info("Tag key:" + tag.getKey() + " Value:" + tag.getValue());
			if(tag.getKey().equals("PlatForm")){
				instance.setPlatform(tag.getValue().toLowerCase().equals("linux")?PlatformKind.linux:PlatformKind.windows);
			}else if(tag.getKey().equals("NodeName")){
				instance.setHostName(tag.getValue());
				instance.setInstanceId(tag.getValue());
				
			}
		}

		instance.setState(InstanceStateKind.pending);
		
		AzurePendingInstances cond = AzurePendingInstances.getInstance();
		cond.addAzureInstance(instance,getRegion().getName(),getCledential().getAccessKey());
		
		logger.info("Create Instance end");
		return instance;
	}

	private Configuration getConfiguration() throws Exception{
		Configuration config = AzureUtil.getConfiguration(getCledential().getAccessKey(), getCledential().getSecretKey());;
		return config;
	}

	@Override
	public void deleteInstance(String instanceId) throws CloudManagerFault {
		Logger logger = Logger.getLogger(this.getClass());

		try {
	    	logger.info("deleteInstance start: " + instanceId );
			AzureDeleteInstance azureDeleteInstance = new AzureDeleteInstance(getConfiguration(),instanceId);
			if (AzureOptionPropertyConstants.azure_api_mode.value().equalsIgnoreCase("async")) {
				AzureAsyncExecution.getSingleton().put(azureDeleteInstance);
			}else{
				azureDeleteInstance.execute();
			}			
		}catch(ServiceException e){
			logger.error("deleteInstance: "+ instanceId + " (" + e.getErrorCode() +") "+ e.getMessage());
			throw new CloudManagerFault(e.getMessage(), e.getErrorCode(), e);
		}catch(Exception e){
			logger.error("deleteInstance: "+ instanceId + " " + e.getMessage());
			throw new CloudManagerFault(e.getMessage(), e);
		} finally {
	    	logger.info("deleteInstance end: " + instanceId );
	    }		
	}
	
	protected static PlatformKind getOSType(String ostype) {
		if (Pattern.compile(AzureOptionPropertyConstants.azure_os_win_pattern.value(), Pattern.CASE_INSENSITIVE).matcher(ostype).matches())
			return PlatformKind.windows;
		else if (Pattern.compile(AzureOptionPropertyConstants.azure_os_linux_pattern.value(), Pattern.CASE_INSENSITIVE).matcher(ostype).matches())
			return PlatformKind.linux;
		return PlatformKind.other;
	}
	
	@Override
	public Instance getInstance(String instanceId) throws CloudManagerFault {
		Logger logger = Logger.getLogger(this.getClass());
		Instance instance = null;
		try{
			if(instanceId != null){
				logger.debug("getting Instance: "+getRegion().getName() + " " + instanceId);
				
				AzurePendingInstances cond = AzurePendingInstances.getInstance();
				for(Instance pendingInstance:cond.getAzureInstances(getRegion().getName(),getCledential().getAccessKey())){
					if(pendingInstance.getInstanceId().equals(instanceId)){
						instance = pendingInstance;
					}
				}
				
				if(instance==null){
					instance = new Instance();
					List<String> instanceIds = new ArrayList<String>();
					instanceIds.add(instanceId);
					CloudInstanceDao dao = getInstanceDAO(instanceId);
					Configuration config = getConfiguration();
					ComputeManagementClient computeManagementClient = ComputeManagementService.create(config);
					DeploymentOperations deploymentOperations = computeManagementClient.getDeploymentsOperations();
					for(HostedService hostService:computeManagementClient.getHostedServicesOperations().list().getHostedServices()){
						if(hostService.getServiceName().equals(dao.getZone())){
							for (DeploymentSlot slot:AzureUtil.getDeploymentSlots()) {
				 				DeploymentGetResponse deploymentGetResponse = null;
					 			try {
					 				deploymentGetResponse = deploymentOperations.getBySlot(hostService.getServiceName(),slot);
					 			}
					 			catch (ServiceException e) {
					 				if (e.getErrorCode().equals("ResourceNotFound"))
					 					continue;
					 				logger.error("getInstances: (" + e.getErrorCode() +") "+ e.getMessage());
					 				throw new CloudManagerFault(e.getMessage(), e.getErrorCode(), e);
					 			}
							
								for (Role role: deploymentGetResponse.getRoles()) {
									if(role.getRoleName().equals(instanceId)){
										for(RoleInstance roleinstance:deploymentGetResponse.getRoleInstances()){
											if(roleinstance.getRoleName().equalsIgnoreCase(role.getRoleName())){
												List<Tag> tags = new ArrayList<>();
												instance=convertInstance(hostService,role,roleinstance,slot.toString(),tags);
											}
										}
										return instance;
									}
								}
				 			}
						}
					}
				}
				
				logger.debug("Successful in getting Instance: "+getRegion().getName() + " " + instanceId);
			}else{
				instance = new Instance();
				instance.setState(InstanceStateKind.terminated);
			}
		}catch(Exception e){
			e.printStackTrace();
		}
		return instance;
	}

	@Override
	public List<Instance> getInstances(String...instanceIds) throws CloudManagerFault {
		return getInstances(Arrays.asList(instanceIds));
	}
	
	public List<String> getInstanceByCloudService(String cloudServiceName) throws CloudManagerFault {
		List<String> instanceIDs = new ArrayList<>();
		for(Instance instance:getInstances(new ArrayList<String>())){
			if(instance.getZone().equals(cloudServiceName)){
				instanceIDs.add(instance.getInstanceId());
			}
		}
		return instanceIDs;
	}
	
	@Override
	public List<Instance> getInstances(List<String> instanceIds) throws CloudManagerFault {
		Logger logger = Logger.getLogger(this.getClass());
		List<Instance> instances = new ArrayList<>();
		try {
        	logger.debug("getting Instances: "+getRegion().getName() + (!instanceIds.isEmpty() ? (" "+instanceIds.toString()) : ""));        	

        	AzurePendingInstances cond = AzurePendingInstances.getInstance();
			instances.addAll(cond.getAzureInstances(getRegion().getName(),getCledential().getAccessKey()));
        	
        	Configuration config = getConfiguration();
			ManagementClient managementClient = ManagementService.create(config);
			ComputeManagementClient computeManagementClient = ComputeManagementService.create(config);

			HostedServiceOperations hostedServicesOperations = computeManagementClient.getHostedServicesOperations();
			DeploymentOperations deploymentOperations = computeManagementClient.getDeploymentsOperations();
			AffinityGroupOperations affinityGroupOperations = managementClient.getAffinityGroupsOperations();
			AffinityGroupListResponse affinityGroupListResponse = affinityGroupOperations.list();
	 		HostedServiceListResponse hostedServiceListResponse = hostedServicesOperations.list();
	 		for (HostedService hostService: hostedServiceListResponse.getHostedServices()) {
				if (AzureUtil.compareLocation(getRegion().getName(),hostService.getProperties().getLocation(),affinityGroupListResponse.getAffinityGroups())) {
		 			for (DeploymentSlot slot:AzureUtil.getDeploymentSlots()) {
		 				DeploymentGetResponse deploymentGetResponse = null;
			 			try {
			 				deploymentGetResponse = deploymentOperations.getBySlot(hostService.getServiceName(),slot);
			 			}
			 			catch (ServiceException e) {
			 				if (e.getErrorCode().equals("ResourceNotFound"))
			 					continue;
			 				logger.error("getInstances: (" + e.getErrorCode() +") "+ e.getMessage());
			 				throw new CloudManagerFault(e.getMessage(), e.getErrorCode(), e);
			 			}
					
						for (Role role: deploymentGetResponse.getRoles()) {
							for(RoleInstance roleinstance:deploymentGetResponse.getRoleInstances()){
								if (!instanceIds.isEmpty()) {
									if (!instanceIds.contains(roleinstance.getRoleName())) {
										continue;
									}
								}
								if(roleinstance.getRoleName().equalsIgnoreCase(role.getRoleName())){
									if(!cond.isExist(role.getRoleName(),getRegion().getName(),getCledential().getAccessKey())){
										List<Tag> tags = new ArrayList<>();
										instances.add(convertInstance(hostService,role,roleinstance,slot.toString(),tags));
									}
								}
							}
						}
		 			}
		 		}
			}
        	logger.debug("Successful in getting Instances:"+getRegion().getName() + (!instanceIds.isEmpty() ? (" "+instanceIds.toString()) : ""));

		}catch(ServiceException e){
			logger.error("getInstances: (" + e.getErrorCode() +") "+ e.getMessage());
			throw new CloudManagerFault(e.getMessage(), e.getErrorCode(), e);
		}catch(Exception e){
			logger.error("getInstances: " + e.getMessage());
			throw new CloudManagerFault(e.getMessage(), e);
	    }
		return instances;
	}

	protected Instance convertInstance(HostedService hostedService,com.microsoft.windowsazure.management.compute.models.Role role, com.microsoft.windowsazure.management.compute.models.RoleInstance roleinstance, String slot, List<Tag> tags) throws CloudManagerFault {
		InstanceStateKind state;
		Instance instance = new Instance();
		instance.setResourceType(RT_Instance);
		instance.setInstanceId(role.getRoleName());
		instance.setName(role.getRoleName());
		instance.setFlavor(slot + (hostedService.getProperties() == null ? "" : (hostedService.getProperties().getAffinityGroup() == null ? "" : (hostedService.getProperties().getAffinityGroup().equals("") ? "" : ("-" + hostedService.getProperties().getAffinityGroup())))));
		instance.setZone(hostedService.getServiceName());
		instance.setImageId(role.getOSVirtualHardDisk().getSourceImageName());
		instance.setTags(tags);
		if (role.getOSVirtualHardDisk().getOperatingSystem().isEmpty())
			throw new InternalManagerError("not found ostype of vm created. vm=" + role.getRoleName()); 
		instance.setPlatform(getOSType(role.getOSVirtualHardDisk().getOperatingSystem()));

		if(roleinstance.getPowerState().toString().toLowerCase().equals("started")){
			state=InstanceStateKind.running;
		}else{
			state=InstanceStateKind.stopped;
		}
		instance.setState(state);
		
		instance.setIpAddress(roleinstance.getIPAddress()==null?null:roleinstance.getIPAddress().getHostAddress());

		instance.setActualResource(role);

		return instance;
	}

	@Override
	public InstanceStateChange startInstance(String instanceId) throws CloudManagerFault {
		
		Logger logger = Logger.getLogger(this.getClass());
		try{
			logger.info("startInstance() start:" + instanceId);
			AzurePendingStartInstance azureStartInstances = new AzurePendingStartInstance(getConfiguration(),instanceId,getCledential().getAccessKey());
			if (AzureOptionPropertyConstants.azure_api_mode.value().equalsIgnoreCase("async")) {
				AzureAsyncExecution.getSingleton().put(azureStartInstances);
			}else{
				azureStartInstances.execute();
			}
		}catch(ServiceException e){
			logger.error("startInstance: "+ instanceId + " (" + e.getErrorCode() +") "+ e.getMessage());
			throw new CloudManagerFault(e.getMessage(), e.getErrorCode(), e);
		}catch(Exception e){
			logger.error("startInstance: "+ instanceId +" "+ e.getMessage());
			throw new CloudManagerFault(e.getMessage(), e);
	    }
		finally {
			logger.info("startInstance() end:" + instanceId);
		}
		
		InstanceStateChange change = new InstanceStateChange();
		change.setPreviousState(InstanceStateKind.running);
		change.setCurrentState(InstanceStateKind.running);
		change.setInstanceId(instanceId);
		return change;
	}

	@Override
	public InstanceStateChange stopInstance(String instanceId) throws CloudManagerFault {
 		
		Logger logger = Logger.getLogger(this.getClass());
		try{
			logger.info("stopInstance() start:" + instanceId);
			AzurePendingStopInstance azureStopInstance = new AzurePendingStopInstance(getConfiguration(),instanceId,getCledential().getAccessKey());
			if (AzureOptionPropertyConstants.azure_api_mode.value().equalsIgnoreCase("async")) {
				AzureAsyncExecution.getSingleton().put(azureStopInstance);
			}else{
				azureStopInstance.execute();
			}
		}catch(ServiceException e){
			logger.error("stopInstance: (" + e.getErrorCode() +") "+ e.getMessage());
			throw new CloudManagerFault(e.getMessage(), e.getErrorCode(), e);
		}catch(Exception e){
			logger.error("stopInstance: " + e.getMessage());
			throw new CloudManagerFault(e.getMessage(), e);
	    }
		finally {
			logger.info("stopInstance() end:" + instanceId);
		}
		
		InstanceStateChange change = new InstanceStateChange();
		change.setPreviousState(InstanceStateKind.shutting_down);
		change.setCurrentState(InstanceStateKind.shutting_down);
		change.setInstanceId(instanceId);
		return change;
	}
	

	@Override
	public List<String> getInstanceFlavors() throws CloudManagerFault {
		List<String> instanceTypes = new ArrayList<>();
		/*for (InstanceType type: InstanceType.values()) {
			instanceTypes.add(type.toString());
		}*/
		return instanceTypes;
	}

	@Override
	public List<Zone> getZones() throws CloudManagerFault {
		HostedServiceOperations hostedServicesOperations;
		ComputeManagementClient computeManagementClient;
		List<Zone> zones = new ArrayList<Zone>(); 
		Logger logger = Logger.getLogger(this.getClass());
		
		try {
			Configuration config = getConfiguration();
			computeManagementClient = ComputeManagementService.create(config);
			
	 		hostedServicesOperations = computeManagementClient.getHostedServicesOperations();
	 		HostedServiceListResponse hostedServiceListResponse = hostedServicesOperations.list();
	 		for(HostedService hostedService : hostedServiceListResponse.getHostedServices()){
	 			Zone zone = new Zone();
	 			zone.setName(hostedService.getServiceName());
	 			zones.add(zone);
	 		}

		}catch(ServiceException e){
			logger.error("getZones: (" + e.getErrorCode() +") "+ e.getMessage());
			throw new CloudManagerFault(e.getMessage(), e.getErrorCode(), e);
		}catch(Exception e){
			logger.error("getZones: " + e.getMessage());
			throw new CloudManagerFault(e.getMessage(), e);
	    }
		
		logger.info("successful in get Cloud Services");
		return zones;
	}

	@Override
	public void attachStorage(final String instanceId, final String storageId, String deviceName) throws CloudManagerFault {
	}

	@Override
	public void detachStorage(final String instanceId, final String storageId) throws CloudManagerFault {
		Logger logger = Logger.getLogger(this.getClass());
		try{
			logger.info("detachStorage() start");
			AzureDetachStorage azureDetachStorage = new AzureDetachStorage(getConfiguration(),instanceId,storageId);
			if (AzureOptionPropertyConstants.azure_api_mode.value().equalsIgnoreCase("async")) {
				AzureAsyncExecution.getSingleton().put(azureDetachStorage);
			}else{
				azureDetachStorage.execute();
			}
		}catch(ServiceException e){
			logger.error("detachStorage: (" + e.getErrorCode() +") "+ e.getMessage());
			throw new CloudManagerFault(e.getMessage(), e.getErrorCode(), e);
		}catch(Exception e){
			logger.error("detachStorage: " + e.getMessage());
			throw new CloudManagerFault(e.getMessage(), e);
	    }
		finally {
			logger.info("detachStorage() end");			
		}
	}

	@Override
	public Storage createStorage(String name, String flavor, int size, String snapshotId, String zone, String storageDetail) throws CloudManagerFault {
		Storage storage = new Storage();
		storage.setName(name);
		storage.setStorageId(name);
		storage.setSize(size);
		storage.setSnapshotId(snapshotId);
		storage.setZone(zone);
		storage.setResourceType(RT_Storage);
		storage.setState(StorageStateKind.creating);
		storage.setFlavor("Data");
		StorageAttachment sa = new StorageAttachment();
		sa = new StorageAttachment(name,"DataDisk",StorageAttachmentStateKind.attached,new Date());
		storage.setStorageAttachment(sa);
		return storage;
	}
	
	@Override
	public void deleteStorage(String storageId) throws CloudManagerFault {

		Logger logger = Logger.getLogger(this.getClass());
		
		try {
			logger.info("deleteStorage() start");
			AzureDeleteStorage azureDeleteStorage = new AzureDeleteStorage(getConfiguration(),storageId,getRegion().getName());
			if (AzureOptionPropertyConstants.azure_api_mode.value().equalsIgnoreCase("async")) {
				AzureAsyncExecution.getSingleton().put(azureDeleteStorage);
			}else{
				azureDeleteStorage.execute();
			}
		}catch(ServiceException e){
			logger.error("deleteStorage: " + storageId + "(" + e.getErrorCode() +") "+ e.getMessage());
			throw new CloudManagerFault(e.getMessage(), e.getErrorCode(), e);
		}catch(Exception e){
			logger.error("deleteStorage: " + storageId + " " + e.getMessage());
			throw new CloudManagerFault(e.getMessage(), e);
	    } finally {
	    	logger.info("deleteStorage end: " + storageId );
	    }
	}

	@Override
	public Storage getStorage(String storageId) throws CloudManagerFault {
		return new Storage();
	}

	@Override
	public List<Storage> getStorages(String... storageIds) throws CloudManagerFault {
		return getStorages(Arrays.asList(storageIds));
	}
	
	public static Storage convertStorage(VirtualMachineDisk dataDisk,StorageAccount storageAccount){
		Storage storage = new Storage();
		storage.setStorageId(dataDisk.getName());
		storage.setName(storageAccount.getName()+"-"+dataDisk.getName());
		storage.setSize(dataDisk.getLogicalSizeInGB());
		storage.setZone(dataDisk.getUsageDetails()==null?"":dataDisk.getUsageDetails().getHostedServiceName());
		storage.setResourceType(RT_Storage);
		storage.setFlavor(dataDisk.getOperatingSystemType()==null?"Data":dataDisk.getOperatingSystemType());
		StorageAttachment sa = new StorageAttachment();
		if(dataDisk.getUsageDetails() != null){
			storage.setState(StorageStateKind.in_use);
			sa = new StorageAttachment(dataDisk.getUsageDetails().getRoleName(),dataDisk.getOperatingSystemType()==null?"DataDisk":"OSDisk",StorageAttachmentStateKind.attached,new Date());
		}else{
			storage.setState(StorageStateKind.available);
			sa = null;
		}
		storage.setStorageAttachment(sa);
		return storage;
	}
	
	@Override
	public List<Storage> getStorages(List<String> storageIds) throws CloudManagerFault {
		List<Storage> storages = new ArrayList<>();
		Logger logger = Logger.getLogger(this.getClass());
		
		try {
        	logger.debug("getting Storages: "+getRegion().getName() + (!storageIds.isEmpty() ? (" "+storageIds.toString()) : ""));

        	Configuration config = getConfiguration();
			ManagementClient managementClient = ManagementService.create(config);
			ComputeManagementClient computeManagementClient = ComputeManagementService.create(config);
			StorageManagementClient storageManagementClient = StorageManagementService.create(config);

			AffinityGroupOperations affinityGroupOperations = managementClient.getAffinityGroupsOperations();
	    	StorageAccountOperations storageAccountOperations = storageManagementClient.getStorageAccountsOperations();
			
	 		AffinityGroupListResponse affinityGroupListResponse = affinityGroupOperations.list();
	    	StorageAccountListResponse storageServiceListResponse = storageAccountOperations.list();
        	VirtualMachineDiskListResponse vmds = computeManagementClient.getVirtualMachineDisksOperations().listDisks();
        	computeManagementClient.getHostedServicesOperations().list();
        	for(VirtualMachineDisk vmd : vmds.getDisks()){
           		if(AzureUtil.compareLocation(getRegion().getName(),vmd.getLocation(),vmd.getAffinityGroup(),affinityGroupListResponse.getAffinityGroups())) {
            		for (StorageAccount sa: storageServiceListResponse.getStorageAccounts()) {
        				String match = sa.getUri().toString().substring(sa.getUri().toString().lastIndexOf("/")+1);
                		if(vmd.getMediaLinkUri().toString().indexOf("http://"+match+".")==0
                				|| vmd.getMediaLinkUri().toString().indexOf("https://"+match+".")==0 ){
                			storages.add(convertStorage(vmd,sa));
                		}
                	}
        		}
        	}
        	logger.debug("Successful in getting Storages:"+getRegion().getName() + (!storageIds.isEmpty() ? (" "+storageIds.toString()) : ""));

		}catch(ServiceException e){
			logger.error("getStorages: (" + e.getErrorCode() +") "+ e.getMessage());
			throw new CloudManagerFault(e.getMessage(), e.getErrorCode(), e);
		}catch(Exception e){
			logger.error("getStorages: " + e.getMessage());
			throw new CloudManagerFault(e.getMessage(), e);
	    }

		return storages;
	}

	@Override
	public List<String> getStorageFlavors() throws CloudManagerFault {
		List<String> storageTypes = new ArrayList<>();
		/*for (VolumeType type: VolumeType.values()) {
			storageTypes.add(type.toString());
		}*/
		return storageTypes;
	}

	@Override
	public StorageBackup createStorageBackup(String storageId, String name, String description, String backupOption) throws CloudManagerFault {
		final StorageBackup storageBackup = new StorageBackup();
		VirtualMachineDisk target_disk = null;
		StorageAccountOperations storageAccountOperations = null;
		Logger logger = Logger.getLogger(this.getClass());
		String storageAccountName = "";
		String containerName = "";
		String vhdFileName = "";
		
		AzureUtil.AzureServiceRequestFilter requestFilter = null;
        AzureUtil.AzureServiceResponseFilter responseFilter = null;
		
		try {
			Configuration config = getConfiguration();
			ComputeManagementClient computeManagementClient = ComputeManagementService.create(config);
			StorageManagementClient storageManagementClient = StorageManagementService.create(config);
			storageAccountOperations = storageManagementClient.getStorageAccountsOperations();
        	VirtualMachineDiskListResponse vmds = computeManagementClient.getVirtualMachineDisksOperations().listDisks();
        	for(VirtualMachineDisk vmd : vmds.getDisks()){
           		if(vmd.getName().equals(storageId)) {
           			target_disk = new VirtualMachineDisk();
           			target_disk=vmd;
           			
           			storageAccountName = AzureUtil.getStorageAccountNameFromVHD(vmd.getMediaLinkUri().toString());
           			containerName = AzureUtil.getContainerNameFromVHD(vmd.getMediaLinkUri().toString());
           			vhdFileName = AzureUtil.getVHDFileNameFromVHD(vmd.getMediaLinkUri().toString());
           			break;
        		}
        	}
        	
        	if(target_disk!=null){
        		Configuration blobconfig = null;
        		blobconfig = AzureUtil.getBlobConfiguration(storageAccountName, storageAccountOperations.getKeys(storageAccountName).getPrimaryKey());
        		BlobContract blobContract = null;
        			
      	        //スナップショット作成
    	        responseFilter = new AzureUtil.AzureServiceResponseFilter(); 
    	        
    	        //リクエストのURLにパラメーターを追加し、snapshot作成を実行
    	        requestFilter = new AzureUtil.AzureServiceRequestFilter();
    	        requestFilter.changeURL(vhdFileName, vhdFileName+"?comp=snapshot");
    	        
    	        blobContract = BlobService.create(blobconfig).withRequestFilterLast(requestFilter).withResponseFilterLast(responseFilter);

            	logger.info("BlobContract.createBlockBlob(" + containerName +","+ vhdFileName +") Create snapshot");
    	        blobContract.createBlockBlob(containerName, vhdFileName, null);
    	        String snapshot = responseFilter.getSnapShot();
               	logger.info("RequestId:" + responseFilter.getRequestId() + " Create snapshot: " + containerName+"/"+vhdFileName+"?snapshot="+snapshot + " snapshot-id="+AzureUtil.getConvertSnapshotId(snapshot));
    	        
    			String backUpName = storageAccountName + "_" +containerName + "_" + AzureUtil.getConvertSnapshotId(snapshot);
    			
            	responseFilter.close();
            	
            	storageBackup.setName(name);
            	storageBackup.setDescription(description);
            	storageBackup.setStorageBackupId(backUpName);
            	storageBackup.setStorageId(storageId);
            	storageBackup.setCreateTime(new Date());
     	        	
        	}
        	
		}catch(ServiceException e){
			logger.error("createStorageBackup: (" + e.getErrorCode() +") "+ e.getMessage());
			throw new CloudManagerFault(e.getMessage(), e.getErrorCode(), e);
		}catch(Exception e){
			logger.error("createStorageBackup: " + e.getMessage());
			throw new CloudManagerFault(e.getMessage(), e);
	    }
		
		return storageBackup;

	}

	@Override
	public void deleteStorageBackup(String storageBackupId) throws CloudManagerFault {

		Logger logger = Logger.getLogger(this.getClass());
	   	logger.info("deleteStorageBackup start: " + storageBackupId );
	   	try {
			AzureDeleteStorageBackup azureDeleteStorageBackup = new AzureDeleteStorageBackup(getConfiguration(),storageBackupId);
			if (AzureOptionPropertyConstants.azure_api_mode.value().equalsIgnoreCase("async")) {
				AzureAsyncExecution.getSingleton().put(azureDeleteStorageBackup);
			}else{
				azureDeleteStorageBackup.execute();
			}
		}catch(ServiceException e){
	    	logger.error("Deleting Storage backup is failed...");
			logger.error("deleteStorageBackup: (" + e.getErrorCode() +") "+ e.getMessage());
			throw new CloudManagerFault(e.getMessage(), e.getErrorCode(), e);
		}catch(Exception e){
	    	logger.error("Deleting Storage backup is failed...");
			logger.error("deleteStorageBackup: " + e.getMessage());
			throw new CloudManagerFault(e.getMessage(), e);
	    } finally {
	    	logger.info("deleteStorageBackup end: " + storageBackupId );
	    }		
	}
	
	@Override
	public List<Snapshot> getSnapshots(String...snapshotIds) throws CloudManagerFault {
		return getSnapshotsWithFilter(new Filter("snapshot-id", snapshotIds));
	}

	@Override
	public List<Snapshot> getSnapshots(List<String> snapshotIds) throws CloudManagerFault {
		return getSnapshotsWithFilter(new Filter("snapshot-id", snapshotIds));
	}

	@Override
	public List<Snapshot> getSnapshotsWithFilter(Filter...filters) throws CloudManagerFault {
		return getSnapshotsWithFilter(Arrays.asList(filters));
	}

	@Override
	public List<Snapshot> getSnapshotsWithFilter(List<Filter> filters) throws CloudManagerFault {
		Logger logger = Logger.getLogger(this.getClass());
		List<Snapshot> snapShots = new ArrayList<>();
		StorageAccountOperations storageAccountOperations = null;
		AzureUtil.AzureServiceResponseFilter response = new AzureUtil.AzureServiceResponseFilter();
		
		try {
			Configuration config = getConfiguration();
			ComputeManagementClient computeManagementClient = ComputeManagementService.create(config);
        	VirtualMachineDiskListResponse vmds = computeManagementClient.getVirtualMachineDisksOperations().listDisks();
        	for(VirtualMachineDisk vmd : vmds.getDisks()){
           			
        		String storageAccountName = AzureUtil.getStorageAccountNameFromVHD(vmd.getMediaLinkUri().toString());
        		String containerName = AzureUtil.getContainerNameFromVHD(vmd.getMediaLinkUri().toString());
        		String vhdFileName = AzureUtil.getVHDFileNameFromVHD(vmd.getMediaLinkUri().toString());
        		
        		StorageManagementClient storageManagementClient = StorageManagementService.create(config);
       			storageAccountOperations = storageManagementClient.getStorageAccountsOperations();
        		Configuration blobconfig = null;
        		blobconfig = AzureUtil.getBlobConfiguration(storageAccountName, storageAccountOperations.getKeys(storageAccountName).getPrimaryKey());
    			BlobContract blobContract = BlobService.create(blobconfig);

    			blobContract.getBlob(containerName, vhdFileName);
    			
       			for(StorageAccount sa:storageAccountOperations.list()){
       				if(vmd.getMediaLinkUri().toString().contains(sa.getName())){
       					AzureUtil.AzureServiceRequestFilter request = new AzureUtil.AzureServiceRequestFilter();
       					request.changeURL("?", containerName+"?restype=container&include=snapshots&");
       					response.setUseBody();
       					blobContract = BlobService.create(config).withRequestFilterLast(request).withResponseFilterLast(response);
       		            storageAccountOperations = storageManagementClient.getStorageAccountsOperations();
       					blobContract.listContainers();
       					
       					JAXBContext jaxbContext = JAXBContext.newInstance(ListBlobsResult.class);     
       					Unmarshaller jaxbUnmarshaller = jaxbContext.createUnmarshaller();
       					ListBlobsResult listBlobsResult = (ListBlobsResult)jaxbUnmarshaller.unmarshal(response.getInputStream());
       					response.close();
       					
       					for (Blob blob:listBlobsResult.getBlobs()) {
         					logger.debug("ListBlobsResult: "+containerName+"/"+blob.getName()+blob.getSnapshot() != null ? (" snapshot="+blob.getSnapshot()) : "");
         					Snapshot ss = new Snapshot();
         					ss.setSnapshotId(AzureUtil.getConvertSnapshotId(blob.getSnapshot()));
         					ss.setName(blob.getName());
           	                snapShots.add(ss);
       					} 
       					logger.info("Successful in getting Snapshots");
       				}
       			}
        	}
        	
		}catch(ServiceException e){
			logger.error("getSnapshotsWithFilter: (" + e.getErrorCode() +") "+ e.getMessage());
			throw new CloudManagerFault(e.getMessage(), e.getErrorCode(), e);
		}catch(Exception e){
			logger.error("getSnapshotsWithFilter: " + e.getMessage());
			throw new CloudManagerFault(e.getMessage(), e);
        } finally {
        	if (response != null) {
				response.close();
				response = null;
        	}
        }
		
		return snapShots;
	}

	@Override
	public void deleteInstanceBackup(String instanceBackupId) throws CloudManagerFault {
		Logger logger = Logger.getLogger(this.getClass());

		try{
		   	logger.info("deleteInstanceBackup start: " + instanceBackupId );
			AzureDeleteInstanceBackup azureDeleteInstanceBackup = new AzureDeleteInstanceBackup(getConfiguration(),instanceBackupId);
			if (AzureOptionPropertyConstants.azure_api_mode.value().equalsIgnoreCase("async")) {
				AzureAsyncExecution.getSingleton().put(azureDeleteInstanceBackup);
			}else{
				azureDeleteInstanceBackup.execute();
			}
		}catch(ServiceException e){
			logger.error("deleteInstanceBackup: (" + e.getErrorCode() +") "+ e.getMessage());
			throw new CloudManagerFault(e.getMessage(), e.getErrorCode(), e);
		}catch(Exception e){
			logger.error("deleteInstanceBackup: " + e.getMessage());
			throw new CloudManagerFault(e.getMessage(), e);
	    } finally {
	    	logger.info("deleteInstanceBackup end: " + instanceBackupId );
	    }		
	}

	@Override
	public List<Image> getImages(String...imageIds) throws CloudManagerFault {
		return getImagesWithFilter(new Filter("image-id", imageIds));
	}

	@Override
	public List<Image> getImages(List<String> imageIds) throws CloudManagerFault {
		return getImagesWithFilter(new Filter("image-id", imageIds));
	}

	@Override
	public List<Image> getImagesWithFilter(Filter...filters) throws CloudManagerFault {
		return getImagesWithFilter(Arrays.asList(filters));
	}

	@Override
	public List<Image> getImagesWithFilter(List<Filter> filters) throws CloudManagerFault {
		/*AmazonEC2 ec2 = AzureUtil.createAmazonEC2(credential, region.getEndpoint(EP_TYPE_EC2).getLocation());

		List<com.amazonaws.services.ec2.model.Filter> awsFilters = new ArrayList<>();
		for (Filter filter: filters) {
			awsFilters.add(new com.amazonaws.services.ec2.model.Filter().withName(filter.getName()).withValues(filter.getValues()));
		}

		com.amazonaws.services.ec2.model.DescribeImagesRequest request = new com.amazonaws.services.ec2.model.DescribeImagesRequest().withFilters(awsFilters);
		com.amazonaws.services.ec2.model.DescribeImagesResult result = ec2.describeImages(request);

		List<Image> images = new ArrayList<>();
		for(com.amazonaws.services.ec2.model.Image awsImage: result.getImages()){
			Image image = new Image();
			image.setResourceType(RT_Image);
			image.setName(awsImage.getName());
			image.setImageId(awsImage.getImageId());
			image.setDescription(awsImage.getDescription());
			image.setActualResource(awsImage);
			images.add(image);
		}
		return images;*/
		return new ArrayList<>();
	}

	@Override
	public Snapshot getSnapshot(String snapshotId) throws CloudManagerFault {
		List<Snapshot> snapshots = getSnapshots(snapshotId);
		if (snapshots.isEmpty()) {
			throw ErrorCode.Resource_InvalidSnapshot_NotFound.cloudManagerFault(snapshotId);
		}
		return snapshots.get(0);
	}

	@Override
	public Image getImage(String imageId) throws CloudManagerFault {
		List<Image> images = getImages(imageId);
		if (images.isEmpty()) {
			throw ErrorCode.Resource_InvalidImageID_NotFound.cloudManagerFault(imageId);
		}
		return images.get(0);
	}

	@Override
	public Instance restoreInstance(String facilityId, String instanceBackupId, String name, String flavor, String zone, String instanceDetail, List<Tag> tags) throws CloudManagerFault {
		return new Instance();
	}

	@Override
	public Storage restoreStorage(String storageBackupId, String name,String flavor, Integer size, String zone, String storageDetail) throws CloudManagerFault {
		return new Storage();
	}

	@Override
	public StorageBackup getStorageBackup(String storageBackupId) throws CloudManagerFault {
		return new StorageBackup();
	}

	@Override
	public List<StorageBackup> getStorageBackups(String... storageBackupId) throws CloudManagerFault {
		return getStorageBackups(Arrays.asList(storageBackupId));
	}
	
	private CloudStorageBackupDao getStorageBackupDAO(String storageId){
		Logger logger = Logger.getLogger(this.getClass());
		try{
	    	ICloudContext context = SessionService.current().get(ICloudContext.class);
			EntityManagerEx em = SessionService.current().getEntityManagerEx();
			Query query = em.createQuery("SELECT b FROM CloudStorageBackupDao b " +
										"WHERE b.storageBackupId   = :target " +
										  "AND b.region            = :region " +
										  "AND b.accountResourceId = :accountResourceId ");
			query.setParameter("target", storageId);
			query.setParameter("region", context.getCurrentRegion().getRegion());
			query.setParameter("accountResourceId", context.getAccessDestionation().getCloudAccountResource().getAccountResourceId());
			
			@SuppressWarnings("unchecked")
			List<CloudStorageBackupDao> backups = new ArrayList<>((List<CloudStorageBackupDao>)query.getResultList());
			
			return backups.get(0);
			
		}catch(Exception e){
 			logger.error("getStorageBackupDAO: " + e.getMessage());
	    }
		return null;
	}
	
	private CloudInstanceDao getInstanceDAO(String instanceId){
		Logger logger = Logger.getLogger(this.getClass());
		try{
	    	ICloudContext context = SessionService.current().get(ICloudContext.class);
			EntityManagerEx em = SessionService.current().getEntityManagerEx();
			Query query = em.createQuery("SELECT b FROM CloudInstanceDao b " +
										"WHERE b.instanceId        = :instanceId " +
										  "AND b.region            = :region " +
										  "AND b.accountResourceId = :accountResourceId ");
			query.setParameter("instanceId", instanceId);
			query.setParameter("region", context.getCurrentRegion().getRegion());
			query.setParameter("accountResourceId", context.getAccessDestionation().getCloudAccountResource().getAccountResourceId());
			
			@SuppressWarnings("unchecked")
			List<CloudInstanceDao> backups = new ArrayList<>((List<CloudInstanceDao>)query.getResultList());
			
			return backups.get(0);
			
		}catch(Exception e){
 			logger.error("getStorageBackupDAO: " + e.getMessage());
	    }
		return null;
	}
	
	@Override
	public List<StorageBackup> getStorageBackups(List<String> storageBackupIds) throws CloudManagerFault {
		List<StorageBackup> storageBackups = new ArrayList<>();
		Logger logger = Logger.getLogger(this.getClass());
		
		StorageAccountOperations storageAccountOperations = null;
		AzureUtil.AzureServiceRequestFilter request = new AzureUtil.AzureServiceRequestFilter();
		AzureUtil.AzureServiceResponseFilter response = new AzureUtil.AzureServiceResponseFilter();
		
		String snapshotID = "";
    	VirtualMachineDiskGetResponse vmd = new VirtualMachineDiskGetResponse();
    	String storageAccountName = "";
    	String containerName = "";
    	String vhdFileName = "";
		
		try {
			Configuration config = getConfiguration();
			ComputeManagementClient computeManagementClient = ComputeManagementService.create(config);
        	
			for(String storageBackupId:storageBackupIds){
				
				CloudStorageBackupDao dao = getStorageBackupDAO(storageBackupId);
				
				vmd = computeManagementClient.getVirtualMachineDisksOperations().getDisk(dao.getBackupedData().getStorageId());
				
				storageAccountName = AzureUtil.getStorageAccountNameFromVHD(vmd.getMediaLinkUri().toString());
        		containerName = AzureUtil.getContainerNameFromVHD(vmd.getMediaLinkUri().toString());
        		vhdFileName = AzureUtil.getVHDFileNameFromVHD(vmd.getMediaLinkUri().toString());
				
				snapshotID = AzureUtil.getConvertSnapshotValue(storageBackupId.substring(storageBackupId.lastIndexOf("_")+1));
				
				StorageManagementClient storageManagementClient = StorageManagementService.create(config);
       			storageAccountOperations = storageManagementClient.getStorageAccountsOperations();
				
       			Configuration blobconfig = null;
        		blobconfig = AzureUtil.getBlobConfiguration(storageAccountName, storageAccountOperations.getKeys(storageAccountName).getPrimaryKey());
    			
        		//パラメータに?snapshot=を追加
				request.changeURL(vhdFileName, vhdFileName+"?snapshot="+snapshotID);
				
				GetBlobResult blobRes = null;
				BlobContract blobContract = BlobService.create(blobconfig).withRequestFilterLast(request).withResponseFilterLast(response);
				try{
					blobRes = blobContract.getBlob(containerName, vhdFileName);
				}catch(Exception e){
					logger.info("Snapshot is anavailable : vhd_file=" + vhdFileName + ", snapshotId="+snapshotID);
					continue;
				}
				
				if(blobRes!=null){
					StorageBackup storageBackup=new StorageBackup();
					storageBackup.setStorageBackupId(storageBackupId);
					storageBackups.add(storageBackup);
					logger.info("Get snapshot in success..." + containerName +","+ vhdFileName +","+ "snapshot="+ snapshotID +","+ storageBackupId);
				}
			}
		}catch(ServiceException e){
        	logger.error("Get snapshot in failed... snapshot="+snapshotID);
			logger.error("getStorageBackups: (" + e.getErrorCode() +") "+ e.getMessage());
			throw new CloudManagerFault(e.getMessage(), e.getErrorCode(), e);
		}catch(Exception e){
        	logger.error("Get snapshot in failed... snapshot="+snapshotID);
			logger.error("getStorageBackups: " + e.getMessage());
			throw new CloudManagerFault(e.getMessage(), e);
	    } finally {
	    	if (response != null) {
				response.close();
				response = null;
	    	}	
	    }
		return storageBackups;
	}

	@Override
	public InstanceBackup getInstanceBackup(String instanceBackupId) throws CloudManagerFault {
		List<String> instanceBackupIds = new ArrayList<>();
		instanceBackupIds.add(instanceBackupId);
		
		List<InstanceBackup> InstanceBackups =  getInstanceBackups(instanceBackupIds);
		
		return InstanceBackups.get(0);
	}

	@Override
	public List<InstanceBackup> getInstanceBackups(String... instanceBackupIds) throws CloudManagerFault {
		return getInstanceBackups(Arrays.asList(instanceBackupIds));
	}

	@Override
	public List<InstanceBackup> getInstanceBackups(List<String> instanceBackupIds) throws CloudManagerFault {
		List<InstanceBackup> instanceBackups = new ArrayList<>();
		Logger logger = Logger.getLogger(this.getClass());
		
		try{
			Configuration config = getConfiguration();
			ComputeManagementClient computeManagementClient = ComputeManagementService.create(config);
			VirtualMachineOSImageOperations virtualMachineOSImageOperations = computeManagementClient.getVirtualMachineOSImagesOperations();
			
			for(VirtualMachineOSImage vmimage : virtualMachineOSImageOperations.list()){
				if(instanceBackupIds.contains(vmimage.getName())){
					InstanceBackup bkup = new InstanceBackup();
					bkup.setInstanceBackupId(vmimage.getName());
					bkup.setDescription(vmimage.getDescription());
					if(vmimage.getOperatingSystemType().equals("Linux")){
						bkup.setPlatform(PlatformKind.linux);
			        }else if(vmimage.getOperatingSystemType().equals("Windows")){
			        	bkup.setPlatform(PlatformKind.windows);
			        }else{
			        	bkup.setPlatform(PlatformKind.other);
			        }
					BackupedData bkupdata = new BackupedData();
					bkup.setBackupedData(bkupdata);
					instanceBackups.add(bkup);
				}
			}
			logger.info("Successful in getting instances " + getClass().getSimpleName() + "...");
			
		}catch(ServiceException e){
			logger.error("getInstanceBackups: (" + e.getErrorCode() +") "+ e.getMessage());
			throw new CloudManagerFault(e.getMessage(), e.getErrorCode(), e);
		}catch(Exception e){
			logger.error("getInstanceBackups: " + e.getMessage());
			throw new CloudManagerFault(e.getMessage(), e);
		}
		
		return instanceBackups;
	}

	@Override
	public List<LoadBalancer> getLoadBalancers() throws CloudManagerFault {
		return new ArrayList<>();
	}

	@Override
	public void registerInstanceToLoadBalancer(String lbId, String instanceId) throws CloudManagerFault {
		
	}

	@Override
	public void unregisterInstanceToLoadBalancer(String lbId, String instanceId) throws CloudManagerFault {
		
	}

	@Override
	public InstanceBackup createInstanceBackup(String instanceId, String name, String description, Boolean noReboot, String backupOption) throws CloudManagerFault {
		return createInstanceBackup(instanceId, name, description, noReboot, null, backupOption);
	}

	@Override
	public InstanceBackup createInstanceBackup(String instanceId, String name, String description, Boolean noReboot, List<String> storageIds, String backupOption) throws CloudManagerFault {
		InstanceBackup instanceBackup = new InstanceBackup();
		AzureUtil.AzureServiceRequestFilter request = new AzureUtil.AzureServiceRequestFilter();
        AzureUtil.AzureServiceResponseFilter response = new AzureUtil.AzureServiceResponseFilter();
        Logger logger = Logger.getLogger(this.getClass());
        SimpleDateFormat sdf = new SimpleDateFormat("yyyyMMddhhmmss");
		sdf.setTimeZone(TimeZone.getTimeZone("GMT"));
		
		try {
			Configuration config = getConfiguration();
			ComputeManagementClient computeManagementClient = ComputeManagementService.create(config);
			ManagementClient managementClient = ManagementService.create(config);
			Role targetRole = null;
			HashMap<String,String> attachedDisks = new HashMap<String,String>();
			
			HostedServiceOperations hostedServicesOperations = computeManagementClient.getHostedServicesOperations();
			DeploymentOperations deploymentOperations = computeManagementClient.getDeploymentsOperations();
			VirtualMachineOSImageOperations virtualMachineOSImageOperations = computeManagementClient.getVirtualMachineOSImagesOperations();
			StorageManagementClient storageManagementClient = StorageManagementService.create(config);
			StorageAccountOperations storageAccountOperations = storageManagementClient.getStorageAccountsOperations();
			AffinityGroupOperations affinityGroupOperations = managementClient.getAffinityGroupsOperations();
			AffinityGroupListResponse affinityGroupListResponse = affinityGroupOperations.list();
	 		HostedServiceListResponse hostedServiceListResponse = hostedServicesOperations.list();
	 		
	 		for (HostedService hostService: hostedServiceListResponse.getHostedServices()) {
				if (AzureUtil.compareLocation(getRegion().getName(),hostService.getProperties().getLocation(),affinityGroupListResponse.getAffinityGroups())) {
		 			for (DeploymentSlot slot:AzureUtil.getDeploymentSlots()) {
		 				DeploymentGetResponse deploymentGetResponse = null;
			 			try {
			 				deploymentGetResponse = deploymentOperations.getBySlot(hostService.getServiceName(),slot);
			 			}
			 			catch (ServiceException e) {
			 				if (e.getErrorCode().equals("ResourceNotFound"))
			 					continue;
			 				logger.error("createInstanceBackup: (" + e.getErrorCode() +") "+ e.getMessage());
			 				throw new CloudManagerFault(e.getMessage(), e.getErrorCode(), e);
			 			}
					
						for(Role role:deploymentGetResponse.getRoles()){
							if(role.getRoleName().equals(instanceId)){
								targetRole=role;
							}
						}
						break;
		 			}
		 		}
				if(targetRole!=null){break;};
			}
	 		
	 		//storageIds==nullのとき、インスタンスに紐付いた全てのストレージも含めてバックアップ(AWS版の実装に合わせる)
	 		if(storageIds==null){
	 			String key="";
		 		Integer num=0;
		 		for(DataVirtualHardDisk dvhd:targetRole.getDataVirtualHardDisks()){
		 			StorageBackup bkup_Data=createStorageBackup(dvhd.getName(),"","","");
		 			String ssid = bkup_Data.getStorageBackupId().substring(bkup_Data.getStorageBackupId().lastIndexOf("_")+1);
		 			ssid = AzureUtil.getConvertSnapshotValue(ssid);
		 			if(dvhd.getLogicalUnitNumber()!=null){
		 				key="LUN"+Integer.toString(dvhd.getLogicalUnitNumber());
		 			}else{
		 				key="LUNNULL"+Integer.toString(num);
		 				num+=1;
		 			}
		 			attachedDisks.put(key, dvhd.getMediaLink().toString()+"?snapshot="+ssid);
		 		}
	 		}
	 		
	 		String storageAccountName = AzureUtil.getStorageAccountNameFromVHD(targetRole.getOSVirtualHardDisk().getMediaLink().toString());
	 		String containerName = AzureUtil.getContainerNameFromVHD(targetRole.getOSVirtualHardDisk().getMediaLink().toString());
	 		StorageBackup bkup_OS = createStorageBackup(targetRole.getOSVirtualHardDisk().getName(),"","","");
	 		String vhdFileName = AzureUtil.getVHDFileNameFromVHD(targetRole.getOSVirtualHardDisk().getMediaLink().toString());
	 		String snapshotID = AzureUtil.getConvertSnapshotValue(bkup_OS.getStorageBackupId().substring(bkup_OS.getStorageBackupId().lastIndexOf("_")+1));
	 		String x_ms_copy_source = "/"+storageAccountName+"/"+containerName+"/"+vhdFileName+"?snapshot="+snapshotID;
	 		
	 		//スナップショットを作成
	 		request.setHeader("x-ms-copy-source",x_ms_copy_source);
	 		Configuration blobconfig = null;
			blobconfig = AzureUtil.getBlobConfiguration(storageAccountName, storageAccountOperations.getKeys(storageAccountName).getPrimaryKey());
			BlobContract blobContract = BlobService.create(blobconfig).withRequestFilterLast(request).withResponseFilterLast(response);
			
			
			//スナップショットを元にVMイメージを作成
			CreateBlobOptions opt = new CreateBlobOptions();
			opt.setMetadata(attachedDisks);
			
			String backUpName =instanceId+"_BackUp_"+sdf.format(new Date())+".vhd";
			
	 		logger.info("BlobContract.createBlockBlob(" + containerName +","+ backUpName +") " +" Metadata="+ opt.getMetadata() + "x-ms-copy-source="+ x_ms_copy_source);
			blobContract.createBlockBlob(containerName, backUpName, null, opt);
			
			String image_src = AzureUtil.getVHDUrl(storageAccountName, containerName, backUpName);
			
	 		VirtualMachineOSImageCreateParameters vmiParam = new VirtualMachineOSImageCreateParameters();
	 		vmiParam.setName(name);
	 		vmiParam.setDescription(description);
	 		vmiParam.setMediaLinkUri(new URI(image_src));
	 		vmiParam.setLabel(name);
	 		vmiParam.setOperatingSystemType(targetRole.getOSVirtualHardDisk().getOperatingSystem());
	 		
	 		logger.info("VirtualMachineOSImageOperations.create(" + name +","+ targetRole.getOSVirtualHardDisk().getOperatingSystem() +")");
	 		VirtualMachineOSImageCreateResponse operationResponse = virtualMachineOSImageOperations.create(vmiParam);
	 		logger.info("RequestId:" + operationResponse.getRequestId());
		        
	        for(;;){
        		OperationStatusResponse operationStatusResponse = computeManagementClient.getOperationStatus(operationResponse.getRequestId());
     			if (!operationStatusResponse.getStatus().equals(OperationStatus.InProgress)) {
     				logger.info("creating " + getClass().getSimpleName() + "...");
     				if(operationStatusResponse.getStatus().equals(OperationStatus.Succeeded)){
     					logger.info("Successful in creating InstanceBackup " + getClass().getSimpleName() + "...");
     				}else{
     					logger.error("failed in creating InstanceBackup " + operationStatusResponse.getError());
	 					throw new CloudManagerFault(operationStatusResponse.getError().getMessage(),new Exception(operationStatusResponse.getError().getCode()));
     				}
     				break;
     			}
        	}
	        
	        //取得したスナップショットの削除
	        request = new AzureUtil.AzureServiceRequestFilter();
	        response = new AzureUtil.AzureServiceResponseFilter();
	        request.changeURL(vhdFileName, vhdFileName+"?snapshot="+snapshotID);
	        BlobContract blobContract_del = BlobService.create(blobconfig).withRequestFilterLast(request).withResponseFilterLast(response);
	        blobContract_del.deleteBlob(containerName, vhdFileName);
	        
	        instanceBackup.setInstanceId(instanceId);
	        instanceBackup.setCreateTime(new Date());
	        instanceBackup.setName(name);
	        instanceBackup.setDescription(description);
	        instanceBackup.setInstanceBackupId(name);
	        if(targetRole.getOSVirtualHardDisk().getOperatingSystem().toLowerCase().equals("linux")){
	        	instanceBackup.setPlatform(PlatformKind.linux);
	        }else if(targetRole.getOSVirtualHardDisk().getOperatingSystem().toLowerCase().equals("windows")){
	        	instanceBackup.setPlatform(PlatformKind.windows);
	        }else{
	        	instanceBackup.setPlatform(PlatformKind.other);
	        }
	        instanceBackup.setResourceType("image");
	 		
		}catch(ServiceException e){
			logger.error("createInstanceBackup: (" + e.getErrorCode() +") "+ e.getMessage());
			throw new CloudManagerFault(e.getMessage(), e.getErrorCode(), e);
		}catch(Exception e){
			logger.error("createInstanceBackup: " + e.getMessage());
			throw new CloudManagerFault(e.getMessage(), e);
		}

		return instanceBackup;
	}
}