/*
Copyright (C) 2013 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
*/
package com.clustercontrol.cloud.azure;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.apache.log4j.Logger;

import com.clustercontrol.cloud.CloudManagerBaseService;
import com.clustercontrol.cloud.CloudPlugin;
import com.clustercontrol.cloud.ICloudOption;
import com.clustercontrol.cloud.IResourceManagement;
import com.clustercontrol.cloud.IUserManagement;
import com.clustercontrol.cloud.InternalManagerError;
import com.clustercontrol.cloud.PluginFault;
import com.clustercontrol.cloud.azure.factory.AzureAsyncThread;
import com.clustercontrol.cloud.azure.factory.AzureOption;
import com.clustercontrol.cloud.azure.factory.AzureResourceManagement;
import com.clustercontrol.cloud.azure.factory.AzureUserManagement;
import com.clustercontrol.cloud.azure.util.AzureConstants;
import com.clustercontrol.cloud.azure.util.AzureUtil;
import com.clustercontrol.cloud.commons.CloudConstants;
import com.clustercontrol.cloud.registry.AbstractObjectChangedListener;
import com.clustercontrol.cloud.registry.IObjectChangedService;
import com.clustercontrol.cloud.registry.ObjectRegistryService;
import com.clustercontrol.cloud.util.proxy.IProxyStore;
import com.clustercontrol.repository.bean.NodeInfo;
import com.clustercontrol.repository.bean.NodeVariableInfo;
import com.clustercontrol.ws.azure.AzureOptionEndpointImpl;
import com.microsoft.windowsazure.Configuration;

public class AzureManagerBaseService extends CloudPlugin {
	public final static String id = "com.clustercontrol.cloud.azure.AzureManagerBaseService";
	
	private static AzureManagerBaseService singleton;
	AzureAsyncThread azureAsyncThread = null;
	
	/**
	 * CloudServiceのコンストラクタ
	 */
	public AzureManagerBaseService() {
		super();
		singleton = this;
	}

	@Override
	public Set<String> getDependency() {
		Set<String> set = new HashSet<>();
		set.add(CloudManagerBaseService.class.getName());
		return set;
	}

	@Override
	public void create() {
		Logger logger = Logger.getLogger(this.getClass());

		// メイン処理
		logger.info("creating " + getClass().getSimpleName() + "...");
		
		//　プロパティ一覧の表示
		for (AzureOptionPropertyConstants value: AzureOptionPropertyConstants.values()) {
			logger.info(value.id + " = " + value.value() + " (" + value.defaultValue() + ")");
		}
		
		logger.info("successful in creating " + getClass().getSimpleName() + "...");
	}

	@Override
	public void activate() {
		Logger logger = Logger.getLogger(this.getClass());

		// メイン処理
		logger.info("starting " + getClass().getSimpleName() + "...");
		
		logger.info("successful in starting " + getClass().getSimpleName() + "...");
	}

	@Override
	public void deactivate() {
		Logger logger = Logger.getLogger(this.getClass());

		// メイン処理
		logger.info("stopping " + getClass().getSimpleName() + "...");
		
		logger.info("successful in stopping " + getClass().getSimpleName() + "...");
	}

	@Override
	public void destroy() {
		Logger logger = Logger.getLogger(this.getClass());
		
		if (azureAsyncThread != null) {
			azureAsyncThread.terminate();
		}

		// メイン処理
		logger.info("destroying " + getClass().getSimpleName() + "...");
		
		logger.info("successful in destroying " + getClass().getSimpleName() + "...");
	}
	
	public static AzureManagerBaseService getSingleton() {
		return singleton;
	}

	@Override
	public String getPluginId() {
		return id;
	}

	@Override
	public void initialize() {
		Logger logger = Logger.getLogger(this.getClass());

		// メイン処理
		logger.info("initializing " + getClass().getSimpleName() + "...");

		AzureUtil.setBuilder(Configuration.getInstance());
		if (azureAsyncThread == null) {
			azureAsyncThread = new AzureAsyncThread();
		}

		ObjectRegistryService.registry().put(ICloudOption.class, AzureConstants.TYPE_ID, AzureOption.class);
		ObjectRegistryService.registry().put(IUserManagement.class, AzureConstants.TYPE_ID, AzureUserManagement.class);
		ObjectRegistryService.registry().put(IResourceManagement.class, AzureConstants.TYPE_ID, AzureResourceManagement.class);
		
		CloudManagerBaseService.getSingleton().publish("/HinemosWS/AzureOptionEndpoint", new AzureOptionEndpointImpl());

		String proxyHost = AzureOptionPropertyConstants.azure_client_config_proxyHost.value();
		int proxyPort = Integer.valueOf(AzureOptionPropertyConstants.azure_client_config_proxyPort.value());
		if (proxyHost != null && !proxyHost.isEmpty() && proxyPort > 0) {
			String proxyUsername = AzureOptionPropertyConstants.azure_client_config_proxyUsername.value();
			String proxyPassword = AzureOptionPropertyConstants.azure_client_config_proxyPassword.value();
			IProxyStore proxyStore = ObjectRegistryService.registry().get(IProxyStore.class);
			
			List<String> destinations = new ArrayList<>();
			try {
				destinations.add(new URI(AzureOptionPropertyConstants.azure_base_uri.value()).getHost());
			}
			catch (URISyntaxException e) {
				throw new InternalManagerError(e);
			}
			proxyStore.add(
				proxyHost,
				proxyPort,
				proxyUsername,
				proxyPassword,
				destinations.toArray(new String[destinations.size()])
				);
		}
		
		IObjectChangedService service = ObjectRegistryService.registry().get(IObjectChangedService.class);
		service.addObjectChangedListener(CloudConstants.Node_Instance_Modify, NodeInfo.class, new AbstractObjectChangedListener<NodeInfo>() {
			@Override
			public void preModified(String type, NodeInfo object) throws PluginFault {
				if (!AzureConstants.SERVICE_ID.equals(object.getCloudService())) return;
				
				// リソース監視用のノード変数を追加する処理。
				NodeVariableInfo instaceId = null;
				List<NodeVariableInfo> variables = object.getNodeVariableInfo();
				for (NodeVariableInfo variable: variables) {
					if (variable.getNodeVariableName().equals(AzureConstants.CloudWatch_IntanceId)) {
						variable.setNodeVariableValue(object.getCloudResourceId());
						instaceId = variable;
						break;
					}
				}
				
				if (instaceId == null) {
					List<NodeVariableInfo> newVariables = new ArrayList<>(variables);
					newVariables.add(new NodeVariableInfo(AzureConstants.CloudWatch_IntanceId, object.getCloudResourceId()));
					object.setNodeVariableInfo(new ArrayList<>(newVariables));
				}
			}
		});
		
		service.addObjectChangedListener(CloudConstants.Node_Instance_Add, NodeInfo.class, new AbstractObjectChangedListener<NodeInfo>() {
			@Override
			public void preAdded(String type, NodeInfo object) throws PluginFault {
				if (!AzureConstants.SERVICE_ID.equals(object.getCloudService())) return;

				// リソース監視用のノード変数を追加する処理。
				object.getNodeVariableInfo().add(new NodeVariableInfo(AzureConstants.CloudWatch_IntanceId, object.getCloudResourceId()));
			}
		});
		
		service.addObjectChangedListener(CloudConstants.Node_Instance_Clear, NodeInfo.class, new AbstractObjectChangedListener<NodeInfo>() {
			@Override
			public void preModified(String type, NodeInfo object) throws PluginFault {
				if (!AzureConstants.SERVICE_ID.equals(object.getCloudService())) return;

				// リソース監視用のノード変数を削除する処理。
				for (NodeVariableInfo var: object.getNodeVariableInfo()) {
					if (AzureConstants.CloudWatch_IntanceId.equals(var.getNodeVariableName())) {
						object.getNodeVariableInfo().remove(var);
						break;
					}
				}
			}
		});

		if (azureAsyncThread != null) {
			azureAsyncThread.start();
		}
		
		logger.info("successful in initializing " + getClass().getSimpleName() + "...");
	}
}
