/*
Copyright (C) 2013 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */
package com.clustercontrol.cloud.azure.base.resources;

import java.util.ArrayList;
import java.util.List;

import com.clustercontrol.cloud.azure.base.Activator;
import com.clustercontrol.cloud.commons.util.AccountControlUtil;
import com.clustercontrol.cloud.presenter.CloudModelException;
import com.clustercontrol.ws.azure.AzureCreateInstanceRequest;
import com.clustercontrol.ws.azure.AzureCreateStorageRequest;
import com.clustercontrol.ws.azure.AzureOptionEndpoint;
import com.clustercontrol.ws.cloud.CloudEndpoint;
import com.clustercontrol.ws.cloud.CloudManagerFault_Exception;
import com.clustercontrol.ws.cloud.CloudRegion;
import com.clustercontrol.ws.cloud.CloudService;
import com.clustercontrol.ws.cloud.InvalidRole_Exception;
import com.clustercontrol.ws.cloud.InvalidUserPass_Exception;
import com.clustercontrol.ws.cloud.Zone;

public class CloudResourceProviderImpl implements CloudResourceProvider {
	private AzureOptionEndpoint azureEndpoint;
	private CloudEndpoint cloudEndpoint;
	private String serviceId;
	private String roleId;

	public CloudResourceProviderImpl(String accountResourceId) {
		this.azureEndpoint = Activator.getEndpointManager().getEndpoint(AzureOptionEndpoint.class);
		this.cloudEndpoint = Activator.getEndpointManager().getEndpoint(CloudEndpoint.class);

		try {
			for(CloudService cloudService: cloudEndpoint.getAllCloudServices()){
				if(cloudService.getCloudServiceId().equals("Azure")){
					this.serviceId = cloudService.getCloudServiceId();
				}
			}
		} catch (CloudManagerFault_Exception | InvalidRole_Exception | InvalidUserPass_Exception e) {
			new CloudModelException(e);
		}

		this.roleId = AccountControlUtil.getRoleId(accountResourceId);
	}

	@Override
	public List<String> getAllRegion() {
		try {
			List<String> regions = new ArrayList<String>();
			for (CloudRegion r: cloudEndpoint.getCloudRegionsByService(serviceId)) {
				regions.add(r.getRegion());
			}
			return regions;
		}
		catch (Exception e) {
			throw new IllegalStateException(e);
		}
	}

	@Override
	public List<AvailabilityZone> getAvailabilityZones(String region) {
		try {
			List<AvailabilityZone> azs = new ArrayList<AvailabilityZone>();
			for (Zone wsAZ: cloudEndpoint.getZones(roleId, region)) {
				AvailabilityZone az = new AvailabilityZone();
				az.setZoneName(wsAZ.getName());
				azs.add(az);
			}
			return azs;
		}
		catch (Exception e) {
			throw new IllegalStateException(e);
		}
	}



	@Override
	public List<String> getAllInstanceType() {
		try {
			return cloudEndpoint.getInstanceFlavorsByService(serviceId);
		}
		catch (Exception e) {
			throw new IllegalStateException(e);
		}
	}



	@Override
	public List<String> getAllVolumeType() {
		try {
			List<String> volumeTypes = new ArrayList<String>();
			for (String v: cloudEndpoint.getStorageFlavorsByService(serviceId)) {
				volumeTypes.add(v);
			}
			return volumeTypes;
		}
		catch (Exception e) {
			throw new IllegalStateException(e);
		}
	}

	@Override
	public List<String> getAllServiceName() {
		try {
			//TODO
			return null;
		}
		catch (Exception e) {
			throw new IllegalStateException(e);
		}
	}

	public List<com.clustercontrol.ws.azure.InstanceInfo> getInstanceByCloudService(String regionName,String cloudServiceName){
		try {
			List<com.clustercontrol.ws.azure.InstanceInfo> azureInstance = new ArrayList<com.clustercontrol.ws.azure.InstanceInfo>();
			for (com.clustercontrol.ws.azure.InstanceInfo v: azureEndpoint.getInstanceByCloudService(roleId,regionName,cloudServiceName)){
				azureInstance.add(v);
			}
			return azureInstance;
		}
		catch (Exception e) {
			throw new IllegalStateException(e);
		}
	}

	public List<String> getStorageAccountByRegion(String regionName){
		try {
			List<String> azureStorageAccount = new ArrayList<String>();
			for (String v: azureEndpoint.getStorageAccountByRegion(roleId,regionName)){
				azureStorageAccount.add(v);
			}
			return azureStorageAccount;
		}
		catch (Exception e) {
			throw new IllegalStateException(e);
		}
	}

	public String getRegionByCloudService(String regionName,String cloudServiceName){
		try {
			return azureEndpoint.getRegionByCloudService(roleId,regionName,cloudServiceName);
		}
		catch (Exception e) {
			throw new IllegalStateException(e);
		}
	}

	public List<String> getStorageAccountByCloudService(String regionName,String cloudServiceName){
		try {
			List<String> azureStorageAccount = new ArrayList<String>();
			for (String v: azureEndpoint.getStorageAccountByCloudService(roleId,regionName,cloudServiceName)){
				azureStorageAccount.add(v);
			}
			return azureStorageAccount;
		}
		catch (Exception e) {
			throw new IllegalStateException(e);
		}
	}

	public List<String> getContainerByStorageAccount(String regionName,String cloudServiceName){
		try {
			List<String> containers = new ArrayList<String>();
			for (String v: azureEndpoint.getContainerByStorageAccount(roleId,regionName,cloudServiceName)){
				containers.add(v);
			}
			return containers;
		}
		catch (Exception e) {
			throw new IllegalStateException(e);
		}
	}

	public void addStorage(String regionName,AzureCreateStorageRequest request){
		try {
			azureEndpoint.addStorage(roleId, regionName, request);
		}
		catch (Exception e) {
			throw new IllegalStateException(e);
		}
	}

	public void createInstance(String regionName,AzureCreateInstanceRequest request){
		try {
			azureEndpoint.createInstance(roleId, regionName, request);
		}
		catch (Exception e) {
			throw new IllegalStateException(e);
		}
	}

	public List<com.clustercontrol.ws.azure.AzureImage> getImages(String region, String filter) {
		try {
			List<com.clustercontrol.ws.azure.AzureImage> images = new ArrayList<com.clustercontrol.ws.azure.AzureImage>();
			for (com.clustercontrol.ws.azure.AzureImage v: azureEndpoint.getImagesWithFilter(roleId,region,filter)){
				images.add(v);
			}
			return images;
		}
		catch (Exception e) {
			throw new IllegalStateException(e);
		}
	}

	public List<com.clustercontrol.ws.azure.DataVirtualHardDisks> getDataVirtualHardDisk(String region,String instanceId){
		try {
			List<com.clustercontrol.ws.azure.DataVirtualHardDisks> dvhds = new ArrayList<com.clustercontrol.ws.azure.DataVirtualHardDisks>();
			for (com.clustercontrol.ws.azure.DataVirtualHardDisks v: azureEndpoint.getAttacheableDataVirtualHardDisk(roleId,region,instanceId)){
				dvhds.add(v);
			}
			return dvhds;
		}
		catch (Exception e) {
			throw new IllegalStateException(e);
		}
	}
	
	public com.clustercontrol.ws.azure.SnapShotInfo getSnapShotInfo(String region,String snapShotId,String storageId){
		try {
			return azureEndpoint.getSnapShotInfo(roleId,region,snapShotId,storageId);
		}
		catch (Exception e) {
			throw new IllegalStateException(e);
		}
	}
	
	public void restoreStorage(String region,com.clustercontrol.ws.azure.AzureRestoreStorageRequest request){
		try {
			azureEndpoint.restoreStorage(roleId,region,request);
		}
		catch (Exception e) {
			throw new IllegalStateException(e);
		}
	}
	
	public void detachStorage(String region, String instanceID, String storageID){
		try {
			azureEndpoint.detachStorage(roleId,region,instanceID,storageID);
		}
		catch (Exception e) {
			throw new IllegalStateException(e);
		}
	}
	
	public void atachStorage(String region, String instanceID, String storageID, AzureCreateStorageRequest request){
		try {
			azureEndpoint.atachStorage(roleId, region,storageID, request);
		}
		catch (Exception e) {
			throw new IllegalStateException(e);
		}
	}
	
}