/*
Copyright (C) 2013 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */
package com.clustercontrol.cloud.commons.util;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IViewPart;

import com.clustercontrol.cloud.Activator;
import com.clustercontrol.cloud.presenter.CloudModelException;
import com.clustercontrol.cloud.presenter.IAccountResource;
import com.clustercontrol.cloud.presenter.ICloudUser;
import com.clustercontrol.cloud.presenter.IFacility;
import com.clustercontrol.cloud.ui.dialogs.ChangeActiveCloudUserDialog;
import com.clustercontrol.util.Messages;
import com.clustercontrol.ws.access.AccessEndpoint;
import com.clustercontrol.ws.access.InvalidRole_Exception;
import com.clustercontrol.ws.cloud.AvailableRole;
import com.clustercontrol.ws.cloud.CloudEndpoint;
import com.clustercontrol.ws.cloud.CloudManagerFault_Exception;
import com.clustercontrol.ws.cloud.InvalidUserPass_Exception;

public class AccountControlUtil {
	public static class NotFoundAccountResourceException extends Exception {
		/**
		 * 
		 */
		private static final long serialVersionUID = -8532898225608136194L;

		public NotFoundAccountResourceException() {
			super("accountResource is null");
		}
	}
	
	//指定したアカウントリソースのアクティブユーザのロールIDを返します。
	public static String getRoleId(String accountResourceId){
		return getAccountResource(accountResourceId).getActiveUser().getRoleId();
	}

	//ログインユーザが所属するロールIDのリストを返します。
	public static List<String> getOwnerRoleIdList() {
		//ロール情報取得
//		List<RoleInfo> infoList = null;
		try {
//			infoList = getEndpoint(AccessEndpoint.class).getRoleInfoList();
			return getEndpoint(AccessEndpoint.class).getOwnerRoleIdList();
		} catch (InvalidRole_Exception e1) {
			// 権限なし
			MessageDialog.openInformation(null, Messages.getString("message"),
					Messages.getString("message.accesscontrol.16"));

			// 一覧の参照権限がない場合、ユーザ自身の情報を表示する
//			infoList = new ArrayList<RoleInfo>();

		} catch (Exception e1) {
			// 上記以外の例外
			System.out.println(e1.getMessage());
			MessageDialog.openError(
					null,
					Messages.getString("failed"),
					Messages.getString("message.hinemos.failure.unexpected") + ", " + e1.getMessage());
		}
//		if(infoList != null){
//			List<String> ownerLoleList = new ArrayList<String>();
//			for(RoleInfo roleInfo: infoList){
//				ownerLoleList.add(roleInfo.getId());
//			}
//			return ownerLoleList;
//		}
		
		return null;
	}

	//クラウドユーザに割り当て可能なロールIDのリストを返します。
	public static List<AvailableRole> getAssignableRoles(){
//		List<String> roleIds = getOwnerRoleIdList();
//		List<String> usedRoleIds = new ArrayList<String>();
//		for(String roleId: getAllCloudUsersRoleMap().values()){
//			usedRoleIds.add(roleId);
//		}
//		
//		if(roleIds != null && !roleIds.isEmpty()){
//			for(String roleId: new ArrayList<String>(roleIds)){
//				for(String usedRoleId: new ArrayList<String>(usedRoleIds)){
//					if(roleId.equals(usedRoleId)){
//						roleIds.remove(roleId);
//						usedRoleIds.remove(usedRoleId);
//					}
//				}
//			}
//			for(String roleId: new ArrayList<String>(roleIds)){
//				if(RoleIdConstant.INTERNAL.equals(roleId)){
//					roleIds.remove(roleId);
//					break;
//				}
//			}
//		}
//		return roleIds;
		
		try {
			return getEndpoint(CloudEndpoint.class).getAvailableRoles();
		}
		catch (CloudManagerFault_Exception
				| com.clustercontrol.ws.cloud.InvalidRole_Exception
				| InvalidUserPass_Exception e) {
			throw new CloudModelException(e.getMessage(), e);
		}
	}

	//エンドポイントを返します。
	private static <E> E getEndpoint(Class<E> clazz){
		return Activator.getEndpointManager().getEndpoint(clazz);
	}

	//全クラウドユーザのクラウドユーザIDとロールIDのマップを返します。
	public static Map<String, String> getAllCloudUsersRoleMap() {
		List<ICloudUser> cloudUsers = new ArrayList<>();
		for(IAccountResource accountResource: Activator.getDefault().getCloudResourceManager().getAccountResourceManager().getAccountResources()){
			cloudUsers.addAll(Arrays.asList(accountResource.getCloudUserManager().getCloudUsers()));
		}
		Map<String, String> cloudUsersRoleMap = new HashMap<String, String>();
		for(ICloudUser cloudUser: cloudUsers){
			cloudUsersRoleMap.put(cloudUser.getCloudUserId(), cloudUser.getRoleId());
		}
		return cloudUsersRoleMap;
	}

	//指定のアカウントリソースに紐づくクラウドユーザのクラウドユーザIDとロールIDのマップを返します。
	public static Map<String, String> getCloudUsersRoleMapByAccountResource(String accountResourceId) {
		IAccountResource accountResource = getAccountResource(accountResourceId);
		List<ICloudUser> cloudUsers = new ArrayList<ICloudUser>((Arrays.asList(accountResource.getCloudUserManager().getCloudUsers())));
		Map<String, String> cloudUsersRoleMap = new HashMap<String, String>();
		for(ICloudUser cloudUser: cloudUsers){
			cloudUsersRoleMap.put(cloudUser.getCloudUserId(), cloudUser.getRoleId());
		}
		return cloudUsersRoleMap;
	}

	//指定のアカウントリソースを返します。
	public static IAccountResource getAccountResource(String accountResourceId){
		return Activator.getDefault().getCloudResourceManager().getAccountResourceManager().getAccountResource(accountResourceId);
	}

	//指定したアカウントリソースにアクティブユーザがセットされているかを返します。セットされていない場合、該当するクラウドユーザが単一の場合はそのユーザをセットしTrueを返します。
	public static Boolean isInitializedActiveUser(String accountResourceId) throws CloudModelException, NotFoundAccountResourceException {
		IAccountResource accountResource = getAccountResource(accountResourceId);
		if(accountResource != null){
			if(accountResource.getActiveUser() == null){
				List<ICloudUser> cloudUsers = getAvailableCloudUsers(accountResourceId);
				if(cloudUsers.size() == 0){
					throw new CloudModelException("available cloudUser is empty");
				} else if(cloudUsers.size() == 1){
					accountResource.setActiveUser(cloudUsers.get(0));
					return true;
				} else if(accountResource.getAccountId() != null && !accountResource.getAccountId().isEmpty()){
					for(ICloudUser cloudUser: cloudUsers){
						if(cloudUser.getCloudUserId().equals(accountResource.getAccountId())){
							accountResource.setActiveUser(cloudUser);
							return true;
						}
						return false;
					}
				} else {
					return false;
				}
			}
			return true;
		} else {
			throw new NotFoundAccountResourceException();
		}
	}
	
	//指定したアカウントリソースに紐づくクラウドユーザのうち、ログインユーザが所属しているオーナーロールに紐づいているものだけを返します。
	public static List<ICloudUser> getAvailableCloudUsers(String accountResourceId){
		List<ICloudUser> cloudUsers = Arrays.asList(getAccountResource(accountResourceId).getCloudUserManager().getCloudUsers());
		List<ICloudUser> tmpCloudUsers = new ArrayList<ICloudUser>();
		List<String> ownerRoleIds = getOwnerRoleIdList();
		for(ICloudUser cloudUser: cloudUsers){
			for(String ownerRoleId: new ArrayList<String>(ownerRoleIds)){
				if(cloudUser.getRoleId().equals(ownerRoleId)){
					tmpCloudUsers.add(cloudUser);
					ownerRoleIds.remove(ownerRoleId);
					break;
				}
			}
		}
		return tmpCloudUsers;
	}
	
	public static Boolean selectActiveUser(IViewPart viewPart) throws NotFoundAccountResourceException {

		MessageManager bundle_messages = MessageManager.getInstance("messages");
		
		IStructuredSelection selection = (IStructuredSelection) viewPart.getSite().getSelectionProvider().getSelection();
		final IFacility facility = (IFacility) selection.getFirstElement();
		final IAccountResource accountResource = AccountControlUtil.getAccountResource(facility.getAccountResourceId());

		ChangeActiveCloudUserDialog dialog = new ChangeActiveCloudUserDialog(viewPart.getSite().getShell());

		dialog.setBehavior(new ChangeActiveCloudUserDialog.DialogBehavior(){

			@Override
			public void setupAccountResource(Text txtAccountResource) {
				super.setupAccountResource(txtAccountResource);
				txtAccountResource.setText(accountResource.getAccountResourceId());
			}

			@Override
			public void setupCloudUser(Combo cmbCloudUser) {
				for(ICloudUser cloudUser: AccountControlUtil.getAvailableCloudUsers(facility.getAccountResourceId())){
					String label = cloudUser.getCloudUserName() + "(" + cloudUser.getCloudUserId() + ")";
					cmbCloudUser.add(label);
					cmbCloudUser.setData(label, cloudUser);
				}
				if(accountResource.getActiveUser() != null &&
					cmbCloudUser.indexOf(accountResource.getActiveUser().getCloudUserName() + "(" + accountResource.getActiveUser().getCloudUserId() + ")") != -1
				){
					cmbCloudUser.select(cmbCloudUser.indexOf(accountResource.getActiveUser().getCloudUserName() + "(" + accountResource.getActiveUser().getCloudUserId() + ")"));
				} else if(cmbCloudUser.getItemCount() > 0) {
					cmbCloudUser.select(0);
				}
			}
		});
		
		if (dialog.open() == Window.OK) {
			if (MessageDialog.openConfirm(
				null,
				Messages.getString("confirmed"),
				bundle_messages.getString("message.change_active_cloud_user_confirm"))) {

				ChangeActiveCloudUserDialog.DialogOutput output = dialog.getOutput();
				accountResource.setActiveUser(output.getCloudUser());
				
				// 成功報告ダイアログを生成
				MessageDialog.openInformation(
					null,
					Messages.getString("successful"),
					bundle_messages.getString("message.change_active_cloud_user_result"));
				
			}
		}

		return isInitializedActiveUser(accountResource.getAccountResourceId());
	}
}
