/*

Copyright (C) 2006 - 2008 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.snmptrap.session;

import java.util.ArrayList;

import javax.persistence.EntityExistsException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.commons.bean.SettingUpdateInfo;
import com.clustercontrol.commons.util.HinemosSessionContext;
import com.clustercontrol.commons.util.JpaTransactionManager;
import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.fault.InvalidSetting;
import com.clustercontrol.fault.MonitorNotFound;
import com.clustercontrol.snmptrap.bean.SnmpTrapMasterInfo;
import com.clustercontrol.snmptrap.entity.SnmpTrapMibMasterData;
import com.clustercontrol.snmptrap.factory.AddMibMaster;
import com.clustercontrol.snmptrap.factory.AddOidMaster;
import com.clustercontrol.snmptrap.factory.DeleteMibMaster;
import com.clustercontrol.snmptrap.factory.DeleteOidMaster;
import com.clustercontrol.snmptrap.factory.ModifyMibMaster;
import com.clustercontrol.snmptrap.factory.ModifyOidMaster;
import com.clustercontrol.snmptrap.factory.SelectMibMaster;
import com.clustercontrol.snmptrap.util.SnmpTrapMasterCache;
import com.clustercontrol.snmptrap.util.SnmpTrapMibMasterCache;

/**
 * SNMPトラップ監視を制御するSession Bean <BR>
 * 
 * @version 4.0.0
 * @since 2.1.0
 * 
 */
public class MonitorSnmpTrapControllerBean {

	private static Log m_log = LogFactory.getLog( MonitorSnmpTrapControllerBean.class );

	/**
	 * MIB一覧を取得します。<BR>
	 * 
	 * 戻り値はMIBの名前(String)のArrayListです。
	 * 
	 * @throws HinemosUnknown
	 */
	public ArrayList<String> getMibList() throws HinemosUnknown {
		m_log.debug("getMibList() : start");

		JpaTransactionManager jtm = null;

		// MIB一覧を取得
		SelectMibMaster master = new SelectMibMaster();
		ArrayList<String> list = null;
		try {
			jtm = new JpaTransactionManager();
			jtm.begin();

			list = master.getMibList();
			jtm.commit();
		} catch (Exception e) {
			m_log.warn("getMibList() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(),e );
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();
		}
		m_log.debug("getMibList() : end");
		return list;
	}

	/**
	 * 監視マスタ情報一覧を取得します。<BR>
	 * 
	 * 
	 * 戻り値はSnmpTrapMasterInfoのArrayListです。
	 * 
	 * @return MIB一覧
	 * @throws HinemosUnknown
	 * @see com.clustercontrol.snmptrap.bean.SnmpTrapMasterInfo
	 */
	public ArrayList<SnmpTrapMasterInfo> getMasterList(String mib) throws HinemosUnknown{
		m_log.debug("getMasterList() : start mib = " + mib);

		JpaTransactionManager jtm = null;

		// 監視マスタ情報一覧を取得
		SelectMibMaster master = new SelectMibMaster();
		ArrayList<SnmpTrapMasterInfo> list = null;
		try {
			jtm = new JpaTransactionManager();
			jtm.begin();

			list = master.getMasterList(mib);
			jtm.commit();
		} catch (Exception e) {
			m_log.warn("getMasterList() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(),e );
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();
		}
		m_log.debug("getMasterList() : end mib = " + mib);
		return list;
	}


	/**
	 * MIBの内容を取得します。<BR>
	 * 
	 * 戻り値はMIBのDTO(SnmpTrapMibMasterData)です。
	 * 
	 * @return MIB一覧
	 * @throws MonitorNotFound
	 * @throws HinemosUnknown
	 */
	public SnmpTrapMibMasterData getMib(String Mib) throws MonitorNotFound, HinemosUnknown{
		m_log.debug("getMib() : start mib = " + Mib);

		JpaTransactionManager jtm = null;
		SelectMibMaster  selector = new SelectMibMaster();

		SnmpTrapMibMasterData data = null;
		try {
			jtm = new JpaTransactionManager();
			jtm.begin();

			data = selector.getMibMasterData(Mib);
			jtm.commit();
		} catch (MonitorNotFound e) {
			jtm.rollback();
			throw e;
		} catch (Exception e) {
			m_log.warn("getMib() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();
		}
		m_log.debug("getMib() : end mib = " + Mib);
		return data;
	}

	/**
	 * MIB情報の登録
	 *
	 * Database commit後に実施すべきユーザ実装が存在するため、トランザクション開始はユーザが制御する。
	 * 
	 * @param MibData
	 * @return
	 * @throws HinemosUnknown
	 */
	public boolean addMibMaster(SnmpTrapMibMasterData MibData) throws HinemosUnknown {
		m_log.debug("addMibMaster() : start mib = " + MibData.getMib());

		JpaTransactionManager jtm = null;
		String loginUser = (String)HinemosSessionContext.instance().getProperty(HinemosSessionContext.LOGIN_USER_ID);
		AddMibMaster adder = new AddMibMaster();
		boolean flag;
		try {
			// トランザクションがすでに開始されている場合は処理終了
			jtm = new JpaTransactionManager();
			jtm.begin(true);

			flag = adder.add(MibData , loginUser);

			SnmpTrapMibMasterCache.clear();

			jtm.commit();

			// read-committedのため、commit後に外部コンポーネントに通知する
			SettingUpdateInfo.getInstance().setSnmptrapMonitorUpdateTime(System.currentTimeMillis());

		} catch (EntityExistsException e) {
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (Exception e) {
			m_log.warn("addMibMaster() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();
		}
		m_log.debug("addMibMaster() : end mib = " + MibData.getMib());
		return flag;
	}

	/**
	 * MIB情報の変更
	 * 
	 * Database commit後に実施すべきユーザ実装が存在するため、トランザクション開始はユーザが制御する。
	 * 
	 * @param MibData
	 * @return
	 * @throws MonitorNotFound
	 * @throws HinemosUnknown
	 */
	public boolean modifyMibMaster(SnmpTrapMibMasterData MibData) throws MonitorNotFound, HinemosUnknown{
		m_log.debug("modifyMibMaster() : start mib = " + MibData.getMib());

		JpaTransactionManager jtm = null;
		String loginUser = (String)HinemosSessionContext.instance().getProperty(HinemosSessionContext.LOGIN_USER_ID);
		ModifyMibMaster modifier = new ModifyMibMaster();

		boolean flag;
		try {
			// トランザクションがすでに開始されている場合は処理終了
			jtm = new JpaTransactionManager();
			jtm.begin(true);

			flag = modifier.modify(MibData , loginUser);

			SnmpTrapMibMasterCache.clear();

			jtm.commit();

			// read-committedのため、commit後に外部コンポーネントに通知する
			SettingUpdateInfo.getInstance().setSnmptrapMonitorUpdateTime(System.currentTimeMillis());
		} catch (MonitorNotFound e) {
			jtm.rollback();
			throw e;
		} catch (Exception e) {
			m_log.warn("modifyMibMaster() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();
		}
		m_log.debug("modifyMibMaster() : end mib = " + MibData.getMib());
		return flag;
	}

	/**
	 * MIB情報の削除
	 * 
	 * Database commit後に実施すべきユーザ実装が存在するため、トランザクション開始はユーザが制御する。
	 * 
	 * @param mib
	 * @return
	 * @throws MonitorNotFound
	 * @throws HinemosUnknown
	 */
	public boolean deleteMibMaster(String Mib) throws MonitorNotFound, HinemosUnknown{
		m_log.debug("deleteMibMaster() : start mib = " + Mib);

		JpaTransactionManager jtm = null;
		DeleteMibMaster eraser= new DeleteMibMaster();
		boolean flag;
		try {
			// トランザクションがすでに開始されている場合は処理終了
			jtm = new JpaTransactionManager();
			jtm.begin(true);

			flag = eraser.delete(Mib);

			SnmpTrapMibMasterCache.clear();

			jtm.commit();

			// read-committedのため、commit後に外部コンポーネントに通知する
			SettingUpdateInfo.getInstance().setSnmptrapMonitorUpdateTime(System.currentTimeMillis());

		} catch (MonitorNotFound e) {
			jtm.rollback();
			throw e;
		} catch (Exception e) {
			m_log.warn("deleteMibMaster() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();
		}
		m_log.debug("deleteMibMaster() : start mib = " + Mib);
		return flag;
	}


	/**
	 * OID情報の登録
	 *
	 * Database commit後に実施すべきユーザ実装が存在するため、トランザクション開始はユーザが制御する。
	 *
	 * @param MibData
	 * @return
	 * @throws HinemosUnknown
	 */
	public boolean addOidMaster(SnmpTrapMasterInfo Oid) throws HinemosUnknown {
		m_log.debug("addOidMaster() : start Uei = " + Oid.getUei());

		JpaTransactionManager jtm = null;
		String loginUser = (String)HinemosSessionContext.instance().getProperty(HinemosSessionContext.LOGIN_USER_ID);
		AddOidMaster adder = new AddOidMaster();

		boolean flag;
		try {
			// トランザクションがすでに開始されている場合は処理終了
			jtm = new JpaTransactionManager();
			jtm.begin(true);

			flag = adder.add(Oid , loginUser);

			SnmpTrapMasterCache.clear();

			// read-committedのため、commit後に外部コンポーネントに通知する
			SettingUpdateInfo.getInstance().setSnmptrapMonitorUpdateTime(System.currentTimeMillis());

			jtm.commit();

		} catch (EntityExistsException e) {
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (Exception e) {
			m_log.warn("addOidMaster() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();
		}
		m_log.debug("addOidMaster() : end Uei = " + Oid.getUei());
		return flag;
	}

	/**
	 * OID情報の変更
	 * 
	 * Database commit後に実施すべきユーザ実装が存在するため、トランザクション開始はユーザが制御する。
	 * 
	 * @param MibData
	 * @return
	 * @throws MonitorNotFound
	 * @throws HinemosUnknown
	 */
	public boolean modifyOidMaster(SnmpTrapMasterInfo Oid) throws MonitorNotFound, HinemosUnknown{
		m_log.debug("modifyOidMaster() : start Uei = " + Oid.getUei());

		JpaTransactionManager jtm = null;
		String loginUser = (String)HinemosSessionContext.instance().getProperty(HinemosSessionContext.LOGIN_USER_ID);
		ModifyOidMaster modifier = new ModifyOidMaster();
		boolean flag;
		try {
			// トランザクションがすでに開始されている場合は処理終了
			jtm = new JpaTransactionManager();
			jtm.begin(true);

			flag = modifier.modify(Oid , loginUser);

			SnmpTrapMasterCache.clear();

			jtm.commit();

			// read-committedのため、commit後に外部コンポーネントに通知する
			SettingUpdateInfo.getInstance().setSnmptrapMonitorUpdateTime(System.currentTimeMillis());
		} catch (MonitorNotFound e) {
			jtm.rollback();
			throw e;
		} catch (Exception e) {
			m_log.warn("modifyOidMaster() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();
		}
		m_log.debug("modifyOidMaster() : end Uei = " + Oid.getUei());
		return flag;
	}

	/**
	 * OID情報の削除
	 * 
	 * Database commit後に実施すべきユーザ実装が存在するため、トランザクション開始はユーザが制御する。
	 * 
	 * @param mib
	 * @return
	 * @throws MonitorNotFound
	 * @throws HinemosUnknown
	 */
	public boolean deleteOidMaster(SnmpTrapMasterInfo oid) throws MonitorNotFound, HinemosUnknown{
		m_log.debug("deleteOidMaster() : start Uei = " + oid.getUei());

		JpaTransactionManager jtm = null;
		DeleteOidMaster eraser= new DeleteOidMaster();
		boolean flag;
		try {
			// トランザクションがすでに開始されている場合は処理終了
			jtm = new JpaTransactionManager();
			jtm.begin(true);

			flag = eraser.delete(oid);

			SnmpTrapMasterCache.clear();

			jtm.commit();

			// read-committedのため、commit後に外部コンポーネントに通知する
			SettingUpdateInfo.getInstance().setSnmptrapMonitorUpdateTime(System.currentTimeMillis());
		} catch (MonitorNotFound e) {
			jtm.rollback();
			throw e;
		} catch (InvalidSetting e) {
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage());
		} catch (Exception e) {
			m_log.warn("delteOidMaster() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();
		}
		m_log.debug("deleteOidMaster() : end Uei = " + oid.getUei());
		return flag;
	}
}
