/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.monitor.run.composite;

import java.util.ArrayList;

import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.ui.PlatformUI;

import com.clustercontrol.dialog.ValidateResult;
import com.clustercontrol.monitor.run.dialog.StringValueInfoCreateDialog;
import com.clustercontrol.monitor.run.util.StringValueInfoManager;
import com.clustercontrol.util.Messages;
import com.clustercontrol.ws.monitor.MonitorInfo;
import com.clustercontrol.ws.monitor.MonitorStringValueInfo;

/**
 * 文字列監視の判定情報（重要度）コンポジットクラス<BR>
 * <p>
 * <dl>
 *  <dt>コンポジット</dt>
 *  <dd>値取得の成功時</dd>
 *  <dd>　判定情報一覧コンポジット</dd>
 *  <dd>　「追加」ボタン</dd>
 *  <dd>　「変更」ボタン</dd>
 *  <dd>　「削除」ボタン</dd>
 *  <dd>　「上へ」ボタン</dd>
 *  <dd>　「下へ」ボタン</dd>
 *  <dd>値取得の失敗時</dd>
 *  <dd>　「重要度」 コンボボックス</dd>
 * </dl>
 * 
 * @version 4.0.0
 * @since 2.1.0
 */
public class StringValueInfoComposite extends Composite {

	/** カラム数（タイトル）。 */
	public static final int WIDTH_TITLE = 5;

	/** カラム数（値）。 */
	public static final int WIDTH_VALUE = 2;

	/** 判定情報一覧 コンポジット。 */
	private StringValueListComposite m_infoList = null;

	/** 追加 ボタン。 */
	private Button m_buttonAdd = null;

	/** 変更 ボタン。 */
	private Button m_buttonModify = null;

	/** 削除 ボタン。 */
	private Button m_buttonDelete = null;

	/** 上へ ボタン。 */
	private Button m_buttonUp = null;

	/** 下へ ボタン。 */
	private Button m_buttonDown = null;

	/** メッセージにデフォルトを入れるフラグ(#[LOG_LINE]) */
	private boolean logLineFlag = false;

	/**
	 * インスタンスを返します。
	 * <p>
	 * 初期処理を呼び出し、コンポジットを配置します。
	 * 
	 * @param parent 親のコンポジット
	 * @param style スタイル
	 * @param tableDefine 文字列監視の判定情報一覧のテーブル定義情報（{@link com.clustercontrol.bean.TableColumnInfo}のリスト）
	 * 
	 * @see org.eclipse.swt.SWT
	 * @see org.eclipse.swt.widgets.Composite#Composite(Composite parent, int style)
	 * @see com.clustercontrol.bean.TableColumnInfo#TableColumnInfo(java.lang.String, int, int, int)
	 * @see com.clustercontrol.monitor.run.action.GetStringFilterTableDefine
	 * @see #initialize(ArrayList)
	 */
	public StringValueInfoComposite(Composite parent, int style, ArrayList tableDefine, boolean logLineFlag) {
		super(parent, style);

		this.initialize(tableDefine);
		this.logLineFlag = logLineFlag;
	}

	/**
	 * コンポジットを配置します。
	 */
	private void initialize(ArrayList tableDefine) {

		// 変数として利用されるラベル
		Label label = null;
		// 変数として利用されるグリッドデータ
		GridData gridData = null;

		GridLayout layout = new GridLayout(1, true);
		layout.marginHeight = 0;
		layout.marginWidth = 0;
		layout.numColumns = 15;
		this.setLayout(layout);

		/*
		 * 文字列監視判定情報一覧
		 */
		this.m_infoList = new StringValueListComposite(this, SWT.BORDER, tableDefine);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.verticalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.grabExcessVerticalSpace = true;
		gridData.horizontalSpan = 12;
		this.m_infoList.setLayoutData(gridData);

		/*
		 * 操作ボタン
		 */
		Composite composite = new Composite(this, SWT.NONE);
		layout = new GridLayout(1, true);
		layout.numColumns = 1;
		composite.setLayout(layout);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.verticalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.grabExcessVerticalSpace = true;
		gridData.horizontalSpan = 3;
		composite.setLayoutData(gridData);

		// 追加ボタン
		this.m_buttonAdd = this.createButton(composite, Messages.getString("add"));
		this.m_buttonAdd.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {

				// シェルを取得
				Shell shell = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getShell();

				StringValueInfoCreateDialog dialog = new StringValueInfoCreateDialog(shell, logLineFlag);
				if (dialog.open() == IDialogConstants.OK_ID) {
					StringValueInfoManager.getInstance().add(dialog.getInputData());
					m_infoList.update();
				}
			}
		});

		// 変更ボタン
		this.m_buttonModify = this.createButton(composite, Messages.getString("modify"));
		this.m_buttonModify.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				String identifier = getSelectIdentifier();
				if (identifier != null) {

					// シェルを取得
					Shell shell = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getShell();

					StringValueInfoCreateDialog dialog = new StringValueInfoCreateDialog(shell, identifier);
					if (dialog.open() == IDialogConstants.OK_ID) {
						Table table = m_infoList.getTableViewer().getTable();
						int selectIndex = table.getSelectionIndex();
						StringValueInfoManager.getInstance().modify(dialog.getInputData());
						m_infoList.update();
						table.setSelection(selectIndex);
					}
				}
				else{
					MessageDialog.openWarning(
							null,
							Messages.getString("warning"),
							Messages.getString("message.monitor.30"));
				}
			}
		});

		// 削除ボタン
		this.m_buttonDelete = this.createButton(composite, Messages.getString("delete"));
		this.m_buttonDelete.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				MonitorStringValueInfo info = m_infoList.getFilterItem();

				if (info != null) {
					String detail = info.getDescription();

					String[] args = { detail };
					if (MessageDialog.openConfirm(
							null,
							Messages.getString("confirmed"),
							Messages.getString("message.monitor.31", args))) {

						StringValueInfoManager.getInstance().delete(getSelectIdentifier());
						m_infoList.update();
					}
				}
				else{
					MessageDialog.openWarning(
							null,
							Messages.getString("warning"),
							Messages.getString("message.monitor.30"));
				}
			}
		});

		// 上へボタン
		label = new Label(composite, SWT.NONE);	// ダミー
		this.m_buttonUp = this.createButton(composite, Messages.getString("up"));
		this.m_buttonUp.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				String identifier = getSelectIdentifier();
				if (identifier != null) {
					if (StringValueInfoManager.getInstance().upOrder(identifier)) {
						m_infoList.update();
						selectItem(identifier);
					}
				}
				else{
					MessageDialog.openWarning(
							null,
							Messages.getString("warning"),
							Messages.getString("message.monitor.30"));
				}
			}
		});

		// 下へボタン
		this.m_buttonDown = this.createButton(composite, Messages.getString("down"));
		this.m_buttonDown.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				String identifier = getSelectIdentifier();
				if (identifier != null) {
					if (StringValueInfoManager.getInstance().downOrder(getSelectIdentifier())) {
						m_infoList.update();
						selectItem(identifier);
					}
				}
				else{
					MessageDialog.openWarning(
							null,
							Messages.getString("warning"),
							Messages.getString("message.monitor.30"));
				}
			}
		});
	}

	/**
	 * 引数で指定された監視情報の値を、各項目に設定します。
	 * 
	 * @param info 設定値として用いる監視情報
	 */
	public void setInputData(MonitorInfo info) {

		if(info != null){
			this.m_infoList.setInputData(info);
		}
		// 必須項目を明示
		this.update();
	}

	/**
	 * 引数で指定された監視情報に、入力値を設定します。
	 * <p>
	 * 入力値チェックを行い、不正な場合は認証結果を返します。
	 * 不正ではない場合は、<code>null</code>を返します。
	 * 
	 * @param info 入力値を設定する監視情報
	 * @return 検証結果
	 * 
	 * @see com.clustercontrol.monitor.run.composite.StringValueListComposite#createInputData(MonitorInfo)
	 */
	public ValidateResult createInputData(MonitorInfo info) {

		// 文字列監視判定情報
		ValidateResult validateResult = m_infoList.createInputData(info);
		if(validateResult != null){
			return validateResult;
		}

		return null;
	}

	/* (非 Javadoc)
	 * @see org.eclipse.swt.widgets.Control#setEnabled(boolean)
	 */
	@Override
	public void setEnabled(boolean enabled) {
		this.m_infoList.setEnabled(enabled);
		this.m_buttonAdd.setEnabled(enabled);
		this.m_buttonModify.setEnabled(enabled);
		this.m_buttonDelete.setEnabled(enabled);
		this.m_buttonUp.setEnabled(enabled);
		this.m_buttonDown.setEnabled(enabled);
	}

	/**
	 * ボタンを返します。
	 * 
	 * @param parent 親のコンポジット
	 * @param label ボタンに表示するテキスト
	 * @return ボタン
	 */
	private Button createButton(Composite parent, String label) {
		Button button = new Button(parent, SWT.NONE);

		GridData gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		button.setLayoutData(gridData);

		button.setText(label);

		return button;
	}

	/**
	 * 選択されている判定情報の識別キーを返します。
	 * 
	 * @return 識別キー。選択されていない場合は、<code>null</code>。
	 */
	private String getSelectIdentifier() {
		StructuredSelection selection = (StructuredSelection) this.m_infoList.getTableViewer().getSelection();

		if (selection.getFirstElement() instanceof MonitorStringValueInfo) {
			MonitorStringValueInfo filterInfo = (MonitorStringValueInfo) selection.getFirstElement();

			return filterInfo.getIdentifier();
		} else {
			return null;
		}
	}

	/**
	 * 引数で指定された判定情報の行を選択状態にします。
	 * 
	 * @param identifier 識別キー
	 */
	private void selectItem(String identifier) {
		Table table = this.m_infoList.getTableViewer().getTable();
		TableItem[] items = table.getItems();

		if (items == null || identifier == null) {
			return;
		}

		for (int i = 0; i < items.length; i++) {

			if (items[i].getData() instanceof MonitorStringValueInfo) {
				MonitorStringValueInfo filterInfo = (MonitorStringValueInfo) items[i].getData();
				if (identifier.equals(filterInfo.getIdentifier())) {
					table.select(i);
					return;
				}
			}
		}
	}

	/**
	 * 無効な入力値の情報を設定します。
	 * 
	 * @param id ID
	 * @param message メッセージ
	 * @return 認証結果
	 */
	protected ValidateResult setValidateResult(String id, String message) {

		ValidateResult validateResult = new ValidateResult();
		validateResult.setValid(false);
		validateResult.setID(id);
		validateResult.setMessage(message);

		return validateResult;
	}
}