/*

Copyright (C) 2013 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.calendar.dialog;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;

import com.clustercontrol.util.Messages;

import com.clustercontrol.ws.calendar.CalendarDetailInfo;
import com.clustercontrol.ws.calendar.CalendarPatternInfo;
import com.clustercontrol.ws.calendar.InvalidRole_Exception;
import com.clustercontrol.bean.DayOfWeekConstant;
import com.clustercontrol.bean.RequiredFieldColorConstant;
import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.calendar.bean.DayOfWeekInMonthConstant;
import com.clustercontrol.calendar.bean.MonthConstant;
import com.clustercontrol.calendar.util.CalendarEndpointWrapper;
import com.clustercontrol.composite.action.NumberKeyListener;
import com.clustercontrol.dialog.CommonDialog;
import com.clustercontrol.dialog.ValidateResult;

/**
 * カレンダ詳細設定ダイアログ作成・変更ダイアログクラス<BR>
 * 
 * @version 4.1.0
 * @since 4.1.0
 */
public class CalendarDetailDialog extends CommonDialog{

	// ----- instance フィールド ----- //
	// ログ
	private static Log m_log = LogFactory.getLog( CalendarDetailDialog.class );
	/* ----- 変数 ----- */
	/**
	 * ダイアログの最背面レイヤのカラム数
	 * 最背面のレイヤのカラム数のみを変更するとレイアウトがくずれるため、
	 * グループ化されているレイヤは全てこれにあわせる
	 */
	private final int DIALOG_WIDTH = 8;
	/** 入力値を保持するオブジェクト */
	private CalendarDetailInfo inputData = null;
	/** 入力値の正当性を保持するオブジェクト。 */
	private ValidateResult m_validateResult = null;
	// ----- 共通メンバ変数 ----- //
	private Shell shell = null;
	private Group groupYear = null; //年グループ
	private Group groupMonth = null; //月グループ
	private Group groupDate = null;// 日グループ
	private Group groupAfterDate = null; //上記日程よりx日後設定グループ
	private Group groupTime = null; //時間グループ
	private Group groupOperate = null; //稼動非稼動設定グループ
	/**ラジオボタン**/
	//年グループ
	private Button radioEveryYear = null;
	private Button radioSpecifyYear = null;

	//曜日グループ
	private Button radioAllDay = null;
	private Button radioDayOfWeek = null;
	private Button radioDate = null;
	private Button radioCalPattern = null;
	//稼動/非稼動グループ
	private Button radioOperateON = null;
	private Button radioOperateOFF = null;
	/**コンボボックス**/
	private Combo cmbMonth = null;
	private Combo cmbDayOfWeek = null;
	private Combo cmbDayOfWeekInMonth = null;
	private Combo cmbDays = null;
	private Combo cmbCalPattern = null;
	/**テキスト**/
	//説明
	private Text txtDescription = null;
	//年
	private Text txtYear = null;
	//曜日、日、その他 の日程からx日後
	private Text txtDaysLater = null;
	//時間 - 開始時間
	private Text txtTimeFrom = null;
	//時間 - 終了時間
	private Text txtTimeTo = null;

	// オーナーロールID
	private String ownerRoleId = null;

	/**
	 * 
	 * @return
	 */
	public CalendarDetailInfo getInputData() {
		return this.inputData;
	}
	// ----- コンストラクタ ----- //
	/**
	 * 作成用ダイアログのインスタンスを返します。
	 * 
	 * @param parent
	 *            親のシェルオブジェクト
	 * @param ownerRoleId オーナーロールID
	 */
	public CalendarDetailDialog(Shell parent, String ownerRoleId) {
		super(parent);
		this.ownerRoleId = ownerRoleId;
	}
	/**
	 * 変更用ダイアログのインスタンスを返します。
	 * 
	 * @param parent 親のシェルオブジェクト
	 * @param identifier 変更する文字列監視の判定情報の識別キー
	 * @param ownerRoleId オーナーロールID
	 */
	public CalendarDetailDialog(Shell parent ,int order, String ownerRoleId){
		super(parent);
		this.ownerRoleId = ownerRoleId;
	}
	/**
	 * 作成用ダイアログのインスタンスを返します。
	 * 
	 * @param parent
	 *            親のシェルオブジェクト
	 * @param detailInfo カレンダ詳細情報
	 * @param ownerRoleId オーナーロールID
	 */
	public CalendarDetailDialog(Shell parent, CalendarDetailInfo detailInfo, String ownerRoleId){
		super(parent);
		this.inputData = detailInfo;
		this.ownerRoleId = ownerRoleId;
	}
	// ----- instance メソッド ----- //

	/**
	 * ダイアログエリアを生成します。
	 * 
	 * @param parent
	 *            親のインスタンス
	 */
	@Override
	protected void customizeDialog(Composite parent) {

		shell = this.getShell();
		// タイトル
		//カレンダ[詳細設定作成・変更]
		shell.setText(Messages.getString("dialog.calendar.detail.create.modify"));

		// ラベル
		GridData gridData = new GridData();
		GridLayout layout = new GridLayout(1, true);
		layout.marginWidth = 10;
		layout.marginHeight = 10;
		layout.numColumns = DIALOG_WIDTH;
		parent.setLayout(layout);
		/*
		 * 説明
		 */
		//ラベル
		Label label = new Label(parent, SWT.LEFT);
		gridData = new GridData();
		gridData.horizontalSpan = 3;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText("  " + Messages.getString("description"));
		//テキスト
		txtDescription = new Text(parent, SWT.BORDER);
		gridData = new GridData();
		gridData.horizontalSpan = 4;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		txtDescription.setLayoutData(gridData);
		txtDescription.addModifyListener(new ModifyListener(){
			@Override
			public void modifyText(ModifyEvent arg0) {
				update();
			}
		});
		/*
		 * 「年」設定グループ
		 */
		groupYear = new Group(parent, SWT.NONE);
		layout = new GridLayout(1, true);
		layout.marginWidth = 5;
		layout.marginHeight = 5;
		layout.numColumns = 8;
		groupYear.setLayout(layout);
		gridData = new GridData();
		gridData.horizontalSpan = DIALOG_WIDTH;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		groupYear.setLayoutData(gridData);
		//年
		groupYear.setText(Messages.getString("year"));
		/*
		 * 年
		 */
		//「毎年」ラジオボタン
		this.radioEveryYear = new Button(groupYear, SWT.RADIO);
		//毎年
		this.radioEveryYear.setText(Messages.getString("calendar.detail.every.year"));
		GridData gridYearRadio = new GridData();
		gridYearRadio.horizontalSpan = 8;
		gridYearRadio.horizontalAlignment = GridData.FILL;
		gridYearRadio.grabExcessHorizontalSpace = true;
		this.radioEveryYear.setLayoutData(gridYearRadio);
		this.radioEveryYear.setSelection(true);
		// ラジオボタンのイベント
		this.radioEveryYear.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				update();
			}
		});
		//「毎年」ラジオボタン
		this.radioSpecifyYear = new Button(groupYear, SWT.RADIO);
		//毎年
		this.radioSpecifyYear.setText(Messages.getString("calendar.detail.specify.year"));
		gridYearRadio = new GridData();
		gridYearRadio.horizontalSpan = 3;
		gridYearRadio.horizontalAlignment = GridData.FILL;
		gridYearRadio.grabExcessHorizontalSpace = true;
		this.radioSpecifyYear.setLayoutData(gridYearRadio);
		// ラジオボタンのイベント
		this.radioSpecifyYear.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				update();
			}
		});
		//テキスト
		txtYear = new Text(groupYear, SWT.BORDER);
		gridData = new GridData();
		gridData.horizontalSpan = 2;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		txtYear.setLayoutData(gridData);
		txtYear.setTextLimit(4);
		txtYear.addKeyListener(new NumberKeyListener());
		txtYear.addModifyListener(new ModifyListener(){
			@Override
			public void modifyText(ModifyEvent arg0) {
				if ("-".equals(txtYear.getText())) {
					txtYear.setText("");
				}
				update();
			}
		});
		//ラベル
		label = new Label(groupYear, SWT.LEFT);
		gridData = new GridData();
		gridData.horizontalSpan = 1;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("year"));
		/*
		 * 「月」設定グループ
		 */
		groupMonth = new Group(parent, SWT.NONE);
		layout = new GridLayout(1, true);
		layout.marginWidth = 5;
		layout.marginHeight = 5;
		layout.numColumns = 8;
		groupMonth.setLayout(layout);
		gridData = new GridData();
		gridData.horizontalSpan = DIALOG_WIDTH;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		groupMonth.setLayoutData(gridData);
		//年	月
		groupMonth.setText(Messages.getString("month"));
		/*
		 * 月
		 */
		//ラベル
		Label lblCalID = new Label(groupMonth, SWT.LEFT);
		gridData = new GridData();
		gridData.horizontalSpan = 3;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		lblCalID.setLayoutData(gridData);
		lblCalID.setText(Messages.getString("month"));
		// コンボ
		this.cmbMonth = new Combo(groupMonth, SWT.RIGHT | SWT.BORDER | SWT.READ_ONLY);
		gridData = new GridData();
		gridData.horizontalSpan = 2;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.cmbMonth.setLayoutData(gridData);
		//毎月、1月～12月までをコンボボックスで表示
		for(int i = 0; i < 13 ; i++){
			this.cmbMonth.add(MonthConstant.typeToString(i));
		}
		//初期設定は「毎月」
		this.cmbMonth.setText(MonthConstant.typeToString(0));
		/*
		 * 「日」設定グループ
		 * 
		 */
		groupDate = new Group(parent, SWT.NONE);
		layout = new GridLayout(1, true);
		layout.marginWidth = 5;
		layout.marginHeight = 5;
		layout.numColumns = 8;
		groupDate.setLayout(layout);
		gridData = new GridData();
		gridData.horizontalSpan = DIALOG_WIDTH;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		groupDate.setLayoutData(gridData);
		groupDate.setText(Messages.getString("monthday"));
		//「すべての日」ラジオボタン
		this.radioAllDay = new Button(groupDate, SWT.RADIO);
		//すべての日
		this.radioAllDay.setText(Messages.getString("calendar.detail.everyday"));
		GridData gridDataRadio = new GridData();
		gridDataRadio.horizontalSpan = 3;
		gridDataRadio.horizontalAlignment = GridData.FILL;
		gridDataRadio.grabExcessHorizontalSpace = true;
		this.radioAllDay.setLayoutData(gridDataRadio);
		this.radioAllDay.setSelection(true);
		// ラジオボタンのイベント
		this.radioAllDay.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				update();
			}
		});
		// 空白
		label = new Label(groupDate, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 5;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);

		//「曜日」ラジオボタン
		this.radioDayOfWeek = new Button(groupDate, SWT.RADIO);
		this.radioDayOfWeek.setText(Messages.getString("weekday"));
		gridDataRadio = new GridData();
		gridDataRadio.horizontalSpan = 3;
		gridDataRadio.horizontalAlignment = GridData.FILL;
		gridDataRadio.grabExcessHorizontalSpace = true;
		this.radioDayOfWeek.setLayoutData(gridDataRadio);

		// ラジオボタンのイベント
		this.radioDayOfWeek.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				update();
			}
		});
		//「第x週選択」コンボボックス
		this.cmbDayOfWeekInMonth = new Combo(groupDate, SWT.RIGHT | SWT.BORDER | SWT.READ_ONLY);
		gridData = new GridData();
		gridData.horizontalSpan = 2;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.cmbDayOfWeekInMonth.setLayoutData(gridData);
		//毎週、第1週～第5週までの配列
		String dayOfWeekInMonth[] = new String[6];
		for(int i = 0; i < dayOfWeekInMonth.length ; i++){
			dayOfWeekInMonth[i] = DayOfWeekInMonthConstant.typeToString(i);
			this.cmbDayOfWeekInMonth.add(dayOfWeekInMonth[i]);
		}
		this.cmbDayOfWeekInMonth.addModifyListener(new ModifyListener() {
			@Override
			public void modifyText(ModifyEvent e) {
				update();
			}
		});
			
		//「曜日選択」コンボボックス
		this.cmbDayOfWeek = new Combo(groupDate, SWT.RIGHT | SWT.BORDER | SWT.READ_ONLY);
		gridData = new GridData();
		gridData.horizontalSpan = 2;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.cmbDayOfWeek.setLayoutData(gridData);
		//日曜日から土曜日までの配列
		String dayOfWeek[] = new String[7];
		for(int i = 0; i < dayOfWeek.length ; i++){
			dayOfWeek[i] = DayOfWeekConstant.typeToString(i+1);
			this.cmbDayOfWeek.add(dayOfWeek[i]);
		}
		this.cmbDayOfWeek.addModifyListener(new ModifyListener() {
			@Override
			public void modifyText(ModifyEvent e) {
				update();
			}
		});
		
		// 空白
		label = new Label(groupDate, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 1;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		//日にち
		this.radioDate = new Button(groupDate, SWT.RADIO);
		this.radioDate.setText(Messages.getString("monthday"));
		gridDataRadio = new GridData();
		gridDataRadio.horizontalSpan = 3;
		gridDataRadio.horizontalAlignment = GridData.FILL;
		gridDataRadio.grabExcessHorizontalSpace = true;
		this.radioDate.setLayoutData(gridDataRadio);
		// ラジオボタンのイベント
		this.radioDate.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				update();
			}
		});
		//「日にち」コンボボックス
		this.cmbDays = new Combo(groupDate, SWT.RIGHT | SWT.READ_ONLY);
		gridData = new GridData();
		gridData.horizontalSpan = 2;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.cmbDays.setLayoutData(gridData);
		//1日～31日
		String date[] = new String[31];
		//		date[0] = "";
		//		this.cmbDays.add(date[0]);
		for(int i = 0; i < date.length ; i++){
			date[i] = (i+1) + "";
			this.cmbDays.add(date[i]);
		}
		this.cmbDays.addModifyListener(new ModifyListener() {
			@Override
			public void modifyText(ModifyEvent e) {
				update();
			}
		});
		
		//ラベル
		Label day = new Label(groupDate, SWT.LEFT);
		gridData = new GridData();
		gridData.horizontalSpan = 2;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		day.setLayoutData(gridData);
		day.setText(Messages.getString("monthday"));
		// 空白
		label = new Label(groupDate, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 1;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		//その他
		this.radioCalPattern = new Button(groupDate, SWT.RADIO);
		this.radioCalPattern.setText(Messages.getString("calendar.pattern"));
		gridDataRadio = new GridData();
		gridDataRadio.horizontalSpan = 3;
		gridDataRadio.horizontalAlignment = GridData.FILL;
		gridDataRadio.grabExcessHorizontalSpace = true;
		this.radioCalPattern.setLayoutData(gridDataRadio);
		// ラジオボタンのイベント
		this.radioCalPattern.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				update();
			}
		});

		//「カレンダパターン」の選択コンボボックス
		this.cmbCalPattern = new Combo(groupDate, SWT.RIGHT | SWT.BORDER | SWT.READ_ONLY);
		gridData = new GridData();
		gridData.horizontalSpan = 4;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.cmbCalPattern.setLayoutData(gridData);
		this.cmbCalPattern.add("");
		for(CalendarPatternInfo str : getCalendarPatternList(this.ownerRoleId)){
			this.cmbCalPattern.add(str.getId());
		}
		this.cmbCalPattern.addModifyListener(new ModifyListener() {
			@Override
			public void modifyText(ModifyEvent e) {
				update();
			}
		});
		
		//前後日グループ
		groupAfterDate = new Group(parent, SWT.NONE);
		layout = new GridLayout(1, true);
		layout.marginWidth = 5;
		layout.marginHeight = 5;
		layout.numColumns = 8;
		groupAfterDate.setLayout(layout);
		gridData = new GridData();
		gridData.horizontalSpan = DIALOG_WIDTH;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		groupAfterDate.setLayoutData(gridData);
		groupAfterDate.setText(Messages.getString("calendar.detail.before.after"));
		// 上記の日程からx日後
		label = new Label(groupAfterDate, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 3;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		//上記の日程より
		label.setText(Messages.getString("calendar.detail.after.1") + "");
		//テキスト
		txtDaysLater = new Text(groupAfterDate, SWT.BORDER | SWT.RIGHT);
		gridData = new GridData();
		gridData.horizontalSpan = 2;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		txtDaysLater.setLayoutData(gridData);
		txtDaysLater.setTextLimit(8);
		txtDaysLater.setText("0");
		txtDaysLater.setToolTipText(Messages.getString("calendar.detail.notes") + "");
		txtDaysLater.addKeyListener(new NumberKeyListener());
		txtDaysLater.addModifyListener(new ModifyListener(){
			@Override
			public void modifyText(ModifyEvent arg0) {
				update();
			}
		});
		// 上記の日程からx日後
		label = new Label(groupAfterDate, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 1;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		//日後"
		label.setText(Messages.getString("calendar.detail.after.2") + "");

		/*
		 * 時間設定グループ
		 */
		groupTime = new Group(parent, SWT.NONE);
		layout = new GridLayout(1,true);
		layout.marginWidth = 5;
		layout.marginHeight = 5;
		layout.numColumns = 8;
		groupTime.setLayout(layout);
		gridData = new GridData();
		gridData.horizontalSpan = DIALOG_WIDTH;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		groupTime.setLayoutData(gridData);
		groupTime.setText(Messages.getString("time.period"));

		//時間
		// 開始時刻
		label = new Label(groupTime, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 3;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("start") + Messages.getString("timestamp") + "");

		//テキスト
		txtTimeFrom = new Text(groupTime, SWT.BORDER | SWT.LEFT);
		gridData = new GridData();
		gridData.horizontalSpan = 4;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		txtTimeFrom.setLayoutData(gridData);
		txtTimeFrom.setTextLimit(8);
		txtTimeFrom.setText("00:00:00");
		txtTimeFrom.addModifyListener(new ModifyListener(){
			@Override
			public void modifyText(ModifyEvent arg0) {
				update();
			}
		});
		// 空白
		label = new Label(groupTime, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 1;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);

		// 終了時刻
		label = new Label(groupTime, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 3;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("end") + Messages.getString("timestamp") + "");
		//テキスト
		txtTimeTo = new Text(groupTime, SWT.BORDER | SWT.LEFT);
		gridData = new GridData();
		gridData.horizontalSpan = 4;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		txtTimeTo.setLayoutData(gridData);
		txtTimeTo.setTextLimit(8);
		txtTimeTo.setText("24:00:00");
		txtTimeTo.addModifyListener(new ModifyListener(){
			@Override
			public void modifyText(ModifyEvent arg0) {
				update();
			}
		});
		/*
		 * カレンダ稼動非稼動設定グループ
		 */
		groupOperate = new Group(parent, SWT.NONE);
		layout = new GridLayout(1, true);
		layout.marginWidth = 5;
		layout.marginHeight = 5;
		layout.numColumns = 6;
		groupOperate.setLayout(layout);
		gridData = new GridData();
		gridData.horizontalSpan = DIALOG_WIDTH;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		groupOperate.setLayoutData(gridData);
		groupOperate.setText(Messages.getString("calendar.detail.operation.3"));
		//稼動ボタン
		this.radioOperateON = new Button(groupOperate, SWT.RADIO);
		this.radioOperateON.setText(Messages.getString("calendar.detail.operation.1"));
		this.radioOperateON.setSelection(true);
		gridDataRadio = new GridData();
		gridDataRadio.horizontalSpan = 2;
		gridDataRadio.horizontalAlignment = GridData.FILL;
		gridDataRadio.grabExcessHorizontalSpace = true;
		this.radioOperateON.setLayoutData(gridDataRadio);
		//非稼動ボタン
		this.radioOperateOFF =  new Button(groupOperate, SWT.RADIO);
		this.radioOperateOFF.setText(Messages.getString("calendar.detail.operation.2"));
		gridDataRadio = new GridData();
		gridDataRadio.horizontalSpan = 2;
		gridDataRadio.horizontalAlignment = GridData.FILL;
		gridDataRadio.grabExcessHorizontalSpace = true;
		this.radioOperateOFF.setLayoutData(gridDataRadio);

		shell.pack();
		// 画面中央に
		Display display = shell.getDisplay();
		shell.setLocation((display.getBounds().width - shell.getSize().x) / 2,
				(display.getBounds().height - shell.getSize().y) / 2);

		// 必須入力項目を可視化
		this.update();
		this.reflectCalendar();
	}

	/**
	 * 更新処理
	 * 
	 */
	public void update(){
		// 必須項目を明示

		//ラジオボタン「毎年」を選択
		if(radioEveryYear.getSelection()){
			txtYear.setEnabled(false);
		}else {//ラジオボタン「指定」を選択
			txtYear.setEnabled(true);
		}
		//ラジオボタン「すべての日」を選択
		if(radioAllDay.getSelection()){
			cmbDayOfWeekInMonth.setEnabled(false);
			cmbDayOfWeek.setEnabled(false);
			cmbDays.setEnabled(false);
			txtDaysLater.setEnabled(false);
		}else{
			cmbDayOfWeekInMonth.setEnabled(true);
			cmbDayOfWeek.setEnabled(true);
			cmbDays.setEnabled(true);
			cmbCalPattern.setEnabled(true);
			txtDaysLater.setEnabled(true);
		}
		//ラジオボタン「曜日」を選択
		if(radioDayOfWeek.getSelection()){
			cmbDayOfWeekInMonth.setEnabled(true);
			cmbDayOfWeek.setEnabled(true);
			cmbDays.setEnabled(false);
			cmbCalPattern.setEnabled(false);
		}else{
			cmbDayOfWeekInMonth.setEnabled(false);
			cmbDayOfWeek.setEnabled(false);
		}
		//ラジオボタン「日」を選択
		if(radioDate.getSelection()){
			cmbDayOfWeekInMonth.setEnabled(false);
			cmbDayOfWeek.setEnabled(false);
			cmbDays.setEnabled(true);
			cmbCalPattern.setEnabled(false);
		}else{
			cmbDays.setEnabled(false);
		}
		//ラジオボタン「カレンダパターン」を選択
		if(radioCalPattern.getSelection()){
			cmbDayOfWeekInMonth.setEnabled(false);
			cmbDayOfWeek.setEnabled(false);
			cmbDays.setEnabled(false);
			cmbCalPattern.setEnabled(true);
		}else{
			cmbCalPattern.setEnabled(false);
		}
		//テキスト 指定年
		if(txtYear.getEnabled() && ("").equals(txtYear.getText())){
			txtYear.setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
		}else {
			txtYear.setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
		}
		//コンボボックス 第x週
		if(cmbDayOfWeekInMonth.getEnabled() && ("").equals(cmbDayOfWeekInMonth.getText())){
			cmbDayOfWeekInMonth.setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
		}else {
			cmbDayOfWeekInMonth.setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
		}
		//コンボボックス 曜日
		if(cmbDayOfWeek.getEnabled() && ("").equals(cmbDayOfWeek.getText())){
			cmbDayOfWeek.setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
		}else {
			cmbDayOfWeek.setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
		}
		//コンボボックス 日
		if(cmbDays.getEnabled() && ("").equals(cmbDays.getText())){
			cmbDays.setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
		}else {
			cmbDays.setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
		}
		//コンボボックス カレンダパターン
		if(cmbCalPattern.getEnabled() && ("").equals(cmbCalPattern.getText())){
			cmbCalPattern.setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
		}else {
			cmbCalPattern.setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
		}
		//テキスト 開始時間
		if(("").equals(txtTimeFrom.getText())){
			txtTimeFrom.setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
		}else {
			txtTimeFrom.setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
		}
		//テキスト 終了時間
		if(("").equals(txtTimeTo.getText())){
			txtTimeTo.setBackground(RequiredFieldColorConstant.COLOR_REQUIRED);
		}else {
			txtTimeTo.setBackground(RequiredFieldColorConstant.COLOR_UNREQUIRED);
		}
	}

	/**
	 * ダイアログの情報からカレンダ情報を作成します。
	 * 
	 * @return 入力値の検証結果
	 * 
	 * @see
	 */
	private CalendarDetailInfo createCalendarInfo() {
		/** 開始時間*/
		Long timeFrom = null;
		/** 終了時間*/
		Long timeTo = null;

		inputData = new CalendarDetailInfo();
		/*
		 * 説明
		 */
		if(txtDescription.getText().length() > 0){
			this.inputData.setDescription(txtDescription.getText());
		}

		/*
		 * 年情報 取得
		 */
		if(radioEveryYear.getSelection()){
			//0 は毎年
			this.inputData.setYear(0);
		}else {
			if(txtYear.getText() != null && txtYear.getText().length() > 0){
				try {
					this.inputData.setYear(Integer.parseInt(txtYear.getText()));
				}catch (NumberFormatException e) {
					String[] args = {"[ " +  Messages.getString("year") + " ]"};
					this.setValidateResult(Messages.getString("message.hinemos.1"),
							Messages.getString("message.calendar.51",args));
					return null;
				}
			}else {
				String[] args = {"[ " +  Messages.getString("year") + " ]"};
				this.setValidateResult(Messages.getString("message.hinemos.1"),
						Messages.getString("message.calendar.15",args));
				return null;
			}
		}
		/*
		 * 月グループ情報 取得
		 */
		if(cmbMonth.getText() != null && cmbMonth.getText().length() > 0){
			this.inputData.setMonth(MonthConstant.stringToType(cmbMonth.getText()));
		} else {
			String[] args = {"[ " +  Messages.getString("month") + " ]"};
			this.setValidateResult(Messages.getString("message.hinemos.1"),
					Messages.getString("message.calendar.15",args));
			return null;
		}

		/*
		 * 日グループ情報取得
		 * 
		 * 曜日選択ラジオボタン
		 * 0 : 全ての日
		 * 1 : 曜日
		 * 2 : 日
		 * 3 : その他
		 */
		//すべての日
		if(radioAllDay.getSelection()){
			this.inputData.setDayType(0);
		}
		//曜日
		else if(radioDayOfWeek.getSelection()){
			//第x週テキスト
			if(cmbDayOfWeekInMonth.getText() != null
					&& cmbDayOfWeekInMonth.getText().length() > 0){
				this.inputData.setDayOfWeekInMonth(
						DayOfWeekInMonthConstant.stringToType(cmbDayOfWeekInMonth.getText()));
			}else {
				String[] args = {"[ " +  Messages.getString("calendar.detail.xth") + " ]"};
				this.setValidateResult(Messages.getString("message.hinemos.1"),
						Messages.getString("message.calendar.15",args));
				return null;
			}
			//曜日テキスト
			if(cmbDayOfWeek.getText() != null &&
					cmbDayOfWeek.getText().length() > 0){
				this.inputData.setDayOfWeek(DayOfWeekConstant.stringToType(cmbDayOfWeek.getText()));
			}else {
				String[] args = {"[ " +  Messages.getString("weekday") + " ]"};
				this.setValidateResult(Messages.getString("message.hinemos.1"),
						Messages.getString("message.calendar.15",args));
				return null;
			}
			this.inputData.setDayType(1);
		}
		//日
		else if(radioDate.getSelection()){
			//日テキスト
			if(cmbDays.getText() != null
					&& cmbDays.getText().length() > 0){
				this.inputData.setDate(Integer.parseInt(cmbDays.getText()));
			}else {
				String[] args = {"[ " +  Messages.getString("monthday") + " ]"};
				this.setValidateResult(Messages.getString("message.hinemos.1"),
						Messages.getString("message.calendar.15",args));
				return null;
			}
			this.inputData.setDayType(2);
		}
		//カレンダパターン
		else if(radioCalPattern.getSelection()){
			//カレンダパターンテキスト
			if(cmbCalPattern.getText() != null && cmbCalPattern.getText().length() > 0){
				this.inputData.setCalPatternId(cmbCalPattern.getText());
			}else {
				String[] args = {"[ " +  Messages.getString("calendar.pattern") + " ]"};
				this.setValidateResult(Messages.getString("message.hinemos.1"),
						Messages.getString("message.calendar.15",args));
				return null;
			}
			this.inputData.setDayType(3);
		}
		//ラジオボタンで、いずれか選択状態にあるはずだが、念のため...未選択のときアラート
		else {
			String[] args = {"[ " +  Messages.getString("calendar.detail.date.type") + " ]"};
			this.setValidateResult(Messages.getString("message.hinemos.1"),
					Messages.getString("message.calendar.15",args));
			return null;
		}
		//前後日
		if(txtDaysLater.getText().length() > 0){
			int daysLater = 0;
			try {
				daysLater = Integer.parseInt(txtDaysLater.getText());
			} catch (Exception e) {
				String[] args = {"[ " +  Messages.getString("calendar.detail.before.after") + " ]"};
				this.setValidateResult(Messages.getString("message.hinemos.1"),
						Messages.getString("message.calendar.53",args));
				return null;
			}
			if (-32768 > daysLater || daysLater > 32767) {
				String[] args = {Messages.getString("calendar.detail.before.after"), 
						"-32768", "32767"};
				this.setValidateResult(Messages.getString("message.hinemos.1"),
						Messages.getString("message.calendar.52",args));
				return null;
			}
			this.inputData.setAfterday(daysLater);
		}

		//開始時間、終了時間
		Date dateTimeFrom = null;
		Date dateTimeTo = null;
		try {
			SimpleDateFormat sdfYmd = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss");
			String strHour48 = "1970/01/03 00:00:00";
			Date date = sdfYmd.parse(strHour48);
			long hour48 = date.getTime();
			dateTimeFrom = (new SimpleDateFormat("HH:mm:ss")).parse(txtTimeFrom.getText());
			dateTimeTo = (new SimpleDateFormat("HH:mm:ss")).parse(txtTimeTo.getText());
			timeFrom = dateTimeFrom.getTime();
			timeTo = dateTimeTo.getTime();
			if(timeFrom >= hour48){
				String[] args = {"[ " +  Messages.getString("start")
						+ Messages.getString("time") + " ]",
						Messages.getString("calendar.detail.hour48")};
				this.setValidateResult(Messages.getString("message.hinemos.1"),
						Messages.getString("message.calendar.32",args));
				return null;
			} else {
				this.inputData.setTimeFrom(timeFrom);
			}
			if(timeFrom >= timeTo){
				String[] args = {"[ " +  Messages.getString("end")
						+ Messages.getString("time") + " ]",
						"[ " +  Messages.getString("start")
						+ Messages.getString("time") + " ]"};
				this.setValidateResult(Messages.getString("message.hinemos.1"),
						Messages.getString("message.calendar.31",args));
				return null;
			}
			else if(timeTo > hour48){
				String[] args = {"[ " +  Messages.getString("end")
						+ Messages.getString("time") + " ]",
						Messages.getString("calendar.detail.hour48")};
				this.setValidateResult(Messages.getString("message.hinemos.1"),
						Messages.getString("message.calendar.33",args));
				return null;
			}
			else {
				this.inputData.setTimeTo(timeTo);
			}
		} catch (ParseException e) {
			//「HH:mm:ss」形式に変換できないものが入力値として与えられた場合
			String tmp = "";
			if (dateTimeFrom == null) {
				tmp = "start";
			}
			else {
				tmp = "end";
			}
			this.setValidateResult(Messages.getString("message.hinemos.1"),
					(Messages.getString(tmp) + Messages.getString("message.hinemos.6")));
			//e.printStackTrace();
			return null;
		}
		//稼動/非稼動取得
		if (radioOperateON.getSelection()) {
			this.inputData.setOperateFlg(ValidConstant.BOOLEAN_VALID);
		} else {
			this.inputData.setOperateFlg(ValidConstant.BOOLEAN_INVALID);
		}
		return inputData;
	}
	/**
	 * ダイアログにカレンダ情報を反映します。
	 * 
	 * @param calInfo
	 *            設定値として用いる監視情報
	 */
	private void reflectCalendar() {
		// 初期表示
		CalendarDetailInfo detailInfo = null;
		if(this.inputData != null){
			// 変更の場合、情報取得
			detailInfo = this.inputData;
			//ここで、getSelection() firstElementを取得する
		}
		else{
			// 作成の場合
			detailInfo = new CalendarDetailInfo();
		}
		//カレンダ詳細情報取得
		if(detailInfo != null){
			//説明
			if(detailInfo.getDescription() != null){
				this.txtDescription.setText(detailInfo.getDescription());
			}
			//年
			if(detailInfo.getYear() != null){
				if(detailInfo.getYear() == 0){
					this.radioEveryYear.setSelection(true);
					this.radioSpecifyYear.setSelection(false);
				}else {
					this.radioSpecifyYear.setSelection(true);
					this.radioEveryYear.setSelection(false);
					this.txtYear.setText(String.valueOf(detailInfo.getYear()));
				}
			}
			//月
			if (detailInfo.getMonth() != null) {
				this.cmbMonth.setText(MonthConstant.typeToString(detailInfo.getMonth()));
			}
			//日
			if(detailInfo.getDayType() != null){
				//すべての日
				if(detailInfo.getDayType() == 0){
					this.radioAllDay.setSelection(true);
					this.radioDayOfWeek.setSelection(false);
					this.radioDate.setSelection(false);
					this.radioCalPattern.setSelection(false);
				}
				//曜日
				else if (detailInfo.getDayType() == 1){
					this.radioAllDay.setSelection(false);
					this.radioDayOfWeek.setSelection(true);
					this.radioDate.setSelection(false);
					this.radioCalPattern.setSelection(false);
				}
				//日
				else if (detailInfo.getDayType() == 2){
					this.radioAllDay.setSelection(false);
					this.radioDayOfWeek.setSelection(false);
					this.radioDate.setSelection(true);
					this.radioCalPattern.setSelection(false);
				}
				//その他
				else if (detailInfo.getDayType() == 3){
					this.radioAllDay.setSelection(false);
					this.radioDayOfWeek.setSelection(false);
					this.radioDate.setSelection(false);
					this.radioCalPattern.setSelection(true);
				}else {
					//FIXME 出力メッセージ要追加
					m_log.error("Error Of DayType[" + detailInfo.getDayType() + "]");
				}
				//第x週
				if(detailInfo.getDayOfWeekInMonth() != null){
					//取得した数値を文字列に変換
					String str = DayOfWeekInMonthConstant.typeToString(detailInfo.getDayOfWeekInMonth());
					this.cmbDayOfWeekInMonth.setText(str);
				}
				//曜日
				if(detailInfo.getDayOfWeek() != null){
					String str = DayOfWeekConstant.typeToString(detailInfo.getDayOfWeek());
					this.cmbDayOfWeek.setText(str);
				}
				//日
				if(detailInfo.getDate() != null){
					this.cmbDays.setText(String.valueOf(detailInfo.getDate()));
				}
				//カレンダパターン
				if(detailInfo.getCalPatternId() != null){
					this.cmbCalPattern.setText(detailInfo.getCalPatternId());
				}
				//上記の日程からx日後
				if(detailInfo.getAfterday() != null){
					this.txtDaysLater.setText(String.valueOf(detailInfo.getAfterday()));
				}
			}
			/**
			 * 時間
			 * 開始時間、終了時間
			 * 24:00:00は、00:00:00と表示される
			 * そのため、変換処理を行う。
			 */
			SimpleDateFormat sdf = new SimpleDateFormat("HH:mm:ss");
			SimpleDateFormat sdfYmd = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss");
			SimpleDateFormat sdfHH = new SimpleDateFormat("HH");
			SimpleDateFormat sdfMmSs = new SimpleDateFormat("mm:ss");
			String strHour24 = "1970/01/02 00:00:00";
			String strHour48 = "1970/01/03 00:00:00";

			Date date = null;
			Date date2 = null;
			try {
				date = sdfYmd.parse(strHour24);
				date2 = sdfYmd.parse(strHour48);
			} catch (ParseException e) {
				e.printStackTrace();
			}
			long hour24 = date.getTime();
			long hour48 = date2.getTime();
			if(detailInfo.getTimeFrom() != null){
				if(hour48 <= detailInfo.getTimeFrom()){
					String strHH = sdfHH.format(detailInfo.getTimeFrom());
					Integer hh = Integer.parseInt(strHH);
					hh = hh + 48;
					this.txtTimeFrom.setText(String.valueOf(hh) + ":" + sdfMmSs.format(detailInfo.getTimeFrom()));
				}else if(hour24 <= detailInfo.getTimeFrom()){
					String strHH = sdfHH.format(detailInfo.getTimeFrom());
					Integer hh = Integer.parseInt(strHH);
					hh = hh + 24;
					this.txtTimeFrom.setText(String.valueOf(hh) + ":" + sdfMmSs.format(detailInfo.getTimeFrom()));
				}else {
					//開始時間
					this.txtTimeFrom.setText(sdf.format(detailInfo.getTimeFrom()));
				}
			}
			if(detailInfo.getTimeTo() != null){
				if(hour48 <= detailInfo.getTimeTo()){
					String strHH = sdfHH.format(detailInfo.getTimeTo());
					Integer hh = Integer.parseInt(strHH);
					hh = hh + 48;
					this.txtTimeTo.setText(String.valueOf(hh) + ":" + sdfMmSs.format(detailInfo.getTimeTo()));
				}else if(hour24 <= detailInfo.getTimeTo()){
					String strHH = sdfHH.format(detailInfo.getTimeTo());
					Integer hh = Integer.parseInt(strHH);
					hh = hh + 24;
					this.txtTimeTo.setText(String.valueOf(hh) + ":" + sdfMmSs.format(detailInfo.getTimeTo()));
				}else {
					//終了時間
					this.txtTimeTo.setText(sdf.format(detailInfo.getTimeTo()));
				}
			}
			//稼動/非稼動取得
			if (detailInfo.isOperateFlg() != null) {
				if(detailInfo.isOperateFlg()){
					radioOperateON.setSelection(true);
					radioOperateOFF.setSelection(false);
				} else {
					radioOperateON.setSelection(false);
					radioOperateOFF.setSelection(true);
				}
			}
		}
		// カレンダIDが必須項目であることを明示
		this.update();
	}
	/**
	 * 入力値をCalendarDetailInfoListに登録します。
	 *
	 * @return true：正常、false：異常
	 *
	 * @see com.clustercontrol.dialog.CommonDialog#action()
	 */
	@Override
	protected boolean action() {
		createCalendarInfo();
		return true;
	}

	/**
	 * 入力値チェックをします。
	 * 
	 * @return 検証結果
	 * 
	 * @see com.clustercontrol.dialog.CommonDialog#validate()
	 */
	@Override
	protected ValidateResult validate() {
		// 入力値生成
		this.inputData = this.createCalendarInfo();
		if (this.inputData != null) {
			return super.validate();
		} else {
			return m_validateResult;
		}
	}
	/**
	 * 無効な入力値の情報を設定します。
	 * 
	 * @param id ID
	 * @param message メッセージ
	 */
	protected void setValidateResult(String id, String message) {

		this.m_validateResult = new ValidateResult();
		this.m_validateResult.setValid(false);
		this.m_validateResult.setID(id);
		this.m_validateResult.setMessage(message);
	}

	/**
	 * ＯＫボタンのテキストを返します。
	 * 
	 * @return ＯＫボタンのテキスト
	 */
	@Override
	protected String getOkButtonText() {
		return Messages.getString("ok");
	}

	/**
	 * キャンセルボタンのテキストを返します。
	 * 
	 * @return キャンセルボタンのテキスト
	 */
	@Override
	protected String getCancelButtonText() {
		return Messages.getString("cancel");
	}

	/**
	 * カレンダ詳細ダイアログ
	 * その他項目を取得する。
	 * 
	 * @param ownerRoleId オーナーロールID
	 * @return
	 */
	private List<CalendarPatternInfo> getCalendarPatternList(String ownerRoleId){
		//カレンダ詳細ダイアログカレンダパターン項目
		List<CalendarPatternInfo> calPatternList = null;
		//カレンダパターン情報取得
		try {
			calPatternList = CalendarEndpointWrapper.getCalendarPatternList(ownerRoleId);
		} catch (InvalidRole_Exception e) {
			// 権限なし
			MessageDialog.openInformation(null, Messages.getString("message"),
					Messages.getString("message.accesscontrol.16"));
		} catch (Exception e) {
			// 上記以外の例外
			m_log.warn("update(), " + e.getMessage(), e);
			MessageDialog.openError(
					null,
					Messages.getString("failed"),
					Messages.getString("message.hinemos.failure.unexpected") + ", " + e.getMessage());
			return null;
		}
		return calPatternList;
	}
}