package com.clustercontrol.calendar.util;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.calendar.bean.CalendarPatternInfo;
import com.clustercontrol.calendar.bean.YMD;
import com.clustercontrol.calendar.model.CalPatternDetailInfoEntity;
import com.clustercontrol.calendar.model.CalPatternInfoEntity;
import com.clustercontrol.fault.CalendarNotFound;
import com.clustercontrol.fault.InvalidRole;

public class CalendarPatternCache {

	private static Log m_log = LogFactory.getLog( CalendarPatternCache.class );
	private static Object calendarPatternLock = new Object();
	private static ConcurrentHashMap<String, CalendarPatternInfo> map =
			new ConcurrentHashMap<String,CalendarPatternInfo>();

	/**
	 * キャッシュをリフレッシュする。
	 * カレンダパターンの登録、変更、削除時に呼ぶ。
	 */
	public static void remove(String id) {
		m_log.info("remove() calendar pattern cache is removed");

		synchronized (calendarPatternLock) {
			map.remove(id);
		}
	}

	/**
	 * id一致するCalendarPatternInfoを返す。
	 * 一致しなければ、キャッシュに追加する
	 * @param id
	 * @return
	 * @throws CalendarNotFound
	 * @throws InvalidRole
	 */
	public static CalendarPatternInfo getCalendarPatternInfo(String id) throws CalendarNotFound, InvalidRole {
		CalendarPatternInfo ret = null;
		synchronized (calendarPatternLock) {
			ret = map.get(id);
			if (ret != null) {
				return ret;
			}
			ret = getCalendarPatternInfoDB(id);
			map.put(id, ret);
		}
		return ret;
	}

	private static CalendarPatternInfo getCalendarPatternInfoDB(String id) throws CalendarNotFound, InvalidRole {
		//カレンダ取得
		CalPatternInfoEntity entity = null;

		entity = QueryUtil.getCalPatternInfoPK(id);

		//カレンダ情報のDTOを生成
		CalendarPatternInfo ret = new CalendarPatternInfo();
		//id
		ret.setId(entity.getCalPatternId());
		//名前
		ret.setName(entity.getCalPatternName());
		//オーナーロールID
		ret.setOwnerRoleId(entity.getOwnerRoleId());

		//登録者
		ret.setRegUser(entity.getRegUser());
		//登録日時
		if (entity.getRegDate() != null) {
			ret.setRegDate(entity.getRegDate().getTime());
		}
		//更新者
		ret.setUpdateUser(entity.getUpdateUser());
		//更新日時
		if (entity.getUpdateDate() != null) {
			ret.setUpdateDate(entity.getUpdateDate().getTime());
		}
		//カレンダ詳細情報
		ArrayList<YMD> ymdList = getCalPatternDetailList(id);
		for(YMD ymd : ymdList){
			m_log.trace("YMD : " + ymd.yyyyMMdd());
		}
		ret.getYmd().addAll(ymdList);
		return ret;
	}
	/**
	 * カレンダパターン詳細情報一覧を取得します。
	 * @param id
	 * @return カレンダパターン詳細情報のリスト
	 */
	private static ArrayList<YMD> getCalPatternDetailList(String id) {
		ArrayList<YMD> list = new ArrayList<YMD>();

		//カレンダパターンID別の情報を取得
		List<CalPatternDetailInfoEntity> ct = QueryUtil.getCalPatternDetailByCalPatternId(id);
		for (CalPatternDetailInfoEntity cal : ct) {
			YMD info = new YMD();
			//年
			info.setYear(cal.getId().getYearNo());
			//月
			info.setMonth(cal.getId().getMonthNo());
			//日
			info.setDay(cal.getId().getDayNo());
			list.add(info);
		}
		//昇順ソート
		Collections.sort(list);
		return list;
	}
}
