#!/bin/bash
#
# Copyright (C) since 2006 NTT DATA Corporation
# 
# This program is free software; you can redistribute it and/or
# Modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation, version 2.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details
#
# Born Shell Common Utilities for Hinemos
#
# <description>
#  This script is designed for Hinemos 
#
# <usage>
#  . hinemos_utility.sh
#
# <return>
#  
#
# <dependency>
#  
#
# $Id:$

export PROG=`basename $0`

MSG_CE011_01="$0: illegal argument."
MSG_CE011_02="Try \`$0 --help' for more information."
MSG_CE012_01="$0: illegal option."
MSG_CE012_02="Try \`$0 --help' for more information."
MSG_CE021="PID file (${JVM_PID_FILE}) exists. Java Virtual Machine may be running."
MSG_CE022="PID file (${JVM_PID_FILE}) does not exists. Java Virtual Machine may be stopped."
MSG_CE023="PID file (${JVM_PID_FILE}) exists. but the process is not Java Virtual Machine process expected."
MSG_CE024="PID file (${PG_PID_FILE}) exists. PostgreSQL may be running."
MSG_CE025="PID file (${PG_PID_FILE}) does not exists. PostgreSQL may be stopped."
MSG_CE026="PID file (${PG_PID_FILE}) exists. but the process is not PostgreSQL process expected."
MSG_CE061="incorrect user."

function ExitIllegalArgumentError() {
	Logging "${MSG_CE011_01}"
	Logging "${MSG_CE011_02}"
	exit 1
}

function ExitIllegalOptionErrorWithoutLogger() {
	# Only the logger output is done here so that the Builtin command may output the following messages to a standard error.
	logger "${PROG} ${MSG_CE012_01}"
	Logging "${MSG_CE012_02}"
	exit 1
}

function CheckUser() {
	CURRENT_USER=`/usr/bin/whoami`
	if [ "x"$1 != "x"${CURRENT_USER} ]
	then
		Logging "${MSG_CE061} current user is not $1."
		exit 6
	fi
}

#
# PG_PID is set to the PosgreSQL process id.
# PG_STATUS is set to the PosgreSQL status.
#
function CheckPostgreSQLStatus() {
	PG_ERROR_BASE_CODE=32
	
	GetPostgreSQLProcessState
	RET=$?
	if [ ${RET} -eq 0 ]
	then
		PG_PID=`head -n 1 ${PG_PID_FILE}`
		PG_STATUS="(running)"
		RETURN_CODE=0
	else
		PG_PID=""
		PG_STATUS="(stopped)"
		RETURN_CODE=$((${PG_ERROR_BASE_CODE} + ${RET}))
	fi
	
	return ${RETURN_CODE}
}

#
# JVM_PID is set to the Java VM process id.
# JVM_STATUS is set to the JVM status.
#
function CheckJVMStatus() {
	JVM_ERROR_BASE_CODE=64
	
	GetJVMProcessState
	RET=$?
	if [ ${RET} -eq 0 ]
	then
		read JVM_PID < ${JVM_PID_FILE}
		JVM_STATUS="(running)"
		RETURN_CODE=0
	else
		JVM_PID=""
		JVM_STATUS="(stopped)"
		RETURN_CODE=$((${JVM_ERROR_BASE_CODE} + ${RET}))
	fi
	
	return ${RETURN_CODE}
}


function CheckHinemosManagerStatus() {
	# PostgreSQL
	CheckPostgreSQLStatus
	PG_EXIT_CODE=$?

	# JVM
	CheckJVMStatus
	JVM_EXIT_CODE=$?

	if [ "x"${PG_PID} != "x" ] && [ "x"${JVM_PID} != "x" ]
	then
		Logging "Hinemos Manager is running..."
	else
		Logging "Hinemos Manager is stopped."
	fi

	# output PID
	Logging "  - Java VM Process ID    : ${JVM_PID} ${JVM_STATUS}"
	Logging "  - PostgreSQL Process ID : ${PG_PID} ${PG_STATUS}"

	return $((${JVM_EXIT_CODE} + ${PG_EXIT_CODE})) 
}

# 0 : running, 1 : stoppped
function GetJVMProcessState() {
	if [ -f ${JVM_PID_FILE} ]
	then
		read JVM_PID < ${JVM_PID_FILE}
		if [ "x"${JVM_PID} != "x" ] && [ `ps --no-headers --pid ${JVM_PID} e | grep "${JAVA_HOME}/bin/java.*com.clustercontrol.HinemosManagerMain" | wc -l` -ne 0 ]
		then
			return 0
		else
			return 2
		fi
	fi
	return 1
}

# 0 : running, 1 : stopped
function GetPostgreSQLProcessState() {
	if [ -f ${PG_PID_FILE} ]
	then
		PG_PID=`head -n 1 ${PG_PID_FILE}`
		if [ "x"${PG_PID} != "x" ] && [ `ps --no-headers --pid ${PG_PID} e | grep "${PG_HOME}/bin/postgres -D ${PGDATA}" | wc -l` -ne 0 ]
		then
			return 0
		else
			return 2
		fi
	fi
	return 1
}

function CheckJVMProcessStopped() {
	GetJVMProcessState
	RET=$?
	if [ ${RET} -eq 0 ]
	then
		Logging "${MSG_CE021}"
		exit 1
	fi
}

function CheckJVMProcessRunning() {
	GetJVMProcessState
	RET=$?
	if [ ${RET} -eq 1 ]
	then
		Logging "${MSG_CE022}"
		exit 1
	elif [ ${RET} -eq 2 ]
	then
		Logging "${MSG_CE023}"
		exit 2
	fi
}

function CheckPostgreSQLProcessStopped() {
	GetPostgreSQLProcessState
	RET=$?
	if [ ${RET} -eq 0 ]
	then
		Logging "${MSG_CE024}"
		exit 1
	fi
}

function CheckPostgreSQLProcessRunning() {
	GetPostgreSQLProcessState
	RET=$?
	if [ ${RET} -eq 1 ]
	then
		Logging "${MSG_CE025}"
		exit 1
	elif [ ${RET} -eq 2 ]
	then
		Logging "${MSG_CE026}"
		exit 1
	fi
}

function AskPostgreSQLPasswd() {
	if [ "x${PGPASSWORD}" = "x" ]
	then
		stty -echo
		read -p "input a password of Hinemos RDBM Server (default 'hinemos') : " PASSWD
		stty echo
		echo
		export PGPASSWORD=${PASSWD:-hinemos}
	fi
}


function Logging() {
	MODE=$2
	OPTS="-e"

	case ${MODE} in
		withline)
			EchoLine
			Logging "$1"
			EchoLine
			return 0
			;;
 		noreturn)
			OPTS="${OPTS} -n"
			;;
		*)
			;;
	esac

	echo ${OPTS} "$1" >&2

	/usr/bin/logger "${PROG} $1"
}

function EchoOK() {
	Logging "[ OK ]"
}

function EchoNG() {
	Logging "[ NG ]"
}

function EchoLine() {
	Logging "--------------------------------------------------"
}

