/*
Copyright (C) 2014 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */

package com.clustercontrol.ws.cloudn;

import java.util.List;

import javax.jws.WebMethod;
import javax.jws.WebParam;
import javax.jws.WebService;

import com.clustercontrol.cloud.CloudManagerFault;
import com.clustercontrol.fault.InvalidRole;
import com.clustercontrol.fault.InvalidUserPass;

/**
 * クラウド管理オプションの Web サービス API 定義。
 *
 *
 */
@WebService(targetNamespace = "http://cloudn.ws.clustercontrol.com")
public interface CloudnOptionEndpoint {
	/**
	 * cloudn に登録されているネットワークの一覧を取得します。
	 * 取得したネットワークは、VPCタイプ OpenNW へインスタンスを作成する際に指定できます。
	 * 
	 * @param roleId クラウドユーザーに紐づくロール ID。
	 * @param region リージョン名。
	 * @return ネットワークの一覧。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	List<Network> getNetworks(@WebParam(name = "roleId") String roleId, @WebParam(name = "region") String region) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * cloudn に登録されているセキュリティグループの一覧を取得します。
	 * 取得したセキュリティグループは、FLATタイプのインスタンスを作成する際に指定できます。
	 * 
	 * @param roleId クラウドユーザーに紐づくロール ID。
	 * @param region リージョン名。
	 * @return セキュリティグループの一覧。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	List<SecurityGroup> getSecurityGroups(@WebParam(name = "roleId") String roleId, @WebParam(name = "region") String region) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * インスタンスのパスワードをリセットします。
	 * Cloudnで作成される際に、インスタンスの作成元となるテンプレートが
	 * パスワードリセットを許可している必要があります。
	 * また、リセット後のパスワードは、getPassword を使用して取得できます。
	 * 
	 * @param roleId クラウドユーザーに紐づくロール ID。
	 * @param region リージョン名。
	 * @param instanceId インスタンスID。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	void resetPassword(@WebParam(name = "roleId") String roleId, @WebParam(name = "region") String region, @WebParam(name = "instanceId") String instanceId) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * cloudn で実行されている非同期ジョブの結果を確認し、その内容を返します。
	 * 結果には、コマンドの成否・取得したパスワード・エラー情報などが含まれます。
	 * 
	 * @param roleId クラウドユーザーに紐づくロール ID。
	 * @param region リージョン名。
	 * @param instanceId インスタンスID。
	 * @return パスワード取得コマンドの実行結果。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	PasswordResult getPassword(@WebParam(name = "roleId") String roleId, @WebParam(name = "region") String region, @WebParam(name = "instanceId") String instanceId) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * cloudn で実行されている非同期ジョブ（ストレージのアタッチ）の
	 * 結果を確認し、その内容を返します。
	 * 結果には、コマンドの成否・エラー情報などが含まれます。
	 * 
	 * @param roleId クラウドユーザーに紐づくロール ID。
	 * @param region リージョン名。
	 * @param storageId ストレージID。
	 * @param instanceId インスタンスID。
	 * @return ストレージアタッチ処理の実行結果。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	AttachStorageResult getAttachStorageResult(@WebParam(name = "roleId") String roleId, @WebParam(name = "region") String region, @WebParam(name = "storageId") String storageId, @WebParam(name = "instanceId") String instanceId) throws CloudManagerFault, InvalidUserPass, InvalidRole;

	/**
	 * cloudn で実行されている非同期ジョブ（インスタンスの削除）の
	 * 結果を確認し、その内容を返します。
	 * 結果には、コマンドの成否・エラー情報などが含まれます。
	 * 
	 * @param roleId クラウドユーザーに紐づくロール ID。
	 * @param region リージョン名。
	 * @param instanceId インスタンスID。
	 * @return インスタンス削除処理の実行結果。
	 * @throws CloudManagerFault
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 */
	@WebMethod
	DeleteInstanceResult getDeleteInstanceResult(@WebParam(name = "roleId") String roleId, @WebParam(name = "region") String region, @WebParam(name = "instanceId") String instanceId) throws CloudManagerFault, InvalidUserPass, InvalidRole;
}