/*
Copyright (C) 2014 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */
package com.clustercontrol.cloud.cloudn.util;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import org.apache.log4j.Logger;

import com.clustercontrol.cloud.CloudManagerFault;
import com.clustercontrol.cloud.Messages;
import com.clustercontrol.cloud.IResourceManagement.ICredential;
import com.clustercontrol.cloud.cloudn.rest.CloudnCompute;
import com.clustercontrol.cloud.cloudn.rest.CreateTagsRequest;
import com.clustercontrol.cloud.cloudn.rest.ListDiskOfferingsResponse;
import com.clustercontrol.cloud.cloudn.rest.ListServiceOfferingsResponse;
import com.clustercontrol.cloud.cloudn.rest.ListZonesResponse;
import com.clustercontrol.cloud.cloudn.rest.RTag;
import com.clustercontrol.cloud.cloudn.rest.Zone;
import com.clustercontrol.cloud.cloudn.rest.api.CloudnEndpoint;
import com.clustercontrol.cloud.cloudn.rest.api.RestApplicationBuilder;
import com.clustercontrol.cloud.cloudn.rest.api.RestfulApplicationException;
import com.clustercontrol.cloud.util.CloudMessageUtil;
import com.clustercontrol.cloud.util.HinemosUtil;
import com.clustercontrol.cloud.util.Tuple;
import com.google.common.base.Preconditions;

public class CloudnUtil {
	private CloudnUtil() {
	}
	
	public static CloudnCompute createCloudnCompute(ICredential credential, String location) throws CloudManagerFault {
		return createCloudnEndpoint(CloudnCompute.class, credential, location);
	}

	public static <T extends CloudnEndpoint> T createCloudnEndpoint(Class<T> interfaceClass, ICredential credential, String location) throws CloudManagerFault {
		Preconditions.checkNotNull(credential);
		return createCloudnEndpoint(interfaceClass, credential.getAccessKey(), credential.getSecretKey(), location);
	}
	
	public static <T extends CloudnEndpoint> T createCloudnEndpoint(Class<T> interfaceClass, String accessKey, String secretKey, String location) throws CloudManagerFault {
		Preconditions.checkNotNull(interfaceClass);
		Preconditions.checkNotNull(accessKey);
		Preconditions.checkNotNull(secretKey);
		Preconditions.checkNotNull(location);
		T endpoint = RestApplicationBuilder.buildRestApplicationProxy(interfaceClass, accessKey, secretKey);
		endpoint.setEndpoint("https://" + location);
		return endpoint;
	}
	
	private static ThreadLocal<Map<Tuple, CloudnEndpoint>> endpointMap = new ThreadLocal<Map<Tuple, CloudnEndpoint>>() {
		@Override
		protected Map<Tuple, CloudnEndpoint> initialValue() {
			return new HashMap<>();
		}
	};
	
	@SuppressWarnings("unchecked")
	public static <T extends CloudnEndpoint> T getCloudnEndpoint(Class<T> interfaceClass, String accessKey, String secretKey, String location) throws CloudManagerFault {
		try {
			Tuple t = Tuple.build(interfaceClass, accessKey, secretKey, location);
			Map<Tuple, CloudnEndpoint> emap = endpointMap.get();
			CloudnEndpoint e = emap.get(t);
			if (e == null) {
				e = createCloudnEndpoint(interfaceClass, accessKey, secretKey, location);
				emap.put(t, e);
			}
			return (T)e; 
		}
		catch (RestfulApplicationException e) {
			throw new CloudManagerFault(e.getMessage(), e.getErrorCode(), e);
		}
	}
	
	private static Map<String, Map<String, ListServiceOfferingsResponse.ServiceOffering>> serviceOfferingMap = new ConcurrentHashMap<>();
	
	public static Map<String, ListServiceOfferingsResponse.ServiceOffering> getInstanceFlavors(String accessKey, String secretKey, String location) throws CloudManagerFault {
		Map<String, ListServiceOfferingsResponse.ServiceOffering> smap = serviceOfferingMap.get(location);
		if (smap == null) {
			try {
				CloudnCompute endpoint = getCloudnEndpoint(CloudnCompute.class, accessKey, secretKey, location);
				ListServiceOfferingsResponse res = endpoint.listServiceOfferings();

				smap = new LinkedHashMap<>();
				for (ListServiceOfferingsResponse.ServiceOffering so: res.serviceOfferings) {
					smap.put(so.name, so);
				}
				
				// 非同期アクセスしているので、後勝ちの put になる。
				// 関数の特性上問題なし。
				// 先だろうが、後だろうが同じ内容のものが入る。
				// 入っているインスタンスの同一性 (equals) は保障しない。
				serviceOfferingMap.put(location, Collections.unmodifiableMap(smap));
			}
			catch (RestfulApplicationException e) {
				throw new CloudManagerFault(e.getMessage(), e.getErrorCode(), e);
			}
		}
		return smap;
	}

	private static Map<String, Map<String, Zone>> zoneMap = new ConcurrentHashMap<>();
	
	public static Map<String, Zone> getZones(String accessKey, String secretKey, String location) throws CloudManagerFault {
		Map<String, Zone> map = zoneMap.get(location);
		if (map == null) {
			try {
				CloudnCompute endpoint = getCloudnEndpoint(CloudnCompute.class, accessKey, secretKey, location);
				ListZonesResponse res = endpoint.listZones();

				map = new LinkedHashMap<>();
				for (Zone zone: res.zone) {
					map.put(zone.name, zone);
				}
				
				// 非同期アクセスしているので、後勝ちの put になる。
				// 関数の特性上問題なし。
				// 先だろうが、後だろうが同じ内容のものが入る。
				// 入っているインスタンスの同一性 (equals) は保障しない。
				zoneMap.put(location, Collections.unmodifiableMap(map));
			}
			catch (RestfulApplicationException e) {
				throw new CloudManagerFault(e.getMessage(), e.getErrorCode(), e);
			}
		}
		return map;
	}
	
	private static Map<String, Map<String, ListDiskOfferingsResponse.DiskOffering>> diskOfferingMap = new ConcurrentHashMap<>();
	
	public static Map<String, ListDiskOfferingsResponse.DiskOffering> getStorageFlavors(String accessKey, String secretKey, String location) throws CloudManagerFault {
		Map<String, ListDiskOfferingsResponse.DiskOffering> smap = diskOfferingMap.get(location);
		if (smap == null) {
			try {
				CloudnCompute endpoint = getCloudnEndpoint(CloudnCompute.class, accessKey, secretKey, location);
				ListDiskOfferingsResponse res = endpoint.listDiskOfferings();

				smap = new LinkedHashMap<>();
				for (ListDiskOfferingsResponse.DiskOffering so: res.diskOfferings) {
					smap.put(so.name, so);
				}
				
				// 非同期アクセスしているので、後勝ちの put になる。
				// 関数の特性上問題なし。
				// 先だろうが、後だろうが同じ内容のものが入る。
				// 入っているインスタンスの同一性 (equals) は保障しない。
				diskOfferingMap.put(location, Collections.unmodifiableMap(smap));
			}
			catch (RestfulApplicationException e) {
				throw new CloudManagerFault(e.getMessage(), e.getErrorCode(), e);
			}
		}
		return smap;
	}
	
	public static void addTag(CloudnCompute compute, String resourceId, String resourcetype, String key, String value) {
		addTags(compute, resourceId, resourcetype, new RTag(key, value));
	}

	public static void addTags(CloudnCompute compute, String resourceId, String resourcetype, List<RTag> tags) {
		if (tags.isEmpty()) return;
		
		// Name タグをつける際、作成直後のリソースが見つからずに失敗する場合があったので、リトライをする。
		int count = 0;
		int interval = 1000;
		for (;;) {
			try {
				CreateTagsRequest createTagsRequest = new CreateTagsRequest();
				createTagsRequest.withTags(tags).withResourceids(resourceId).withResourcetype(resourcetype);
				compute.createTags(createTagsRequest);
				break;
			}
			catch (RestfulApplicationException e) {
				++count;
	
				Logger logger = Logger.getLogger(CloudnUtil.class);
				logger.debug(e.getMessage(), e);
				
				if (count == 5) {
					throw e;
				}
				
				try {
					Thread.sleep(interval);
					interval += interval;
				} catch (InterruptedException e1) {
				}
			}
		}
	}

	public static void addTags(CloudnCompute compute, String resourceId, String resourcetype, RTag...tags) {
		addTags(compute, resourceId, resourcetype, Arrays.asList(tags));
	}
	
	public static void notify_Cloudn_Instance_Exception(
			String accountResourceId,
			String instanceId,
			String message,
			Exception exception
			) {
		notify_Cloudn_Instance_ErrorMessage(accountResourceId, instanceId, message, CloudMessageUtil.getExceptionStackTrace(exception));
	}

	public static void notify_Cloudn_Instance_ErrorMessage(
			String accountResourceId,
			String instanceId,
			String message,
			String messageOrg
			) {
		HinemosUtil.notifyInternalMessage(
			HinemosUtil.Priority.FAILURE,
			CloudnConstants.pluginId_cloudn,
			accountResourceId,
			Messages.messages().getString("message.autoupdate.error.detail1", new Object[]{accountResourceId, instanceId}),
			CloudnConstants.InternalScopeText,
			"CloudnOption",
			message,
			messageOrg);
	}
}
