/*
Copyright (C) 2014 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */
package com.clustercontrol.cloud.cloudn.factory;

import java.io.IOException;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.persistence.Query;

import org.apache.log4j.Logger;
import org.eclipse.jdt.annotation.NonNull;
import org.eclipse.jdt.annotation.NonNullByDefault;
import org.eclipse.jdt.annotation.Nullable;

import com.clustercontrol.cloud.CloudManagerFault;
import com.clustercontrol.cloud.Filter;
import com.clustercontrol.cloud.ICloudContext;
import com.clustercontrol.cloud.IResourceManagement.Storage.StorageAttachment;
import com.clustercontrol.cloud.InternalManagerError;
import com.clustercontrol.cloud.Messages;
import com.clustercontrol.cloud.SessionService;
import com.clustercontrol.cloud.bean.Image;
import com.clustercontrol.cloud.bean.InstanceStateChange;
import com.clustercontrol.cloud.bean.InstanceStateKind;
import com.clustercontrol.cloud.bean.PlatformKind;
import com.clustercontrol.cloud.bean.Snapshot;
import com.clustercontrol.cloud.bean.StorageAttachmentStateKind;
import com.clustercontrol.cloud.bean.StorageStateKind;
import com.clustercontrol.cloud.bean.Tag;
import com.clustercontrol.cloud.bean.Zone;
import com.clustercontrol.cloud.cloudn.CloudnOptionPropertyConstants;
import com.clustercontrol.cloud.cloudn.rest.AttachVolumeResponse;
import com.clustercontrol.cloud.cloudn.rest.CloudnCompute;
import com.clustercontrol.cloud.cloudn.rest.CreateSnapshotRequest;
import com.clustercontrol.cloud.cloudn.rest.CreateSnapshotResponse;
import com.clustercontrol.cloud.cloudn.rest.CreateTagsRequest;
import com.clustercontrol.cloud.cloudn.rest.CreateTemplateRequest;
import com.clustercontrol.cloud.cloudn.rest.CreateTemplateResponse;
import com.clustercontrol.cloud.cloudn.rest.CreateVolumeRequest;
import com.clustercontrol.cloud.cloudn.rest.CreateVolumeResponse;
import com.clustercontrol.cloud.cloudn.rest.DeployVirtualMachineRequest;
import com.clustercontrol.cloud.cloudn.rest.DeployVirtualMachineResponse;
import com.clustercontrol.cloud.cloudn.rest.DestroyVirtualMachineResponse;
import com.clustercontrol.cloud.cloudn.rest.DetachVolumeRequest;
import com.clustercontrol.cloud.cloudn.rest.ListDiskOfferingsResponse;
import com.clustercontrol.cloud.cloudn.rest.ListDiskOfferingsResponse.DiskOffering;
import com.clustercontrol.cloud.cloudn.rest.ListOSTypesResponse;
import com.clustercontrol.cloud.cloudn.rest.ListOsTypesRequest;
import com.clustercontrol.cloud.cloudn.rest.ListPublicIpAddressesResponse;
import com.clustercontrol.cloud.cloudn.rest.ListServiceOfferingsResponse;
import com.clustercontrol.cloud.cloudn.rest.ListSnapshotsRequest;
import com.clustercontrol.cloud.cloudn.rest.ListSnapshotsResponse;
import com.clustercontrol.cloud.cloudn.rest.ListTemplateRequest;
import com.clustercontrol.cloud.cloudn.rest.ListTemplatesResponse;
import com.clustercontrol.cloud.cloudn.rest.ListVirtualMachinesRequest;
import com.clustercontrol.cloud.cloudn.rest.ListVirtualMachinesResponse;
import com.clustercontrol.cloud.cloudn.rest.ListVolumesRequest;
import com.clustercontrol.cloud.cloudn.rest.ListVolumesResponse;
import com.clustercontrol.cloud.cloudn.rest.Nic;
import com.clustercontrol.cloud.cloudn.rest.QueryAsyncJobResultResponse;
import com.clustercontrol.cloud.cloudn.rest.RTag;
import com.clustercontrol.cloud.cloudn.rest.Template;
import com.clustercontrol.cloud.cloudn.rest.VirtualMachine;
import com.clustercontrol.cloud.cloudn.rest.Volume;
import com.clustercontrol.cloud.cloudn.rest.api.RestfulApplicationException;
import com.clustercontrol.cloud.cloudn.util.Cider;
import com.clustercontrol.cloud.cloudn.util.CloudnConstants;
import com.clustercontrol.cloud.cloudn.util.CloudnUtil;
import com.clustercontrol.cloud.commons.PropertyStringConstants;
import com.clustercontrol.cloud.dao.CloudInstanceDao;
import com.clustercontrol.cloud.dao.CloudStorageDao;
import com.clustercontrol.cloud.persistence.EntityManagerEx;
import com.clustercontrol.cloud.persistence.TransactionException;
import com.clustercontrol.cloud.persistence.Transactional;
import com.clustercontrol.cloud.util.Tuple;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectReader;
import com.fasterxml.jackson.databind.ObjectWriter;


@NonNullByDefault(false)
@Transactional(Transactional.TransactionType.Supported)
public class CloudnResourceManagement_VPC extends CloudnResourceManagementBase {
	public static String TEMP_HOSTNAME = "TEMP-HOSTNAME";
	
	public static String RT_Instance = "COMPUTE";
	public static String RT_Image = "COMPUTE_Template";
	public static String RT_Storage = "COMPUTE_Volume";
	public static String RT_Snapshot = "COMPUTE_Snapshot";
	public static String RT_LoadBalancer = "LBA";
	public static String RT_InstanceBackup = "InstanceBackup";
	public static String RT_StorageBackup = "StorageBackup";
	
	protected Cider cider = null;

	public static class AttachedStorage {
		public AttachedStorage() {
		}
		public AttachedStorage(String name, String storageId, String snapshotId) {
			this.name = name;
			this.storageId = storageId;
			this.snapshotId = snapshotId;
		}
		public String name;
		public String storageId;
		public String snapshotId;
	}

	public static class BackupedInstanceDetail {
		public String instanceId;
		public String name;
		public String flavor;
		public String zone;
		public List<AttachedStorage> attachedStorages;
		public List<Tag> tags;
		public Date createTime;
		public InstanceDetail instanceDetail;
	}

	public static class BackupedStorageDetail {
		public String storageId;
		public String name;
		public String flavor;
		public String zone;
		public Integer size;
		public Date createTime;
		public String description;
		public StorageDetail storageDetail;
	}

	public static class StorageDetail {
		public String volumeName;
	}
	
	protected InstanceStateKind instanceStateKind(String state) {
		if ("Destroyed".equals(state)) {
			return InstanceStateKind.terminated;
		}
		else if ("Starting".equals(state)) {
			return InstanceStateKind.pending;
		}
		else if ("Error".equals(state)) {
			return InstanceStateKind.terminated;
		}
		else if ("Expunging".equals(state)) {
			return InstanceStateKind.shutting_down;
		}
		else if ("Expunged".equals(state)) {
			return InstanceStateKind.terminated;
		}
		else if ("Migrating".equals(state)) {
			return InstanceStateKind.pending;
		}
		for (InstanceStateKind stateKind: InstanceStateKind.values()) {
			if (stateKind.label().equalsIgnoreCase(state)) {
				return stateKind;
			}
		}
		throw new InternalManagerError(state);
	}
	
	protected @NonNull CloudnCompute getCloudnEndpoint() throws CloudManagerFault {
		return CloudnUtil.getCloudnEndpoint(
				CloudnCompute.class,
				getCledential().getAccessKey(),
				getCledential().getSecretKey(),
				getRegion().getEndpoint(getComputeId()).getLocation());
	}
	
	protected @NonNull Map<String, ListServiceOfferingsResponse.ServiceOffering> getInstanceFlavorsInternal() throws CloudManagerFault {
		return CloudnUtil.getInstanceFlavors(
				getCledential().getAccessKey(),
				getCledential().getSecretKey(),
				getRegion().getEndpoint(getComputeId()).getLocation());
	}
	
	protected @NonNull ListServiceOfferingsResponse.ServiceOffering getInstanceFlavorInternal(@NonNull String name) throws CloudManagerFault {
		ListServiceOfferingsResponse.ServiceOffering offering =  getInstanceFlavorsInternal().get(name);
		if (offering == null)
			throw com.clustercontrol.cloud.cloudn.ErrorCode.SERVICE_OFFERING_NOT_FONUD.cloudManagerFault(name);
		return offering;
	}

	protected @NonNull Map<String, ListDiskOfferingsResponse.DiskOffering> getStorageFlavorsInternal() throws CloudManagerFault {
		return CloudnUtil.getStorageFlavors(
				getCledential().getAccessKey(),
				getCledential().getSecretKey(),
				getRegion().getEndpoint(getComputeId()).getLocation());
	}
	
	protected @NonNull ListDiskOfferingsResponse.DiskOffering getStorageFlavorInternal(@NonNull String name) throws CloudManagerFault{
		ListDiskOfferingsResponse.DiskOffering offering =  getStorageFlavorsInternal().get(name);
		if (offering == null)
			throw com.clustercontrol.cloud.cloudn.ErrorCode.DISK_OFFERING_NOT_FONUD.cloudManagerFault(name);
		return offering;
	}

	protected @NonNull Map<String, com.clustercontrol.cloud.cloudn.rest.Zone> getZonesInternal() throws CloudManagerFault {
		return CloudnUtil.getZones(
				getCledential().getAccessKey(),
				getCledential().getSecretKey(),
				getRegion().getEndpoint(getComputeId()).getLocation());
	}
	
	protected @NonNull com.clustercontrol.cloud.cloudn.rest.Zone getZoneInternal(@NonNull String name) throws CloudManagerFault {
		com.clustercontrol.cloud.cloudn.rest.Zone zone =  getZonesInternal().get(name);
		if (zone == null)
			throw com.clustercontrol.cloud.cloudn.ErrorCode.ZONE_NOT_FONUD.cloudManagerFault(name);
		return zone;
	}
	
	public static class InstanceDetail {
		public List<String> networkIds = new ArrayList<>();
		public String group = "";
		public Boolean attachingDisk;
	}
	

	protected @NonNull StorageStateKind getStorageStatus(@NonNull Volume volume){
		StorageStateKind state = null;
		switch(volume.state){
			case "Allocated":
			case "Ready":
				if(volume.virtualmachineid != null && !volume.virtualmachineid.isEmpty()
						&& !(volume.vmstate != null && volume.vmstate.equals("Destroyed"))
				){
					state = StorageStateKind.in_use;
				} else {
					state = StorageStateKind.available;
				}
				break;
			case "Creating":
			case "Migrating":
				state = StorageStateKind.creating;
				break;
			case "Expunged":
				state = StorageStateKind.deleting;
				break;
			default:
				Logger logger = Logger.getLogger(this.getClass());
				logger.info(volume.state);
				state = StorageStateKind.error;
		}
		return state;
	}
	
	protected @NonNull StorageAttachmentStateKind getAttachStatus(@NonNull Volume volume){
		StorageAttachmentStateKind state;
		switch(getStorageStatus(volume)){
			case available:
				state = StorageAttachmentStateKind.detached;
				break;
			case in_use:
				state = StorageAttachmentStateKind.attached;
				break;
			default:
				state = StorageAttachmentStateKind.detached;
		}
		
		return state;
	}
	
	protected static PlatformKind getOSType(String ostype) {
		if (Pattern.compile(CloudnOptionPropertyConstants.cloudn_os_win_pattern.value(), Pattern.CASE_INSENSITIVE).matcher(ostype).matches())
			return PlatformKind.windows;
		else if (Pattern.compile(CloudnOptionPropertyConstants.cloudn_os_linux_pattern.value(), Pattern.CASE_INSENSITIVE).matcher(ostype).matches())
			return PlatformKind.linux;
		return PlatformKind.other;
	}
	
	protected Cider getCider() {
		if (cider == null) {
			try {
				cider = new Cider(CloudnOptionPropertyConstants.cloudn_private_cidr.value());
			}
			catch (Exception e) {
				throw new InternalManagerError(e);
			}
		}
		return cider;
	}
	
	protected void setupIpAddress(Instance instance, VirtualMachine vm) throws CloudManagerFault {
		if (CloudnOptionPropertyConstants.cloudn_node_ip.match(PropertyStringConstants.qublic)) {
			CloudnCompute compute = getCloudnEndpoint();
			for (ListPublicIpAddressesResponse.PublicIpAddress a: compute.listPublicIpAddresses().publicipaddresses) {
				if (vm.id.equals(a.virtualmachineid)) {
					instance.setIpAddress(a.ipaddress);
					break;
				}
			}
		}
		else {
			for (Nic nic: vm.nic) {
				if (getCider().matches(nic.ipaddress)) {
					instance.setIpAddress(nic.ipaddress);
					break;
				}
			}
		}
		instance.setHostName(TEMP_HOSTNAME);
	}
	
	protected static class InstanceInfo {
		public final Instance instance;
//		public final VirtualMachine vm;
		public final DeployVirtualMachineResponse responce;
		
		public InstanceInfo(Instance instance, VirtualMachine vm, DeployVirtualMachineResponse responce) {
			this.instance = instance;
//			this.vm = vm;
			this.responce = responce;
		}
	}
	
	protected DeployVirtualMachineRequest getCreateInstanceRequest(String id, String name, String flavor, String imageId, String zone, String instanceDetail, List<Tag> tags) throws CloudManagerFault, JsonProcessingException, IOException{
		ObjectMapper om = new ObjectMapper();
		ObjectReader or = om.reader(InstanceDetail.class);
		InstanceDetail detail = or.readValue(instanceDetail);
		
		DeployVirtualMachineRequest deployRequest = new DeployVirtualMachineRequest();
		deployRequest.serviceofferingid = getInstanceFlavorInternal(flavor).id; // 必須
		deployRequest.templateid = parseImageId(imageId).get(0, String.class); // 必須
		deployRequest.zoneid = getZoneInternal(zone).id; // 必須
//		request.diskofferingid = getStorageFlavorName(detail.);
		deployRequest.displayname = id;
		deployRequest.group = detail.group;
//		deployRequest.name = name + '-' + new Date().getTime();
		String s = "";
		for (String networkId: detail.networkIds) {
			if (!s.isEmpty()) s += ',';
			s += networkId;
		}
		deployRequest.networkids = s; // 必須
		
		return deployRequest;
	}
	
	protected InstanceInfo internalCreateInstance(String id, String name, String flavor, String imageId, String zone, String instanceDetail, List<Tag> tags) throws CloudManagerFault {
		CloudnCompute compute = getCloudnEndpoint();
		try {
			DeployVirtualMachineRequest deployRequest = getCreateInstanceRequest(id, name, flavor, imageId, zone, instanceDetail, tags);
			
			DeployVirtualMachineResponse resoponse = compute.deployVirtualMachine(deployRequest);
			ListVirtualMachinesResponse vmlist = compute.listVirtualMachines(new ListVirtualMachinesRequest().withId(resoponse.id));
			if (vmlist.virtualMachines.isEmpty())
				throw com.clustercontrol.cloud.cloudn.ErrorCode.VM_NOT_FONUD.cloudManagerFault(resoponse.id); 

			final VirtualMachine vm = vmlist.virtualMachines.get(0);
			SessionService.current().addRollbackAction(
					new SessionService.RolebackAction() {
						@Override
						public void rollback() throws TransactionException {
							try {
								CloudnCompute compute = getCloudnEndpoint();
								compute.destroyVirtualMachine(vm.id);
							}
							catch (CloudManagerFault e) {
								throw new TransactionException(e);
							}
						}
					});

			List<RTag> cloudnTags = new ArrayList<>();
			for (Tag t: tags) {
				cloudnTags.add(new RTag().withKey(t.getKey()).withValue(t.getValue()));
			}
			CloudnUtil.addTags(compute, resoponse.id, "userVM", cloudnTags);
			
			Instance instance = new Instance();
			instance.setResourceType(RT_Instance);
			instance.setInstanceId(vm.id);
			instance.setName(name);
			instance.setFlavor(flavor);
			instance.setZone(vm.zonename);
			instance.setImageId(vm.templateid);
			ListOSTypesResponse ostypesList = compute.listOsTypes(new ListOsTypesRequest().withId(vm.guestosid));
			if (ostypesList.ostypes.isEmpty())
				throw new InternalManagerError("not found ostype of vm created. vm=" + vm.id); 
			instance.setPlatform(getOSType(ostypesList.ostypes.get(0).description));
			instance.setState(instanceStateKind(vm.state));
			if (vm.nic.isEmpty())
				throw new InternalManagerError("expect vm has one nic at least. vm=" + vm.id); 
			
			setupIpAddress(instance, vm);
			
			ListVolumesResponse volumelist = compute.listVolumes(new ListVolumesRequest().withVirtualMachineId(resoponse.id));
			List<Instance.BlockDeviceMapping> blockDeviceMappings = new ArrayList<>();
			for (Volume v: volumelist.volumes) {
				blockDeviceMappings.add(new Instance.BlockDeviceMapping(v.id, "", v.state));
			}
			instance.setBlockDeviceMappings(blockDeviceMappings);

			List<Tag> cloudTags = new ArrayList<>();
			for (RTag tag: cloudnTags) {
				cloudTags.add(new Tag(tag.key, tag.value));
			}
			
			if(vm.passwordenabled){
				// パスワード取得用ジョブID登録
				CloudnAsynchJobTrackingOperator.registJobId(Tuple.build("getPassword", instance.getInstanceId(), getAccountResourceId()), resoponse.jobid);
			}
			
			instance.setTags(cloudTags);
			instance.setActualResource(vm);

			return new InstanceInfo(instance, vm, resoponse);
		}
		catch (RestfulApplicationException e) {
			throw new CloudManagerFault(e.getMessageWithoutErrorCode(), e.getErrorCode(), e);
		}
		catch (Exception e) {
			throw new InternalManagerError(e);
		}
	}
	
	@Override
	public Instance createInstance(String id, String name, String flavor, String imageId, String zone, String instanceDetail, List<Tag> tags) throws CloudManagerFault {
		return internalCreateInstance(id, name, flavor, imageId, zone, instanceDetail, tags).instance;
	}

	@Override
	public void deleteInstance(String instanceId) throws CloudManagerFault {
		try {
			// 関連している EC2 インスタンスが存在するので削除。
			CloudnCompute compute = getCloudnEndpoint();
			
			// Startingが作成直後なのか、通常の起動処理のものなのかが判別できないため抑止を行わない。
//			if(compute.listVirtualMachines(new ListVirtualMachinesRequest().withId(instanceId)).virtualMachines.get(0).state.equals("Starting")){
//				throw com.clustercontrol.cloud.cloudn.ErrorCode.INSTANCE_CANNOT_DELETE_WHILE_LAUNCHING.cloudManagerFault(instanceId);
//			}
			
			DestroyVirtualMachineResponse response = compute.destroyVirtualMachine(instanceId);
			
			// 追跡用のジョブIDを登録
			CloudnAsynchJobTrackingOperator.registJobId(Tuple.build("getDeleteInstanceResult", instanceId, getAccountResourceId()), response.jobid);
		}
		catch (RestfulApplicationException e) {
			throw new CloudManagerFault(e.getMessageWithoutErrorCode(), e.getErrorCode(), e);
		}
	}

	protected Instance convertInstance(VirtualMachine vm) throws CloudManagerFault {
		Instance instance = new Instance();
		instance.setResourceType(RT_Instance);
		instance.setInstanceId(vm.id);
		instance.setName(vm.displayname);
		instance.setFlavor(vm.serviceofferingname);
		instance.setZone(vm.zonename);
		instance.setImageId(vm.templatename);
		ListOSTypesResponse ostypesList = getCloudnEndpoint().listOsTypes(new ListOsTypesRequest().withId(vm.guestosid));
		if (ostypesList.ostypes.isEmpty())
			throw new InternalManagerError("not found ostype of vm created. vm=" + vm.id); 
		instance.setPlatform(getOSType(ostypesList.ostypes.get(0).description));
		instance.setState(instanceStateKind(vm.state));
		if (vm.nic.isEmpty())
			throw new InternalManagerError("expect vm has one nic at least. vm=" + vm.id); 
		
		setupIpAddress(instance, vm);

		ListVolumesResponse volumelist = getCloudnEndpoint().listVolumes(new ListVolumesRequest().withVirtualMachineId(vm.id));
		List<Instance.BlockDeviceMapping> blockDeviceMappings = new ArrayList<>();
		for (Volume v: volumelist.volumes) {
			blockDeviceMappings.add(new Instance.BlockDeviceMapping(v.id, "", v.state));
		}
		instance.setBlockDeviceMappings(blockDeviceMappings);

		List<Tag> cloudTags = new ArrayList<>();
		for (com.clustercontrol.cloud.cloudn.rest.Tag tag: vm.tags) {
			cloudTags.add(new Tag(tag.key, tag.value));
		}
		instance.setTags(cloudTags);
		instance.setActualResource(vm);

		return instance;
	}

	@Override
	public Instance getInstance(String instanceId) throws CloudManagerFault {
		try {
			CloudnCompute compute = getCloudnEndpoint();
			VirtualMachine vm = compute.listVirtualMachines(new ListVirtualMachinesRequest().withId(instanceId)).virtualMachines.get(0);
			if(!isCorrectedInstance(vm)){
				throw ErrorCode.Resource_InvalidInstanceID_NotFound.cloudManagerFault(instanceId);
			}
			return convertInstance(vm);
		}
		catch (RestfulApplicationException e) {
			// cloudn は、エラーの内容が明瞭でないので、とりあえず失敗したら、インスタンスが無かった例外を返す。
			throw ErrorCode.Resource_InvalidInstanceID_NotFound.cloudManagerFault(instanceId);
		}
	}

	@Override
	public List<Instance> getInstances(String...instanceIds) throws CloudManagerFault {
		return getInstances(Arrays.asList(instanceIds));
	}

	@Override
	public List<Instance> getInstances(List<String> instanceIds) throws CloudManagerFault {
		try {
			CloudnCompute compute = getCloudnEndpoint();
			ListVirtualMachinesResponse result = compute.listVirtualMachines();

			List<Instance> instances = new ArrayList<>();
			if (instanceIds.isEmpty()) {
				for (VirtualMachine vm: result.virtualMachines) {
					if(isCorrectedInstance(vm)){
						instances.add(convertInstance(vm));
					}
				}
			}
			else {
				for (VirtualMachine vm: result.virtualMachines) {
					for (String instanceId: instanceIds) {
						if (instanceId.equals(vm.id) && !(instanceStateKind(vm.state).equals(InstanceStateKind.stopping) && !isRegisteredInstance(vm.id))) {
							instances.add(convertInstance(vm));
							instanceIds.remove(instanceId);
							// ここでちゃんと break しないと、
							// instanceIds から instanceId を抜いているので、
							// 次の for の判定で例外がスローされる。
							break;
						}
					}
				}
			}
			
			return instances;
		}
		catch (RestfulApplicationException e) {
			throw new CloudManagerFault(e.getMessageWithoutErrorCode(), e.getErrorCode(), e);
		}
	}

	@Override
	public InstanceStateChange startInstance(String instanceId) throws CloudManagerFault {
		try {
			CloudnCompute compute = getCloudnEndpoint();
			compute.startVirtualMachine(instanceId);

			InstanceStateChange change = new InstanceStateChange();
			change.setInstanceId(instanceId);
			change.setCurrentState(InstanceStateKind.pending);
			change.setPreviousState(InstanceStateKind.pending);

			return change;
		}
		catch (RestfulApplicationException e) {
			throw new CloudManagerFault(e.getMessageWithoutErrorCode(), e.getErrorCode(), e);
		}
	}

	@Override
	public InstanceStateChange stopInstance(String instanceId) throws CloudManagerFault {
		try {
			CloudnCompute compute = getCloudnEndpoint();
			compute.stopVirtualMachine(instanceId);

			InstanceStateChange change = new InstanceStateChange();
			change.setInstanceId(instanceId);
			change.setCurrentState(InstanceStateKind.stopping);
			change.setPreviousState(InstanceStateKind.stopping);

			return change;
		}
		catch (RestfulApplicationException e) {
			throw new CloudManagerFault(e.getMessageWithoutErrorCode(), e.getErrorCode(), e);
		}
	}

	@Override
	public List<String> getInstanceFlavors() throws CloudManagerFault {
		List<String> l = new ArrayList<>();
		Map<String, ListServiceOfferingsResponse.ServiceOffering> map = getInstanceFlavorsInternal();
		for (ListServiceOfferingsResponse.ServiceOffering value: map.values()) {
			l.add(value.name);
		}
		return l;
	}

	@Override
	public List<Zone> getZones() throws CloudManagerFault {
		List<Zone> zones = new ArrayList<>();
		Map<String, com.clustercontrol.cloud.cloudn.rest.Zone> map = getZonesInternal();
		for (com.clustercontrol.cloud.cloudn.rest.Zone value: map.values()) {
			Zone zone = new Zone();
			zone.setName(value.name);
			zones.add(zone);
		}
		return zones;
	}

	@Override
	public void attachStorage(final String instanceId, final String storageId, String deviceName) throws CloudManagerFault {
		try {
			CloudnCompute compute = getCloudnEndpoint();
			
			int retryCount = 0;
			//削除済みインスタンスに紐づいていないかを確認
			Volume volume = compute.listVolumes(new ListVolumesRequest().withId(storageId)).volumes.get(0);
			if(volume.vmstate != null && volume.vmstate.equals("Destroyed")){
				compute.detachVolume(new DetachVolumeRequest().withId(storageId));
				retryCount = 3;
			}

			//デタッチに時間が掛かった時のためにリトライを行う
			for(;;){
				try{
					AttachVolumeResponse response = compute.attachVolume(storageId, instanceId);

					// 追跡用のジョブIDを登録
					CloudnAsynchJobTrackingOperator.registJobId(Tuple.build("getAttachStorageResult", storageId, instanceId, getAccountResourceId()), response.jobid);
					break;
				}catch (RestfulApplicationException e){
					if(retryCount <= 0){
						throw e;
					}
					--retryCount;

					try {
						Thread.sleep(1000);
					} catch (InterruptedException e1) {
					}
				}
			}
		}
		catch (RestfulApplicationException e) {
//			Logger logger = Logger.getLogger(this.getClass());
//			logger.warn("error request: " + e.getRequestUrl());
			throw new CloudManagerFault(e.getMessageWithoutErrorCode(), e.getErrorCode(), e);
		}
		finally {
			SessionService.current().addRollbackAction(
					new SessionService.RolebackAction() {
						@Override
						public void rollback() throws TransactionException {
							try {
								CloudnCompute compute = getCloudnEndpoint();
								DetachVolumeRequest request = new DetachVolumeRequest();
								request.id = storageId;
								compute.detachVolume(request);
							}
							catch (CloudManagerFault e) {
								throw new TransactionException(e);
							}
						}
					});
		}
	}

	@Override
	public void detachStorage(final String instanceId, final String storageId) throws CloudManagerFault {
		try {
			CloudnCompute compute = getCloudnEndpoint();
			
			DetachVolumeRequest request = new DetachVolumeRequest();
			request.id = storageId;
//			DetachVolumeResponse response = compute.detachVolume(request);
			compute.detachVolume(request);
		}
		catch (RestfulApplicationException e) {
//			Logger logger = Logger.getLogger(this.getClass());
//			logger.warn("error request: " + e.getRequestUrl());
			throw new CloudManagerFault(e.getMessageWithoutErrorCode(), e.getErrorCode(), e);
		}
		//　完全に元の形でアタッチすることができないため、ロールバックによるアタッチは行わない
//		finally{
//			SessionService.current().addRollbackAction(
//					new SessionService.RolebackAction() {
//						@Override
//						public void rollback() throws TransactionException {
//							try {
//								CloudnCompute_VPC compute = getCloudnEndpoint();
//								//元の位置にアタッチできることが保証できない。
//								compute.attachVolume(storageId, instanceId);
//							}
//							catch (CloudManagerFault e) {
//								throw new TransactionException(e);
//							}
//						}
//					});
//		}
	}

	@Override
	public Storage createStorage(String name, String flavor, int size, String snapshotId, String zone, String storageDetail) throws CloudManagerFault {
		CreateVolumeResponse response = null;
		try {
			CloudnCompute compute = getCloudnEndpoint();
			
			CreateVolumeRequest request = new CreateVolumeRequest();
			request.name = name;
			
			if(flavor != null && ! flavor.isEmpty()){
				request.diskofferingid = getStorageFlavorInternal(flavor).id;
			} else if(snapshotId != null && ! snapshotId.isEmpty()) {
				List<com.clustercontrol.cloud.cloudn.rest.Snapshot> snapshots = compute.listSnapshots(new ListSnapshotsRequest().withId(snapshotId)).snapshots;  
				if(snapshots.size() > 0 && snapshots.get(0).volumetype.equals(CloudnConstants.ROOT_VOLUME_TYPE)){
					throw com.clustercontrol.cloud.cloudn.ErrorCode.STORAGE_CANNOT_CREATE_FROM_SNAPSHOT_OF_ROOT_VOLUME.cloudManagerFault(snapshotId);
				}
				request.snapshotid = snapshotId;
			}

			request.zoneid = getZoneInternal(zone).id;
			
			response = compute.createVolume(request);
			
			return getStorage(response.id);
		}
		catch (RestfulApplicationException e) {
//			Logger logger = Logger.getLogger(this.getClass());
//			logger.warn("error request: " + e.getRequestUrl());
			throw new CloudManagerFault(e.getMessageWithoutErrorCode(), e.getErrorCode(), e);
		}
		finally {
			if(response != null){
				final String id = response.id;
				SessionService.current().addRollbackAction(
					new SessionService.RolebackAction() {
						@Override
						public void rollback() throws TransactionException {
							try {
								CloudnCompute compute = getCloudnEndpoint();
								compute.deleteVolume(id);
							}
							catch (CloudManagerFault e) {
								throw new TransactionException(e);
							}
						}
					});
			}
		}
	}

	@Override
	public void deleteStorage(String storageId) throws CloudManagerFault {
		try {
			CloudnCompute compute = getCloudnEndpoint();

			int retryCount = 0;
			//削除済みインスタンスに紐づいていないかを確認
			Volume volume = compute.listVolumes(new ListVolumesRequest().withId(storageId)).volumes.get(0);
			if(volume.vmstate != null && volume.vmstate.equals("Destroyed")){
				compute.detachVolume(new DetachVolumeRequest().withId(storageId));
				retryCount = 3;
			}

			//デタッチに時間が掛かった時のためにリトライを行う
			for(;;){
				try{
					compute.deleteVolume(storageId);
					break;
				}catch (RestfulApplicationException e){
					if(retryCount <= 0){
						throw e;
					}
					--retryCount;

					try {
						Thread.sleep(1000);
					} catch (InterruptedException e1) {
					}
				}
			}
		}
		catch (RestfulApplicationException e) {
//			Logger logger = Logger.getLogger(this.getClass());
//			logger.warn("error request: " + e.getRequestUrl());
			throw new CloudManagerFault(e.getMessageWithoutErrorCode(), e.getErrorCode(), e);
		}
	}

	@Override
	public Storage getStorage(String storageId) throws CloudManagerFault {
		List<Storage> storages = getStorages(storageId);
		
		if(storages.size() < 1){
			throw com.clustercontrol.cloud.cloudn.ErrorCode.STORAGE_NOT_FONUD.cloudManagerFault(storageId);
		}
		
		return storages.get(0);
		
		// どのみちマイグレーションへの考慮が必要なため単体リクエストは行わない。
//		try {
//			CloudnCompute compute = getCloudnEndpoint();
//			ListVolumesRequest request = new ListVolumesRequest();
//			request.id = storageId;
//			
//			ListVolumesResponse response = compute.listVolumes(request);
//
//			if(response.volumes.size() < 1 || !isCorrectedStorage(response.volumes.get(0))){
//				throw com.clustercontrol.cloud.cloudn.ErrorCode.STORAGE_NOT_FONUD.cloudManagerFault(storageId);
//			}
//			
//			Volume volume = response.volumes.get(0);
//
//			return createStorage(volume);
//		}
//		catch (RestfulApplicationException e) {
////			Logger logger = Logger.getLogger(this.getClass());
////			logger.warn("error request: " + e.getRequestUrl());
//			throw new CloudManagerFault(e.getMessageWithoutErrorCode(), e.getErrorCode(), e);
//		}
	}

	public @Nullable Date convertDate(@NonNull String source) {
		try {
			return (new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ssZ")).parse(source);
		} catch (ParseException e) {
			Logger logger = Logger.getLogger(this.getClass());
			logger.info(source);
			return null;
		}
	}
	
	@Override
	public List<Storage> getStorages(String... storageIds) throws CloudManagerFault {
		return getStorages(Arrays.asList(storageIds));
	}
	
	@Override
	public List<Storage> getStorages(List<String> storageIds) throws CloudManagerFault {
		try {
			CloudnCompute compute = getCloudnEndpoint();
			ListVolumesResponse response = compute.listVolumes();

			List<Storage> storages = new ArrayList<Storage>();
			if (storageIds.isEmpty()) {
				for(Volume v: response.volumes){
					if (isCorrectedStorage(v)){
						storages.add(createStorage(v));
					}
				}
			} else {
				for(Volume v: response.volumes){
					for(String id: storageIds){
						if(id.equals(v.id) && isCorrectedStorage(v)){
							storages.add(createStorage(v));
							break;
						}
					}
				}
			}
			
			// ストレージのマイグレーションによるストレージバックアップの挿げ替え。
			{
				ICloudContext context = SessionService.current().get(ICloudContext.class);
				EntityManagerEx em = SessionService.current().getEntityManagerEx();
				Query query = em.createQuery("SELECT s FROM CloudStorageDao s JOIN CloudAccountResourceDao a ON s.accountResourceId = a.accountResourceId JOIN CloudRegionDao r ON a.cloudServiceId = r.cloudServiceId " +
								"WHERE a.accountResourceId = :accountResourceId AND r.region = :region");
				query.setParameter("accountResourceId", context.getAccessDestionation().getCloudAccountResource().getAccountResourceId());
				query.setParameter("region", context.getCurrentRegion().getRegion());
				@SuppressWarnings("unchecked")

				List<CloudStorageDao> difdaos = new ArrayList<>((List<CloudStorageDao>)query.getResultList());
				List<Volume> difvolumes = new ArrayList<>(response.volumes);
				
				List<Tuple> expunged = new ArrayList<>();
				Iterator<CloudStorageDao> daoIter = difdaos.iterator();
				while (daoIter.hasNext()) {
					CloudStorageDao d = daoIter.next();
					
					Iterator<Volume> volumeIter = difvolumes.iterator();
					while (volumeIter.hasNext()) {
						Volume v = volumeIter.next();
						if (v.id.equals(d.getStorageId())) {
							daoIter.remove();
							volumeIter.remove();

							if ("Expunged".equals(v.state) || "Expunging".equals(v.state)) {
								expunged.add(Tuple.build(d, v));
							}
							break;
						}
					}
				}
				
				MigrationProcesser.StorageMigrationHolder migrated = new MigrationProcesser.StorageMigrationHolder();
				for (CloudStorageDao d: difdaos) {
					for (Volume v: response.volumes) {
						if (!d.getStorageId().equals(v.id) && d.getStorageName().equals(v.name)) {
							migrated.add(new MigrationProcesser.StorageMigration().from(d).to(createStorage(v)));
							break;
						}
					}
				}
				for (Tuple t: expunged) {
					CloudStorageDao d = t.get(0, CloudStorageDao.class);
					for (Volume v: response.volumes) {
						if (!d.getStorageId().equals(v.id) && d.getStorageName().equals(v.name)) {
							migrated.add(new MigrationProcesser.StorageMigration().from(d).to(createStorage(v)));
							break;
						}
					}
				}
				
				if (!migrated.isEmpty())
					SessionService.current().set(MigrationProcesser.StorageMigrationHolder.class, migrated);
			}
			return storages;
		}
		catch (RestfulApplicationException e) {
			throw new CloudManagerFault(e.getMessageWithoutErrorCode(), e.getErrorCode(), e);
		}
		catch (Exception e) {
			throw new InternalManagerError(e);
		}
	}
	
	protected Storage createStorage(Volume volume){
		Storage storage = new Storage();
		storage.setResourceType(RT_Storage);
		storage.setStorageId(volume.id);
//		if(volume.tags != null){
//			for (com.clustercontrol.cloud.cloudn.rest.Tag t: volume.tags) {
//			
//			}
//		}
		storage.setName(volume.name);
//		storage.setSnapshotId(volume.);
		storage.setSize(stringToVolumeSize(volume.size));
		storage.setZone(volume.zonename);
		storage.setFlavor(volume.diskofferingname);
		storage.setState(getStorageStatus(volume));

		if(((volume.attached != null && !volume.attached.isEmpty()) || volume.type.equals(CloudnConstants.ROOT_VOLUME_TYPE)) && !(volume.vmstate != null && volume.vmstate.equals("Destroyed"))){
			StorageAttachment sa = new StorageAttachment();
			sa.setInstanceId(volume.virtualmachineid);
			sa.setDevice(volume.deviceid.toString());
			sa.setState(getAttachStatus(volume));
			if(volume.type.equals(CloudnConstants.ROOT_VOLUME_TYPE)){
				sa.setAttachTime(convertDate(volume.created));
			} else {
				sa.setAttachTime(convertDate(volume.attached));
			}
			storage.setStorageAttachment(sa);
		}

		storage.setCreateTime(convertDate(volume.created));
		storage.setActualResource(volume);
		return storage;
	}
	
	protected int stringToVolumeSize(String size){
		long quote = new Long(size);
		return (int) (quote / 1024 / 1024 / 1024);
	}

	@Override
	public List<String> getStorageFlavors() throws CloudManagerFault {
		List<String> flavors = new ArrayList<String>();
		for(DiskOffering offering: getStorageFlavorsInternal().values()){
			flavors.add(offering.name);
		}
		return flavors;
	}

	@Override
	public StorageBackup createStorageBackup(String storageId, String name, String description, String backupOption) throws CloudManagerFault {
		CloudnCompute compute = getCloudnEndpoint();
		CreateSnapshotResponse response = null;
		try {
			Volume volume = compute.listVolumes(new ListVolumesRequest().withId(storageId)).volumes.get(0);
			
			if(volume.type.equals(CloudnConstants.ROOT_VOLUME_TYPE)){
				throw com.clustercontrol.cloud.cloudn.ErrorCode.STORAGE_CANNOT_BACKUP_ROOT_VOLUME.cloudManagerFault(storageId);
			}
			
			response = compute.createSnapshot(new CreateSnapshotRequest().withVolumeid(storageId));

			BackupedStorageDetail backupedData = new BackupedStorageDetail();
			backupedData.storageId = storageId;
			backupedData.name = name;
			
			backupedData.flavor = volume.diskofferingid;
			backupedData.zone = volume.zonename;
			backupedData.size = stringToVolumeSize(volume.size);
			backupedData.createTime = convertDate(volume.created);
			backupedData.description = description;
			
			StorageDetail storageDetail = new StorageDetail();
			storageDetail.volumeName = volume.name;
			backupedData.storageDetail = storageDetail;			

			ObjectMapper om = new ObjectMapper();
			ObjectWriter ow = om.writerWithType(BackupedStorageDetail.class);
			String backupedDetail = ow.writeValueAsString(backupedData);
			this.getStore().put(RT_StorageBackup, response.id, backupedDetail);
			
			compute.createTags(new CreateTagsRequest().withResourceids(response.id).withResourcetype("Snapshot").withTag(new RTag().withKey("SourceStorageId").withValue(storageId)));
			
			int retryCount = 0;
			for(;;){
				try{
					ListSnapshotsResponse createdSnnapshotResponse = compute.listSnapshots(new ListSnapshotsRequest().withId(response.id));
					return createStorageBackup(createdSnnapshotResponse.snapshots.get(0));
				} catch (RestfulApplicationException | IndexOutOfBoundsException | NullPointerException  e) {
					if(e instanceof RestfulApplicationException){
						Logger logger = Logger.getLogger(this.getClass());
						logger.warn("failed request: " + ((RestfulApplicationException)e).getRequestUrl());
					}
					if(retryCount > 3){
						break;
					}
					try {
						Thread.sleep(1000);
					} catch (InterruptedException e1) {
					}
					++retryCount;
				}
			}
			this.getStore().remove(RT_StorageBackup, response.id);
			return null;
		}
		catch (RestfulApplicationException e) {
//			Logger logger = Logger.getLogger(this.getClass());
//			logger.warn("error request: " + e.getRequestUrl());
			throw new CloudManagerFault(e.getMessageWithoutErrorCode(), e.getErrorCode(), e);
		}
		catch (Exception e) {
			throw new InternalManagerError(e);
		}
		finally {
			if(response != null){
				final String snapshotId = response.id;
				SessionService.current().addRollbackAction(new SessionService.RolebackAction() {
					@Override
					public void rollback() throws TransactionException {
						try {
							CloudnCompute compute = getCloudnEndpoint();
							compute.deleteSnapshot(snapshotId, null);
						}
						catch (CloudManagerFault e) {
							throw new TransactionException(e);
						}
					}
				});
			}
		}
	}

	@Override
	public void deleteStorageBackup(String storageBackupId) throws CloudManagerFault {
		try {
			CloudnCompute compute = getCloudnEndpoint();
			compute.deleteSnapshot(storageBackupId, null);
		}
		catch (RestfulApplicationException e) {
//			Logger logger = Logger.getLogger(this.getClass());
//			logger.warn("error request: " + e.getRequestUrl());
			throw new CloudManagerFault(e.getMessageWithoutErrorCode(), e.getErrorCode(), e);
		}

		try {
			getStore().remove(RT_StorageBackup, storageBackupId);
		}
		catch (Exception e) {
			Logger logger = Logger.getLogger(this.getClass());
			logger.error(e.getMessage(), e);
		}
	}

	@Override
	public List<Snapshot> getSnapshots(String...snapshotIds) throws CloudManagerFault {
		return getSnapshotsWithFilter(new Filter("snapshot-id", snapshotIds));
	}

	@Override
	public List<Snapshot> getSnapshots(List<String> snapshotIds) throws CloudManagerFault {
		return getSnapshotsWithFilter(new Filter("snapshot-id", snapshotIds));
	}

	@Override
	public List<Snapshot> getSnapshotsWithFilter(Filter...filters) throws CloudManagerFault {
		return getSnapshotsWithFilter(Arrays.asList(filters));
	}

	@Override
	public List<Snapshot> getSnapshotsWithFilter(List<Filter> filters) throws CloudManagerFault {
		CloudnCompute compute = getCloudnEndpoint();
		
		ListSnapshotsRequest request = new ListSnapshotsRequest();
		List<String> snapshotIds = null;
		for(Filter filter: filters){
			if(filter.getName().equals("zone")){
				request.zoneid = getZoneInternal(filter.getValues().get(0)).id;
			}
			if(filter.getName().equals("snapshot-id")){
				snapshotIds = filter.getValues();
			}
		}
		
		
		List<Snapshot> snapshots = new ArrayList<>();
		try{
			List<com.clustercontrol.cloud.cloudn.rest.Snapshot> orgSnapshots = compute.listSnapshots(request).snapshots;

			if(orgSnapshots != null){
				if(snapshotIds != null){
					List<com.clustercontrol.cloud.cloudn.rest.Snapshot> tmpSnapshots = orgSnapshots;
					orgSnapshots = new ArrayList<com.clustercontrol.cloud.cloudn.rest.Snapshot>();
					for(String id: snapshotIds){
						for(com.clustercontrol.cloud.cloudn.rest.Snapshot snapshot: tmpSnapshots){
							if(id.equals(snapshot.id)){
								orgSnapshots.add(snapshot);
								break;
							}
						}
					}
				}
				for(com.clustercontrol.cloud.cloudn.rest.Snapshot restSnapshot: orgSnapshots){
					if(!restSnapshot.volumetype.equals(CloudnConstants.ROOT_VOLUME_TYPE)){
						Snapshot snapshot = new Snapshot();
						snapshot.setResourceType(RT_Snapshot);
						snapshot.setSnapshotId(restSnapshot.id);
						snapshot.setDescription(restSnapshot.name);
						snapshots.add(snapshot);
					}
				}
			}
		} catch (RestfulApplicationException e) {
//			Logger logger = Logger.getLogger(this.getClass());
//			logger.warn("error request: " + e.getRequestUrl());
			throw new CloudManagerFault(e.getMessageWithoutErrorCode(), e.getErrorCode(), e);
		}

		return snapshots;
	}

	@Override
	public List<Image> getImages(String...imageIds) throws CloudManagerFault {
		return getImagesWithFilter(new Filter("image-id", imageIds));
	}

	@Override
	public List<Image> getImages(List<String> imageIds) throws CloudManagerFault {
		return getImagesWithFilter(new Filter("image-id", imageIds));
	}

	@Override
	public List<Image> getImagesWithFilter(Filter...filters) throws CloudManagerFault {
		return getImagesWithFilter(Arrays.asList(filters));
	}

	@Override
	public List<Image> getImagesWithFilter(List<Filter> filters) throws CloudManagerFault {
		List<ImageBucket> ibs = internalGetImagesWithFilter(filters);
		List<Image> images = new ArrayList<>();
		for (ImageBucket ib: ibs) {
			images.add(ib.cloudImgae);
		}
		return images;
	}

	protected static class ImageBucket {
		public ImageBucket(Image cloudImgae, Template cloudnImgae) {
			this.cloudImgae = cloudImgae;
			this.cloudnImgae = cloudnImgae;
		}
		public Image cloudImgae;
		public Template cloudnImgae;
		
		public ImageBucket withCloudImgae(Image cloudImgae) {
			this.cloudImgae = cloudImgae;
			return this;
		}
		public ImageBucket withCloudnImgae(Template cloudnImgae) {
			this.cloudnImgae = cloudnImgae;
			return this;
		}
	}
	
	protected static final Pattern imageIdPattern = Pattern.compile("^(.*)\\+([^\\+]*)$");

	protected @Nullable static Tuple parseImageId(String imageId) throws CloudManagerFault {
		Matcher m = imageIdPattern.matcher(imageId);
		if (!m.matches()) throw com.clustercontrol.cloud.cloudn.ErrorCode.IMAGE_INVALID_ID.cloudManagerFault(imageId);
		String templateId = m.group(1);
		String zoneName = m.group(2);
		return Tuple.build(templateId, zoneName);
	}
	
	protected @Nullable static String createImageId(String imageId, String zoneId) throws CloudManagerFault {
		return imageId + "+" + zoneId;
	}

	protected List<ImageBucket> internalGetImagesWithFilter(Filter...filters) throws CloudManagerFault {
		return internalGetImagesWithFilter(Arrays.asList(filters));
	}
	
	protected List<ImageBucket> internalGetImagesWithFilter(List<Filter> filters) throws CloudManagerFault {
		CloudnCompute compute = getCloudnEndpoint();
		ListTemplateRequest request = new ListTemplateRequest().withTemplatefilter("executable");
		
		List<Tuple> ts = new ArrayList<>();
		for (Filter f: filters) {
			switch (f.getName()) {
			case "category":
				if (f.getValues().isEmpty())
					break;
				switch (f.getValues().get(0)) {
				case "official":
				case "featured":
					request.withTemplatefilter("featured");
					break;
				case "my":
				case "self":
					request.withTemplatefilter("self");
					break;
				case "all":
				case "executable":
					request.withTemplatefilter("executable");
					break;
				default:
					break;
				}
				break;
			case "zone":
				request.withZoneId(getZonesInternal().get(f.getValues().get(0)).id);
				break;
			case "image-id":
				if (f.getValues().size() == 1) {
					try {
						Tuple t = parseImageId(f.getValues().get(0));
						request.withId(t.get(0, String.class));
					}
					catch(Exception e) {
						throw com.clustercontrol.cloud.cloudn.ErrorCode.FILTER_INVALID.cloudManagerFault(f.getName(), f.getValues().get(0));
					}
				}
				else {
					for (Object value: f.getValues()) {
						try {
							Tuple t = parseImageId(value.toString());
							ts.add(t);
						}
						catch(Exception e) {
							// 他のフィルターが有効かもしれないので、ループ継続
							Logger.getLogger(this.getClass()).warn(com.clustercontrol.cloud.cloudn.ErrorCode.FILTER_INVALID.cloudManagerFault(f.getName(), f.getValues().get(0)).getMessage());
						}
					}
					if (!f.getValues().isEmpty() && ts.isEmpty())
						throw com.clustercontrol.cloud.cloudn.ErrorCode.FILTER_INVALID.cloudManagerFault(f.getName(), f.getValues().get(0));
				}
				break;
			default:
				break;
			}
		}
		ListTemplatesResponse response = compute.listTemplates(request);
		
		List<ImageBucket> bs = new ArrayList<>();
		for(com.clustercontrol.cloud.cloudn.rest.Template template: response.template){
			// サイズが 1 の場合、既にリクエストに ID を含めているのでフィルター済み。
			// フィルター用の ID が複数あるなら、以下の処理でフィルターする。
			if (ts.size() > 1) {
				boolean match = false;
				for (Tuple t: ts) {
					if (template.id.equals(t.get(0)) && template.zonename.equals(t.get(1))){
						match = true;
						break;
					}
				}
				if (!match)
					continue;
			}
			Image image = new Image();
			image.setResourceType(RT_Image);
			image.setName(template.name);
			image.setImageId(template.id + "+" + template.zonename);
			image.setDescription(template.displaytext);
			image.setActualResource(template);
			bs.add(new ImageBucket(image, template));
		}
		return bs;
	}
	
	@Override
	public Snapshot getSnapshot(String snapshotId) throws CloudManagerFault {
		List<Snapshot> snapshots = getSnapshots(snapshotId);
		if (snapshots.isEmpty()) {
			throw ErrorCode.Resource_InvalidSnapshot_NotFound.cloudManagerFault(snapshotId);
		}
		return snapshots.get(0);
	}
	
	@Override
	public Image getImage(String imageId) throws CloudManagerFault {
		List<Image> images = getImages(imageId);
		if (images.isEmpty()) {
			throw ErrorCode.Resource_InvalidImageID_NotFound.cloudManagerFault(imageId);
		}
		return images.get(0);
	}

	protected RestoreInfo internalRestoreInstance(String facilityId, String instanceBackupId, String name, String flavor, String zone/*この情報は未使用。*/, String instanceDetail, List<Tag> tags) throws CloudManagerFault, JsonProcessingException, IOException{
		String value = getStore().get(RT_InstanceBackup, instanceBackupId);

		ObjectMapper om = new ObjectMapper();
		BackupedInstanceDetail backuped = om.reader(BackupedInstanceDetail.class).readValue(value);

		backuped.name = name != null ? name: backuped.name;
		backuped.flavor = flavor != null ? flavor: backuped.flavor;
//		backuped.zone = zone != null ? zone: backuped.zone;

		if (instanceDetail != null) {
			InstanceDetail override = om.reader(InstanceDetail.class).readValue(instanceDetail);

			if (backuped.instanceDetail == null) {
				backuped.instanceDetail = override;
			}
			else {
				backuped.instanceDetail.group = override.group != null ? override.group: backuped.instanceDetail.group;
				backuped.instanceDetail.networkIds = override.networkIds != null ? override.networkIds: backuped.instanceDetail.networkIds;
				backuped.instanceDetail.attachingDisk = override.attachingDisk != null ? override.attachingDisk: backuped.instanceDetail.attachingDisk;
			}
		}

		List<Tag> tagsTemp = new ArrayList<>(tags);
		for (Tag t: backuped.tags) {
			if (tagsTemp.isEmpty()) {
				break;
			}
			for (Iterator<Tag> iter = tagsTemp.iterator(); iter.hasNext(); ) {
				Tag override = iter.next();
				if (t.getKey().equals(override.getKey())) {
					t.setValue(override.getValue());
					iter.remove();
					break;
				}
			}
		}
		for (Tag t: tagsTemp) {
			backuped.tags.add(t);
		}
		String backupedDetail = om.writerWithType(InstanceDetail.class).writeValueAsString(backuped.instanceDetail);

		return new RestoreInfo(internalCreateInstance(facilityId, backuped.name, backuped.flavor, instanceBackupId, backuped.zone, backupedDetail, backuped.tags), backuped.attachedStorages, backuped.instanceDetail.attachingDisk);
	}
	
	protected static class RestoreInfo{
		private InstanceInfo instanceInfo;
		private List<AttachedStorage> attachStorages;
		private Boolean attachingDisk;
		public RestoreInfo(InstanceInfo instanceInfo, List<AttachedStorage> attachStorages, Boolean attachingDisk){
			this.instanceInfo = instanceInfo;
			this.attachStorages = attachStorages;
			this.attachingDisk = attachingDisk;
		}
		public InstanceInfo getInstanceInfo() {return instanceInfo;}
		public List<AttachedStorage> getAttachStorages() {return attachStorages;}
		public Boolean isAttachingDisk() {return attachingDisk;}
	}
	
	@Override
	public Instance restoreInstance(String facilityId, String instanceBackupId, String name, String flavor, String zone/*この情報は未使用。*/, String instanceDetail, List<Tag> tags) throws CloudManagerFault {
		try {
			RestoreInfo restoreInfo = internalRestoreInstance(facilityId, instanceBackupId, name, flavor, zone, instanceDetail, tags);
			final InstanceInfo instanceInfo = restoreInfo.instanceInfo;
			SessionService.current().addRollbackAction(
				new SessionService.RolebackAction() {
					@Override
					public void rollback() throws TransactionException {
						try {
							try {
								CloudnCompute compute = getCloudnEndpoint();
								compute.destroyVirtualMachine(instanceInfo.instance.getInstanceId());
							}
							catch (CloudManagerFault e) {
								throw new TransactionException(e);
							}
						}
						catch (RestfulApplicationException e) {
							throw new TransactionException(e);
						}
					}
				});
			
			final List<String> createdStorages = new ArrayList<>();
			CloudnCompute compute = getCloudnEndpoint();
			for (AttachedStorage backupedStorage: restoreInfo.getAttachStorages()) {
				final CreateVolumeResponse response = compute.createVolume(new CreateVolumeRequest().withName(backupedStorage.name).withSnapshotid(backupedStorage.snapshotId));
				SessionService.current().addRollbackAction(
					new SessionService.RolebackAction() {
						@Override
						public void rollback() throws TransactionException {
							try {
								try {
									CloudnCompute compute = getCloudnEndpoint();
									compute.deleteVolume(response.id);
								}
								catch (CloudManagerFault e) {
									throw new TransactionException(e);
								}
							}
							catch (RestfulApplicationException e) {
								throw new TransactionException(e);
							}
						}
					});
				CloudnUtil.addTags(compute, response.id, "Volume", new RTag("ownerInstanceId", instanceInfo.instance.getInstanceId()), new RTag("originalStorageId", backupedStorage.storageId), new RTag("sourceBackupId", backupedStorage.snapshotId));
				createdStorages.add(response.id);
			}

			if (restoreInfo.isAttachingDisk() != null && restoreInfo.isAttachingDisk() && !createdStorages.isEmpty()) {
				SessionService.scheduleWithFixedDelay(new Runnable() {
					@Override
					public void run() {
						ICloudContext context = SessionService.current().get(ICloudContext.class);
						CloudnCompute compute;
						QueryAsyncJobResultResponse response;
						try {
							compute = getCloudnEndpoint();
							response = compute.queryAsyncJobResult(instanceInfo.responce.jobid);
						}
						catch (RestfulApplicationException | CloudManagerFault e) {
							CloudnUtil.notify_Cloudn_Instance_Exception(
								context.getAccessDestionation().getCloudAccountResource().getAccountResourceId(),
								instanceInfo.instance.getInstanceId(),
								Messages.messages().getString("message.validation.cloudn.instance.fail_to_get_status_of_instance_on_tracking", instanceInfo.instance.getInstanceId()),
								e);
							throw new InternalManagerError(e);
						}

						switch (response.jobstatus) {
						case "0":
							break;
						case "1":
							for (String storageId: createdStorages) {
								try {
									compute.attachVolume(storageId, instanceInfo.instance.getInstanceId());
								}
								catch (Exception e) {
									CloudnUtil.notify_Cloudn_Instance_Exception(
										context.getAccessDestionation().getCloudAccountResource().getAccountResourceId(),
										instanceInfo.instance.getInstanceId(),
										Messages.messages().getString("message.validation.cloudn.instance.fail_to_attach_storage_on_restoring", instanceInfo.instance.getInstanceId(), storageId),
										e);
								}
							}
							// 終了のために RuntimeException をスロー。
							throw new RuntimeException();
						case "2":
							String errorMessage = "errorcode=" + response.jobresult.errorcode + ",errortext=" + response.jobresult.errortext;
							CloudnUtil.notify_Cloudn_Instance_ErrorMessage(
								context.getAccessDestionation().getCloudAccountResource().getAccountResourceId(),
								instanceInfo.instance.getInstanceId(),
								Messages.messages().getString("message.validation.cloudn.instance.fail_to_get_status_of_instance_on_tracking", instanceInfo.instance.getInstanceId()),
								errorMessage);
							throw new InternalManagerError(errorMessage);
						default:
							throw new InternalManagerError("got unexpected status number. status=" + response.jobstatus);
						}
					}
				}, 0, 10000, TimeUnit.MILLISECONDS);
			}
			return instanceInfo.instance;
		}
		catch (Exception e) {
			throw new InternalManagerError(e);
		}
	}
	
	@Override
	public Storage restoreStorage(String storageBackupId, String name,String flavor, Integer size, String zone, String storageDetail) throws CloudManagerFault {
		try {
			String value = getStore().get(RT_StorageBackup, storageBackupId);

			ObjectMapper om = new ObjectMapper();
			ObjectReader or = om.reader(BackupedStorageDetail.class);
			BackupedStorageDetail backuped = or.readValue(value);

			backuped.name = name != null ? name: backuped.name;
			backuped.flavor = flavor != null ? flavor: backuped.flavor;
			backuped.zone = zone != null ? zone: backuped.zone;
			backuped.size = size != null ? size: backuped.size;
			
//			ObjectWriter ow = om.writerWithType(StorageDetail.class);
//			String backupedDetail = ow.writeValueAsString(backuped.storageDetail);
			return createStorage(backuped.name, backuped.flavor, backuped.size, storageBackupId, backuped.zone, null);
		}
		catch (Exception e) {
			throw new InternalManagerError(e);
		}
	}

	protected StorageBackup createStorageBackup(com.clustercontrol.cloud.cloudn.rest.Snapshot snapshot){
		try {
			ObjectMapper om = new ObjectMapper();
			ObjectReader or = om.reader(BackupedStorageDetail.class);
			ObjectWriter ow = om.writerWithType(StorageDetail.class);
			
			String backupDetail = getStore().get(RT_StorageBackup, snapshot.id);
			BackupedStorageDetail backuped = or.readValue(backupDetail);

			StorageBackup storageBackup = new StorageBackup();
			storageBackup.setResourceType(RT_StorageBackup);
			storageBackup.setStorageBackupId(snapshot.id);
			storageBackup.setName(backuped.name);
			storageBackup.setStorageId(backuped.storageId);
			storageBackup.setDescription(backuped.description);
			storageBackup.setCreateTime(backuped.createTime);
			storageBackup.setBackupedData(new StorageBackup.BackupedData());
			storageBackup.getBackupedData().setName(backuped.name);
			storageBackup.getBackupedData().setFlavor(backuped.flavor);
			storageBackup.getBackupedData().setZone(backuped.zone);
			storageBackup.getBackupedData().setStorageDetail(ow.writeValueAsString(backuped.storageDetail));
			storageBackup.setActualResource(snapshot);

			return storageBackup;
		}
		catch (Exception e) {
			throw new InternalManagerError(e);
		}
	}
	
	@Override
	public StorageBackup getStorageBackup(String storageBackupId) throws CloudManagerFault {
		try {
			CloudnCompute compute = getCloudnEndpoint();
			ListSnapshotsResponse response = compute.listSnapshots(new ListSnapshotsRequest().withId(storageBackupId));
			com.clustercontrol.cloud.cloudn.rest.Snapshot snapshot = response.snapshots.get(0);
			
			if(response.snapshots.get(0).state.equals("Error")){
				return null;
			}
			
			return createStorageBackup(snapshot);					
		}
		catch (RestfulApplicationException e) {
			throw new CloudManagerFault(e.getMessage(), e.getErrorCode(), e);
		}
		catch (Exception e) {
			throw new InternalManagerError(e);
		}
	}

	@Override
	public List<StorageBackup> getStorageBackups(String... storageBackupId) throws CloudManagerFault {
		return getStorageBackups(Arrays.asList(storageBackupId));
	}

	@Override
	public List<StorageBackup> getStorageBackups(List<String> storageBackupIds) throws CloudManagerFault {
		List<StorageBackup> storageBackups = new ArrayList<StorageBackup>();
		try {
			CloudnCompute compute = getCloudnEndpoint();
			ListSnapshotsResponse response = compute.listSnapshots();
			
			List<String> filterdList = new ArrayList<String>(getStore().getIds(RT_StorageBackup));
			if (!storageBackupIds.isEmpty()) {
				List<String> tempfilterdList = new ArrayList<>();
				for (String storageBackupId1: storageBackupIds) {
					for (String storageBackupId2: filterdList) {
						if (storageBackupId1.equals(storageBackupId2)) {
							tempfilterdList.add(storageBackupId1);
							break;
						}
					}
				}
				filterdList = tempfilterdList.isEmpty() ? filterdList: tempfilterdList;
			}
			
			for (Iterator<String> iter = filterdList.iterator(); iter.hasNext();) {
				String storageBackupId = iter.next();
				
				for(com.clustercontrol.cloud.cloudn.rest.Snapshot snapshot: response.snapshots){
					if (storageBackupId.equals(snapshot.id) && !snapshot.state.equals("Error")) {
						storageBackups.add(createStorageBackup(snapshot));
						iter.remove();
						break;
					}
				}
			}
			
			for (String storageBackupId: filterdList) {
				try {
					getStore().remove(RT_StorageBackup, storageBackupId);
				}
				catch (Exception e) {
					Logger.getLogger(this.getClass()).warn(e.getMessage(), e);
				}
			}
		}
		catch (RestfulApplicationException e) {
			throw new CloudManagerFault(e.getMessage(), e.getErrorCode(), e);
		}
		catch (Exception e) {
			throw new InternalManagerError(e);
		}
		
		return storageBackups;
	}

	@Override
	public List<LoadBalancer> getLoadBalancers() throws CloudManagerFault {
		return Collections.emptyList();
	}

	@Override
	public void registerInstanceToLoadBalancer(String lbId, String instanceId) throws CloudManagerFault {

	}

	@Override
	public void unregisterInstanceToLoadBalancer(String lbId, String instanceId) throws CloudManagerFault {
	}

	@Override
	public InstanceBackup createInstanceBackup(String instanceId, String name, String description, Boolean noReboot, String backupOption) throws CloudManagerFault {
		return createInstanceBackup(instanceId, name, description, noReboot, null, backupOption);
	}

	@Override
	public InstanceBackup createInstanceBackup(String instanceId, String name, String description, Boolean noReboot, List<String> storageIds, String backupOption) throws CloudManagerFault {
		try {
			CloudnCompute compute = getCloudnEndpoint();

			if (noReboot != null && noReboot)
				throw com.clustercontrol.cloud.cloudn.ErrorCode.INSTANCE_BACKUP_NEED_VM_TO_BE_STOPPED.cloudManagerFault(instanceId); 

			VirtualMachine vm;
			{
				ListVirtualMachinesResponse response = compute.listVirtualMachines(new ListVirtualMachinesRequest().withId(instanceId));
				if (!response.virtualMachines.isEmpty()) {
					vm = response.virtualMachines.get(0);
				}
				else {
					throw com.clustercontrol.cloud.cloudn.ErrorCode.VM_NOT_FONUD.cloudManagerFault(instanceId); 
				}
			}
			
			if (!"stopped".equalsIgnoreCase(vm.state))
				throw com.clustercontrol.cloud.cloudn.ErrorCode.INSTANCE_BACKUP_NEED_VM_TO_BE_STOPPED.cloudManagerFault(instanceId); 
			
			String backupId = null;
			List<AttachedStorage> snapshotIds = new ArrayList<>();
			ListVolumesResponse volumelist = compute.listVolumes(new ListVolumesRequest().withVirtualMachineId(vm.id));
			for (Volume v: volumelist.volumes) {
				if(v.type.equals(CloudnConstants.ROOT_VOLUME_TYPE)){
					CreateTemplateResponse response = compute.createTemplate(new CreateTemplateRequest()
							.withName(name)
							.withDisplaytext(description)
							.withOstypeid(vm.guestosid)
							.withVolumeid(v.id));
					final String templateId = response.id;
					SessionService.current().addRollbackAction(
							new SessionService.RolebackAction() {
								@Override
								public void rollback() throws TransactionException {
									try {
										try {
											CloudnCompute compute = getCloudnEndpoint();
											String zonId = null;
											compute.deleteTemplate(templateId, zonId);
										}
										catch (CloudManagerFault e) {
											throw new TransactionException(e);
										}
									}
									catch (RestfulApplicationException e) {
										throw new TransactionException(e);
									}
								}
							});
					backupId = response.id;
				}
				else {
					if (storageIds == null || storageIds.contains(v.id)) {
						CreateSnapshotResponse response = compute.createSnapshot(new CreateSnapshotRequest().withVolumeid(v.id));
						final String snapshotId = response.id;
						SessionService.current().addRollbackAction(
								new SessionService.RolebackAction() {
									@Override
									public void rollback() throws TransactionException {
										try {
											try {
												CloudnCompute compute = getCloudnEndpoint();
												String zonId = null;
												compute.deleteSnapshot(snapshotId, zonId);
											}
											catch (CloudManagerFault e) {
												throw new TransactionException(e);
											}
										}
										catch (RestfulApplicationException e) {
											throw new TransactionException(e);
										}
									}
								});
						snapshotIds.add(new AttachedStorage(v.name, v.id, response.id));
					}
				}
			}
			
			if (backupId == null) throw new InternalManagerError("not create instancebackup. instance=" + instanceId);

			compute.createTags(new CreateTagsRequest().withResourceids(backupId).withResourcetype("Template").withTag(new RTag().withKey("SourceInstanceId").withValue(instanceId)));
			
			ListTemplatesResponse response = compute.listTemplates(new ListTemplateRequest().withId(backupId).withTemplatefilter("self"));
			Template template = null;
			if (response.template.isEmpty()){
//				throw new InternalManagerError("not found template created. template=" + backupId);
				// Flat版の場合　時間が掛かるため、Templateが取れなかった場合でも続行できるように変更
				template = new Template();
				template.id = backupId;
				template.zonename = vm.zonename;
				template.created = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ssZ").format(new Date());
			} else {
				template = response.template.get(0);
			}

			registStoredInstanceBackupData(vm, template, snapshotIds);

			final InstanceBackup instanceBackup = createInternalInstanceBackup(createImageId(template.id, template.zonename));

			instanceBackup.setName(name);
			instanceBackup.setDescription(description);
			
			ListOSTypesResponse ostypesList = compute.listOsTypes(new ListOsTypesRequest().withId(vm.guestosid));
			if (ostypesList.ostypes.isEmpty())
				throw new InternalManagerError("not found ostype of vm created. vm=" + vm.id); 
			instanceBackup.setPlatform(getOSType(ostypesList.ostypes.get(0).description));
			instanceBackup.setActualResource(template);

			return instanceBackup;
		}
		catch (CloudManagerFault e) {
			throw e;
		}
		catch (RestfulApplicationException e) {
			throw new CloudManagerFault(e.getMessageWithoutErrorCode(), e.getErrorCode(), e);
		}
		catch (Exception e) {
			throw new InternalManagerError(e);
		}
	}
	
	protected void registStoredInstanceBackupData(VirtualMachine vm, Template template, List<AttachedStorage> snapshotIds) throws JsonProcessingException, CloudManagerFault{
		String backupId = createImageId(template.id, template.zonename);
		BackupedInstanceDetail backupedData = new BackupedInstanceDetail();
		backupedData.instanceId = vm.id;
		backupedData.name = vm.displayname;
		backupedData.flavor = vm.serviceofferingid;
		backupedData.zone = vm.zonename;
		backupedData.createTime = convertDate(template.created);
		backupedData.attachedStorages = new ArrayList<>(snapshotIds);
		backupedData.tags = new ArrayList<>();
		for (com.clustercontrol.cloud.cloudn.rest.Tag t: vm.tags) {
			backupedData.tags.add(new Tag(t.key, t.value));
		}
		backupedData.instanceDetail = new InstanceDetail();
		backupedData.instanceDetail.networkIds = new ArrayList<>();
		for (Nic nic: vm.nic) {
			backupedData.instanceDetail.networkIds.add(nic.networkid);
		}
		backupedData.instanceDetail.group = vm.group;

		ObjectMapper om = new ObjectMapper();
		ObjectWriter ow = om.writerWithType(BackupedInstanceDetail.class);
		String backupedDetail = ow.writeValueAsString(backupedData);
		getStore().put(RT_InstanceBackup, backupId, backupedDetail);
	}
	
	protected void deleteRelationalSnapshot(String instanceBackupId) throws CloudManagerFault{
		CloudnCompute compute = getCloudnEndpoint();
		String resource = getStore().get(RT_InstanceBackup, instanceBackupId);
		ObjectMapper om = new ObjectMapper();
		ObjectReader or = om.reader(BackupedInstanceDetail.class);
		BackupedInstanceDetail detail;
		try {
			detail = or.readValue(resource);
		} catch (IOException e1) {
			throw new InternalManagerError(e1);
		}
		
		for (AttachedStorage snapshotId: detail.attachedStorages) {
			try {
				compute.deleteSnapshot(snapshotId.snapshotId, null);
			}
			catch (RestfulApplicationException e) {
				// イメージは消しているので、メッセージだけ記録して処理続行。
				Logger logger = Logger.getLogger(this.getClass());
				logger.error(e.getMessage(), e);
			}
		}
	}
	
	protected InstanceBackup createInternalInstanceBackup(String id) throws CloudManagerFault, JsonProcessingException, IOException{
		ObjectMapper om = new ObjectMapper();
		ObjectReader or = om.reader(BackupedInstanceDetail.class);
		ObjectWriter ow = om.writerWithType(InstanceDetail.class);

		String backupDetail = getStore().get(RT_InstanceBackup, id);
		BackupedInstanceDetail backuped = or.readValue(backupDetail);
		
		InstanceBackup instanceBackup = new InstanceBackup();
		instanceBackup.setResourceType(RT_InstanceBackup);
		instanceBackup.setInstanceBackupId(id);
		instanceBackup.setInstanceId(backuped.instanceId);
		instanceBackup.setCreateTime(backuped.createTime);
		instanceBackup.setBackupedData(new InstanceBackup.BackupedData());
		instanceBackup.getBackupedData().setName(backuped.name);
		instanceBackup.getBackupedData().setFlavor(backuped.flavor);
		instanceBackup.getBackupedData().setZone(backuped.zone);
		instanceBackup.getBackupedData().setTags(backuped.tags);
		instanceBackup.getBackupedData().setInstanceDetail(ow.writeValueAsString(backuped.instanceDetail));
		
		return instanceBackup;
	}
	
	@Override
	public InstanceBackup getInstanceBackup(String instanceBackupId) throws CloudManagerFault {
		List<InstanceBackup> l = getInstanceBackups(instanceBackupId);
		return l.isEmpty() ? null: l.get(0);
	}
	@Override
	public List<InstanceBackup> getInstanceBackups(String... instanceBackupIds) throws CloudManagerFault {
		return getInstanceBackups(Arrays.asList(instanceBackupIds));
	}
	@Override
	public List<InstanceBackup> getInstanceBackups(List<String> instanceBackupIds) throws CloudManagerFault {
		try {
			List<Filter> filters = Collections.emptyList();
			if (instanceBackupIds.isEmpty())
				filters = Arrays.asList(new Filter("category", "self"));
			else
				filters = Arrays.asList(new Filter("category", "self"), new Filter("image-id", instanceBackupIds));
			
			List<ImageBucket> response = internalGetImagesWithFilter(filters);
			
			List<String> filterdList = new ArrayList<String>(getStore().getIds(RT_InstanceBackup));
			if (!instanceBackupIds.isEmpty()) {
				List<String> tempfilterdList = new ArrayList<>();
				for (String instanceBackupId1: instanceBackupIds) {
					for (String instanceBackupId2: filterdList) {
						if (instanceBackupId1.equals(instanceBackupId2)) {
							tempfilterdList.add(instanceBackupId1);
							break;
						}
					}
				}
				filterdList = tempfilterdList.isEmpty() ? filterdList: tempfilterdList;
			}
			
			List<InstanceBackup> instanceBackups = new ArrayList<>();
			Iterator<String> iter = filterdList.iterator();
			while (iter.hasNext()) {
				String instanceBackupId = iter.next();
				for (ImageBucket ib: response) {
					if (instanceBackupId.equals(ib.cloudImgae.getImageId())) {
						InstanceBackup instanceBackup = createInternalInstanceBackup(instanceBackupId);

						instanceBackup.setName(ib.cloudImgae.getName());
						instanceBackup.setDescription(ib.cloudImgae.getDescription());
						instanceBackup.setPlatform(getOSType(ib.cloudnImgae.ostypename));
						instanceBackup.setActualResource(ib);
						
						instanceBackups.add(instanceBackup);
						
						iter.remove();
						break;
					}
				}
			}
			
			for (String instanceBackupId: filterdList) {
				try {
					if(isInstanceBackupCreating(instanceBackupId)){
						InstanceBackup instanceBackup = createInternalInstanceBackup(instanceBackupId);
						instanceBackup.setName("");	//イメージ名　取得不可　TODO
						instanceBackup.setDescription("");	//イメージ　説明（表示名）　取得不可　TODO
						instanceBackup.setPlatform(null);	//プラットフォーム情報　取得不可　TODO
						instanceBackup.setActualResource(null);	//イメージ情報　取得不可　TODO
						instanceBackups.add(instanceBackup);
					} else {
						getStore().remove(RT_InstanceBackup, instanceBackupId);
					}
				}
				catch (Exception e) {
					Logger.getLogger(this.getClass()).warn(e.getMessage(), e);
				}
			}
			
			return instanceBackups;
		}
		catch (CloudManagerFault e) {
			throw e;
		}
		catch (RestfulApplicationException e) {
			throw new CloudManagerFault(e.getMessageWithoutErrorCode(), e.getErrorCode(), e);
		}
		catch (Exception e) {
			throw new InternalManagerError(e);
		}
	}
	
	protected boolean isInstanceBackupCreating(String instanceBackupId) throws CloudManagerFault, JsonProcessingException, IOException{
		ObjectMapper om = new ObjectMapper();
		ObjectReader or = om.reader(BackupedInstanceDetail.class);
		BackupedInstanceDetail detail = or.readValue(getStore().get(RT_InstanceBackup, instanceBackupId));
		
		return new Date().getTime() - detail.createTime.getTime() < Long.parseLong(CloudnOptionPropertyConstants.cloudn_flat_until_template_created.value());
	}
	
	@Override
	public void deleteInstanceBackup(String instanceBackupId) throws CloudManagerFault {
		CloudnCompute compute = getCloudnEndpoint();
		
		ImageBucket t = null;
		try {
			List<ImageBucket> response = internalGetImagesWithFilter(new Filter("image-id", instanceBackupId), new Filter("category", "self"));
			if (!response.isEmpty()) {
				compute.deleteTemplate((t = response.get(0)).cloudnImgae.id, null);
			}
		}
		catch (RestfulApplicationException e) {
			// イメージは消しているので、メッセージだけ記録して処理続行。
			Logger logger = Logger.getLogger(this.getClass());
			logger.error(e.getMessage(), e);
		}
		
		// 紐付いているスナップショットも削除。
		if (t != null) {
			deleteRelationalSnapshot(instanceBackupId);
		}
		
		try {
			getStore().remove(RT_InstanceBackup, instanceBackupId);
		}
		catch (Exception e) {
			Logger logger = Logger.getLogger(this.getClass());
			logger.error(e.getMessage(), e);
		}
	}

	@Override
	protected String getComputeId() {
		return EP_TYPE_COMPUTE_VPC;
	}
	
	@Override
	protected String getServiceId() {
		return SERVICE_VPC_ID;
	}

	@Override
	protected String getDefaultRegion() {
		return DEFAULT_REGION;
	}
	
	//Cloudnオプションにおける取捨選択
	protected Boolean isCorrectedInstance(VirtualMachine virtualMachine){
		return !(instanceStateKind(virtualMachine.state).equals(InstanceStateKind.stopping) && !isRegisteredInstance(virtualMachine.id));
	}

	protected Boolean isCorrectedInstance(String instanceId) throws RestfulApplicationException, CloudManagerFault{
		List<VirtualMachine> virtualMachines = getCloudnEndpoint().listVirtualMachines(new ListVirtualMachinesRequest().withId(instanceId)).virtualMachines;
		if(virtualMachines.size() < 1){
			return false;
		}
		return isCorrectedInstance(virtualMachines.get(0));
	}
	
	//Cloudnオプションにおける取捨選択
	protected Boolean isCorrectedStorage(Volume volume) throws RestfulApplicationException, CloudManagerFault{
		return (volume.virtualmachineid == null	|| !volume.type.equals(CloudnConstants.ROOT_VOLUME_TYPE) ||
					(
						!(
							volume.vmstate != null
							&& volume.vmstate.equals("Destroyed")
						)
						&& isCorrectedInstance(volume.virtualmachineid)
					) 
				)
				&& !volume.state.equalsIgnoreCase("expunged") 
				&& !volume.state.equalsIgnoreCase("expunging");
	}
	
	protected Boolean isRegisteredInstance(String instanceId){
		EntityManagerEx em = SessionService.current().getEntityManagerEx();
		ICloudContext context = SessionService.current().get(ICloudContext.class);
		CloudInstanceDao instance =em.find(CloudInstanceDao.class, new CloudInstanceDao.CloudInstancePK(instanceId, context.getCurrentRegion().getRegion(), context.getAccessDestionation().getCloudAccountResource().getAccountResourceId()));
		return instance != null;
	}
	
	protected String getAccountResourceId(){
		return SessionService.current().get(ICloudContext.class).getAccessDestionation().getCloudAccountResource().getAccountResourceId();
	}
}